/*
	Description: stdout viewer

	Author: Marco Costalba (C) 2006

	Copyright: See COPYING file that comes with this distribution

*/
#include <qlabel.h>
#include <qtextedit.h>
#include <qpushbutton.h>
#include <qstatusbar.h>
#include <qmessagebox.h>
#include <qapplication.h>
#include <qcursor.h>
#include "myprocess.h"
#include "git.h"
#include "consoleimpl.h"

ConsoleImpl::ConsoleImpl(const QString& nm, Git* g) :
             Console(0, 0, Qt::WDestructiveClose), name(nm), git(g) {

	QFont f = textLabelCmd->font();
	f.setBold(true);
	textLabelCmd->setFont(f);
	setCaption("\'" + name + "\' output window - QGit");
}

void ConsoleImpl::pushButtonOk_clicked() {

	close();
}

void ConsoleImpl::closeEvent(QCloseEvent* ce) {

	if (proc && proc->isRunning())
		if (QMessageBox::question(this, "Action output window - QGit",
		    "Action is still running.\nAre you sure you want to close "
		    "the window and leave the action running in background?",
		    "&Yes", "&No", QString::null, 1, 1) == 1) {
			ce->ignore();
			return;
		}
	if (QApplication::overrideCursor())
		QApplication::restoreOverrideCursor();

	Console::closeEvent(ce);
}

bool ConsoleImpl::start(const QString& cmd, const QString& cmdArgs) {

	statusBar()->message("Executing \'" + name + "\' action...");

	QString t(cmd.section('\n', 1, 0xffffffff, QString::SectionIncludeLeadingSep));

	// in case of a multi-sequence, line arguments are bounded to first command only
	QString txt = cmd.section('\n', 0, 0).append(cmdArgs).append(t);
	textLabelCmd->setText(txt);

	if (t.stripWhiteSpace().isEmpty())
		// any one-line command followed by a newline would fail
		proc = git->runAsync(cmd.stripWhiteSpace(), this);
	else
		proc = git->runAsScript(cmd, this); // wrap in a script

	if (proc.isNull())
		deleteLater();
	else
		QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

	return !proc.isNull();
}

void ConsoleImpl::on_procDataReady(const QString& data) {

	textEditOutput->append(data);
}

void ConsoleImpl::on_eof() {

	QApplication::restoreOverrideCursor();
	statusBar()->message("End of \'" + name + "\' execution.");
	pushButtonStop->setEnabled(false);
	emit customAction_exited(name);
}

void ConsoleImpl::pushButtonStop_clicked() {

	git->cancelProcess(proc);
	on_eof();
}
