/*
	Description: start-up repository opening and reading

	Author: Marco Costalba (C) 2005-2006

	Copyright: See COPYING file that comes with this distribution

*/
#include <unistd.h> // usleep()
#include <qapplication.h>
#include <qsettings.h>
#include <qeventloop.h>
#include <qregexp.h>
#include <qtextcodec.h>
#include "exceptionmanager.h"
#include "rangeselectimpl.h"
#include "lanes.h"
#include "myprocess.h"
#include "cache.h"
#include "annotate.h"
#include "mainimpl.h"
#include "dataloader.h"
#include "git.h"

#define POST_MSG(x) QApplication::postEvent(parent(), new MessageEvent(x))

using namespace QGit;

const QStringList Git::getArgs(bool askForRange, bool* quit) {

	static bool startup = true; // it's OK to be unique among qgit windows
	if (startup) {
		curRange = "";
		for (int i = 1; i < qApp->argc(); i++) {
			// in arguments with spaces double quotes
			// are stripped by Qt, so re-add them
			QString arg(qApp->argv()[i]);
			if (arg.contains(' '))
				arg.prepend('\"').append('\"');

			curRange.append(arg + ' ');
		}
	}
	if (    askForRange
	    &&  testFlag(RANGE_SELECT_F)
	    && (!startup || curRange.isEmpty())) {

		RangeSelectImpl rs((QWidget*)parent(), &curRange,
		                    getAllRefNames(TAG, !optOnlyLoaded), quit, this);
		rs.exec(); // modal execution
		if (*quit)
			return QStringList();
	}
	startup = false;
	return MyProcess::splitArgList(curRange);
}

const QString Git::getBaseDir(bool* changed, SCRef wd, bool* ok, QString* gd) {
// we could run from a subdirectory, so we need to get correct directories

	QString runOutput, tmp(workDir);
	workDir = wd;
	errorReportingEnabled = false;
	bool ret = run("git rev-parse --git-dir", &runOutput); // run under newWorkDir
	errorReportingEnabled = true;
	workDir = tmp;
	runOutput = runOutput.stripWhiteSpace();
	if (!ret || runOutput.isEmpty()) {
		*changed = true;
		if (ok)
			*ok = false;
		return wd;
	}
	// 'git rev-parse --git-dir' output could be a relative
	// to working dir (as ex .git) or an absolute path
	QDir d(runOutput.startsWith("/") ? runOutput : wd + "/" + runOutput);
	*changed = (d.absPath() != gitDir);
	if (gd)
		*gd = d.absPath();
	if (ok)
		*ok = true;
	d.cdUp();
	return d.absPath();
}

Git::Reference* Git::lookupReference(SCRef sha, bool create) {

	RefMap::iterator it(refsShaMap.find(sha));
	if (it == refsShaMap.end() && create)
		it = refsShaMap.insert(sha, Reference());

	return (it != refsShaMap.end() ? &(*it) : NULL);
}

bool Git::getRefs() {

	// check for a StGIT stack
	QDir d(gitDir);
	QString stgCurBranch;
	if (d.exists("patches")) { // early skip
		errorReportingEnabled = false;
		isStGIT = run("stg branch", &stgCurBranch); // slow command
		errorReportingEnabled = true;
		stgCurBranch = stgCurBranch.stripWhiteSpace();
	} else
		isStGIT = false;

	// check for a merge and read current branch sha
	isMergeHead = d.exists("MERGE_HEAD");
	QString curBranchSHA, curBranchName;
	if (!run("git rev-parse HEAD", &curBranchSHA))
		return false;

	if (!run("git branch", &curBranchName))
		return false;

	curBranchSHA = curBranchSHA.stripWhiteSpace();
	curBranchName = curBranchName.prepend('\n').section("\n*", 1);
	curBranchName = curBranchName.section('\n', 0, 0).stripWhiteSpace();

	// read refs, normally unsorted
	QString runOutput;
	if (!run("git show-ref -d", &runOutput))
		return false;

	refsShaMap.clear();
	QString prevRefSha;
	QStringList patchNames, patchShas;
	const QStringList rLst(QStringList::split('\n', runOutput));
	FOREACH_SL (it, rLst) {

		SCRef revSha = (*it).left(40);
		SCRef refName = (*it).mid(41);

		if (refName.startsWith("refs/patches/")) {

			// save StGIT patch sha, to be used later
			SCRef patchesDir("refs/patches/" + stgCurBranch + "/");
			if (refName.startsWith(patchesDir)) {
				patchNames.append(refName.mid(patchesDir.length()));
				patchShas.append(revSha);
			}
			// StGIT patches should not be added to refs,
			// but an applied StGIT patch could be also an head or
			// a tag in this case will be added in another loop cycle
			continue;
		}
		// one rev could have many tags
		Reference* cur = lookupReference(revSha, optCreate);

		if (refName.startsWith("refs/tags/")) {

			if (refName.endsWith("^{}")) { // tag dereference

				// we assume that a tag dereference follows strictly
				// the corresponding tag object in rLst. So the
				// last added tag is a tag object, not a commit object
				cur->tags.append(refName.mid(10, refName.length() - 13));

				// store tag object. Will be used to fetching
				// tag message (if any) when necessary.
				cur->tagObj = prevRefSha;

				// tagObj must be removed from ref map
				refsShaMap.remove(prevRefSha);

			} else
				cur->tags.append(refName.mid(10));

			cur->type |= TAG;

		} else if (refName.startsWith("refs/heads/")) {

			cur->branches.append(refName.mid(11));
			cur->type |= BRANCH;
			if (curBranchSHA == revSha) {
				cur->type |= CUR_BRANCH;
				cur->currentBranch = curBranchName;
			}
		} else if (refName.startsWith("refs/remotes/") && !refName.endsWith("HEAD")) {

			cur->remoteBranches.append(refName.mid(13));
			cur->type |= RMT_BRANCH;

		} else if (!refName.startsWith("refs/bases/") && !refName.endsWith("HEAD")) {

			cur->refs.append(refName);
			cur->type |= REF;
		}
		prevRefSha = revSha;
	}
	if (isStGIT && !patchNames.isEmpty())
		parseStGitPatches(patchNames, patchShas);

	return !refsShaMap.empty();
}

void Git::parseStGitPatches(SCList patchNames, SCList patchShas) {

	patchesStillToFind = 0;

	// get patch names and status of current branch
	QString runOutput;
	if (!run("stg series", &runOutput))
		return;

	const QStringList pl(QStringList::split('\n', runOutput));
	FOREACH_SL (it, pl) {

		SCRef status = (*it).left(1);
		SCRef patchName = (*it).mid(2);

		bool applied = (status == "+" || status == ">");
		int pos = patchNames.findIndex(patchName);
		if (pos == -1) {
			dbp("ASSERT in Git::parseStGitPatches(), patch %1 "
			    "not found in references list.", patchName);
			continue;
		}
		Reference* cur = lookupReference(patchShas[pos], optCreate);
		cur->stgitPatch = patchName;
		cur->type |= (applied ? APPLIED : UN_APPLIED);

		if (applied)
			patchesStillToFind++;
	}
}

const QStringList Git::getOthersFiles() {
// add files present in working directory but not in git archive

	QString runCmd("git ls-files --others");
	QSettings settings;
	QString exFile(settings.readEntry(APP_KEY + EX_KEY, EX_DEF));
	if (!exFile.isEmpty()) {
		QString path = (exFile.startsWith("/")) ? exFile : workDir + "/" + exFile;
		if (QFile::exists(path))
			runCmd.append(" --exclude-from=" + quote(exFile));
	}
	QString exPerDir(settings.readEntry(APP_KEY + EX_PER_DIR_KEY, EX_PER_DIR_DEF));
	if (!exPerDir.isEmpty())
		runCmd.append(" --exclude-per-directory=" + quote(exPerDir));

	QString runOutput;
	run(runCmd, &runOutput);
	return QStringList::split('\n', runOutput);
}

const Rev* Git::fakeWorkDirRev(SCRef parent, SCRef log, SCRef longLog, int idx, FileHistory* fh) {

	QString date(QString::number(QDateTime::currentDateTime().toTime_t()) + " +0200");
	QString data(ZERO_SHA + ' ' + parent + "\ntree ");
	data.append(ZERO_SHA);
	data.append("\nparent " + parent);
	data.append("\nauthor Working Dir " + date);
	data.append("\ncommitter Working Dir " + date);
	data.append("\n\n    " + log + '\n');
	data.append(longLog);

	QByteArray* ba = new QByteArray(data.length() + 1);
	memcpy(ba->data(), data.ascii(), data.length());
	ba->at(data.length()) = '\0';
	fh->rowData.append(ba);
	int dummy;
	Rev* c = new Rev(*ba, 0, idx, &dummy);
	c->isDiffCache = true;
	c->lanes.append(EMPTY);
	return c;
}

const RevFile* Git::fakeWorkDirRevFile(const WorkingDirInfo& wd) {

	RevFile* rf = new RevFile();
	parseDiffFormat(*rf, wd.diffIndex);
	rf->onlyModified = false;

	FOREACH_SL (it, wd.otherFiles) {

		appendFileName(*rf, *it);
		rf->status.append(RevFile::UNKNOWN);
		rf->mergeParent.append(1);
	}
	RevFile cachedFiles;
	parseDiffFormat(cachedFiles, wd.diffIndexCached);
	for (uint i = 0; i < rf->status.count(); i++)
		if (findFileIndex(cachedFiles, filePath(*rf, i)) != -1)
			rf->status[i] |= RevFile::IN_INDEX;
	return rf;
}

void Git::getDiffIndex() {

	QString status;
	if (!run("git status", &status)) // git status refreshes the index, run as first
		return;

	if (!run("git diff-index --no-color HEAD", &_wd.diffIndex))
		return;

	// check for files already updated in cache, we will
	// save this information in status third field
	if (!run("git diff-index --no-color --cached HEAD", &_wd.diffIndexCached))
		return;

	// get any file not in tree
	_wd.otherFiles = getOthersFiles();

	// now mockup a RevFile
	revsFiles.insert(ZERO_SHA, fakeWorkDirRevFile(_wd));

	// then mockup the corresponding Rev
	QString parent;
	if (!run("git rev-parse HEAD", &parent))
		return;

	parent = parent.section('\n', 0, 0);
	SCRef log = (isNothingToCommit() ? "Nothing to commit" : "Working dir changes");
	const Rev* r = fakeWorkDirRev(parent, log, status, revData.revOrder.count(), &revData);
	revData.revs.insert(ZERO_SHA, r);
	revData.revOrder.append(ZERO_SHA);

	// finally send it to GUI
	emit newRevsAdded(&revData, revData.revOrder);
}

void Git::parseDiffFormatLine(RevFile& rf, SCRef line, int parNum) {

	if (line[1] == ':') { // it's a combined merge

		/* For combined merges rename/copy information is useless
		 * because nor the original file name, nor similarity info
		 * is given, just the status tracks that in the left/right
		 * branch a renamed/copy occurred (as example status could
		 * be RM or MR). For visualization purposes we could consider
		 * the file as modified
		 */
		appendFileName(rf, line.section('\t', -1));
		setStatus(rf, "M");
		rf.mergeParent.append(parNum);
	} else { // faster parsing in normal case

		if (line[98] == '\t') {
			appendFileName(rf, line.mid(99));
			setStatus(rf, line.at(97));
			rf.mergeParent.append(parNum);
		} else
			// it's a rename or a copy, we are not in fast path now!
			setExtStatus(rf, line.mid(97), parNum);
	}
}

void Git::setStatus(RevFile& rf, SCRef rowSt) {

	char status = rowSt.at(0).latin1();
	switch (status) {
	case 'M':
	case 'T':
		rf.status.append(RevFile::MODIFIED);
		break;
	case 'D':
		rf.status.append(RevFile::DELETED);
		rf.onlyModified = false;
		break;
	case 'A':
		rf.status.append(RevFile::NEW);
		rf.onlyModified = false;
		break;
	case '?':
		rf.status.append(RevFile::UNKNOWN);
		rf.onlyModified = false;
		break;
	default:
		dbp("ASSERT in Git::setStatus, unknown status <%1>. "
		    "'MODIFIED' will be used instead.", rowSt);
		rf.status.append(RevFile::MODIFIED);
		break;
	}
}

void Git::setExtStatus(RevFile& rf, SCRef rowSt, int parNum) {

	const QStringList sl(QStringList::split('\t', rowSt));
	if (sl.count() != 3) {
		dbp("ASSERT in setExtStatus, unexpected status string %1", rowSt);
		return;
	}
	// we want store extra info with format "orig --> dest (Rxx%)"
	// but git give us something like "Rxx\t<orig>\t<dest>"
	SCRef type = sl[0];
	SCRef orig = sl[1];
	SCRef dest = sl[2];
	const QString extStatusInfo(orig + " --> " + dest + " (" + type + "%)");

	/*
	   NOTE: we set rf.extStatus size equal to position of latest
	         copied/renamed file. So it can have size lower then
	         rf.count() if after copied/renamed file there are
	         others. Here we have no possibility to know final
	         dimension of this RefFile. We are still in parsing.
	*/

	// simulate new file
	appendFileName(rf, dest);
	rf.mergeParent.append(parNum);
	rf.status.append(RevFile::NEW);
	rf.extStatus.resize(rf.status.size());
	rf.extStatus[rf.status.size() - 1] = extStatusInfo;

	// simulate deleted orig file only in case of rename
	if (type.at(0) == 'R') { // renamed file
		appendFileName(rf, orig);
		rf.mergeParent.append(parNum);
		rf.status.append(RevFile::DELETED);
		rf.extStatus.resize(rf.status.size());
		rf.extStatus[rf.status.size() - 1] = extStatusInfo;
	}
	rf.onlyModified = false;
}

void Git::parseDiffFormat(RevFile& rf, SCRef buf) {

	int parNum = 1, startPos = 0, endPos = buf.find('\n');
	while (endPos != -1) {

		SCRef line = buf.mid(startPos, endPos - startPos);
		if (line[0] == ':') // avoid sha's in merges output
			parseDiffFormatLine(rf, line, parNum);
		else
			parNum++;

		startPos = endPos + 1;
		endPos = buf.find('\n', endPos + 99);
	}
}

bool Git::startParseProc(SCList initCmd, FileHistory* fh) {

	DataLoader* dl = new DataLoader(this, fh); // auto-deleted when done

	connect(this, SIGNAL(cancelLoading(const FileHistory*)),
	        dl, SLOT(on_cancel(const FileHistory*)));

	connect(dl, SIGNAL(newDataReady(const FileHistory*)),
	        this, SLOT(on_newDataReady(const FileHistory*)));

	connect(dl, SIGNAL(loaded(const FileHistory*, ulong, int,
	        bool, const QString&, const QString&)), this,
	        SLOT(on_loaded(const FileHistory*, ulong, int,
	        bool, const QString&, const QString&)));

	return dl->start(initCmd, workDir);
}

bool Git::startRevList(SCList args, FileHistory* fh) {

	const QString baseCmd("git rev-list --header --parents --boundary --default HEAD");
	QStringList initCmd(QStringList::split(' ', baseCmd));
	if (!isMainHistory(fh)) {
		// fetch history from all branches so any revision in
		// main view that changes the file is always found
		// NOTE: we don't use '--remove-empty' option because
		// in case a file is deleted and then a new file with
		// the same name is created again in the same directory
		// then, with this option, file history is truncated to
		// the file deletion revision.
		initCmd += getAllRefSha(BRANCH | RMT_BRANCH);
		initCmd += "--";
	} else
		initCmd += "--topo-order";

	return startParseProc(initCmd + args, fh);
}

bool Git::startUnappliedList() {

	// TODO truncate sha to avoid overflow in 'git rev-list'
	// command line arguments
	const QStringList unAppliedSha(getAllRefSha(UN_APPLIED));
	if (unAppliedSha.isEmpty())
		return false;

	// WARNING: with this command git rev-list could send spurious
	// revs so we need some filter out logic during loading
	QStringList initCmd(QStringList::split(' ', "git rev-list --header --parents"));
	initCmd += unAppliedSha;
	initCmd += QString::fromLatin1("^HEAD");
	return startParseProc(initCmd, &revData);
}

bool Git::stop(bool saveCache) {
// normally called when changing directory or closing

	EM_RAISE(exGitStopped);

	// stop all data sending from process and asks them
	// to terminate. Note that process could still keep
	// running for a while although silently
	emit cancelAllProcesses(); // non blocking

	if (cacheNeedsUpdate && saveCache) {

		cacheNeedsUpdate = false;
		if (!filesLoadingCurSha.isEmpty()) // we are in the middle of a loading
			revsFiles.remove(filesLoadingCurSha); // remove partial data

		if (!revsFiles.isEmpty()) {
			POST_MSG("Saving cache. Please wait...");
			EM_PROCESS_EVENTS_NO_INPUT; // to paint the message
			if (!Cache::save(gitDir, revsFiles, dirNamesVec, fileNamesVec))
				dbs("ERROR unable to save file names cache");
		}
	}
	return allProcessDeleted();
}

void Git::clearRevs() {

	revData.clear("");
	patchesStillToFind = 0; // TODO TEST WITH FILTERING
	firstNonStGitPatch = "";
	_wd.clear();
	revsFiles.remove(ZERO_SHA);
}

void Git::clearFileNames() {

	revsFiles.clear();
	fileNamesMap.clear();
	dirNamesMap.clear();
	dirNamesVec.clear();
	fileNamesVec.clear();
	cacheNeedsUpdate = false;
}

bool Git::init(SCRef wd, bool askForRange, QStringList* filterList, bool* quit) {
// normally called when changing git directory. Must be called after stop()

	*quit = false;
	bool filteredLoading = (filterList != NULL);
	clearRevs();
	try {
		setThrowOnStop(true);

		// check if repository is valid
		bool repoChanged;
		workDir = getBaseDir(&repoChanged, wd, &isGIT, &gitDir);

		if (repoChanged) {
			clearFileNames();
			fileCacheAccessed = false;
		}
		if (!isGIT) {
			setThrowOnStop(false);
			return false;
		}
		const QString msg1("Path is '" + workDir + "'    Loading ");
		QStringList args;
		if (!filteredLoading) {

			// update text codec according to repo settings
			bool dummy;
			QTextCodec::setCodecForCStrings(getTextCodec(&dummy));

			// load references
			POST_MSG(msg1 + "refs...");
			if (!getRefs())
				dbs("WARNING: no tags or heads found");

			// startup input range dialog
			POST_MSG("");
			args = getArgs(askForRange, quit); // must be called with refs loaded
			if (*quit) {
				setThrowOnStop(false);
				return false;
			}
			// load StGit unapplied patches, must be after getRefs()
			if (isStGIT) {
				loadingUnAppliedPatches = startUnappliedList();
				if (loadingUnAppliedPatches) {

					POST_MSG(msg1 + "StGIT unapplied patches...");
					while (loadingUnAppliedPatches) {
						// WARNING we are in setRepository() now
						usleep(20000);
						EM_PROCESS_EVENTS;
					}
					revData.lns->clear(); // again to reset lanes
				}
			}
			// load working dir files
			if (testFlag(DIFF_INDEX_F)) {
				POST_MSG(msg1 + "working directory changed files...");
				getDiffIndex(); // blocking, we are in setRepository() now
			}

		} else { // filteredLoading
			args += getAllRefSha(BRANCH | RMT_BRANCH);
			args += "--";
			args += *filterList;
		}
		POST_MSG(msg1 + "revisions...");
		if (!startRevList(args, &revData))
			dbs("ERROR: unable to start git-rev-list loading");

		setThrowOnStop(false);
		return true;

	} catch (int i) {

		setThrowOnStop(false);

		if (isThrowOnStopRaised(i, "initializing")) {
			EM_THROW_PENDING;
			return false;
		}
		const QString info("Exception \'" + EM_DESC(i) + "\' "
		                   "not handled in init...re-throw");
		dbs(info);
		throw;
	}
}

void Git::on_newDataReady(const FileHistory* fh) {

	emit newRevsAdded(fh , fh->revOrder);
}

void Git::on_loaded(const FileHistory* fh, ulong byteSize, int loadTime,
                    bool normalExit, SCRef cmd, SCRef errorDesc) {

	if (!errorDesc.isEmpty()) {
		MainExecErrorEvent* e = new MainExecErrorEvent(cmd, errorDesc);
		QApplication::postEvent(parent(), e);
	}
	if (normalExit) { // do not send anything if killed

		on_newDataReady(fh);

		if (!loadingUnAppliedPatches) {

			uint kb = byteSize / 1024;
			float mbs = (float)byteSize / loadTime / 1000;
			QString tmp;
			tmp.sprintf("Loaded %i revisions  (%i KB),   "
			            "time elapsed: %i ms  (%.2f MB/s)",
			            fh->revs.count(), kb, loadTime, mbs);

			emit loadCompleted(fh, tmp);

			if (isMainHistory(fh))
				// check for revisions modified files out of fast path
				// let the dust to settle down, so that the first
				// revision is shown to user without noticeable delay
				QTimer::singleShot(500, this, SLOT(loadFileNames()));
		}
	}
	if (loadingUnAppliedPatches)
		loadingUnAppliedPatches = false;
}

void Git::populateFileNamesMap() {

	for (uint i = 0; i < dirNamesVec.count(); ++i)
		dirNamesMap.insert(dirNamesVec[i], i);

	for (uint i = 0; i < fileNamesVec.count(); ++i)
		fileNamesMap.insert(fileNamesVec[i], i);
}

void Git::loadFileNames() {
// warning this function is not re-entrant, background
// activity after repository has been loaded

	if (!fileCacheAccessed) { // deferred file names cache load

		fileCacheAccessed = true;
		bool isWorkingDirRevFile = (getFiles(ZERO_SHA) != NULL);
		clearFileNames(); // any already created RevFile will be lost

		if (Cache::load(gitDir, revsFiles, dirNamesVec, fileNamesVec))
			populateFileNamesMap();
		else
			dbs("ERROR: unable to load file names cache");

		if (isWorkingDirRevFile) // re-add ZERO_SHA with new file names indices
			revsFiles.insert(ZERO_SHA, fakeWorkDirRevFile(_wd));
	}

	QString diffTreeBuf;
	FOREACH (StrVect, it, revData.revOrder) {
		if (!revsFiles.find(*it)) {
			const Rev* c = revLookup(*it);
			if (c->parentsCount() == 1) // skip initials and merges
				diffTreeBuf.append(*it).append('\n');
		}
	}
	if (!diffTreeBuf.isEmpty()) {
		filesLoadingPending = filesLoadingCurSha = "";
		const QString runCmd("git diff-tree --no-color -r -C --stdin");
		runAsync(runCmd, this, diffTreeBuf);
	}
	indexTree();
}

int Git::addChunk(FileHistory* fh, const QByteArray& ba, int start) {

	RevMap& r = fh->revs;
	int nextStart;
	Rev* rev = new Rev(ba, start, r.count(), &nextStart); // only here we create a new rev

	if (nextStart == -1) { // half chunk detected
		delete rev;
		return -1;
	}

	SCRef sha = rev->sha();

	if (isStGIT) {
		if (loadingUnAppliedPatches) { // filter out possible spurious revs
			Reference* rf = lookupReference(sha);
			if (!(rf && (rf->type & UN_APPLIED))) {
				delete rev;
				return nextStart;
			}
		}
		// remove StGIT spurious revs filter
		if (!firstNonStGitPatch.isEmpty() && firstNonStGitPatch == sha)
			firstNonStGitPatch = "";

		// StGIT called with --all option creates spurious revs so filter
		// out unknown revs until no more StGIT patches are waited and
		// firstNonStGitPatch is reached
		if (!(firstNonStGitPatch.isEmpty() && patchesStillToFind == 0) &&
		    !loadingUnAppliedPatches && isMainHistory(fh)) {

			Reference* rf = lookupReference(sha);
			if (!(rf && (rf->type & APPLIED))) {
				delete rev;
				return nextStart;
			}
		}
		if (r.find(sha)) {
			// StGIT unapplied patches could be sent again by
			// git rev-list as example if called with --all option.
			if (r[sha]->isUnApplied) {
				delete rev;
				return nextStart;
			}
			dbp("ASSERT: addChunk sha <%1> already received", sha);
		}
	}
	if (r.isEmpty() && !isMainHistory(fh)) {
		bool added = copyDiffIndex(fh, sha);
		rev->orderIdx = added ? 1 : 0;
	}
	r.insert(sha, rev);
	fh->revOrder.append(sha);

	if (isStGIT) {
		// updateLanes() is called too late, after loadingUnAppliedPatches
		// has been reset so update the lanes now.
		if (loadingUnAppliedPatches) {

			Rev* c = const_cast<Rev*>(revLookup(sha, fh));
			c->isUnApplied = true;
			c->lanes.append(UNAPPLIED);

		} else if (patchesStillToFind > 0 || !isMainHistory(fh)) { // try to avoid costly lookup

			Reference* rf = lookupReference(sha);
			if (rf && (rf->type & APPLIED)) {

				Rev* c = const_cast<Rev*>(revLookup(sha, fh));
				c->isApplied = true;
				if (isMainHistory(fh)) {
					patchesStillToFind--;
					if (patchesStillToFind == 0)
						// any rev will be discarded until
						// firstNonStGitPatch arrives
						firstNonStGitPatch = c->parent(0);
				}
			}
		}
	}
	return nextStart;
}

bool Git::copyDiffIndex(FileHistory* fh, SCRef parent) {
// must be called with empty revs and empty revOrder

	if (!fh->revOrder.isEmpty() || !fh->revs.isEmpty()) {
		dbs("ASSERT in copyDiffIndex: called with wrong context");
		return false;
	}
	const Rev* r = revLookup(ZERO_SHA);
	if (!r)
		return false;

	const RevFile* files = getFiles(ZERO_SHA);
	if (!files || findFileIndex(*files, fh->fileName) == -1)
		return false;

	// insert a custom ZERO_SHA rev with proper parent
	const Rev* rf = fakeWorkDirRev(parent, "Working dir changes", "dummy", 0, fh);
	fh->revs.insert(ZERO_SHA, rf);
	fh->revOrder.append(ZERO_SHA);
	return true;
}

void Git::setLane(SCRef sha, FileHistory* fh) {

	Lanes* l = fh->lns;
	uint i = fh->firstFreeLane;
	const QValueVector<QString>& shaVec(fh->revOrder);
	for (uint cnt = shaVec.count(); i < cnt; ++i) {
		SCRef curSha = shaVec[i];
		Rev* r = const_cast<Rev*>(revLookup(curSha, fh));
		if (r->lanes.count() == 0)
			updateLanes(*r, *l, curSha);

		if (curSha == sha)
			break;
	}
	fh->firstFreeLane = ++i;
}

void Git::updateLanes(Rev& c, Lanes& lns, SCRef sha) {
// we could get third argument from c.sha(), but we are in fast path here
// and c.sha() involves a deep copy, so we accept a little redundancy

	if (lns.isEmpty())
		lns.init(sha);

	bool isDiscontinuity;
	bool isFork = lns.isFork(sha, isDiscontinuity);
	bool isMerge = (c.parentsCount() > 1);
	bool isInitial = (c.parentsCount() == 0);

	if (isDiscontinuity)
		lns.changeActiveLane(sha); // uses previous isBoundary state

	lns.setBoundary(c.isBoundary()); // update must be here

	if (isFork)
		lns.setFork(sha);
	if (isMerge)
		lns.setMerge(c.parents());
	if (c.isApplied)
		lns.setApplied();
	if (isInitial)
		lns.setInitial();

	lns.getLanes(c.lanes); // here lanes are snapshotted

	SCRef nextSha = (isInitial) ? "" : c.parent(0);
	lns.nextParent(nextSha);

	if (c.isApplied)
		lns.afterApplied();
	if (isMerge)
		lns.afterMerge();
	if (isFork)
		lns.afterFork();
	if (lns.isBranch())
		lns.afterBranch();

//	QString tmp = "", tmp2;
//	for (uint i = 0; i < c.lanes.count(); i++) {
//		tmp2.setNum(c.lanes[i]);
//		tmp.append(tmp2 + "-");
//	}
//	qDebug("%s %s",tmp.latin1(), c.sha.latin1());
}

void Git::on_procDataReady(const QByteArray& fileChunk) {

	if (filesLoadingPending.isEmpty())
		filesLoadingPending = QString(fileChunk);
	else
		filesLoadingPending.append(fileChunk); // add to previous half lines

	RevFile* rf = revsFiles[filesLoadingCurSha];
	int nextEOL = filesLoadingPending.find('\n');
	int lastEOL = -1;
	while (nextEOL != -1) {

		SCRef line(filesLoadingPending.mid(lastEOL + 1, nextEOL - lastEOL - 1));
		if (line.constref(0) != ':') {
			SCRef sha = line.left(40);
			if (!rf || sha != filesLoadingCurSha) { // new commit
				rf = new RevFile();
				revsFiles.insert(sha, rf);
				filesLoadingCurSha = sha;
				cacheNeedsUpdate = true;
			} else
				dbp("ASSERT: repeated sha %1 in file names loading", sha);
		} else // line.constref(0) == ':'
			parseDiffFormatLine(*rf, line, 1);

		lastEOL = nextEOL;
		nextEOL = filesLoadingPending.find('\n', lastEOL + 1);
	}
	if (lastEOL != -1)
		filesLoadingPending.remove(0, lastEOL + 1);
}

void Git::appendFileName(RevFile& rf, SCRef name) {

	int idx = name.findRev('/') + 1;
	SCRef dr = name.left(idx);
	SCRef nm = name.mid(idx);

	QMap<QString, int>::const_iterator it(dirNamesMap.find(dr));
	if (it == dirNamesMap.constEnd()) {
		int idx = dirNamesVec.count();
		dirNamesMap.insert(dr, idx);
		dirNamesVec.append(dr);
		rf.dirs.append(idx);
	} else
		rf.dirs.append(*it);

	it = fileNamesMap.find(nm);
	if (it == fileNamesMap.constEnd()) {
		int idx = fileNamesVec.count();
		fileNamesMap.insert(nm, idx);
		fileNamesVec.append(nm);
		rf.names.append(idx);
	} else
		rf.names.append(*it);
}

static bool vecContains(const QValueVector<int>& vect, int value) {

	for (uint i = 0, end = vect.count(); i < end; i++)
		if (vect[i] == value)
			return true;
	return false;
}

void Git::updateDescMap(const Rev* r,uint idx, QMap<QPair<uint, uint>, bool>& dm,
                        QMap<uint, QValueVector<int> >& dv) {

	QValueVector<int> descVec;
	if (r->descRefsMaster != -1) {

		const Rev* tmp = revLookup(revData.revOrder[r->descRefsMaster]);
		const QValueVector<int>& nr = tmp->descRefs;

		for (uint i = 0; i < nr.count(); i++) {

			if (!dv.contains(nr[i])) {
				dbp("ASSERT descendant for %1 not found", r->sha());
				return;
			}
			const QValueVector<int>& dvv = dv[nr[i]];

			// copy the whole vector instead of each element
			// in the first iteration of the loop below
			descVec = dvv; // quick (shared) copy

			for (uint y = 0; y < dvv.count(); y++) {

				uint v = (uint)dvv[y];
				QPair<uint, uint> key = qMakePair(idx, v);
				QPair<uint, uint> keyN = qMakePair(v, idx);
				dm.insert(key, true);
				dm.insert(keyN, false);

				// we don't want duplicated entry, otherwise 'dvv' grows
				// greatly in repos with many tagged development branches
				if (i > 0 && !vecContains(descVec, v)) // i > 0 is rare, no
					descVec.append(v);             // need to optimize
			}
		}
	}
	descVec.append(idx);
	dv.insert(idx, descVec);
}

void Git::mergeBranches(Rev* p, const Rev* r) {

	int r_descBrnMaster = (checkRef(r->sha(), BRANCH | RMT_BRANCH) ? r->orderIdx : r->descBrnMaster);

	if (p->descBrnMaster == r_descBrnMaster || r_descBrnMaster == -1)
		return;

	// we want all the descendant branches, so just avoid duplicates
	const QValueVector<int>& src1 = revLookup(revData.revOrder[p->descBrnMaster])->descBranches;
	const QValueVector<int>& src2 = revLookup(revData.revOrder[r_descBrnMaster])->descBranches;
	QValueVector<int> dst(src1);
	for (uint i = 0; i < src2.count(); i++)
		if (qFind(src1.constBegin(), src1.constEnd(), src2[i]) == src1.constEnd())
			dst.append(src2[i]);

	p->descBranches = dst;
	p->descBrnMaster = p->orderIdx;
}

void Git::mergeNearTags(bool down, Rev* p, const Rev* r, const QMap<QPair<uint, uint>, bool>& dm) {

	bool isTag = checkRef(r->sha(), TAG);
	int r_descRefsMaster = isTag ? r->orderIdx : r->descRefsMaster;
	int r_ancRefsMaster = isTag ? r->orderIdx : r->ancRefsMaster;

	if (down && (p->descRefsMaster == r_descRefsMaster || r_descRefsMaster == -1))
		return;

	if (!down && (p->ancRefsMaster == r_ancRefsMaster || r_ancRefsMaster == -1))
		return;

	// we want the nearest tag only, so remove any tag
	// that is ancestor of any other tag in p U r
	const StrVect& ro = revData.revOrder;
	SCRef sha1 = down ? ro[p->descRefsMaster] : ro[p->ancRefsMaster];
	SCRef sha2 = down ? ro[r_descRefsMaster] : ro[r_ancRefsMaster];
	const QValueVector<int>& src1 = down ? revLookup(sha1)->descRefs : revLookup(sha1)->ancRefs;
	const QValueVector<int>& src2 = down ? revLookup(sha2)->descRefs : revLookup(sha2)->ancRefs;
	QValueVector<int> dst(src1);

	for (uint s2 = 0; s2 < src2.count(); s2++) {

		bool add = false;
		for (uint s1 = 0; s1 < src1.count(); s1++) {

			if (src2[s2] == src1[s1]) {
				add = false;
				break;
			}
			QPair<uint, uint> key = qMakePair((uint)src2[s2], (uint)src1[s1]);

			if (!dm.contains(key)) { // could be empty if all tags are independent
				add = true; // could be an independent path
				continue;
			}
			add = (down && dm[key]) || (!down && !dm[key]);
			if (add)
				dst[s1] = -1; // mark for removing
			else
				break;
		}
		if (add)
			dst.append(src2[s2]);
	}
	QValueVector<int>& nearRefs = (down) ? p->descRefs : p->ancRefs;
	int& nearRefsMaster = (down) ? p->descRefsMaster : p->ancRefsMaster;

	nearRefs.clear();
	for (uint s2 = 0; s2 < dst.count(); s2++)
		if (dst[s2] != -1)
			nearRefs.append(dst[s2]);

	nearRefsMaster = p->orderIdx;
}

void Git::indexTree() {

	const StrVect& ro = revData.revOrder;
	if (ro.count() == 0)
		return;

	// we keep the pairs(x, y). Value is true if x is
	// ancestor of y or false if y is ancestor of x
	QMap<QPair<uint, uint>, bool> descMap;
	QMap<uint, QValueVector<int> > descVect;

	// walk down the tree from latest to oldest,
	// compute children and nearest descendants
	for (uint i = 0, cnt = ro.count(); i < cnt; i++) {

		uint type = checkRef(ro[i]);
		bool isB = (type & (BRANCH | RMT_BRANCH));
		bool isT = (type & TAG);

		const Rev* r = revLookup(ro[i]);

		if (isB) {
			Rev* rr = const_cast<Rev*>(r);
			if (r->descBrnMaster != -1) {
				SCRef sha = ro[r->descBrnMaster];
				rr->descBranches = revLookup(sha)->descBranches;
			}
			rr->descBranches.append(i);
		}
		if (isT) {
			updateDescMap(r, i, descMap, descVect);
			Rev* rr = const_cast<Rev*>(r);
			rr->descRefs.clear();
			rr->descRefs.append(i);
		}
		for (uint y = 0; y < r->parentsCount(); y++) {

			Rev* p = const_cast<Rev*>(revLookup(r->parent(y)));
			if (p) {
				p->childs.append(i);

				if (p->descBrnMaster == -1)
					p->descBrnMaster = isB ? r->orderIdx : r->descBrnMaster;
				else
					mergeBranches(p, r);

				if (p->descRefsMaster == -1)
					p->descRefsMaster = isT ? r->orderIdx : r->descRefsMaster;
				else
					mergeNearTags(optGoDown, p, r, descMap);
			}
		}
	}
	// walk backward through the tree and compute nearest tagged ancestors
	for (int i = ro.count() - 1; i >= 0; i--) {

		const Rev* r = revLookup(ro[i]);
		bool isTag = checkRef(ro[i], TAG);

		if (isTag) {
			Rev* rr = const_cast<Rev*>(r);
			rr->ancRefs.clear();
			rr->ancRefs.append(i);
		}
		for (uint y = 0; y < r->childs.count(); y++) {

			Rev* c = const_cast<Rev*>(revLookup(ro[r->childs[y]]));
			if (c) {
				if (c->ancRefsMaster == -1)
					c->ancRefsMaster = isTag ? r->orderIdx:r->ancRefsMaster;
				else
					mergeNearTags(!optGoDown, c, r, descMap);
			}
		}
	}
}

// ********************************* Rev **************************

const QString Rev::mid(int start, int len) const {

	// warning no sanity check is done on arguments
	return QString::fromAscii(&(ba[start]), len);
}

const QString Rev::parent(int idx) const {

	return mid(start + startOfs + 41 + 41 * idx, 40);
}

const QStringList Rev::parents() const {

	if (parentsCnt == 0)
		return QStringList();

	int ofs = start + startOfs + 41;
	return QStringList::split(' ', mid(ofs, 41 * parentsCnt - 1));
}

int Rev::indexData() { // fast path here, less then 4% of load time
/*
	When git-rev-list is called with --header option, after the first
	line with the commit sha, the following information is produced

	- one line with "tree"
	- an arbitrary amount of "parent" lines
	- one line with "author"
	- one line with "committer"
	- zero or more non blank lines with other info, as the encoding
	- one blank line
	- zero or one line with log title
	- zero or more lines with log message
	- a terminating '\0'
*/
	int last = ba.size() - 1;

	// take in account --boundary and --left-right options
	startOfs = uint(ba.at(start) == '-' || ba.at(start) == '<' || ba.at(start) == '>');
	boundary = startOfs && ba.at(start) == '-';

	parentsCnt = 0;
	int idx = start + startOfs + 40;
	while (idx < last && ba[idx] == ' ') {
		idx += 41;
		parentsCnt++;
	}
	idx += 47; // idx points to first line '\n', so skip tree line
	while (idx < last && ba[idx] == 'p') //skip parents
		idx += 48;

	idx += 23;
	if (idx > last)
		return -1;

	int lineEnd = ba.find('\n', idx); // author line end
	if (lineEnd == -1)
		return -1;

	lineEnd += 23;
	if (lineEnd > last)
		return -1;

	autStart = idx - 16;
	autLen = lineEnd - idx - 24;
	autDateStart = lineEnd - 39;
	autDateLen = 10;

	idx = ba.find('\n', lineEnd); // committer line end
	if (idx == -1)
		return -1;

	// shortlog could be not '\n' terminated so use committer
	// end of line as a safe start point to find chunk end
	int end = ba.find('\0', idx); // this is the slowest find
	if (end == -1)
		return -1;

	// ok, from here we are sure we have a complete chunk
	while (++idx < end && ba[idx] != '\n') // check for the first blank line
		idx = ba.find('\n', idx);

	sLogStart = idx + 5;
	if (end < sLogStart) { // no shortlog and no longLog

		sLogStart = sLogLen = 0;
		lLogStart = lLogLen = 0;
		return ++end;
	}
	lLogStart = ba.find('\n', sLogStart);
	if (lLogStart != -1 && lLogStart < end) {

		sLogLen = lLogStart++ - sLogStart;
		lLogLen = end - lLogStart;

	} else { // no longLog and no new line at the end of shortlog
		sLogLen = end - sLogStart;
		lLogStart = lLogLen = 0;
	}
	return ++end;
}
