/******************************************************************************
*		       							      *
* calculate.c (part of rCalc)					       	      *
* Copyright 1999, 2000 Gary Benson <rat@spunge.org>		       	      *
*								       	      *
* This program is free software; you can redistribute it and/or modify 	      *
* it under the terms of the GNU General Public License as published by 	      *
* the Free Software Foundation; either version 2 of the License, or    	      *
* (at your option) any later version.				       	      *
*								       	      *
* This program is distributed in the hope that it will be useful,      	      *
* but WITHOUT ANY WARRANTY; without even the implied warranty of       	      *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	       	      *
* GNU General Public License for more details.			       	      *
*								       	      *
* You should have received a copy of the GNU General Public License    	      *
* along with this program; if not, write to the Free Software	       	      *
* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.	       	      *
*								       	      *
******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <gnome.h>
#include <locale.h>

#include "console.h"
#include "config.h"
#include "engine.h"

/*****************************************************************************/

/* Sizes of various items:
*/
#define MAX_EXPRESSION_LEN	1024	/* Maximum length of expression	     */
#define MAX_ITEMS		512	/* Maximum items per expression	     */
#define STACK_SIZE		512	/* Size of number/string stack	     */

/* Decimal point character from locale
*/
static char decimal;

/*****************************************************************************/

/* Initialise the engine. Basically discover what the decimal separator is.
*/
void rCalc_engine_Initialise( void )
{
	struct lconv *locale = localeconv();

	/* Confess to the hacked nature of this hack.
	*/
	if( strlen( locale->decimal_point )!=1 )
	{
		char *lang = getenv( "LANG" );
		char *lc_numeric = getenv( "LC_NUMERIC" );
		if( !lang )		lang = "<null>";
		if( !lc_numeric )	lc_numeric = "<null>";
		
		printf( "Unfortunatly rCalc cannot handle numbers in your\n" );
		printf( "native format. This is being worked on...\n" );
		printf( "\n" );
		printf( "Please email dirtyrat@users.sourceforge.net and\n" );
		printf( "quote the following:\n" );
		printf( "\tVERSION       = `%s'\n", VERSION );
		printf( "\tLANG          = `%s'\n", lang );
		printf( "\tLC_NUMERIC    = `%s'\n", lc_numeric );
		printf( "\tdecimal_point = `%s'\n", locale->decimal_point );
		printf( "\n" );
		printf( "Thank you.\n" );
		printf( "\n" );
	}
	decimal = locale->decimal_point[0];
	Debug( "Decimal point character is `%c'.", decimal );
}

/*****************************************************************************/

/* Table of unary functions.
*/
static double ufMinus	( double x );
static double ufRound	( double x );
static double ufSin	( double x );
static double ufCos	( double x );
static double ufTan	( double x );
static double ufASin	( double x );
static double ufACos	( double x );
static double ufATan	( double x );

typedef struct
{
	char		*name;
	double		(*func)(double);
	HELPTOPIC	helpTopic;
}
FUNCTION;

#define NUM_FUNCTIONS	18

static FUNCTION Function[NUM_FUNCTIONS]=
{
	{ "-",		ufMinus, 	helpNoHelp	},

	/* Engine | Functions | Name of function */
	{ N_("int"),	ufRound,	helpInt		},

	/* Engine | Functions | Name of function */
	{ N_("abs"),	fabs,		helpAbs		},

	/* Engine | Functions | Name of function */
	{ N_("ln"),	log,		helpLogs	},

	/* Engine | Functions | Name of function */
	{ N_("log"),	log10,		helpLogs	},

	/* Engine | Functions | Name of function */
	{ N_("sqrt"),	sqrt,		helpSqrt	},

	/* Engine | Functions | Name of function */
	{ N_("sin"),	ufSin,		helpTrig	},

	/* Engine | Functions | Name of function */
	{ N_("cos"),	ufCos,		helpTrig	},

	/* Engine | Functions | Name of function */
	{ N_("tan"),	ufTan,		helpTrig	},

	/* Engine | Functions | Name of function */
	{ N_("asin"),	ufASin,		helpTrig	},

	/* Engine | Functions | Name of function */
	{ N_("acos"),	ufACos,		helpTrig	},

	/* Engine | Functions | Name of function */
	{ N_("atan"),	ufATan,		helpTrig	},

	/* Engine | Functions | Name of function */
	{ N_("sinh"),	sinh,		helpHype	},

	/* Engine | Functions | Name of function */
	{ N_("cosh"),	cosh,		helpHype	},

	/* Engine | Functions | Name of function */
	{ N_("tanh"),	tanh,		helpHype	},

	/* Engine | Functions | Name of function */
	{ N_("asinh"),	asinh,		helpHype	},

	/* Engine | Functions | Name of function */
	{ N_("acosh"),	acosh,		helpHype	},

	/* Engine | Functions | Name of function */
	{ N_("atanh"),	atanh,		helpHype	},
};
	
/*****************************************************************************/

/* Observer functions for the table of functions
** These do not report ufMinus (unary minus). 
*/
int rCalc_engine_GetNumFunctions( void )
{
	return NUM_FUNCTIONS-1;
}
char *rCalc_engine_GetFunctionName( int i )
{
	return _(Function[i+1].name);
}
HELPTOPIC rCalc_engine_GetFunctionHelp( int i )
{
	return Function[i+1].helpTopic;
}

/*****************************************************************************/

/* The functions themselves.
*/
static double ufMinus( double x )	{ return -x; }
static double ufRound( double x )	{ return floor(x+0.5); }

static double DtoR = M_PI / 180.0;
static double RtoD = 180.0 / M_PI;

static double ufSin( double x )
{
	return cfgCalc->angleMode==angleDegrees ? sin(x*DtoR) : sin(x);
}
static double ufCos( double x )
{
	return cfgCalc->angleMode==angleDegrees ? cos(x*DtoR) : cos(x);
}
static double ufTan( double x )
{
	return cfgCalc->angleMode==angleDegrees ? tan(x*DtoR) : tan(x);
}
static double ufASin( double x )
{
	return cfgCalc->angleMode==angleDegrees ? RtoD*asin(x) : asin(x);
}
static double ufACos( double x )
{
	return cfgCalc->angleMode==angleDegrees ? RtoD*acos(x) : acos(x);
}
static double ufATan( double x )
{
	return cfgCalc->angleMode==angleDegrees ? RtoD*atan(x) : atan(x);
}

/*****************************************************************************/

/* Table of builtin constants.
*/
#define NUM_CONSTANTS	2
static VARIABLE Constant[NUM_CONSTANTS]=
{
	/* Engine | Constants | Name of constant */
	{ N_("pi"),		M_PI },
	/* Engine | Constants | Name of constant */
	{ N_("e"),		M_E  },
};

/*****************************************************************************/

/* Observer functions for the table of constants.
*/
int rCalc_engine_GetNumConstants( void )
{
	return NUM_CONSTANTS;
}
char *rCalc_engine_GetConstantName( int i )
{
	return _(Constant[i].name);
}
double rCalc_engine_GetConstantValue( int i )
{
	return Constant[i].value;
}
HELPTOPIC rCalc_engine_GetConstantHelp( int i )
{
	return helpConstants;
}

/*****************************************************************************/

/* Handlers for variables
*/
static boolean validVariableName( char *name )
{
	int len,num,i;

	len = strlen(name);
	if( len<1 )
	{
		/* Engine | Bad variable name */
		Error( _("variable name has zero length.") );
		return FALSE;
	}
	else if( len>MAX_VARIABLE_NAME_LEN )
	{
		/* Engine | Bad variable name */
		Error( _("%s: variable name too long (>%d characters)."),
			name, MAX_VARIABLE_NAME_LEN );
		return FALSE;
	}
	else if( !isalpha( name[0] ) )
	{
		/* Engine | Bad variable name */
		Error( _("%s: variable name must start with a letter."),
		       name );
		return FALSE;
	}

	for( i=1; i<len; i++ )
	{
		if( isalnum( name[i] ) ) continue;

		/* Engine | Bad variable name */
		Error( _("%s: variable name must be alphanumeric."), name );
		return FALSE;
	}

	num = rCalc_engine_GetNumCommands();
	for( i=0; i<num; i++ )
	{
		if( strcasecmp( name, rCalc_engine_GetCommandName( i ) ) )
			continue;
		/* Engine | Bad variable name */
		Error( _("%s: variable name is a reserved word."), name );
		return FALSE;
	}
	for( i=0; i<NUM_FUNCTIONS; i++ )
	{
		if( strcasecmp( name, Function[i].name ) ) continue;

		/* Engine | Bad variable name */
		Error( _("%s: variable name is a reserved word."), name );
		return FALSE;
	}
	for( i=0; i<NUM_CONSTANTS; i++ )
	{
		if( strcasecmp( name, Constant[i].name ) ) continue;

		/* Engine | Bad variable name */
		Error( _("%s: variable name is a reserved word."), name );
		return FALSE;
	}

	return TRUE;
}

static boolean setVariableValue( char *name, double value )
{
	VARIABLE *Variable = cfgCalc->variables;
	int i;

	/* Check that its name is valid
	*/
	if( !validVariableName(name) ) return FALSE;

	/* If it already exists then overwrite it.
	*/
	for( i=0; i<cfgCalc->nVariables; i++ )
	{
		if( !strcasecmp( Variable[i].name, name) )
		{
			Debug( "%s: overwriting variable.", name );
			strcpy( Variable[i].name, name );
			Variable[i].value = value;

			cfgCalc->variables_mod = TRUE;
			
			return TRUE;
		}
	}

	/* It doesn't exist, so create a new one.
	*/
	Debug( "%s: creating new variable.", name );
	if( cfgCalc->nVariables == MAX_VARIABLES )
	{
		/* Engine | too many variables! */
		Error( _("out of variable storage space.") );
		return FALSE;
	}
	strcpy( Variable[cfgCalc->nVariables].name, name );
	Variable[cfgCalc->nVariables].value = value;
	cfgCalc->nVariables++;
	
	cfgCalc->variables_mod = TRUE;

	return TRUE;
}

static boolean getVariableValue( char *name, double *value_p )
{
	VARIABLE *Variable  = cfgCalc->variables;
	int	 nVariables = cfgCalc->nVariables;
	int i;

	/* Check that its name is valid
	*/
	if( !validVariableName(name) ) return FALSE;

	/* Find the variable.
	*/
	for( i=0; i<nVariables; i++ )
	{
		if( !strcasecmp(Variable[i].name,name) )
		{
			*value_p = Variable[i].value;
			return TRUE;
		}
	}
	/* Engine | The specified variable does not exist */
	Error( _("%s: undefined variable."), name );
	return FALSE;
}

/*****************************************************************************/

/* Stack handler - vital for expression handling.
** infixToPostfix() uses the stack to store pointers to strings,
** and evaluatePostfix() uses it to store numbers.
*/
typedef union
{
	char	*Str;
	double	 Num;
}
StackItem;

static StackItem stack_p[STACK_SIZE];
static int stack_i=0;

#define stackSize()	(stack_i)
#define resetStack()	(stack_i=0)

static boolean pushString( char *str )
{
	if( stack_i>=STACK_SIZE )
	{
		/* Engine | Out of stack */
		/* nb. This should *never* occur */
		Error( _("out of stack space.") );
		return FALSE;
	}
	else
	{
		stack_p[stack_i].Str = str;
		stack_i++;
		return TRUE;
	}
}
static boolean pushNumber( double num )
{
	if( stack_i>=STACK_SIZE )
	{
		Error( _("out of stack space.") );
		return FALSE;
	}
	else
	{
		stack_p[stack_i].Num = num;
		stack_i++;
		return TRUE;
	}
}

static boolean popString( char **str_p )
{
	if( stack_i <= 0 )
	{
		/* Engine | The expression contains some kind of error */
		Error( _("malformed expression.") );
		return FALSE;
	}
	else
	{
		stack_i--;
		*str_p = stack_p[stack_i].Str;
		return TRUE;
	}
}
static boolean popNumber( double *num_p )
{
	if( stack_i <= 0 )
	{
		Error( _("malformed expression.") );
		return FALSE;
	}
	else
	{
		stack_i--;
		*num_p = stack_p[stack_i].Num;
		return TRUE;
	}
}

static boolean topString( char **str_p )
{
	if( stack_i <= 0 )
	{
		Error( _("malformed expression.") );
		return FALSE;
	}
	else
	{
		*str_p = stack_p[stack_i-1].Str;
		return TRUE;
	}
}
/*  static boolean topNumber( double *num_p ) */
/*  { */
/*  	if( stack_i <= 0 ) */
/*  	{ */
/*  		Error( _("malformed expression.") ); */
/*  		return FALSE; */
/*  	} */
/*  	else */
/*  	{ */
/*  		*num_p = stack_p[stack_i-1].Num; */
/*  		return TRUE; */
/*  	} */
/*  } */

/*****************************************************************************/

/* Global variables used in the evaluation functions
*/
static char  exprBuffer [MAX_EXPRESSION_LEN*2];
static char *exprInfix  [MAX_ITEMS+1];
static char *exprPostfix[MAX_ITEMS+1];
static char *expression;

/* Remove all whitespace from a string
*/
static void stripSpace( char *in )
{
	char *out=in;

	while( *in )
	{
		if( !isspace(*in) ) *out++=*in;
		in++;
	}
	*out=0;
}

/*****************************************************************************/

/* Read in one item, returning NULL if the end of the list is reached.
** If the item is one of ()^*+-/ then it is stored like that. Multi-
** character items are given a prefix ( [N]umber, [V]ariable, [C]onstant,
** unary [F]unction ) for ease of recognition.
** Warning: This function assumes that there is some space at the end
** of the string which it may also use.
*/
static boolean readItem( char **p, char **last, int *nInfixItems )
{
	char *expr=*p;
	char *item=*p;
	char *sp,*ep;

	/* Check for the end of the expression.
	*/
	if(*expr==0)
	{
		exprInfix[*nInfixItems]=NULL;
		*last = NULL;
		return TRUE;
	}

	/* Check we are not running out of room.
	*/
	if( *nInfixItems==MAX_ITEMS )
	{
		/* Engine | The expression is too long */
		/* nb. This should *never* occur */
		Error( _("expresson contains too many terms (>%d)."),
		       MAX_ITEMS );
		return FALSE;
	}

	/* Parse this next piece of expression.
	*/
	if( *expr=='-' )
	{
		/* Item is a minus
		*/
		int unary;

		/* Unary or binary?
		*/
		if(*last==NULL)		unary=1;
		else if(**last=='N')	unary=0;
		else if(**last=='C')	unary=0;
		else if(**last=='V')	unary=0;
		else if(**last==')')	unary=0;
		else			unary=1;

		if( unary )		/* Convert to F- */
		{
			/* Insert the prefix
			*/
			for( sp=item, ep = sp+strlen(sp); ep>=sp; ep-- )
				*(ep+1)=*ep;
			*item='F';
			expr++;
		}
		expr++;
	}
	else if( strchr("()^/*+",*expr) )
	{
		/* Item is a single character operator (not minus - see above)
		*/
		expr++;
	}
	else if((*expr=='.')||(*expr==decimal)||(isdigit(*expr)))
	{
		/* Item is a number
		*/
		int d=0,e=0;

		if((*expr=='.')||(*expr==decimal))
		{
			d=1;
			*expr = decimal;
		}
		
		expr++;
		while( *expr!=0 )
		{
			if( (*expr=='e')||(*expr=='E') )
			{
				if(e)
				{
					/* Engine | A number is invalid in some way */
					Error( _("%s: malformed number."),*p );
					return FALSE;
				}
				else d=e=1;

				expr++;
				if(*expr==0)
				{
					Error( _("%s: malformed number."),*p );
					return FALSE;
				}
				if((*expr=='-')||(*expr=='+')) expr++;
				if(!isdigit(*expr))
				{
					Error( _("%s: malformed number."),*p );
					return FALSE;
				}
			}
			else if((*expr=='.')||(*expr==decimal))
			{
				*expr = decimal;
				if(d)
				{
					Error( _("%s: malformed number."),*p );
					return FALSE;
				}
				else d=1;
				expr++;
				if(!isdigit(*expr))
				{
					Error( _("%s: malformed number."),*p );
					return FALSE;
				}
			}
			else if( isdigit(*expr) )
			{
				expr++;
			}
			else break;
		}

		/* Add the N prefix
		*/
		for( sp=item, ep = sp+strlen(sp); ep>=sp; ep-- ) *(ep+1)=*ep;
		*item='N';
		expr++;
	}
	else if(isalpha(*expr))
	{
		/* Item is a variable / special operator / function
		*/
		int length,i;

		while( isalnum(*expr) ) expr++;
		length=expr-item;

		/* Make room for the prefix
		*/
		for( sp=item, ep = sp+strlen(sp); ep>=sp; ep-- ) *(ep+1)=*ep;
		expr++;

		/* Function, constant or variable?
		*/
		*item='V';
		for(i=0;i<NUM_FUNCTIONS;i++)
		{
			if(strlen(Function[i].name)!=length)
				continue;
			if(strncasecmp(item+1,Function[i].name,length))
				continue;
			*item='F';
			break;
		}
		for(i=0;i<NUM_CONSTANTS;i++)
		{
			if(strlen(Constant[i].name)!=length)
				continue;
			if(strncasecmp(item+1,Constant[i].name,length))
				continue;
			*item='C';
			break;
		}
	}
	else
	{
		Error( _("malformed expression.") );
		return FALSE;
	}


	/* Split the string
	*/
	for( sp=expr, ep = sp+strlen(sp); ep>=sp; ep-- ) *(ep+1)=*ep;

	*sp = 0;	/* The end of this item	*/
	expr++;		/* The start of the next item */

	*p = expr;	/* 'this' item for next time round */
	*last = item;	/* 'last' item for next time round */

	exprInfix[(*nInfixItems)++] = item;

	return TRUE;
}

/*****************************************************************************/

/* Using the function readItem() above, split the expression into
** a NULL terminated array of items.
*/
static boolean breakInfix(void)
{
	int nInfixItems;
	char *expr,*c;
	boolean s;

	/* Strip any whitespace
	*/
	stripSpace( expression );
	if( !strlen(expression) ) return FALSE;

	/* Break the expression
	*/
	for( c=NULL, nInfixItems=0, expr=expression;
	     (s=readItem(&expr,&c,&nInfixItems)) ; )
	{
		if(!s) return FALSE;
		if(c==NULL) break;
	}
	if( nInfixItems<1 ) return FALSE;

	return TRUE;
}

/*****************************************************************************/

/* Hack alert! Wait for rCalc 0.3.0 for a nice fix */
static int priority_of( char operator )
{
	switch( operator )
	{
	case 'F': return 0;
	case '^': return 1;
	case '/': return 2;
	case '*': return 2;
	case '+': return 4;
	case '-': return 4;
	case '(': return 6;
	case ')': return 7;
	default:
		Error(_("internal error"));
		Debug( "%c: illegal priority.", operator );
		return 0;
	}
}

/* Convert expression from infix (normal human style) to
** postfix (reverse polish). Basically, infix requires
** brackets to remove ambiguity, whereas postfix does not.
** In addition, postfix is very easy to evaluate using
** a computer. For more details, see:
** http://renoir.vill.edu/~cassel/1200/revpolish.html
*/
static boolean infixToPostfix( void )
{
	char *Operators="F^/*+-()";
	char *item,*p,*q;
	char **in,**out;

	/* The stack should be empty - if not then we have a leak.
	*/
	if( stackSize() )
	{
		Debug( "data left on stack - clearing." );
		resetStack();
	}

	/* Process the expression
	*/
	for( out=exprPostfix, in=exprInfix; *in; in++ )
	{
		item = *in;

		if( *item == '(' )
		{
			/* Push onto the stack
			*/
			if( !pushString( item ) ) return(FALSE);
		}
		else if( *item == ')' )
		{
			/* Pop the stack to the output until '(' is found 
			*/
			do
			{
				if( !popString(&p) ) return FALSE;
				if( *p != '(' ) *out++ = p;
			}
			while( *p != '(' );
		}
		else if( strchr("NHVC", *item ) )
		{
			/* Operand - write to output string
			*/
			*out++ = item;
		}
		else if( (p=strchr( Operators, *item ))!=NULL )
		{
			int prio_p = priority_of( *p );
			int prio_q;
			
			/* Operator - pop higher priority items
			*/
			while(1)
			{
				if( !stackSize()) break;
				if( !topString(&q) ) return FALSE;
				prio_q = priority_of( *q );

				/* Low number = high priority */
				if( prio_q>prio_p ) break;

				if( !popString( out++ ) ) return FALSE;
			}
			if( !pushString( item ) ) return FALSE;
		}
		else
		{
			/* Engine | Programmer error */
			Error( _("internal error.") );
			Debug( "%s: illegal item.", item );
			return FALSE;
		}
	}

	/* Pop the remainder of the stack to the output string, and write the
	** terminating NULL.
	*/
	while( stackSize()>0 ) popString( out++ );
	*out = NULL;

	return TRUE;
}

/*****************************************************************************/

/* Evaluate the postfix expression.
*/
static boolean evaluatePostfix( double *result_p )
{
	char **postfix;
	double x,y;
	int i;


	/* The stack should be empty - if not then we have a leak.
	*/
	if( stackSize() )
	{
		Debug( "data left on stack - clearing." );
		resetStack();
	}

	/* Evaluate the expression.
	*/
	for( postfix = exprPostfix; *postfix; postfix++ )
	{
		switch( *postfix[0] )
		{
			/* A number - push it
			*/
			case 'N':
				if( !pushNumber( atof( (*postfix)+1 ) ) )
					return FALSE;
				break;

			/* Item is a variable - push its value
			*/
			case 'V':
				if( !getVariableValue( (*postfix)+1, &x) )
					return FALSE;
				if( !pushNumber(x) )
					return FALSE;
				break;

			/* Item is a constant - push its value
			*/
			case 'C':
				for( i=0; i<NUM_CONSTANTS; i++ )
				{
					if( strcasecmp( (*postfix)+1,
							Constant[i].name ))
						continue;
					
					if( !pushNumber( Constant[i].value ) )
						return FALSE;
					break;
				}
				if( i==NUM_CONSTANTS )
				{
					Error( _("internal error.") );
					Debug( "%s: undefined constant.",
					       (*postfix)+1 );
					return FALSE;
				}
				break;

			/* Item is a unary function - evaluate it
			*/
			case 'F':
				for( i=0; i<NUM_FUNCTIONS; i++ )
				{
					if( strcasecmp( (*postfix)+1,
							Function[i].name ))
						continue;
					
					if( !popNumber(&x) )
						return FALSE;
					if( !pushNumber(Function[i].func(x)) )
						return FALSE;
					break;
				}
				if( i==NUM_FUNCTIONS )
				{
					Error( _("internal error.") );
					Debug( "%s: undefined function.",
					       (*postfix)+1 );
					return FALSE;
				}
				break;

			/* Item is a binary function - evaluate it
			*/
			case '+':
				if( !popNumber(&y) )	return FALSE;
				if( !popNumber(&x) )	return FALSE;
				if( !pushNumber(x+y) )	return FALSE;
				break;

			case '-':
				if( !popNumber(&y) )	return FALSE;
				if( !popNumber(&x) )	return FALSE;
				if( !pushNumber(x-y) )	return FALSE;
				break;

			case '*':
				if( !popNumber(&y) )	return FALSE;
				if( !popNumber(&x) )	return FALSE;
				if( !pushNumber(x*y) )	return FALSE;
				break;

			case '/':
				if( !popNumber(&y) )	return FALSE;
				if( !popNumber(&x) )	return FALSE;
				if( y==0.0 )
				{
					/* Engine | division by zero */
					Error( _("division by zero.") );
					return FALSE;
				}
				if( !pushNumber(x/y) )	return FALSE;
				break;

			case '^':
				if( !popNumber(&y) )	return FALSE;
				if( !popNumber(&x) ) 	return FALSE;
				if( !pushNumber( pow(x,y) ) ) return FALSE;
				break;

			default:
				Error( _("internal error.") );
				Debug( "%d: illegal item.", *postfix[0] );
				return FALSE;
		}
	}

	/* Get the answer off the stack
	*/
	if( stackSize()!=1 )
	{
		Error( _("malformed expression.") );
		Debug( "%d items on stack after evaluation.", stackSize() );
		return FALSE;
	}
	popNumber(result_p);
	return TRUE;
}

/*****************************************************************************/

void rCalc_engine_Evaluate( char *input )
{
	char *lhs,*rhs;
	char *name;
	double result;
	
	/* Make a copy of the input string
	*/
	if( strlen(input)>MAX_EXPRESSION_LEN )
	{
		/* Engine | the expression is too long */
		Error( _("expression too long (>%d characters)"),
		       MAX_EXPRESSION_LEN );
		return;
	}
	strcpy( exprBuffer, input );
	input = exprBuffer;

	/* Remove all whitespace.
	*/
	stripSpace( input );

	/* Look for an equals sign.
	*/
	if( (rhs = strchr(input,'='))==NULL )
	{
		name	   = "Ans";
		expression = input;
	}
	else
	{
		lhs = input;
		*rhs = 0;
		rhs++;

		if( strchr( rhs,'=' ) )
		{
			/* Engine | too many '=' in expression */
			Error( _("too many equalities.") );
			return;
		}

		name 	   = lhs;
		expression = rhs;
	}

	/* Evaluate the expression and store the result.
	*/
	if( !breakInfix() ) return;
	if( !infixToPostfix() )  return;
	if( !evaluatePostfix( &result ) ) return;
	if( !setVariableValue( name, result ) ) return;

	/* Print the result
	*/
	Print("\t%s = %.16g\n", name, result );
}

/*** end of calculate.c ******************************************************/






