genRepos <- function (repName, urlBase, urlPath="",
                      repType=c("package", "vignette"),
                      repRelLevel=c("release", "devel"), dir =".") {

    repType <- match.arg(repType)
    repRelLevel <- match.arg(repRelLevel)

    outFile <- file.path(dir, "replisting")
    out <- paste("repname: ", repName, "\nreptype: ", repType,
                 "\nrepaddrBase: ", urlBase, "\nrepaddrPath: ",
                 urlPath, "\nreprellevel: ", repRelLevel, "\n\n",
                 sep = "")
    cat(out, file = outFile)
    curDir <- getwd()
    on.exit(setwd(curDir), add=TRUE)

    if (!file.exists(dir)) {
        stop(paste("Directory parameter",dir,"does not exist"))
    }

    setwd(dir)

    switch(repType,
           "package" = genPkgRepos(dir),
           "vignette" = genVigRepos(dir),
           stop("Repository type ", repType, " not currently supported"))

    parseThemesXML()

    return(list(repName=repName, repType = repType, repaddrBase =
                urlBase, repaddrPath=urlPath,
                repRelLevel=repRelLevel, repDir=dir))
}


genPkgRepos <- function(dataDir) {

    ## Create a temporary directory for unpacking packages
    tmpDir <- tempfile()
    on.exit(unlink(tmpDir,recursive=TRUE),add=TRUE)
    dir.create(tmpDir)

    tmpInfo <- file.path(tmpDir,"info")
    dir.create(tmpInfo)

    curDir <- getwd()
    on.exit(setwd(curDir),add=TRUE)

    ## check to make sure this works
    setwd(dataDir)

    ## Get list of packages to unpack
    pkgs <- dir(pattern="*\\.tar\\.gz|*\\.tgz|*\\.zip")
    PACKin <- SrcStat <- WinStat <- NULL
    for (pkg in pkgs) {
        ## Get extension/pkg split out
        ext <- strsplit(pkg,"\\.")[[1]]
        ext <- ext[length(ext)]

        ## Subset the result of unpackExtract as R-devel now attaches
        ## a line to the end of DESCRIPTION, but if there were
        ## blank lines at the end of the normal block, this will
        ## cause there to be multiple rows in the matrix
        DESC <-  unpackExtractPkgDESC(pkg)[1,,drop=FALSE]
        if (!is.null(DESC)) {
            samePack <- which(PACKin[,"Package"]==DESC[,"Package"])
            if ((length(samePack) == 0) ||
                (all(PACKin[samePack,"Version",drop=FALSE]!=DESC[,"Version"]))) {
                PACKin <- rbind(PACKin, DESC)
            }
            if (!is.na(DESC[,"Built"])) {
                parts <- strsplit(DESC[,"Built"],"; ")
                pkgRvers <- strsplit(parts[[1]][1]," ")[[1]][2]
                pkgDate <- parts[[1]][3]
            }
            else {
                pkgRvers <- ""
                pkgDate <- date()
            }
            newStatus <- c(DESC[,"Package"],
                           DESC[,"Version"],
                           file.path(dataDir,pkg),
                           "OK", pkgRvers, pkgDate)
            switch(ext,
                   "gz"=SrcStat<-rbind(SrcStat,newStatus),
                   "tgz"=SrcStat<-rbind(SrcStat,newStatus),
                   "zip"=WinStat<-rbind(WinStat,newStatus)
                   )
        }
    }
    ##rbind seems to add unwanted rownames
    if( !is.null(SrcStat) ) row.names(SrcStat) <- NULL
    if( !is.null(WinStat) ) row.names(WinStat) <- NULL
    fields <- c("Package", "Version", "File", "Status", "Rvers","Date")

    write.dcf(PACKin,file.path(tmpInfo,"PACKAGES.in"))

    if (!is.null(SrcStat)) {
        colnames(SrcStat) <- fields
        write.dcf(SrcStat,file.path(tmpInfo,"Source.status"))
    }
    if (!is.null(WinStat)) {
        colnames(WinStat) <- fields
        write.dcf(WinStat,file.path(tmpInfo,"Win32.status"))
    }

    df <- buildPkgDf(infoDir=tmpInfo)
    setwd(curDir)
    saveDfRda(df)
}

genVigRepos <- function(dataDir) {
    ## Create a temporary directory for holding temp data
    tmpDir <- tempfile()
    on.exit(unlink(tmpDir,recursive=TRUE),add=TRUE)
    dir.create(tmpDir)
    tmpInfo <- file.path(tmpDir,"info")
    dir.create(tmpInfo)
    curDir <- getwd()
    on.exit(setwd(curDir),add=TRUE)

    setwd(dataDir)
    ## Get list of vignettes to use
    vigList <- getPkgVigList(".",baseVigDesc,".",pkgVers=FALSE)
    vigList<- mergeVigListStrings(vigList)
    vigList <- filterVigList(vigList)
    ## Push this into a data frame
    tmpVigDf <- data.frame(I(vigList[[1]]))
    for (i in 2:length(vigList))
        tmpVigDf <- cbind(tmpVigDf,I(vigList[[i]]))
    tmpVigDf <- t(tmpVigDf)
    rownames(tmpVigDf) <- 1:nrow(tmpVigDf)
    tmpVigDf <- as.data.frame(tmpVigDf)
    ## !! Temporarily just remove those w/o a PDFpath.
    ## !!! Need to try to build their PDFs
    tmpVigDf <- tmpVigDf[!is.na(tmpVigDf$PDFpath),]
    ## Write to DCF file
    write.dcf(tmpVigDf,file=file.path(tmpInfo,"Vignettes.in"))

    vigDf <- buildVigDf(infoDir=tmpInfo)
    setwd(curDir)
    saveDfRda(vigDf)
}


unpackExtractPkgDESC <- function(pkg) {
    ## Unpacks the package, extracts DESCRIPTION info
    cDir <- getwd()
    on.exit(setwd(cDir),add=TRUE)

    tmpDir <- tempfile()
    on.exit(unlink(tmpDir,recursive=TRUE),add=TRUE)
    dir.create(tmpDir)

    file.copy(pkg,tmpDir)
    setwd(tmpDir)
    ## Get extension/pkg split out
    ext <- strsplit(pkg,"\\.")[[1]]
    ext <- ext[length(ext)]
    ret <- switch(ext,
                  "gz"=unpackSourcePkg(pkg),
                  "tgz"=unpackSourcePkg(pkg),
                  "zip"=unpackZipPkg(pkg),
                  stop(paste("Don't know how to unpack package",pkg))
                  )
    if (ret != 0)
        return(NULL)

    tmpPkg <- gsub(paste(".",ext,sep=""), "", pkg)
    pkgDir <- strsplit(tmpPkg,"_[[:digit:]]*.")[[1]][1]
    DESC <- file.path(pkgDir,"DESCRIPTION")
    if (!file.exists(DESC)) {
        return(NULL)
    }

    fields <- c("Package","Version", "Keywords", "Depends", "Title",
                "Suggests", "Imports", "Replaces", "Description", "URL",
                "Author", "Maintainer", "License", "Status",
                "Priority", "Built","Bundle","BundleDescription",
                "Contains", "SystemRequirements", "ReleaseLevel")

    dFile <- read.dcf(DESC,fields)

    ## Detect if this is a package bundle
    if (!is.na(dFile[1,"Bundle"])) {
        ## If this is a source bundle we want to treat it like
        ## a normal package, but need to fill in the
        ## package/description fields w/ appropriate info
        dFile[1,"Package"] <- dFile[1,"Bundle"]
        dFile[1,"Description"] <- dFile[1,"BundleDescription"]
    }

    ## clean up from behind
    unlink(pkgDir,recursive=TRUE)
    return(dFile)
}

unpackSourcePkg <- function(pkg) {
    z <- try(system(paste("gzip -dc", pkg, "| tar -xf -")))
    z
}

unpackZipPkg <- function(pkg) {
    OST <- .Platform$OS.type
    if (OST == "unix") {
        zip <- getOption("unzip")
        system(paste(zip,"-q",pkg))
    }
    else {
        zip.unpack(pkg,".")
    }

    return(0)
}


getArgRepos <- function(rep) {
    ## Determine if rep is a directory or an online repository
    ## used for functions where one can pass in a rep or a local
    ## directory that is a rep (but might not be online)
    if (is(rep, "character")) {
        if ((!file.exists(file.path(rep,"replisting")))||
            (!file.exists(file.path(rep,"repdatadesc.rda")))) {
            stop(paste(rep,
                       "does not seem to be a valid repository directory"))
        }
        else {
            load(file.path(rep,"repdatadesc.rda"))
            repL <- read.dcf(file.path(rep,"replisting"))
            rep <- buildReposEntry(new("replisting",replisting=repL),
                                   new("repdatadesc",
                                       repdatadesc=reposDF))
        }
    }
    else if (!is(rep,"ReposEntry"))
        stop(paste("Don't know how to handle passed rep argument:",rep))

    return(rep)
}

genReposHtml <- function (rep, filename="index.html", outDir=".",
                          headerInfo="")
{
    rep <- getArgRepos(rep)
    if (!file.exists(outDir))
        dir.create(outDir)
    if(repType(rep) == "package") {
        GenPkgListingHTML(file.path(outDir,filename), rep,
                          headerInfo)
    }
    else if (repType(rep) == "vignette") {
        GenVigListingHTML(file.path(outDir, filename), rep,
                          headerInfo)
    }
    else {
        stop(paste("Do not know how to construct HTML pages for",
                   "repositories of type:", repType(rep)))
    }
}

##idea here is to write a function that creates the package listings
GenPkgListingHTML <- function(filename, rep, headerInfo="",
                              upFile)
{
  if( missing(filename) )
    filename <- "packagelistingindex.html"

  outFile <- file(filename, "w")
  on.exit(close(outFile))
  rep <- getArgRepos(rep)
  ## Write header
  t1 <- paste("<TITLE>",repName(rep),"</TITLE>")
  cat("<html>", "<head>", t1, headerInfo,
        "</head> <body>", file = outFile, sep = "\n")

  cat(file=outFile, "<table border=0 align=left>",
         "<tr> <td><b>Package</b></td> <td><b>Title</b></td>",
           "<td><b>Version</b></td> </tr>", sep="\n")
  reposDF <- repdataframe(rep)
  pfilenames <- vector("character", length=nrow(reposDF))
  pN <- rownames(reposDF)
  pNames <- reposDF$Package
  for(i in 1:nrow(reposDF) ) {
    cat(file=outFile, "<tr>")
    pfilenames[i] <- genPkgListing(pN[i], reposDF = reposDF,
                                   upFile=filename, rep=rep,
                                   outDir=dirname(filename))
    cat(file=outFile, "<td><a href=\"" , pfilenames[i], "\"> ", pNames[i], "</a></td>\n",
      sep="")
    ## we need a short description
    cat(file=outFile,"<td>", as.character(reposDF[i,"Title"][[1]]),
         "</td>\n", sep="")
    cat(file=outFile, "<td>", as.character(reposDF[i,"Version"][[1]]),
         "</td>\n", sep="")
     cat(file=outFile, "</tr>")
  }
  if( !missing(upFile) )
    cat(file=outFile, "<h3 align=left> <a href=", upFile,
      ">Return to Index</a></h3><br> \n", sep="")

  cat(file=outFile, "</table>", "</body>",  "</html>", sep="\n" )
}

GenVigListingHTML <- function(filename, rep, headerInfo="", upFile) {
    if (missing(filename))
        filename <- "viglistingindex.html"
    outFile <- file(filename, "w")
    on.exit(close(outFile))

    rep <- getArgRepos(rep)

    reposDF <- repdataframe(rep)
    reposDF <- reposDF[sort(reposDF$VignettePackage,index.return=TRUE)$ix,]

    ## Write header
    t1 <- paste("<TITLE>",repName(rep),"</TITLE>")
    cat("<html>", "<head>", t1, headerInfo,
        "</head><body>", file = outFile, sep = "\n")
    cat("<h3 align=left>",repName(rep),"</h3>",file=outFile, sep="")
    cat(file=outFile, "<table border=0 align=left>",
        "<tr> <td><b>Vignette</b></td> <td><b>Package</b></td>",
        "</tr> <tr>", sep="\n")
    for (i in 1:nrow(reposDF)) {
        vigName <- reposDF$VignetteTitle[i]
        if (vigName == "Untitled")
            vigName <- reposDF$VignetteIndexEntry[i]
        vigPkg <- reposDF$VignettePackage[i]
        if (vigPkg == ".")
            vigPkg == "None"
        vigPath <- paste(repURL(rep),reposDF$PDFpath[i],sep="/")
        cat(file=outFile,"<tr>\n","<td><a href=\"",vigPath,
            "\"> ",vigName,"</a></td>\n<td>",vigPkg,
            "</td>\n</tr>\n",sep="")
    }
    if(!missing(upFile))
        cat(file=outFile, "<h3 align=left> <a href=", upFile,
            ">Return to Index</a></h3><br> \n", sep="")

    cat(file=outFile, "</table>", "</body>",  "</html>", sep="\n" )
}


##should return some data -- such as the file name used?
genPkgListing <- function(pkgN, reposDF, filename, upFile, rep,
                          outDir=".")
{
    nVers <- strsplit(pkgN, ":v:")[[1]]
    repRow <- which(row.names(reposDF)==pkgN)
    if( missing(filename) )
       filename <- paste(paste(nVers[1], nVers[2], sep="v"), ".html", sep="")
    ##for now we overwrite
    filename <- file.path(outDir, filename)
    outfile <- file(filename, "w")
    on.exit(close(outfile))
    cat(file=outfile, "<html>",  "<title>", nVers[1], "</title>", "<body>",
      sep="\n")
    if( !missing(upFile) )
       cat(file=outfile, "<h3 align=left> <a href=", upFile,
         ">Return to package listing</a></h3><br> \n", sep="")
    cat(file=outfile, " <h1 align=left> Package: ", nVers[1],
          "</h1> <div align=center> </div> \n <br> \n", sep="")
    cat(file=outfile, "<b>Description:</b>", reposDF[pkgN, "Description"],
          "<br>\n", sep="")
    cat(file=outfile, "<b>Version:</b>", nVers[2], "<br>\n", sep="")
    cat(file=outfile,"<b>Author:</b>",reposDF[pkgN,"Author"],
        "<br\n",sep="")
    cat(file=outfile,"<b>Maintainer:</b>",reposDF[pkgN,"Maintainer"],
        "<br>\n",sep="")
    cat(file=outfile, "<b>Dependencies:</b>", reposDF[pkgN, "Depends"][[1]],
         "<br>\n",sep="")
    cat(file=outfile,"<b>License:</b>",reposDF[pkgN,"License"],
        "<br><br><br>\n",sep="")

    ## Set download links
    if (length(repRow) > 0) {
        osS <- reposDF$OSspecific[[repRow]]
        if (length(osS) > 0)
            for (i in 1:length(osS)) {
                curOS <- osS[[i]]
                if (curOS$Status == "OK")
                    cat(file=outfile,"<a href=\"",
                        file.path(repURL(rep), curOS$File),
                        "\">",names(osS)[i],
                        " package download</a><br>", sep="")
            }

    }
    cat(file=outfile, "</body> \n </html>")
    return(filename)
}


filterVigList <- function(vigList) {
    newVigList <- list()
    fields <- c("VignetteIndexEntry","VignettePackage", "VignetteTitle",
                   "VignetteVersion", "VignetteDepends",
                   "VignetteKeywords", "PDFpath")
    for (i in 1:length(vigList)) {
        tmp <- list()
        for (j in 1:length(fields)) {
            tmp[[j]] <- as.character(vigList[[i]][fields[j]])
        }
        names(tmp) <- fields
        newVigList[[i]] <- tmp
    }
    names(newVigList) <- names(vigList)
    return(newVigList)
}

mergeVigListStrings <- function(vList) {
    for (i in 1:length(vList)) {
        if ((!is.null(vList[[i]]$VignetteDepends))&&
            (!is.na(vList[[i]]$VignetteDepends))) {
            if (length(vList[[i]]$VignetteDepends > 1)) {
                vList[[i]]$VignetteDepends <-
                    paste(vList[[i]]$VignetteDepends, collapse=", ")
            }
        }
        else {
            vList[[i]]$VignetteDepends <- NA
        }
        if ((!is.null(vList[[i]]$VignetteKeywords))&&
            (!is.na(vList[[i]]$VignetteKeywords))) {
            if (length(vList[[i]]$VignetteKeywords > 1)) {
                vList[[i]]$VignetteKeywords <-
                    paste(vList[[i]]$VignetteKeywords, collapse=", ")
            }
        }
        else {
            vList[[i]]$VignetteKeywords <- NA
        }
    }
    return(vList)
}

