/*
 * Resource manager daemon
 *
 * Copyright (C) 2001-2002, Olaf Kirch <okir@lst.de>
 */

#include "resmgrd.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/signal.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <ctype.h>
#include <pwd.h>
#include <grp.h>

struct conn *
rsm_connect(const char *path)
{
	struct sockaddr_un un;
	struct conn	*conn;

	conn = (struct conn *) calloc(1, sizeof(*conn));
	conn->fd = -1;
	conn->passfd = -1;
	conn->acceptfd = 1;

	conn->fd = socket(PF_LOCAL, SOCK_STREAM, 0);
	if (conn->fd < 0)
		goto fail;

	memset(&un, 0, sizeof(un));
	un.sun_family = AF_LOCAL;
	strcpy(un.sun_path, path);
	if (connect(conn->fd, (struct sockaddr *) &un, SUN_LEN(&un)) < 0)
		goto fail;

	return conn;

fail:	rsm_close(conn);
	return NULL;
}

static int
rsm_recv_char(struct conn *conn, char *cp)
{
	char		control[1024];
	struct msghdr	msg;
	struct iovec	iov;
	int		len;

	memset(&msg, 0, sizeof(msg));
	msg.msg_iov	= &iov;
	msg.msg_iovlen	= 1;
	iov.iov_base	= cp;
	iov.iov_len	= 1;

	if (conn->acceptfd && conn->passfd < 0) {
		msg.msg_controllen = sizeof(control);
		msg.msg_control	= &control;
	}

	len = recvmsg(conn->fd, &msg, 0);
	if (len < 0)
		return -1;

	if (msg.msg_controllen) {
		struct cmsghdr	*cmsg;
		unsigned int	left, clen;

		/* Note: the Linux CMSG_{FIRST,NXT}HDR macros are
		 * somewhat broken when it comes to receiving more
		 * than one cmsg */
		cmsg = (struct cmsghdr *) msg.msg_control;
		left = msg.msg_controllen;
		while (cmsg != NULL) {
			if (left < sizeof(*cmsg))
				break;
			clen = CMSG_ALIGN(cmsg->cmsg_len);
			if (clen > left)
				break;
			left -= clen;

#if 0
			syslog(LOG_DEBUG, "CMSG lvl=%d type=%d len=%d",
				cmsg->cmsg_level, cmsg->cmsg_type,
				cmsg->cmsg_len);
#endif
			if (cmsg->cmsg_level != SOL_SOCKET)
				goto next;
			if (cmsg->cmsg_type == SCM_RIGHTS) {
				int	*p, n, m = 0;

				n = cmsg->cmsg_len / CMSG_LEN(sizeof(int));
				p = (int *) CMSG_DATA(cmsg);
				if (n)
					conn->passfd = p[m++];
				while (m < n)
					close(p[m++]);
			}
		next:	cmsg = (struct cmsghdr *) ((caddr_t) cmsg
							+ cmsg->cmsg_len);
		}
	}

	return len;
}
int
rsm_recv(struct conn *conn, char *buffer, unsigned int size)
{
	unsigned int	have, want;
	int		len = 0;
	char		c;

	buffer[0] = '\0';
	have = strlen(conn->buffer);
	want = sizeof(conn->buffer);

	if (have >= want - 1)
		return -1;

	while (have < want - 1) {
		len = rsm_recv_char(conn, &c);

		if (len == 0 || (len < 0 && errno == EAGAIN))
			return 0;

		if (len < 0) {
			/* log("read error on socket: %m"); */
			return -1;
		}

		conn->buffer[have++] = c;
		conn->buffer[have] = '\0';
		if (c == '\n')
			break;
	}

	if (have >= want - 1) {
		/* log("long line from %s, closing connection", conn->user); */
		rsm_printf(conn, "%03d line too long", MSG_ERROR);
		return -1;
	}

	/* Line complete... */
	if (conn->debug)
		printf("%s> %s", conn->user, conn->buffer);

	strncpy(buffer, conn->buffer, size-1);
	buffer[size-1] = '\0';

	memset(conn->buffer, 0, sizeof(conn->buffer));
	return strlen(buffer);
}

/*
 * Send a message to the peer.
 * We don't care about whether we're able to write the full
 * message or not; we don't want to be subject to DoS by
 * a rogue user app that never reads its data.
 */
int
rsm_send(struct conn *conn, const char *mesg, size_t len)
{
	struct { struct cmsghdr	cmsg; int fd; } control;
	struct sigaction act;
	struct msghdr	msg;
	struct iovec	iov;
	int		oerrno;

	memset(&msg, 0, sizeof(msg));
	msg.msg_iov	= &iov;
	msg.msg_iovlen	= 1;
	iov.iov_base	= (char *) mesg;
	iov.iov_len	= len;

	if (conn->passfd >= 0) {
		/* Do the file descriptor passing fandango */
		msg.msg_controllen = CMSG_SPACE(sizeof(int));
		msg.msg_control	= &control;

		control.cmsg.cmsg_len = CMSG_LEN(sizeof(int));
		control.cmsg.cmsg_level = SOL_SOCKET;
		control.cmsg.cmsg_type = SCM_RIGHTS;
		*(int *) CMSG_DATA(&control.cmsg) = conn->passfd;
	}

	/* ignore sigpipe */
	memset(&act, 0, sizeof(act));
	act.sa_handler = SIG_IGN;
	sigaction(SIGPIPE, &act, &act);

	oerrno = 0;
	if (sendmsg(conn->fd, &msg, 0) < 0) {
		oerrno = errno;
		if (conn->debug)
			printf("sendmsg failed: %m\n");
	}

	if (conn->passfd >= 0) {
		close(conn->passfd);
		conn->passfd = -1;
	}

	sigaction(SIGPIPE, &act, NULL);
	if (oerrno) {
		errno = oerrno;
		return -1;
	}

	return 0;
}

/*
 * Enqueue a file descriptor that should be passed to the peer
 */
void
rsm_queue_fd(struct conn *conn, int fd)
{
	conn->passfd = fd;
}

/*
 * Send a message to the client, printf style
 */
int
rsm_printf(struct conn *conn, const char *fmt, ...)
{
	int	res;
	va_list	ap;

	va_start(ap, fmt);
	res = rsm_vprintf(conn, fmt, ap);
	va_end(ap);

	return res;
}

/*
 * Send a message to the client, vprintf style
 */
int
rsm_vprintf(struct conn *conn, const char *fmt, va_list ap)
{
	char	buffer[1024];

	vsnprintf(buffer, sizeof(buffer)-2, fmt, ap);
	strcat(buffer, "\n");

	if (conn->debug)
		printf("< %s", buffer);
	return rsm_send(conn, buffer, strlen(buffer));
}

/*
 * Receive response
 * This function will deal with a single line response only
 */
int
rsm_recv_response(struct conn *conn)
{
	char	respbuf[256], *sp;
	int	code;

	do {
		if (rsm_recv(conn, respbuf, sizeof(respbuf)) < 0)
			return -1;
		if (!isdigit(respbuf[0])) {
			errno = EINVAL;
			return -1;
		}
		code = strtoul(respbuf, &sp, 10);
	} while (*sp == '-');

	return code;
}

void
rsm_close_stream(struct conn *conn)
{
	if (conn->fd >= 0)
		close(conn->fd);
	conn->fd = -1;
}

void
rsm_close(struct conn *conn)
{
	if (conn) {
		rsm_close_stream(conn);
		free(conn);
	}
}
