///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "2D_D.h"
#include "rheolef/piola.h"
#include "rheolef/tensor4.h"
#include "rheolef/ublas_matrix_range.h"
//          [    u0      u1      u2   ]
//
// [ v0 ]   [ A_00      A_01    A_02  ]
// [ v1 ] = [ A_10      A_11    A_12  ]
// [ v2 ]   [ A_20      A_21    A_22  ]
//
//  In the axisymmetric case (r,z):
//      2D_D(0,0) += 2 ur/r vr/r r dr dz 
//                 = 2 ur vr (1/r) dr dz
//
//  In the surfacic case, have also a curvature factor C
//  that depends on the projector P = I - n*n
//
namespace rheolef {
using namespace std;
using namespace ublas;

// curvature factor
template<class T>
static
void
compute_tensor_c (
    const tensor_basic<T>&      invDFt, 
    const tensor_basic<T>&      P, 
    size_t             d,
    size_t             map_d,
    bool               is_on_surface,
    tensor4_basic<T>&           C)
{
 typedef geo_element::size_type size_type;
 C = 0;
 if (! is_on_surface) {
   // ------------------------------------------
   //  calcul de C cas classique
   // ------------------------------------------
   for (size_type r = 0; r < map_d; r++) 
   for (size_type t = 0; t < map_d; t++) {
     for (size_type k = 0; k < d; k++) {
       T sum = 0;
       for (size_type i = 0; i < d; i++) {
         sum += 2*invDFt(i,r)*invDFt(i,t);
       }
       C(k,r,k,t) += sum;
     }
     for (size_type k = 0; k < d; k++)
     for (size_type p = 0; p < d; p++) {
       C(k,r,p,t) += 2*invDFt(p,r)*invDFt(k,t);
     }
   }
 } else {
   // ----------------------------------------------
   // cas surfacique (d > map_d or banded level set)
   // ----------------------------------------------
   // calcul de B 
   tensor4_basic<T> B;
   for (size_type i = 0; i < d; i++)
   for (size_type j = 0; j < d; j++) {
       for (size_type k = 0; k < d;     k++)
       for (size_type r = 0; r < map_d; r++) {
           T sum = 0;
           for (size_type l = 0; l < d; l++) {
	      sum += (P(i,k)*P(l,j) + P(i,l)*P(k,j))*invDFt(l,r);
           }
	   B(i,j,k,r) = sum;
      	}
   }
   // calcul de C cas surfacique  
   for (size_type k = 0; k < d;     k++) 
   for (size_type r = 0; r < map_d; r++) {
       for (size_type p = 0; p < d;     p++)
       for (size_type t = 0; t < map_d; t++) {
           T sum = 0;
           for (size_type i = 0; i < d; i++)
           for (size_type j = 0; j < d; j++) {
	       sum += B(i,j,k,r)*B(i,j,p,t);
           }
           C(k,r,p,t) = sum;
       }
   }
 }
}
template<class T, class M>
void
_2D_D<T,M>::operator() (const geo_element& K, matrix<T>& ak) const
{
  std::vector<size_type> dis_inod_K;
  std::vector<size_type> dis_inod_L;
  base::_omega.dis_inod (K, dis_inod_K);
  reference_element hat_K   = K.variant();
  reference_element tilde_L = hat_K;
  if (base::is_on_band()) {
    // see nfem/plib/form_element.cc : notations are explained in function build_grad_grad()
    size_type first_dis_ie = base::_omega.sizes().ownership_by_dimension [K.dimension()].first_index();
    size_type K_ie = K.dis_ie() - first_dis_ie;
    size_type L_ie = base::get_band().sid_ie2bnd_ie (K_ie);
    const geo_element& L = base::get_band().band() [L_ie];
    base::get_band().band().dis_inod (L, dis_inod_L);
    tilde_L = L.variant();
  }
  size_type d       = base::coordinate_dimension();
  size_type K_map_d = hat_K.dimension();
  size_type L_map_d = tilde_L.dimension();
  bool is_on_surface = (K_map_d < d);
  size_type nx = base:: get_first_basis().size(tilde_L);
  size_type ny = base::get_second_basis().size(tilde_L);
  ak.resize (d*ny, d*nx);
  ak.clear();
  typename quadrature<T>::const_iterator
    first = base::_quad.begin(hat_K),
    last  = base::_quad.end  (hat_K);
  tensor_basic<T> DF, DG, invDG, invDGt, P;
  tensor4_basic<T> C;
  for (size_type q = 0; first != last; first++, q++) {
    rheolef::jacobian_piola_transformation (base::_omega, base::_piola_table, K, dis_inod_K, q, DF);
    T det_DF = rheolef::det_jacobian_piola_transformation (DF, d, K_map_d);
    if (! base::is_on_band()) {
      invDG = pseudo_inverse_jacobian_piola_transformation (DF, d, K_map_d);
      base::_bx_table.evaluate_grad (hat_K, q);
      base::_by_table.evaluate_grad (hat_K, q);
    } else {
      point_basic<T> xq = rheolef::piola_transformation (base::_omega, base::_piola_table, hat_K, dis_inod_K, q);
      point_basic<T> tilde_xq = rheolef::inverse_piola_transformation (base::get_band().band(), tilde_L, dis_inod_L, xq);
      rheolef::jacobian_piola_transformation (base::get_band().band(), base::_piola_table, tilde_L, dis_inod_L, tilde_xq, DG);
      invDG = pseudo_inverse_jacobian_piola_transformation (DG, d, L_map_d);
      base::_bx_table.evaluate_grad (tilde_L, tilde_xq);
      base::_by_table.evaluate_grad (tilde_L, tilde_xq);
    }
    invDGt = trans(invDG);
    if (is_on_surface) map_projector (DF, d, K_map_d, P); // P := I - n*n
    compute_tensor_c (invDGt, P, d, L_map_d, is_on_surface, C);
    T wq = 1;
    wq *= base::is_weighted() ? base::weight(K,q) : T(1);
    wq *= base::weight_coordinate_system (K, dis_inod_K, q);
    wq *= det_DF;
    wq *= (*first).w;
    typename base::quad_const_iterator_grad
        grad_phi       = base::_by_table.begin_grad(),
        last_grad_phi  = base::_by_table.end_grad(),
        first_grad_psi = base::_bx_table.begin_grad(),
        last_grad_psi  = base::_bx_table.end_grad();
    for (size_type i = 0; grad_phi != last_grad_phi; grad_phi++, i++) {
      const point_basic<T>& grad_phi_i = *grad_phi;
      typename base::quad_const_iterator_grad grad_psi = first_grad_psi;
      for (size_type j = 0; grad_psi != last_grad_psi; grad_psi++, j++) {
        const point_basic<T>& grad_psi_j = *grad_psi;
        for (size_type k = 0; k < d; k++)
        for (size_type p = 0; p < d; p++)
        for (size_type r = 0; r < L_map_d; r++)
        for (size_type t = 0; t < L_map_d; t++) {
	  ak(i+k*ny,j+p*nx) += 0.5*C(k,r,p,t)*wq*grad_phi_i[r]*grad_psi_j[t];
	}
      }
    }
  }
  if (base::coordinate_system() == space_constant::cartesian) return;
  // ------------------------------------------------
  // axisymmetric: add the (ur/r) (vr/r) r dr dz term
  //   as : ur vr (1/r) dr dz
  // i.e. using the dual `1/r' extra weight
  // ------------------------------------------------
  ublas::matrix<T> m_invr;
  base::set_use_coordinate_system_dual_weight (true);
  base::build_scalar_mass (K, m_invr);
  base::set_use_coordinate_system_dual_weight (false);
  size_type k = 0;
  if (base::coordinate_system() == space_constant::axisymmetric_zr) k++;
  matrix_range<matrix<T> > dk (ak, range(k*ny,(k+1)*ny), range(k*nx,(k+1)*nx));
  dk += 2.*m_invr;
}
template<class T, class M>
void
_2D_D<T,M>::initialize () const
{
  size_type d = base::get_first_space().get_geo().dimension();
  check_macro (
	d == base::get_first_space().size() && 
	d == base::get_second_space().size(),
	"unsupported non-vectorial space for `2D_D' form");

  if (base::coordinate_system() != space_constant::cartesian) {
    base::set_n_derivative(0); // axisym: have also zero order terms
  } else  {
    base::set_n_derivative(2);
  }
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class _2D_D<Float,sequential>;

#ifdef _RHEOLEF_HAVE_MPI
template class _2D_D<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
