///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/piola.h"

namespace rheolef {

// ------------------------------------------
// inverse piola transformation
// F_K^{-1} : K --> hat(K)
//            x --> hat(x)
// TODO: non-linear case
// ----------------------------------------------------------------------------
template<class T>
static
inline
point_basic<T> 
inv_piola_e (
  const point_basic<T>& x, 
  const point_basic<T>& a, 
  const point_basic<T>& b) 
{
  return point_basic<T>((x[0]-a[0])/(b[0]-a[0]));
}
template<class T>
static
inline
point_basic<T> 
inv_piola_t (
  const point_basic<T>& x, 
  const point_basic<T>& a, 
  const point_basic<T>& b, 
  const point_basic<T>& c) 
{
  T t9 = 1/(-b[0]*c[1]+b[0]*a[1]+a[0]*c[1]+c[0]*b[1]-c[0]*a[1]-a[0]*b[1]);
  T t11 = -a[0]+x[0];
  T t15 = -a[1]+x[1];
  return point_basic<T>((-c[1]+a[1])*t9*t11-(-c[0]+a[0])*t9*t15,
                        ( b[1]-a[1])*t9*t11-( b[0]-a[0])*t9*t15);
}
template<class T>
static
inline
point_basic<T> 
inv_piola_T (
  const point_basic<T>& x, 
  const point_basic<T>& a, 
  const point_basic<T>& b, 
  const point_basic<T>& c, 
  const point_basic<T>& d) 
{
  tensor_basic<T> A;
  point_basic<T> ax;
  for (size_t i = 0; i < 3; i++) {
    ax[i]  = x[i]-a[i];
    A(i,0) = b[i]-a[i];
    A(i,1) = c[i]-a[i];
    A(i,2) = d[i]-a[i];
  }
  tensor_basic<T> inv_A;
  bool is_singular = ! invert_3x3 (A, inv_A);
  check_macro(!is_singular, "inv_piola: singular transformation in a tetrahedron");
  point_basic<T> hat_x = inv_A*ax;
  return hat_x; 
}
template<class T, class M>
point_basic<T>
inverse_piola_transformation (
  const geo_basic<T,M>&         omega,
  reference_element             hat_K,
  const std::vector<size_t>&    dis_inod,
  const point_basic<T>&         x)
{
  check_macro (omega.order() == 1, "inverse piola: mesh order > 1: not yet");
  switch (hat_K.variant()) {
   case reference_element::e: return inv_piola_e (x, omega.dis_node(dis_inod [0]),
                                                     omega.dis_node(dis_inod [1])); 
   case reference_element::t: return inv_piola_t (x, omega.dis_node(dis_inod [0]),
                                                     omega.dis_node(dis_inod [1]),
                                                     omega.dis_node(dis_inod [2]));
   case reference_element::T: return inv_piola_T (x, omega.dis_node(dis_inod [0]),
                                                     omega.dis_node(dis_inod [1]),
                                                     omega.dis_node(dis_inod [2]),
                                                     omega.dis_node(dis_inod [3]));
   default: {
	fatal_macro ("inverse piola: non-linear transformation on '"<<hat_K.name()<<"' element: not yet");
        return point_basic<T>();
   }
  }
}
// calcul P = I - nxn
template<class T>
void
map_projector (
  const tensor_basic<T>& DF,
  size_t d,
  size_t map_d,
  tensor_basic<T>& P)
{
  point_basic<Float> n;
  switch (map_d) {
    case 1: {
      point_basic<Float> t = DF.col(0);
      check_macro (d == map_d+1, "unexpected dimension map_d="<<map_d<<" and d="<<d);
      n = point_basic<T>(t[1],-t[0]);
      break;
    }
    case 2: {
      point_basic<Float> t0 = DF.col(0);
      point_basic<Float> t1 = DF.col(1);
      n = vect(t0,t1);
      break;
    }
    default:
      error_macro ("unexpected dimension "<<map_d);
  }
  n = n/norm(n);
  for (size_t l = 0; l < d; l++) {
    for (size_t m = 0; m < d; m++) {
      P(l,m) = - n[l]*n[m];
    }
    P(l,l) += 1;
  }
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciation2(T,M) 					\
template point_basic<T> inverse_piola_transformation (			\
  const geo_basic<T,M>&         omega,					\
  reference_element             hat_K,					\
  const std::vector<size_t>&    dis_inod,				\
  const point_basic<T>&         x);					\

#define _RHEOLEF_instanciation1(T) 					\
template void map_projector (						\
  const tensor_basic<T>& DF,						\
  size_t d,								\
  size_t map_d,								\
  tensor_basic<T>& P);

_RHEOLEF_instanciation1(Float)
_RHEOLEF_instanciation2(Float,sequential)
#ifdef _RHEOLEF_HAVE_MPI
_RHEOLEF_instanciation2(Float,distributed)
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
