///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/space.h"
#include "rheolef/space_mult.h"
#include "rheolef/piola.h"
#include <sstream>

namespace rheolef {

// ======================================================================
// allocators
// ======================================================================
template <class T, class M>
space_constitution_rep<T,M>::space_constitution_rep (
    const geo_basic<T,M>& omega,
    std::string approx,
    std::string valued)
  : _ownership(),
    _start_by_component(),
    _valued_tag(space_constant::scalar),
    _is_hier(false),
    _scalar_constit(),
    _hier_constit()
{
  if (valued == "scalar") {
    _scalar_constit = scalar_type (omega, approx);
  } else { // multi-valued:
    _is_hier = true;
    size_type d = omega.dimension();
    _valued_tag      = space_constant::valued_tag (valued);
    size_type n_comp = space_constant::n_component (_valued_tag, d, omega.coordinate_system());
    _hier_constit.resize(n_comp);
    space_constitution<T,M> x (omega, approx);
    for (size_type i_comp = 0; i_comp < n_comp; i_comp++) {
      _hier_constit [i_comp] = x;
    }
  }
  initialize();
}
// build a space::constitution hierrachy from a list of spaces
template <class T, class M>
space_constitution_rep<T,M>::space_constitution_rep(const space_mult_list<T,M>& expr)
  : _ownership(),
    _start_by_component(),
    _valued_tag(space_constant::mixed),
    _is_hier(true),
    _scalar_constit(),
    _hier_constit(expr.size())
{
  typename space_constitution_rep<T,M>::hierarchy_type::iterator iter_constit = _hier_constit.begin();
  for (typename space_mult_list<T,M>::const_iterator iter = expr.begin(), last = expr.end(); iter != last; ++iter, ++iter_constit) {
    const space_basic<T,M>& Xi = *iter;
    *iter_constit = Xi.get_constitution();
  }
  initialize();
}
template <class T, class M>
void
space_constitution_rep<T,M>::initialize()
{
  _ownership = distributor (internal_dis_ndof(), internal_comm(), internal_ndof());
  init_start_by_component();
}
template <class T, class M>
bool
space_constitution_rep<T,M>::operator== (const space_constitution_rep<T,M>& V2) const
{
  if (_is_hier != V2._is_hier) return false;
  if (!_is_hier) return (_scalar_constit == V2._scalar_constit);
  // here, two hierarchies:
  if (_hier_constit.size() != V2._hier_constit.size()) return false;
  for (const_iterator iter1 = _hier_constit.begin(), iter2 = V2._hier_constit.begin(), last1 = _hier_constit.end(); iter1 != last1; ++iter1, ++iter2) {
    if (! (*iter1).data().operator==((*iter2).data())) { // recursive call
      return false;
    }
  }
  return true;
}
template <class T, class M>
communicator
space_scalar_constitution_rep<T,M>::internal_comm () const
{
  if (! is_initialized()) { return communicator(); }
  return get_geo().comm();
}
template <class T, class M>
communicator
space_constitution_rep<T,M>::internal_comm () const
{
  if (!_is_hier) {
    return _scalar_constit.get_geo().comm();
  }
  for (const_iterator iter = _hier_constit.begin(), last = _hier_constit.end(); iter != last; ++iter) {
    const space_constitution<T,M>& constit = *iter;
    return constit.data().internal_comm(); // recursive call
  } 
  // empty hierarchical list:
  return communicator();
}
template <class T, class M>
typename space_scalar_constitution_rep<T,M>::size_type
space_scalar_constitution_rep<T,M>::internal_ndof () const
{
  if (! is_initialized()) { return 0; }
  return get_numbering().ndof (get_geo().sizes(), get_geo().map_dimension());
}
template <class T, class M>
typename space_constitution_rep<T,M>::size_type
space_constitution_rep<T,M>::internal_ndof () const
{
  if (!_is_hier) {
    return _scalar_constit.internal_ndof();
  }
  size_type ndof = 0;
  for (const_iterator iter = _hier_constit.begin(), last = _hier_constit.end(); iter != last; ++iter) {
    const space_constitution<T,M>& constit = *iter;
    ndof += constit.data().internal_ndof(); // recursive call
  }
  return ndof;
}
template <class T, class M>
typename space_scalar_constitution_rep<T,M>::size_type
space_scalar_constitution_rep<T,M>::internal_dis_ndof () const
{
  if (! is_initialized()) { return 0; }
  return get_numbering().dis_ndof (get_geo().sizes(), get_geo().map_dimension());
}
template <class T, class M>
typename space_constitution_rep<T,M>::size_type
space_constitution_rep<T,M>::internal_dis_ndof () const
{
  if (!_is_hier) {
    return _scalar_constit.internal_dis_ndof();
  }
  size_type dis_ndof = 0;
  for (const_iterator iter = _hier_constit.begin(), last = _hier_constit.end(); iter != last; ++iter) {
    const space_constitution<T,M>& constit = *iter;
    dis_ndof += constit.data().internal_dis_ndof(); // recursive call
  }
  return dis_ndof;
}
// ======================================================================
// output
// ======================================================================
template<class T, class M>
std::ostream&
put (std::ostream& os, const space_constitution<T,M>& constit, size_t level = 0)
{
  typedef distributor::size_type size_type;
  if (! constit.is_hierarchical()) {
    const space_scalar_constitution<T,M>& scalar_constit = constit.get_scalar();
    os << scalar_constit.get_numbering().name()
       << "(" << scalar_constit.get_geo().name() << ")";
    return os;
  }
  if (constit.valued_tag() == space_constant::vector ||
      constit.valued_tag() == space_constant::tensor) {
    assert_macro (constit.get_hierarchy().size() >= 1, "invalid space_constitution");
    const space_constitution<T,M>& base_constit = *(constit.get_hierarchy().begin());
    std::string spec = (constit.valued_tag() == space_constant::vector) ? "vector" : "tensor";
    os << spec << "(";
    put (os, base_constit, level+1);
    os << ")";
    return os;
  }
  if (level > 0) os << "(";
  typename space_constitution<T,M>::hierarchy_type::const_iterator x = constit.get_hierarchy().begin();
  for (size_type i = 0, n = constit.get_hierarchy().size(); i < n; i++) {
    const space_constitution<T,M>& xi = x[i];
    put (os, xi, level+1); // recursive call
    if (i+1 < n) { os << "*"; }
  }
  if (level > 0) os << ")";
  return os;
}
template<class T, class M>
std::ostream&
operator<< (std::ostream& os, const space_constitution<T,M>& constit)
{
  return put (os, constit);
}
template<class T, class M>
std::string 
space_constitution<T,M>::stamp() const
{
  std::ostringstream ostrstr;
  ostrstr << *this;
  return ostrstr.str ();
}
// ======================================================================
// do_act: block & unblock on a domain
// ======================================================================
template <class T, class M>
void
space_scalar_constitution_rep<T,M>::do_act (const space_act& act)
{
  check_macro (get_numbering().is_continuous(),
    "try to [un]block domain `" << act.get_domain_name() << "' on discontinuous space("
    << get_numbering().name()<<"(" <<_omega.name()<<"))");
  _acts.push_back (act);
}
template <class T, class M>
void
space_constitution_rep<T,M>::do_act (const space_act& act)
{
  typedef distributor::size_type size_type;
  if (! is_hierarchical()) {
    space_scalar_constitution<T,M>& scalar_constit = get_scalar();
    scalar_constit.do_act (act);
    return;
  }
  // hierarchical case:
  for (iterator iter = _hier_constit.begin(), last = _hier_constit.end(); iter != last; ++iter) {
    space_constitution<T,M>& constit = *iter;
    constit.do_act (act); // recursive call
  }
}
// -----------------------------------------------------------------------
// loop on domains: mark blocked dofs (called by space::freeze)
// -----------------------------------------------------------------------
template <class T, class M>
void
space_scalar_constitution_rep<T,M>::build_blocked_flag (
  array<size_type,M>& blocked_flag, // array<bool,M> not supported
  const distributor&  comp_ownership,
  const distributor&  start_by_component) const
{
  check_macro (get_numbering().name() != "", "space with undefined finite element method cannot be used");
  std::vector<size_type> comp_dis_idof_t;
  distributor dof_ownership = blocked_flag.ownership();
  size_type   dis_ndof      = dof_ownership.dis_size();
  bool prev_act = space_act::block;
  for (const_iterator iter = begin(), last = end(); iter != last; iter++) {
    typename space_act::act_type act = (*iter).get_act();
    const std::string&      dom_name = (*iter).get_domain_name();
    // sync all blocked_flags when there is an alternance of block and unblock (bug fixed)
    if (prev_act != act) {
      blocked_flag.dis_entry_assembly();
      prev_act = act;
    }
    const domain_indirect_basic<M>& dom = _omega.get_domain_indirect(dom_name);
    size_type dom_dim = dom.map_dimension();
    distributor ige_ownership = _omega.geo_element_ownership (dom_dim);
    size_type   first_dis_ige = ige_ownership.first_index();
    size_type    last_dis_ige = ige_ownership.last_index();
    bool blk = (act == space_act::block) ? true : false;
    for (size_type ioige = 0, noige = dom.size(); ioige < noige; ioige++) {
      size_type ige     = dom.oige (ioige).index();
      size_type dis_ige = ige + first_dis_ige;
      const geo_element& S = _omega.get_geo_element (dom_dim, ige);
      get_numbering().dis_idof (_omega.sizes(), S, comp_dis_idof_t);
      for (size_type loc_idof = 0, loc_ndof = comp_dis_idof_t.size(); loc_idof < loc_ndof; loc_idof++) {
          size_type comp_dis_idof = comp_dis_idof_t [loc_idof];
          size_type iproc = comp_ownership.find_owner (comp_dis_idof);
          size_type first_comp_dis_idof = comp_ownership.first_index (iproc);
          assert_macro (comp_dis_idof >= first_comp_dis_idof, "unexpected comp_dis_idof");
          size_type comp_idof = comp_dis_idof - first_comp_dis_idof;
          size_type comp_start_idof = start_by_component.size(iproc);
          size_type idof = comp_start_idof + comp_idof;
          size_type first_dis_idof = dof_ownership.first_index(iproc);
          size_type dis_idof = first_dis_idof + idof;
          assert_macro (dis_idof < dis_ndof, "unexpected dis_idof");
          blocked_flag.dis_entry (dis_idof) = blk;
      }
    }
  }
}
template <class T, class M>
void
space_constitution_rep<T,M>::build_blocked_flag_recursive (
  array<size_type,M>&             blocked_flag, // array<bool,M> not supported
  const std::vector<distributor>& start_by_component,
  size_type&                      i_comp) const
{
  if (! is_hierarchical()) {
    const space_scalar_constitution<T,M>& scalar_constit = get_scalar();
    scalar_constit.data().build_blocked_flag (blocked_flag, ownership(), start_by_component [i_comp]);
    i_comp++;
    return;
  }
  // hierarchical case:
  for (const_iterator iter = _hier_constit.begin(), last = _hier_constit.end(); iter != last; ++iter) {
    const space_constitution<T,M>& constit = *iter;
    constit.data().build_blocked_flag_recursive (blocked_flag, start_by_component, i_comp); // recursive call
  }
}
template <class T, class M>
array<typename space_constitution_rep<T,M>::size_type,M>
space_constitution_rep<T,M>::build_blocked_flag() const
{
  array<size_type,M> blocked_flag (ownership(), 0); // array<bool,M> not supported
  size_type i_comp = 0;
  build_blocked_flag_recursive (blocked_flag, _start_by_component, i_comp);
  blocked_flag.dis_entry_assembly();
  return blocked_flag;
}
// ======================================================================
// accessors
// ======================================================================
template <class T, class M>
const geo_basic<T,M>&
space_constitution_rep<T,M>::get_geo() const
{
  if (! is_hierarchical()) {
    return get_scalar().get_geo();
  }
  check_macro (_valued_tag == space_constant::vector || _valued_tag == space_constant::tensor,
	"get_geo: undefined for heterogeneous space products");
  // each component of homogeneous tensorial space product have the same mesh:
  assert_macro (_hier_constit.size() > 0, "get_geo: empty vectorial/tensorial space product");
  const_iterator iter = _hier_constit.begin();
  const space_constitution<T,M>& constit = (*iter);
  return constit.get_scalar().get_geo();
}
template <class T, class M>
const numbering<T,M>&
space_constitution_rep<T,M>::get_numbering() const
{
  if (! is_hierarchical()) {
    return get_scalar().get_numbering();
  }
  check_macro (_valued_tag == space_constant::vector || _valued_tag == space_constant::tensor,
	"get_numbering: undefined for heterogeneous space products");
  // each component of homogeneous tensorial space product have the same basis:
  assert_macro (_hier_constit.size() > 0, "get_numbering: empty vectorial/tensorial space product");
  const_iterator iter = _hier_constit.begin();
  const space_constitution<T,M>& constit = (*iter);
  return constit.get_scalar().get_numbering();
}
// ======================================================================
// ios accessors
// ======================================================================
template <class T, class M>
typename space_constitution_rep<T,M>::size_type
space_constitution_rep<T,M>::ios_ndof () const
{
  if (!_is_hier) {
    communicator comm =_scalar_constit.get_geo().comm();
    size_type dis_ndof =_scalar_constit.get_numbering().dis_ndof (_scalar_constit.get_geo().sizes(), _scalar_constit.get_geo().map_dimension());
    // TODO: memorize ios_ownership : avoid comms & risks of errors
    distributor ios_ownership (dis_ndof, comm, distributor::decide);
    return ios_ownership.size();
  }
  size_type ios_ndof = 0;
  for (const_iterator iter = _hier_constit.begin(), last = _hier_constit.end(); iter != last; ++iter) {
    const space_constitution<T,M>& constit = *iter;
    ios_ndof += constit.ios_ndof(); // recursive call
  }
  return ios_ndof;
}
template <class T, class M>
void
space_constitution_rep<T,M>::set_ios_permutation_recursion (
	array<size_type,M>& idof2ios_dis_idof, 
	size_type&          comp_start_idof,
	size_type&          comp_start_dis_idof) const
{
  if (!_is_hier) {
    // non-hierarchical case:
    array<size_type,M> comp_idof2ios_dis_idof; 
    array<size_type,M> comp_ios_idof2dis_idof; 
    // TODO: avoid allocating comp_idof2ios_dis_idof: numbering supports directly comp_start_{dis_}idof
    // TODO: avoid invert permut here: numbering support no invert perm arg
    _scalar_constit.get_numbering().set_ios_permutations (_scalar_constit.get_geo(), comp_idof2ios_dis_idof, comp_ios_idof2dis_idof);
    // then shift
    size_type     comp_ndof = comp_idof2ios_dis_idof.size();
    size_type comp_dis_ndof = comp_idof2ios_dis_idof.dis_size();
    size_type          ndof = idof2ios_dis_idof.size();
    size_type      dis_ndof = idof2ios_dis_idof.dis_size();
    for (size_type comp_idof = 0; comp_idof < comp_ndof; comp_idof++) {
      size_type comp_ios_dis_idof = comp_idof2ios_dis_idof [comp_idof];
      size_type ios_dis_idof      = comp_start_dis_idof + comp_ios_dis_idof;
      size_type idof              = comp_start_idof + comp_idof;
      assert_macro (idof < ndof, "unexpected idof="<<idof<<" out of range [0:"<<ndof<<"[");
      assert_macro (ios_dis_idof < dis_ndof, "unexpected ios_dis_idof="<<ios_dis_idof<<" out of range [0:"<<dis_ndof<<"[");
      idof2ios_dis_idof [idof]    = ios_dis_idof;
    }
    comp_start_idof     +=     comp_ndof;
    comp_start_dis_idof += comp_dis_ndof;
    return;
  }
  // hierarchical case:
  for (const_iterator iter = _hier_constit.begin(), last = _hier_constit.end(); iter != last; ++iter) {
    const space_constitution<T,M>& constit = *iter;
    constit.set_ios_permutation_recursion (idof2ios_dis_idof, comp_start_idof, comp_start_dis_idof); // recursive call
  }
}
template <class T, class M>
void
space_constitution_rep<T,M>::set_ios_permutations (
    array<size_type,M>& idof2ios_dis_idof,
    array<size_type,M>& ios_idof2dis_idof) const
{
  if (!_is_hier) {
    // non-hierarchical case: direct
    _scalar_constit.get_numbering().set_ios_permutations (_scalar_constit.get_geo(), idof2ios_dis_idof, ios_idof2dis_idof);
    return;
  }
  // hierarchical case:
  // ----------------------------------------------
  // 1) compute permutation
  // ----------------------------------------------
  communicator comm1  = comm();
  distributor dof_ownership = ownership();
  size_type ndof1     = dof_ownership.size();
  size_type dis_ndof1 = dof_ownership.dis_size();
  idof2ios_dis_idof.resize (dof_ownership, std::numeric_limits<size_type>::max());
  size_type comp_start_idof     = 0;
  size_type comp_start_dis_idof = 0;
  set_ios_permutation_recursion (idof2ios_dis_idof, comp_start_idof, comp_start_dis_idof);
  // ----------------------------------------------
  // 2) invert permutation into ios_idof2dis_idof
  // ----------------------------------------------
  size_type ios_ndof1 = ios_ndof();
  distributor ios_dof_ownership (dis_ndof1, idof2ios_dis_idof.comm(), ios_ndof1);
  ios_idof2dis_idof.resize (ios_dof_ownership, std::numeric_limits<size_type>::max());
  idof2ios_dis_idof.reverse_permutation (ios_idof2dis_idof);
}
// ----------------------------------------------------------------------------
// used by field & form_assembly
// ----------------------------------------------------------------------------
template <class T, class M>
void
space_constitution_rep<T,M>::dis_idof (const geo_element& K, std::vector<size_type>& dis_idof_t) const
{
  size_type n_comp = size();
  switch (valued_tag()) {
    case space_constant::scalar: {
      get_numbering().dis_idof (get_geo().sizes(), K, dis_idof_t);
      return;
    }
    case space_constant::vector:
    case space_constant::tensor: {
      // LIMITATION: assume homogeneous vector or tensor valued
      // multi-component case: shift from i_comp=0
      assert_macro (_hier_constit.size() > 0, "unexpected empty vector valued");
      distributor comp_ownership = _hier_constit [0].ownership();
      size_type comp_dis_ndof = get_numbering().dis_ndof (get_geo().sizes(), get_geo().map_dimension());
      std::vector<size_type> comp_dis_idof_t;
      get_numbering().dis_idof (get_geo().sizes(), K, comp_dis_idof_t);
      size_type loc_ndof = comp_dis_idof_t.size();
      dis_idof_t.resize (n_comp*loc_ndof);
      for (size_type loc_idof = 0; loc_idof < loc_ndof; loc_idof++) {
        size_type comp_dis_idof = comp_dis_idof_t [loc_idof];
        size_type iproc               = comp_ownership.find_owner (comp_dis_idof);
        size_type comp_first_dis_idof = comp_ownership.first_index(iproc);
        assert_macro (comp_dis_idof >= comp_first_dis_idof, "unexpected comp_dis_idof");
        size_type comp_idof = comp_dis_idof - comp_first_dis_idof;
        size_type first_dis_idof = ownership().first_index(iproc);
        // ICI
        for (size_type i_comp = 0; i_comp < n_comp; i_comp++) {
          size_type comp_start_idof = _start_by_component [i_comp].size(iproc);
          size_type idof = comp_start_idof + comp_idof;
          size_type dis_idof = first_dis_idof + idof;
          dis_idof_t [i_comp*loc_ndof + loc_idof] = dis_idof;
        }
      }
      return;
    }
    default: {
      fatal_macro ("dis_idof: hybrid multi-component: not yet");
    }
  }
}
// -----------------------------------------------------------------------
// comp_start_idof = _start_by_component [i_comp].size (iproc);
// => build _start_by_component : used by set_external_dof()
// -----------------------------------------------------------------------
template <class T, class M>
typename space_constitution_rep<T,M>::size_type
space_constitution_rep<T,M>::n_component_recursive() const
{
  if (!_is_hier) {
    return 1;
  }
  size_type n_comp = 0;
  for (const_iterator iter = _hier_constit.begin(), last = _hier_constit.end(); iter != last; ++iter) {
    n_comp += (*iter).data().n_component_recursive(); // recursive call
  }
  return n_comp;
}
template <class T, class M>
void
space_constitution_rep<T,M>::init_start_by_component (
	std::vector<distributor>& start_by_component,
	size_type&                i_comp,
	const communicator&	  comm,
	size_type&                comp_start_idof,
	size_type&                comp_start_dis_idof) const
{
  if (!_is_hier) {
    start_by_component [i_comp] = distributor (comp_start_dis_idof, comm, comp_start_idof);
    i_comp++;
    comp_start_idof     += ndof();
    comp_start_dis_idof += dis_ndof();
    return;
  }
  for (const_iterator iter = _hier_constit.begin(), last = _hier_constit.end(); iter != last; ++iter) {
    (*iter).data().init_start_by_component (start_by_component, i_comp, comm,
					 comp_start_idof, comp_start_dis_idof);
  }
}
template <class T, class M>
void
space_constitution_rep<T,M>::init_start_by_component ()
{
  size_type n_comp_rec = n_component_recursive();
  _start_by_component.resize (n_comp_rec);
  size_type i_comp = 0;
  size_type comp_start_idof = 0;
  size_type comp_start_dis_idof = 0;
  init_start_by_component (_start_by_component, i_comp, comm(), comp_start_idof, comp_start_dis_idof);
}
//  ----------------------------------------------------------------------
// get external idofs: used by space<distributed>::freeze
// ---------------------------------------------------------------,--------
template <class T, class M>
void
space_constitution_rep<T,M>::append_external_dof (
        const geo_basic<T,M>&         dom,
        std::set<size_type>&          ext_dof_set,
        const distributor&            dof_ownership,
        const distributor&            start_by_component) const
{
  // assume a scalar (non-multi valued) case:
  distributor comp_ownership = ownership();
  size_type comp_dis_ndof       = comp_ownership.dis_size();
  std::vector<size_type> comp_dis_idof1;
  for (size_type ie = 0, ne = dom.size(); ie < ne; ie++) {
    const geo_element& K = dom [ie];
    dis_idof (K, comp_dis_idof1);
    for (size_type loc_idof = 0, loc_ndof = comp_dis_idof1.size(); loc_idof < loc_ndof; loc_idof++) {
      size_type comp_dis_idof = comp_dis_idof1 [loc_idof];
      assert_macro (comp_dis_idof < comp_dis_ndof, "idof " << comp_dis_idof1 [loc_idof] << " out of range[0:"<< comp_dis_ndof << "[");
      if (! comp_ownership.is_owned (comp_dis_idof)) {
        size_type iproc = comp_ownership.find_owner (comp_dis_idof);
        size_type comp_first_dis_idof = comp_ownership.first_index(iproc);
        assert_macro (comp_dis_idof >= comp_first_dis_idof, "unexpected comp_dis_idof");
        size_type comp_idof = comp_dis_idof - comp_first_dis_idof;
        size_type comp_start_idof = start_by_component.size(iproc);
        size_type idof = comp_start_idof + comp_idof;
        size_type first_dis_idof = dof_ownership.first_index(iproc);
        size_type dis_idof = first_dis_idof + idof;
        assert_macro (! dof_ownership.is_owned (dis_idof), "unexpected dis_idof="<<dis_idof<<" in range ["
		<< first_dis_idof << ":" << dof_ownership.last_index() << "[");
        ext_dof_set.insert (dis_idof);
      }
    }
  }
}
template <class T, class M>
void
space_constitution_rep<T,M>::compute_external_dofs (
        std::set<size_type>&            ext_dof_set,
        const distributor&              dof_ownership,
	const std::vector<distributor>& start_by_component,
	size_type&                      i_comp) const
{
  if (! is_hierarchical()) {
    // loop on the main mesh
    const space_scalar_constitution<T,M>& scalar_constit = get_scalar();
    append_external_dof (scalar_constit.get_geo(), ext_dof_set, dof_ownership, start_by_component [i_comp]);
    // loop on all domains and insert also external dofs
    if (scalar_constit.get_numbering().is_continuous()) {
      for (size_type idom = 0, ndom = scalar_constit.get_geo().n_domain(); idom < ndom; idom++) {
        const geo_basic<T,M>& dom = scalar_constit.get_geo().get_domain (idom);
        append_external_dof (dom, ext_dof_set, dof_ownership, start_by_component [i_comp]);
      }
    }
    i_comp++;
    return;
  }
  // hierarchical case:
  for (const_iterator iter = _hier_constit.begin(), last = _hier_constit.end(); iter != last; ++iter) {
    (*iter).data().compute_external_dofs (ext_dof_set, dof_ownership, start_by_component, i_comp);
  }
}
template <class T, class M>
void
space_constitution_rep<T,M>::compute_external_dofs (
        std::set<size_type>&          ext_dof_set) const
{
  ext_dof_set.clear();
  size_type i_comp = 0;
  compute_external_dofs (ext_dof_set, ownership(), _start_by_component, i_comp);
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class space_scalar_constitution_rep<Float,sequential>;
template class space_constitution_rep<Float,sequential>;
template class space_constitution<Float,sequential>;
template std::ostream& operator<< (std::ostream&, const space_constitution<Float,sequential>&);

#ifdef _RHEOLEF_HAVE_MPI
template class space_scalar_constitution_rep<Float,distributed>;
template class space_constitution_rep<Float,distributed>;
template class space_constitution<Float,distributed>;
template std::ostream& operator<< (std::ostream&, const space_constitution<Float,distributed>&);
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
