///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/cad.h"
using namespace rheolef;
using namespace std;

// -----------------------------------------------------------------------
// geomview/off output
//   geomview does not supports Bezier triangles
//   and visualizes Bezier patches as planar faces...
//   so uses here planar faces : OFF polygonal geomview format
//
// OFF
// NVertices  NFaces  NEdges
// x y z : for reach vertice
// Nv  v[0] v[1] ... v[Nv-1]  color : for each face
//
// In OFF format, there is a header line with the string OFF by itself.
// The next line contains three integers, the number of vertices, the number
// of facets, and the number of edges.
// Say n is the number of vertices.
// The next n lines each contain three floating-point numbers which are vertex
// coordinates in R^3.
// Say m is the number of facets.
// The next m lines contain the specifications of the facets as sequences of integers.
// The first integer in a facet specification is the number of vertices in that facet, say p.
// The next p integers are the indices of the vertices in order around the boundary
// of the facet. These indices are zero-based and refer to the n coordinates.
// Following these p integers, there are optionally three real numbers between 0 and 1
// which form an RGB color specification. 
//
// TODO: redundant points : shared points are repeated in each domain => reduce
//
// TODO: output one file per domain, color it and merge all in a .list file:
// { LIST
//  { < a-dom1.off }
//  { < a-dom2.off }
// }
//

//
// -----------------------------------------------------------------------
static
inline
cad::size_type
make_subdiv (cad::size_type subdiv, cad::size_type deg, bool adapt_degree) 
{
    if (deg == 1)  return 1;
    if (! adapt_degree) return subdiv;
    return subdiv*deg;
}
void
cad_rep::put_geomview(ostream& os, size_type subdiv, bool adapt_degree) const
{
    check_macro (dimension() == 3, "cad: geomview output supports only 3D");

    bool reduce_linear = true; // for debug - obsolete

    //
    // 1rst pass: number of points, of faces
    //
    size_type nn = 0;
    size_type nf = 0;
    for (vector<cad_domain>::const_iterator p = _s.begin(); p != _s.end(); p++) {
        const cad_domain& d = (*p);
	for (vector<cad_element>::const_iterator q = d.begin(); q != d.end(); q++) {
	    const cad_element& S = (*q);
	    size_type subdeg  = make_subdiv (subdiv, S.degree(),  adapt_degree);
	    size_type subdeg2 = make_subdiv (subdiv, S.degree2(), adapt_degree);
            nn += S.n_node_from_degree(subdeg, subdeg2);
            nf += S.n_face_from_degree(subdeg, subdeg2);
        }
    }
    os << "OFF" << endl
       << nn << " " << nf << " 0" << endl
       << endl;
    //
    // 2nd pass: points of subdivision
    //
    for (vector<cad_domain>::const_iterator p = _s.begin(); p != _s.end(); p++) {
        const cad_domain& d = (*p);
	for (vector<cad_element>::const_iterator q = d.begin(); q != d.end(); q++) {
	    const cad_element& S = (*q);
	    size_type subdeg  = make_subdiv (subdiv, S.degree(),  adapt_degree);
	    size_type subdeg2 = make_subdiv (subdiv, S.degree2(), adapt_degree);
            size_type nn_loc = S.n_node_from_degree(subdeg, subdeg2);
	    for (size_type i = 0; i < nn_loc; i++) {
		point x_ref_loc = S.ref_node_from_degree(i, subdeg, subdeg2);
		point x_loc = eval (S, x_ref_loc);
		os << x_loc << endl;
	    }
        }
    }
    os << endl;
    //
    // 3nd pass: connectivity of subdivision
    //
    size_type nn_base = 0;
    for (vector<cad_domain>::const_iterator p = _s.begin(); p != _s.end(); p++) {
	const cad_domain& d = (*p);
	for (vector<cad_element>::const_iterator q = d.begin(); q != d.end(); q++) {
	    const cad_element& S = (*q);
	    size_type subdeg  = make_subdiv (subdiv, S.degree(),  adapt_degree);
	    size_type subdeg2 = make_subdiv (subdiv, S.degree2(), adapt_degree);
	    switch (S.variant()) {
              case cad_element::t: {
		size_type b = nn_base;
                for (size_type i1 = 0; i1 < subdeg; i1++) {
                  for (size_type i0 = 0; i0 < subdeg-i1; i0++) {
                      os << "3 " 
                         << b + i0      << " "
		         << b + (i0+1)  << " "
                         << b + (subdeg+1-i1) + i0
                         << endl;
		      if (i0 == subdeg-i1-1) continue;
                      os << "3 " 
		         << b + (i0+1)  << " "
                         << b + (subdeg+1-i1) + (i0+1) << " "
                         << b + (subdeg+1-i1) + i0
                         << endl;
                  }
		  b += subdeg+1-i1;
                }
		break;
              }
              case cad_element::q: {
                for (size_type i0 = 0; i0 < subdeg; i0++) {
                  for (size_type i1 = 0; i1 < subdeg2; i1++) {
                    os << "4 " 
                       << nn_base +     i1*(subdeg+1) + i0      << " "
		       << nn_base +     i1*(subdeg+1) + (i0+1)  << " "
                       << nn_base + (i1+1)*(subdeg+1) + (i0+1)  << " "
		       << nn_base + (i1+1)*(subdeg+1) + i0
                       << endl;
                  }
                }
		break;
              }
	      default: {
		error_macro ("unexpected type");
	      }
            }
	    nn_base += S.n_node_from_degree(subdeg, subdeg2);
	}
    }
}
int
cad_rep::geomview(string basename, 
	 bool execute, bool clean, bool verbose,
	 size_type subdiv, bool adapt_degree) const
{
    check_macro (dimension() == 3, "cad: geomview output supports only 3D");
    //
    // output on file
    //
    string filename = basename + ".off";
    ofstream off (filename.c_str());
    if (verbose) clog << "! file \"" << filename << "\" created.\n";
    int digits10 = numeric_limits<Float>::digits10;
    off << setprecision(digits10);
    put_geomview(off, subdiv, adapt_degree);
    off.close();
    //
    // run geomview
    //
    int status = 0;
    string command;
    if (execute) {
        command = "geomview " + filename;
        if (verbose) clog << "! " << command << endl;
        status = system (command.c_str());
    }
    //
    // clear data
    //
    if (clean) {
        command = "/bin/rm -f " + filename;
        if (verbose) clog << "! " << command << endl;
        int status = system (command.c_str());
    }
    return status;
}
