//plugincontainer.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2011-2012
 *
 *  This file is part of libroar a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  libroar is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 *  NOTE for everyone want's to change something and send patches:
 *  read README and HACKING! There a addition information on
 *  the license of this document you need to read before you send
 *  any patches.
 *
 *  NOTE for uses of non-GPL (LGPL,...) software using libesd, libartsc
 *  or libpulse*:
 *  The libs libroaresd, libroararts and libroarpulse link this lib
 *  and are therefore GPL. Because of this it may be illigal to use
 *  them with any software that uses libesd, libartsc or libpulse*.
 */

#include "libroar.h"

#define MAX_PLUGINS 64

#define OPT_AUTOAPPSCHED 0x0001

struct roar_plugincontainer {
 size_t refc;
 const struct roar_plugincontainer_callbacks * callbacks;
 void * userdata;
 unsigned int options;
 struct roar_dl_librarypara * default_para;
 struct roar_dl_lhandle * handle[MAX_PLUGINS];
 size_t deprefc[MAX_PLUGINS];
 void * context[MAX_PLUGINS];
 size_t numhandles;
};

static inline void _unload(struct roar_plugincontainer * cont, size_t index) {
 int err = roar_error;

 if ( cont->options & OPT_AUTOAPPSCHED )
  roar_dl_appsched_trigger(cont->handle[index], ROAR_DL_APPSCHED_FREE);

 if ( cont->callbacks != NULL && cont->callbacks->preunload != NULL )
  cont->callbacks->preunload(cont, &(cont->context[index]), cont->handle[index]);

 roar_err_set(err);
 roar_dl_unref(cont->handle[index]);
 err = roar_error;
 cont->handle[index] = NULL;
 cont->numhandles--;

 if ( cont->callbacks != NULL && cont->callbacks->postunload != NULL )
  cont->callbacks->postunload(cont, &(cont->context[index]));

 if ( cont->context[index] != NULL )
  if ( cont->callbacks != NULL && cont->callbacks->freecontext != NULL )
  cont->callbacks->freecontext(cont, &(cont->context[index]));

 if ( cont->context[index] != NULL ) {
  roar_mm_free(cont->context[index]);
  cont->context[index] = NULL;
 }

 roar_err_set(err);
}

static int _loader(struct roar_dl_librarypara * lhandle, void * loader_userdata, enum roar_dl_loadercmd cmd, void * argp) {
 roar_err_set(ROAR_ERROR_NOSYS);
 return -1;
}

static inline void _copy_str(char ** dst, const char * src) {
 for (; *src; src++, (*dst)++) **dst = *src;
 *((*dst)++) = 0;
}

static void * _copy_kv(struct roar_keyval ** copy, struct roar_keyval * src, size_t len) {
 size_t buflen = 0;
 size_t i;
 void * ret;
 char * p;
 struct roar_keyval * c;

 for (i = 0; i < len; i++) {
  if ( src[i].key != NULL )
   buflen += roar_mm_strlen(src[i].key) + 1;
  if ( src[i].value != NULL )
   buflen += roar_mm_strlen(src[i].value) + 1;
 }

 c   = roar_mm_malloc(len * sizeof(struct roar_keyval));
 if ( c == NULL )
  return NULL;

 memset(c, 0, len * sizeof(struct roar_keyval));

 ret = roar_mm_malloc(buflen);
 if ( ret == NULL )
  return NULL;

 memset(ret, 0, buflen);

 p = ret;

 for (i = 0; i < len; i++) {
  if ( src[i].key == NULL ) {
   c[i].key = NULL;
  } else {
   c[i].key = p;
   _copy_str(&p, src[i].key);
  }

  if ( src[i].value == NULL ) {
   c[i].value = NULL;
  } else {
   c[i].value = p;
   _copy_str(&p, src[i].value);
  }
 }

 *copy = c;
 return ret;
}

static struct roar_dl_librarypara * _copy_para(struct roar_dl_librarypara * para, struct roar_plugincontainer * cont) {
 struct roar_dl_librarypara * ret = NULL;
 int err;

 if ( para == NULL || cont == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return NULL;
 }

 ret = roar_dl_para_new(NULL, para->binargv, para->appname, para->abiversion);

 if ( para->argc && para->argv != NULL ) {
  ret->argc       = para->argc;
  ret->args_store = _copy_kv(&(ret->argv), para->argv, para->argc);
  if ( ret->args_store == NULL ) {
   err = roar_error;
   roar_dl_para_unref(ret);
   roar_error = err;
   return NULL;
  }
 }

 ret->notifycore      = para->notifycore;
 ret->container       = cont;
 ret->loader          = _loader;
 ret->loader_userdata = NULL;

 return ret;
}

static struct roar_plugincontainer * _new_init(void) {
 struct roar_plugincontainer * ret = roar_mm_malloc(sizeof(struct roar_plugincontainer));
 if ( ret == NULL )
  return NULL;

 memset(ret, 0, sizeof(struct roar_plugincontainer));

 ret->refc = 1;

 return ret;
}

struct roar_plugincontainer * roar_plugincontainer_new(struct roar_dl_librarypara * default_para) {
 struct roar_plugincontainer * ret = _new_init();
 int err;

 if ( ret == NULL )
  return NULL;

 if ( default_para != NULL ) {
  ret->default_para = _copy_para(default_para, ret);
  if ( ret->default_para == NULL ) {
   err = roar_error;
   roar_plugincontainer_unref(ret);
   roar_err_set(err);
   return NULL;
  }
 }

 return ret;
}

struct roar_plugincontainer * roar_plugincontainer_new_simple(const char * appname, const char * abiversion) {
 struct roar_plugincontainer * ret;
 struct roar_dl_librarypara * para = roar_dl_para_new(NULL, NULL, appname, abiversion);
 int err;

 ret = roar_plugincontainer_new(para);
 err = roar_error;

 roar_dl_para_unref(para);

 roar_err_set(err);
 return ret;
}

int roar_plugincontainer_ref(struct roar_plugincontainer * cont) {
 if ( cont == NULL ) {
  ROAR_DBG("roar_plugincontainer_ref(cont=%p) = -1 // error=FAULT", cont);
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 cont->refc++;

 ROAR_DBG("roar_plugincontainer_ref(cont=%p) = 0", cont);
 return 0;
}

int roar_plugincontainer_unref(struct roar_plugincontainer * cont) {
 size_t i;

 if ( cont == NULL ) {
  ROAR_DBG("roar_plugincontainer_unref(cont=%p) = -1 // error=FAULT", cont);
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 cont->refc--;

 if ( cont->refc ) {
  ROAR_DBG("roar_plugincontainer_unref(cont=%p) = 0", cont);
  return 0;
 }

 if ( cont->callbacks != NULL && cont->callbacks->prefree != NULL )
  cont->callbacks->prefree(cont, &(cont->userdata));

 while (cont->numhandles) {
  for (i = 0; i < MAX_PLUGINS; i++) {
   if ( cont->handle[i] == NULL )
    continue;

   // skip plugins in use by others. We will unload them in a later loop.
   if ( cont->deprefc[i] )
    continue;

   _unload(cont, i);
  }
 }

 // try to free user data...
 if ( cont->userdata != NULL ) {
  if ( cont->callbacks != NULL && cont->callbacks->freeuserdata != NULL ) {
   cont->callbacks->freeuserdata(cont, &(cont->userdata));
  }
 }

 // if still not freed by caller, free it using memmgr.
 if ( cont->userdata != NULL )
  roar_mm_free(cont->userdata);

 if ( cont->default_para != NULL )
  roar_dl_para_unref(cont->default_para);

 roar_mm_free(cont);

 ROAR_DBG("roar_plugincontainer_unref(cont=%p) = 0", cont);
 return 0;
}

int roar_plugincontainer_set_autoappsched(struct roar_plugincontainer * cont, int val) {
 if ( cont == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( val ) {
  cont->options |= OPT_AUTOAPPSCHED;
  return 0;
 } else {
  if ( cont->options & OPT_AUTOAPPSCHED ) {
   roar_err_set(ROAR_ERROR_BUSY);
   return -1;
  } else {
   return 0;
  }
 }
}

int roar_plugincontainer_set_callbacks(struct roar_plugincontainer * cont,
                                       const struct roar_plugincontainer_callbacks * callbacks) {
 if ( cont == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 cont->callbacks = callbacks;

 return 0;
}

int roar_plugincontainer_set_userdata(struct roar_plugincontainer * cont, void * userdata) {
 if ( cont == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 cont->userdata = userdata;

 return 0;
}

void * roar_plugincontainer_get_userdata(struct roar_plugincontainer * cont) {
 if ( cont == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return NULL;
 }

 if ( cont->userdata == NULL ) {
  roar_err_set(ROAR_ERROR_NOENT);
  return NULL;
 }

 return cont->userdata;
}

struct roar_plugincontainer_plugininfo roar_plugincontainer_get_info_by_name (struct roar_plugincontainer * cont,
                                                                              const char * name) {
 struct roar_plugincontainer_plugininfo ret;
 const struct roar_dl_libraryname * libname;
 size_t i;

 memset(&ret, 0, sizeof(ret));

 if ( cont == NULL || name == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return ret;
 }

 for (i = 0; i < MAX_PLUGINS; i++) {
  if ( cont->handle[i] == NULL )
   continue;
  libname = roar_dl_getlibname(cont->handle[i]);
  if ( libname == NULL )
   continue;

  if ( !strcmp(libname->name, name) ) {
   if ( roar_dl_ref(cont->handle[i]) == -1 )
    return ret;

   ret.libname  = libname->name;
   ret.handle   = cont->handle[i];
   ret.rdepends = cont->deprefc[i];
   ret.context  = &(cont->context[i]);
   return ret;
  }
 }

 roar_err_set(ROAR_ERROR_NOENT);
 return ret;
}

struct roar_dl_lhandle * roar_plugincontainer_get_lhandle_by_name (struct roar_plugincontainer * cont,
                                                                   const char * name) {
 struct roar_plugincontainer_plugininfo info = roar_plugincontainer_get_info_by_name(cont, name);

 if ( info.libname == NULL )
  return NULL;

 return info.handle;
}

// plugin loading and unloading:
int roar_plugincontainer_load(struct roar_plugincontainer * cont, const char * name, struct roar_dl_librarypara * para) {
 struct roar_dl_lhandle * ret = roar_plugincontainer_load_lhandle(cont, name, ROAR_DL_FLAG_PLUGIN, 1, para);

 if ( ret == NULL )
  return -1;

 roar_dl_unref(ret);

 return 0;
}

struct roar_dl_lhandle * roar_plugincontainer_load_lhandle    (struct roar_plugincontainer * cont,
                                                               const char * name,
                                                               int flags,
                                                               int ra_init,
                                                               struct roar_dl_librarypara * para) {
 ssize_t idx = -1;
 size_t i;
 int err;

 if ( cont == NULL || name == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return NULL;
 }

 if ( para == NULL ) {
  para = cont->default_para;
  roar_dl_para_ref(para);
 } else {
  para = _copy_para(para, cont);
 }


 // search for a free index.
 if ( cont->numhandles == MAX_PLUGINS ) {
  // return early if we are full.
  roar_dl_para_unref(para);
  roar_err_set(ROAR_ERROR_NOSPC);
  return NULL;
 }
 for (i = 0; i < MAX_PLUGINS; i++) {
  if ( cont->handle[i] == NULL ) {
   idx = i;
   break;
  }
 }

 if ( idx == -1 ) {
  roar_dl_para_unref(para);
  roar_err_set(ROAR_ERROR_NOSPC);
  return NULL;
 }

 cont->context[idx] = NULL; // clear context.
 cont->deprefc[idx] = 0;

 if ( cont->callbacks != NULL && cont->callbacks->preload != NULL )
  cont->callbacks->preload(cont, &(cont->context[idx]), name, flags, para);

 cont->handle[idx] = roar_dl_open(name, flags, 0, para);
 if ( cont->handle[idx] == NULL ) {
  err = roar_error;
  roar_dl_para_unref(para);
  roar_error = err;
  return NULL;
 }

 cont->numhandles++;

 if ( cont->callbacks != NULL && cont->callbacks->postload != NULL )
  cont->callbacks->postload(cont, &(cont->context[idx]), cont->handle[idx], name, flags, para);

 if ( ra_init ) {
  if ( cont->callbacks != NULL && cont->callbacks->prera_init != NULL )
   cont->callbacks->prera_init(cont, &(cont->context[idx]), cont->handle[idx], para);
  if ( roar_dl_ra_init(cont->handle[idx], NULL, para) == -1 ) {
   err = roar_error;

   if ( cont->callbacks != NULL && cont->callbacks->postra_init != NULL )
    cont->callbacks->postra_init(cont, &(cont->context[idx]), cont->handle[idx], para);

   _unload(cont, idx);

   roar_dl_para_unref(para);
   cont->handle[idx] = NULL;
   roar_error = err;
   return NULL;
  }

  if ( cont->callbacks != NULL && cont->callbacks->postra_init != NULL )
   cont->callbacks->postra_init(cont, &(cont->context[idx]), cont->handle[idx], para);

  if ( cont->options & OPT_AUTOAPPSCHED )
   roar_dl_appsched_trigger(cont->handle[idx], ROAR_DL_APPSCHED_INIT);
 }

 roar_dl_para_unref(para);
 roar_dl_ref(cont->handle[idx]);
 return cont->handle[idx];
}

int roar_plugincontainer_unload(struct roar_plugincontainer * cont, const char * name) {
 struct roar_dl_lhandle * lhandle;

 if ( cont == NULL || name == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 lhandle = roar_plugincontainer_get_lhandle_by_name(cont, name);
 if ( lhandle == NULL )
  return -1;

 roar_dl_unref(lhandle); // we can do this early here as the handle stay valid in the container's array.

 return roar_plugincontainer_unload_lhandle(cont, lhandle);
}

int roar_plugincontainer_unload_lhandle(struct roar_plugincontainer * cont, struct roar_dl_lhandle * lhandle) {
 size_t i;

 if ( cont == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( lhandle == NULL ) {
  roar_err_set(ROAR_ERROR_INVAL);
  return -1;
 }

 for (i = 0; i < MAX_PLUGINS; i++) {
  if ( cont->handle[i] == lhandle ) {
   if ( cont->deprefc[i] ) {
    // still in use.
    roar_err_set(ROAR_ERROR_BUSY);
    return -1;
   }
   _unload(cont, i);
   return 0;
  }
 }

 roar_err_set(ROAR_ERROR_NOENT);
 return -1;
}

int                      roar_plugincontainer_ra_init         (struct roar_plugincontainer * cont) {
 size_t i;

 if ( cont == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 for (i = 0; i < MAX_PLUGINS; i++) {
  if ( cont->handle[i] == NULL )
   continue;

  if ( cont->callbacks != NULL && cont->callbacks->prera_init != NULL )
   cont->callbacks->prera_init(cont, &(cont->context[i]), cont->handle[i], cont->default_para);

  roar_dl_ra_init(cont->handle[i], NULL, cont->default_para);

  if ( cont->callbacks != NULL && cont->callbacks->postra_init != NULL )
   cont->callbacks->postra_init(cont, &(cont->context[i]), cont->handle[i], cont->default_para);

  if ( cont->options & OPT_AUTOAPPSCHED )
   roar_dl_appsched_trigger(cont->handle[i], ROAR_DL_APPSCHED_INIT);
 }

 return 0;
}

// appsched:
int roar_plugincontainer_appsched_trigger(struct roar_plugincontainer * cont, enum roar_dl_appsched_trigger trigger) {
 size_t i;
 int ret = -1;
 int rv;

 if ( cont == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( cont->numhandles == 1 ) {
  for (i = 0; i < MAX_PLUGINS; i++) {
   if ( cont->handle[i] == NULL )
    continue;
   return roar_dl_appsched_trigger(cont->handle[i], trigger);
  }
  roar_panic(ROAR_FATAL_ERROR_MEMORY_CORRUPTION, NULL);
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 for (i = 0; i < MAX_PLUGINS; i++) {
  if ( cont->handle[i] == NULL )
   continue;

  rv = roar_dl_appsched_trigger(cont->handle[i], trigger);
  if ( rv == 0 )
   ret = 0;
 }

 return ret;
}

//ll
