/* 
 * ------------------------------------------------------------------
 * Role PlayingDB V2.0 by Deepwoods Software
 * ------------------------------------------------------------------
 * Monster.h - Monster class
 * Created by Robert Heller on Sat Jul 18 16:54:29 1998
 * ------------------------------------------------------------------
 * Modification History: 
 * $Log: Monster.h,v $
 * Revision 1.5  2000/02/11 00:31:27  heller
 * MS-Windows fixes...
 *
 * Revision 1.4  1999/07/13 00:29:40  heller
 * Documentation updates (spelling, punctionation, spelling, caps/nocaps).
 *
 * Revision 1.3  1998/12/27 21:48:19  heller
 * Spelling errors fixed.
 *
 * Revision 1.2  1998/08/19 16:14:58  heller
 * Small fix...
 *
 * Revision 1.1  1998/08/04 21:17:00  heller
 * Initial revision
 *
 * ------------------------------------------------------------------
 * Contents:
 * ------------------------------------------------------------------
 *  
 *     Role Playing DB -- A database package that creates and maintains
 * 		       a database of RPG characters, monsters, treasures,
 * 		       spells, and playing environments.
 * 
 *     Copyright (C) 1995,1998  Robert Heller D/B/A Deepwoods Software
 * 			51 Locke Hill Road
 * 			Wendell, MA 01379-9728
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 *  
 */

#ifndef _MONSTER_H_
#define _MONSTER_H_

#include <Record.h>

/*@ManDoc:
  \typeout{Generated by $Id: Monster.h,v 1.5 2000/02/11 00:31:27 heller Rel $.}
  Basic monster class. Contains all of the information needed to describe a
  monster.
*/

class Monster {
public:
	//@ManDoc: Frequency Types.
	enum FreqType {
		//@ManDoc: This monster is unique (only one exists).
		Unique,
		//@ManDoc: This monster is very rare, only a few exist. 4% probability of occurrence.
		VeryRare,
		//@ManDoc: This monster is rare, not many exist. 11% probability of occurrence.
		Rare,
		//@ManDoc: This monster is uncommon, some exist but are not seen often. 20% probability of occurrence.
		Uncommon,
		//@ManDoc: This monster is common, they are all over the place. 65% probability of occurrence.
		Common,
		//@ManDoc: This is a bogus frequency.  Used to handle uninitialized values.
		BogusFrequency
	};
	//@ManDoc: Intelligence Ratings.
	enum IntelligenceRating {
		//@ManDoc: Unintelligent.  Mindless critter. I score = 0.
		Non,
		//@ManDoc: Animal intelligence.  I score = 1.
		Animal,
		//@ManDoc: Semi-intelligent. I score = 2-4.
		Semi,
		//@ManDoc: Low intelligence. I score = 5-7.
		Low,
		//@ManDoc: Average (Human) intelligence. I score = 8-10.
		Average,
		//@ManDoc: Very intelligent. I score = 11-12.
		Very,
		//@ManDoc: Highly intelligent. I score = 13-14.
		Highly,
		//@ManDoc: Exceptionally intelligent. I score = 15-16.
		Exceptionally,
		//@ManDoc: Genius. I score = 17-18.
		Genius,
		//@ManDoc: Supra-Genius. I score = 19-20.
		SupraGenius,
		//@ManDoc: Godlike intelligence. I score = 21+.
		Godlike, 
		//@ManDoc: This is a bogus intelligence.  Used to handle uninitialized values.
		BogusIntelligence
	};
	//@ManDoc: Method of hit points.
	enum HitType {
		/*@ManDoc: An explicit number of hit points.
		   Typical for unique monsters, like demigods, major dragons,
		   or demons.
		 */
		Points,
		/*@ManDoc: Normal hit dice system.  Typical for monsters that
		  are more common, like orcs.
		 */
		Dice,
		//@ManDoc: This is a bogus value.  Used to handle uninitialized values.
		BogusHitType
	};
private:
	//@ManDoc: Monster's (type) name.
	const char *name;
	//@ManDoc: Monster's alignment.
	const char *alignment;
	//@ManDoc: Monster's treasure type.
	const char *treasureType;
	//@ManDoc: Special attacks.
	const char *specialAttacks;
	//@ManDoc: Special defenses.
	const char *specialDefences;
	//@ManDoc: Psionic abilities.
	const char *psionics;
	//@ManDoc: Other descriptive information.
	const char *comments;
	//@ManDoc: A picture.
	const char *image;
	//@ManDoc: Monster's hitpoints.
	short int hitpoints;
	//@ManDoc: Monster's hit die type (usually 8).
	signed char hdie;
	//@ManDoc: Number of hit dice.
	signed char ndie;
	//@ManDoc: Hit point adjustment.
	signed char hplus;
	//@ManDoc: Armor class.
	signed char armclass;
	//@ManDoc: Speed on land (walking/running).
	signed char move;
	//@ManDoc: Speed in the air (flying).
	signed char move_fly;
	//@ManDoc: Speed in water (swimming).
	signed char move_swim;
	//@ManDoc: Speed in the earth (burrowing).
	signed char move_burrow;
	//@ManDoc: Speed in web.
	signed char move_web;
	//@ManDoc: Percent in lair.
	signed char percentLair;
	//@ManDoc: Number of attacks.
	signed char numatt;
	//@ManDoc: Magical resistance.
	signed char magres;
	//@ManDoc: Range structure (used for damage per attack and number appearing).
	struct Range {
		//@ManDoc: Lower bound.
		signed char l;
		//@ManDoc: Upper bound.
		signed char h;
		};
	//@ManDoc: Damage per attack.
	Range damatt;
	//@ManDoc: Number appearing.
	Range noappearing;
	//@ManDoc: Intelligence.
	IntelligenceRating intelligence;
	//@ManDoc: Frequency.
	FreqType frequency;
	//@ManDoc: Hit type.
	HitType hittype;
	//@ManDoc: Size in inches.
	double size;
	//@ManDoc: Data record.
	Record rawData;
	//@ManDoc: Convert internal record to monster.
	void RecordToMonster ();
	//@ManDoc: Update internal record.
	void UpdateRecord();
	//@ManDoc: Initializer.
	void _Monster(
		//@ManDoc: Hit points.
		int hp, 
		//@ManDoc: Hit dice.
		int hd, 
		//@ManDoc: Number of hit dice.
		int nd, 
		//@ManDoc: Hit point adjustment.
		int hadj, 
		//@ManDoc: Armor class.
		int ac, 
		//@ManDoc: Land speed.
		int m, 
		//@ManDoc: Flying Speed.
		int mf,
		//@ManDoc: Swimming speed.
		int ms, 
		//@ManDoc: Burrowing speed.
		int mb, 
		//@ManDoc: Speed in web.
		int mw, 
		//@ManDoc: Percent in lair.
		int pl, 
		//@ManDoc: Number of attacks.
		int na, 
		//@ManDoc: Minimum damage per attack.
		int daL,
		//@ManDoc: Maximum damage per attack.
		int daH, 
		//@ManDoc: Magical resistance.
		int mr, 
		//@ManDoc: Minimum number appearing.
		int noaL, 
		//@ManDoc: Maximum number appearing.
		int noaH,
		//@ManDoc: Intelligence rating.
		IntelligenceRating i, 
		//@ManDoc: Frequency type.
		FreqType f,
		//@ManDoc: Hit type.
		HitType h,
		//@ManDoc: Size.
		double s, 
		//@ManDoc: Name/Type.
		const char * nm, 
		//@ManDoc: Alignment.
		const char * al, 
		//@ManDoc: Treasure type.
		const char * tt, 
		//@ManDoc: Special attacks.
		const char * sa, 
		//@ManDoc: Special defenses.
		const char * sd, 
		//@ManDoc: Psionics.
		const char * ps, 
		//@ManDoc: Commentary.
		const char * com,
		//@ManDoc: Image.
		const char * im
	) {
			hitpoints = hp;
			hdie = hd;
			ndie = nd;
			hplus = hadj;
			armclass = ac;
			move = m;
			move_fly = mf;
			move_swim = ms;
			move_burrow = mb;
			move_web = mw;
			percentLair = pl;
			numatt = na;
			damatt.l = daL;
			damatt.h = daH;
			magres = mr;
			noappearing.l = noaL;
			noappearing.h = noaH;
			intelligence = i;
			frequency = f;
			hittype = h;
			size = s;
			name = nm;
			alignment = al;
			treasureType = tt;
			specialAttacks = sa;
			specialDefences = sd;
			psionics = ps;
			comments = com;
			image = im;
			UpdateRecord();
		}
public:
	//@ManDoc: Return the name/type.
	const char *Name() const {return name;}
	//@ManDoc: Return the alignment.
	const char *Alignment() const {return alignment;}
	//@ManDoc: Return the treasure type.
	const char *TreasureType() const {return treasureType;}
	//@ManDoc: Return the special attacks.
	const char *SpecialAttacks() const {return specialAttacks;}
	//@ManDoc: Return the special defenses.
	const char *SpecialDefences() const {return specialDefences;}
	//@ManDoc: Return the psionics.
	const char *Psionics() const {return psionics;}
	//@ManDoc: Return the commentary.
	const char *Comments() const {return comments;}
	//@ManDoc: Return the image.
	const char *Image() const {return image;}
	//@ManDoc: Return the hit die sides.
	int HitDieSides() const {return hdie;}
	//@ManDoc: Return the number of hit dice.
	int NumHitDice() const {return ndie;}
	//@ManDoc: Return the hit dice adjustment.
	int HitAdjustment() const {return hplus;}
	//@ManDoc: Return the armor class.
	int ArmorClass() const {return armclass;}
	//@ManDoc: Return the speed on land.
	int LandSpeed() const {return move;}
	//@ManDoc: Return the speed in the air (flying).
	int FlyingSpeed() const {return move_fly;}
	//@ManDoc: Return the speed in the water (swimming).
	int SwimmingSpeed() const {return move_swim;}
	//@ManDoc: Return the speed in the earth (burrowing).
	int BurrowingSpeed() const {return move_burrow;}
	//@ManDoc: Return the speed in web.
	int WebSpeed() const {return move_web;}
	//@ManDoc: Return the percent in lair.
	int PercentInLair() const {return percentLair;}
	//@ManDoc: Return the number of attacks.
	int NumberOfAttacks() const {return numatt;}
	//@ManDoc: Return the magical resistance.
	int MagicalResistance() const {return magres;}
	//@ManDoc: Return the number of hit points.
	int HitPoints() const {return hitpoints;}
	//@ManDoc: Return the damage per attack.
	void DamagePerAttack(
		//@ManDoc: Lower bound.
		int& L,
		//@ManDoc: Upper bound.
		int& H
	) const {L = damatt.l;H = damatt.h;}
	//@ManDoc: Return the number appearing.
	void NumberAppearing(
		//@ManDoc: Lower bound.
		int& L,
		//@ManDoc: Upper bound.
		int& H
	) const {L = noappearing.l;H = noappearing.h;}
	//@ManDoc: Return the intelligence rating.
	IntelligenceRating Intelligence() const {return intelligence;}
	//@ManDoc: Return the frequency.
	FreqType Frequency() const {return frequency;}
	//@ManDoc: Return the hit type.
	HitType Hittype() const {return hittype;}
	//@ManDoc: Return the size.
	double Size() const {return size;}
	//@ManDoc: Default constructor.
	Monster()
	{
//			  hp hd nd had ac m mf ms mb mw pl na da mr noa i
//			  i                         f                h
//			  s   nm       al       tt       
//			  sa       sd       ps      com
		_Monster( 0, 0, 0, 0,  0, 0,0, 0, 0, 0, 0, 0, 0,0, 0, 0,0,
			  Non,Unique,Points,
			  0.0,"","","","","","","","");
	}
	//@ManDoc: Hit point type constructor (unique and very rare monsters).
	Monster(
		//@ManDoc: Hit points.
		int hp,
		//@ManDoc: Armor class.
		int ac, 
		//@ManDoc: Land speed.
		int m, 
		//@ManDoc: Flying speed.
		int mf,
		//@ManDoc: Swimming speed.
		int ms,
		//@ManDoc: Burrowing speed.
		int mb,
		//@ManDoc: Speed in web.
		int mw, 
		//@ManDoc: Percent in lair.
		int pl,
		//@ManDoc: Number of attacks.
		int na,
		//@ManDoc: Minimum damage per attack.
		int daL,
		//@ManDoc: Maximum damage per attack.
		int daH,
		//@ManDoc: Magical resistance.
		int mr,
		//@ManDoc: Minimum number appearing.
		int noaL,
		//@ManDoc: Maximum number appearing.
		int noaH,
		//@ManDoc: Intelligence rating.
		IntelligenceRating i, 
		//@ManDoc: Frequency type.
		FreqType f, 
		//@ManDoc: Size.
		double s,
		//@ManDoc: Name/type.
		const char * nm, 
		//@ManDoc: Alignment.
		const char * al, 
		//@ManDoc: Treasure type.
		const char * tt, 
		//@ManDoc: Special attacks.
		const char * sa, 
		//@ManDoc: Special defenses.
		const char * sd,
		//@ManDoc: Psionics.
		const char * ps, 
		//@ManDoc: Commentary.
		const char * com,
		//@ManDoc: Image.
		const char * im
	) {
			_Monster(hp,0,0,0,ac,m,mf,ms,mb,mw,pl,na,daL,daH,mr,noaL,noaH,i,f,
				 Points,s,nm,al,tt,sa,sd,ps,com,im);
		}
	//@ManDoc: Hit dice type constructor (more common monsters).
	Monster(
		//@ManDoc: Hit dice type (usually 8).
		int hd,
		//@ManDoc: Number of hit dice.
		int nd,
		//@ManDoc: Hit point adjustment.
		int had,
		//@ManDoc: Armor class.
		int ac,
		//@ManDoc: Land speed.
		int m,
		//@ManDoc: Flying speed.
		int mf,
		//@ManDoc: Swimming speed.
		int ms,
		//@ManDoc: Burrowing speed.
		int mb,
		//@ManDoc: Speed in web.
		int mw,
		//@ManDoc: Percent in lair.
		int pl,
		//@ManDoc: Number of attacks.
		int na,
		//@ManDoc: Minimum damage per attack.
		int daL,
		//@ManDoc: Maximum damage per attack.
		int daH,
		//@ManDoc: Magical resistance.
		int mr,
		//@ManDoc: Minimum number appearing.
		int noaL,
		//@ManDoc: Maximum number appearing.
		int noaH,
		//@ManDoc: Intelligence rating.
		IntelligenceRating i, 
		//@ManDoc: Frequency type.
		FreqType f, 
		//@ManDoc: Size.
		double s, 
		//@ManDoc: Name/type.
		const char * nm, 
		//@ManDoc: Alignment.
		const char * al, 
		//@ManDoc: Treasure type.
		const char * tt, 
		//@ManDoc: Special attacks.
		const char * sa, 
		//@ManDoc: Special defenses.
		const char * sd, 
		//@ManDoc: Psionics.
		const char * ps, 
		//@ManDoc: Commentary.
		const char * com,
		//@ManDoc: Image.
		const char * im
	) {
	 		_Monster(0,hd,nd,had,ac,m,mf,ms,mb,mw,pl,na,daL,daH,mr,noaL,noaH,i,f,Dice,
	 	 		 s,nm,al,tt,sa,sd,ps,com,im);
	 	}
	//@ManDoc: Copy constructor (from pointer).
	Monster (const Monster *that);
	//@ManDoc: Copy constructor (from reference).
	Monster (const Monster &that);
	//@ManDoc: Type conversion constructor, from a Record.
	Monster (const Record *rec);
	//@ManDoc: Destructor.
	~Monster() {}
	//@ManDoc: Type conversion: convert to a Record.
	operator const Record  () const {return rawData;}
	//@ManDoc: Update Monster from Record.
	void UpdateFromRecord (const Record &rec)
	{
		rawData = rec;
		RecordToMonster ();
	}
};


/*@ManDoc:
  Monster ``instance''.  Used to implement a working instance of a monster.
 */

class MonsterInstance {
private:
	//@ManDoc: Instance's name.
	const char *instanceName;
	//@ManDoc: What it is an instance of.
	const Monster *instanceOf;
	//@ManDoc: Instance's hit points.
	int instanceHitPoints;
public:
	//@ManDoc: Constructor.
	MonsterInstance(const Monster *iOf, const char *iN = "");
	//@ManDoc: Copy constructor (from pointer).
	MonsterInstance(const MonsterInstance *that);
	//@ManDoc: Copy constructor (from reference).
	MonsterInstance(const MonsterInstance &that);
	//@ManDoc: Destructor.
	~MonsterInstance() {delete (void *)instanceName;}
	//@ManDoc: Return the name.
	const char *InstanceName() const {return instanceName;}
	//@ManDoc: Return the hit points.
	int InstanceHitPoints() const {return instanceHitPoints;}
	//@ManDoc: Adjust the hit points.
	int UpdateInstanceHitPoints(int adj) {
		instanceHitPoints += adj;
		return instanceHitPoints;
	}
	//@ManDoc: Return base monster object.
	const Monster *InstanceOf () const {return instanceOf;}
};
	

#endif // _MONSTER_H_

