/* 
 * ------------------------------------------------------------------
 * Role PlayingDB V2.0 by Deepwoods Software
 * ------------------------------------------------------------------
 * Space.h - Squares and Hexes -- where things happen
 * Created by Robert Heller on Fri Aug 21 10:55:13 1998
 * ------------------------------------------------------------------
 * Modification History: 
 * $Log: Space.h,v $
 * Revision 1.7  2000/02/11 00:31:27  heller
 * MS-Windows fixes...
 *
 * Revision 1.6  1999/07/13 00:29:40  heller
 * Documentation updates (spelling, punctionation, spelling, caps/nocaps).
 *
 * Revision 1.5  1998/12/27 21:48:19  heller
 * Spelling errors fixed.
 *
 * Revision 1.4  1998/12/21 19:58:43  heller
 * Fix error in assignment operator
 *
 * Revision 1.3  1998/12/21 14:47:59  heller
 * Added image field to "Item"
 *
 * Revision 1.2  1998/12/18 21:08:25  heller
 * Changed "NextSpaceFilename" to NextSpaceIndexString"
 *
 * Revision 1.1  1998/08/23 14:46:27  heller
 * Initial revision
 *
 * ------------------------------------------------------------------
 * Contents:
 * ------------------------------------------------------------------
 *  
 */

#ifndef _SPACE_H_
#define _SPACE_H_

#include <Record.h>

#ifndef HAVE_BOOL
typedef enum bool {false=0, true=1};
#define HAVE_BOOL 1
#endif

//@Man: Spaces -- Squares and Hexes, where things happen.
/*@Doc:
  \typeout{Generated by $Id: Space.h,v 1.7 2000/02/11 00:31:27 heller Rel $}
  Space class and related odds and ends.
 */

//@{

#ifdef _MSC_VER

class GeoConstants {
public:
  static const float Width;
  static const float HexSideLength;
  static const float HexPeakHeight;
};

#else

//@ManDoc: Geometric constants.

class GeoConstants {
public:
  //@Man: Width = 100.0
  //@Type: const float
  //@Doc: Space ``width''.
  const float Width = 100.0;
  //@Man: HexSideLength = 57.735
  //@Type: const float
  //@Doc: Side of a hex (computed to give a width of 100).
  const float HexSideLength = 57.735;
  //@Man: HexPeakHeight = 28.8675
  //@Type: const float
  //@Doc: Height of peak above vertical sides.
  const float HexPeakHeight = 28.8675;
};

#endif

class Space;

/*@ManDoc: Exit points and other inter-spatial interconnection points like
  windows and staircases.
 */

class Exit {
public:
	friend class Space;
	//@ManDoc: Exit types.
	enum ExitType {
		//@ManDoc: Plain doorway.
		Doorway,
		//@ManDoc: Plain door.
		Door,
		//@ManDoc: Locked door.
		LockedDoor,
		//@ManDoc: Secret door.
		SecretDoor,
		//@ManDoc: One way door.
		OnewayDoor,
		//@ManDoc: Trapdoor, up.
		TrapDoorUp,
		//@ManDoc: Trapdoor, down.
		TrapDoorDown,
		//@ManDoc: Stairs, up.
		StairsUp,
		//@ManDoc: Stairs, down.
		StairsDown,
		//@ManDoc: Window, unglazed.
		WindowUnglazed,
		//@ManDoc: Window, glazed.
		WindowGlazed,
		//@ManDoc: Chimney or vent.
		Chimney,
		//@ManDoc: Pit or hole.
		Pit,
		//@ManDoc: Unspecified.
		Unspecified
	};
private:
	//@ManDoc: Type of exit.
	ExitType type;
	//@ManDoc: Center X coordinate.
	float xCenter;
	//@ManDoc: Center Y coordinate.
	float yCenter;
	//@ManDoc: Aligned with wall?
	bool wallAligned;
	//@ManDoc: Description.
	const char *descr;
	//@ManDoc: Picture of exit.
	const char *image;
	//@ManDoc: Next space.
	const char *nextSpaceIndexString;
public:
	//@ManDoc: Return type of exit.
	ExitType Type() const {return type;}
	//@ManDoc: Return center X coordinate.
	double XCenter() const {return xCenter;}
	//@ManDoc: Return center Y coordinate.
	double YCenter() const {return yCenter;}
	//@ManDoc: Return wall alignment flag.
	bool WallAligned() const {return wallAligned;}
	//@ManDoc: Return description.
	const char *Description() const {return descr;}
	//@ManDoc: Return picture of exit.
	const char *Image() const {return image;}
	//@ManDoc: Return next space index string.
	const char *NextSpaceIndexString() const {return nextSpaceIndexString;}
	//@ManDoc: Constructor.
	Exit (
		//@ManDoc: Exit type.
		ExitType t = Unspecified,
		//@ManDoc: X coordinate.
		double x = 0.0,
		//@ManDoc: Y coordinate.
		double y = 0.0,
		//@ManDoc: Wall alignment?
		bool wa = true,
		//@ManDoc: Description.
		const char *d = "",
		//@ManDoc: Picture.
		const char *im = "",
		//@ManDoc: Next space.
		const char *ns = ""
	) {
		type = t;
		xCenter = x;
		yCenter = y;
		wallAligned = wa;
		descr = d;
		image = im;
		nextSpaceIndexString = ns;
	}
	//@ManDoc: Destructor.
	~Exit() {}
	//@ManDoc: Assignment operator.
	Exit& operator = (const Exit& source) {
		type = source.type;
		xCenter = source.xCenter;
		yCenter = source.yCenter;
		wallAligned = source.wallAligned;
		descr = source.descr;
		image = source.image;
		return *this;
	}
};

//@ManDoc: Vector of Exits.

class ExitVector {
public:
	friend class Space;
private:
	enum {
		//@Man: initSize = 10
		//@Type: const int
		//@Doc: Initial allocation size.
		initSize = 10,
		//@Man: growSize = 10
		//@Type: const int
		//@Doc: Growth allocation.
		growSize = 10
	};
	//@ManDoc: Element vector.
	Exit *evect;
	//@ManDoc: Allocated size of vector.
	int vSize;
	//@ManDoc: Number of used elements.
	int vCount;
	//@ManDoc: Nearest index.
	int NearestIndex(
		//@ManDoc: X coordinate.
		double x, 
		//@ManDoc: Y coordinate.
		double y
	) const;
public:
	//@ManDoc: Constructor.
	ExitVector();
	//@ManDoc: Destructor.
	~ExitVector();
	//@ManDoc: Insertion function.
	void InsertExit(
		//@ManDoc: Exit object to insert.
		const Exit& source
	);
	//@ManDoc: Index operator.
	const Exit* operator [](
		//@ManDoc: Index.
		int index
	) const {
		if (index >= 0 && index < vCount) return &evect[index];
		else return NULL;
	}
	//@ManDoc: Index function.
	const Exit* Index(
		//@ManDoc: Index.
		int index
	) const { return (*this)[index]; }
	//@ManDoc: Select nearest to (x,y) operator.
	const Exit* operator () (
		//@ManDoc: X coordinate.
		double x,
		//@ManDoc: Y coordinate.
		double y
	) const;
	//@ManDoc: Select nearest to (x,y) function.
	const Exit* Nearest (
		//@ManDoc: X coordinate.
		double x,
		//@ManDoc: Y coordinate.
		double y
	) const { return (*this)(x,y); }
	//@ManDoc: Delete element by index.
	void DeleteAtIndex(
		//@ManDoc: Element to delete.
		int index
	);
	//@ManDoc: Delete element near coordinate.
	void DeleteNear(
		//@ManDoc: X coordinate.
		double x, 
		//@ManDoc: Y coordinate.
		double y
	);
	//@ManDoc: Return element count.
	int ElementCount () const {return vCount;}
};

//@ManDoc: Item in space.

class Item {
public:
	friend class Space;
	//@ManDoc: Item types.
	enum ItemType {
		//@ManDoc: Item is a Character.
		Character,
		//@ManDoc: Item is a Monster.
		Monster,
		//@ManDoc: Item is a Treasure.
		Treasure,
		//@ManDoc: Item is a Trick or Trap.
		TrickTrap,
		//@ManDoc: Item is some random other object.
		Dressing,
		//@ManDoc: Item has no specific type.
		Unspecified
	};
private:
	//@ManDoc: Type of item.
	ItemType type;
	//@ManDoc: Center X coordinate.
	float xCenter;
	//@ManDoc: Center Y coordinate.
	float yCenter;
	//@ManDoc: Image.
	const char *image;
	//@ManDoc: File name.
	const char *filename;
public:
	//@ManDoc: Return type of item.
	ItemType Type() const {return type;}
	//@ManDoc: Return center X coordinate.
	double XCenter() const {return xCenter;}
	//@ManDoc: Return center Y coordinate.
	double YCenter() const {return yCenter;}
	//@ManDoc: Return image.
	const char *Image() const {return image;}
	//@ManDoc: Return file name.
	const char *Filename() const {return filename;}
	//@ManDoc: Constructor.
	Item (
		//@ManDoc: Item type.
		ItemType t = Unspecified,
		//@ManDoc: X coordinate.
		double x = 0.0,
		//@ManDoc: Y coordinate.
		double y = 0.0,
		//@ManDoc: Image.
		const char *im = "",
		//@ManDoc: File name.
		const char *f = ""
	) {
		type = t;
		xCenter = x;
		yCenter = y;
		image = im;
		filename = f;
	}
	//@ManDoc: Destructor.
	~Item() {}
	//@ManDoc: Assignment operator.
	Item& operator = (const Item& source) {
		type = source.type;
		xCenter = source.xCenter;
		yCenter = source.yCenter;
		image = source.image;
		filename = source.filename;
		return *this;
	}
};

//@ManDoc: Vector of Items.

class ItemVector {
public:
	friend class Space;
private:
	enum {
		//@Man: initSize = 10
		//@Type: const int
		//@Doc: Initial allocation size.
		initSize = 10,
		//@Man: growSize = 10
		//@Type: const int
		//@Doc: Growth allocation.
		growSize = 10
	};
	//@ManDoc: Element vector.
	Item *ivect;
	//@ManDoc: Allocated size of vector.
	int vSize;
	//@ManDoc: Number of used elements.
	int vCount;
	//@ManDoc: Nearest index.
	int NearestIndex(
		//@ManDoc: X coordinate.
		double x, 
		//@ManDoc: Y coordinate.
		double y
	) const;
public:
	//@ManDoc: Constructor.
	ItemVector();
	//@ManDoc: Destructor.
	~ItemVector();
	//@ManDoc: Insertion function.
	void InsertItem(
		//@ManDoc: Item object to insert.
		const Item& source
	);
	//@ManDoc: Index operator.
	const Item* operator [](
		//@ManDoc: Index.
		int index
	) const {
		if (index >= 0 && index < vCount) return &ivect[index];
		else return NULL;
	}
	//@ManDoc: Index function.
	const Item* Index(
		//@ManDoc: Index.
		int index
	) const { return (*this)[index]; }
	//@ManDoc: Select nearest to (x,y) operator.
	const Item* operator () (
		//@ManDoc: X coordinate.
		double x,
		//@ManDoc: Y coordinate.
		double y
	) const;
	//@ManDoc: Select nearest to (x,y) function.
	const Item* Nearest (
		//@ManDoc: X coordinate.
		double x,
		//@ManDoc: Y coordinate.
		double y
	) const { return (*this)(x,y); }
	//@ManDoc: Delete element by index.
	void DeleteAtIndex(
		//@ManDoc: Element to delete.
		int index
	);
	//@ManDoc: Delete element near coordinate.
	void DeleteNear(
		//@ManDoc: X coordinate.
		double x, 
		//@ManDoc: Y coordinate.
		double y
	);
	//@ManDoc: Return element count.
	int ElementCount () const {return vCount;}
};


/*@ManDoc: Basic Space class.
 */

class Space {
public:
	//@ManDoc: Space shape.
	enum SpaceShape {
		//@ManDoc: Square.
		Square, 
		//@ManDoc: Hexagon.
		Hexagon, 
		//@ManDoc: Undefined shape.
		Undefined
	};
private:
	//@ManDoc: Space shape.
	SpaceShape shape;
	//@ManDoc: Center point X coordinate.
	double centerX;
	//@ManDoc: Center point Y coordinate.
	double centerY;
	//@ManDoc: List of exits.
	ExitVector exitList;
	//@ManDoc: List of items.
	ItemVector itemList;
	//@ManDoc: Name of the space.
	const char *name;
	//@ManDoc: Description of the space.
	const char *descr;
	//@ManDoc: Background color of the space.
	const char *bgcolor;
	//@ManDoc: Internal storage of the space.
	Record rawData;
	//@ManDoc: Record to Space.
	void RecordToSpace();
	//@ManDoc: Update Record.
	void UpdateRecord();
	void ProcessRecordToExit(int &bytesleft, char *& str);
	void ProcessRecordToItem(int &bytesleft, char *& str);
	int CreateExitRecord(const Exit &element, char *& buffer);
	int CreateItemRecord(const Item &element, char *& buffer);
	void UpdateExitRecord(Exit &element, const char * p);
	void UpdateItemRecord(Item &element, const char * p);
public:
	//@ManDoc: Update Space from Record.
	void UpdateFromRecord (const Record &rec)
	{
		rawData = rec;
		RecordToSpace ();
	}
	//@ManDoc: Constructor.
	Space(
		//@ManDoc: Shape.
		SpaceShape s = Undefined,
		//@ManDoc: Center point X coordinate.
		double x = 0.0,
		//@ManDoc: Center point y coordinate.
		double y = 0.0,
		//@ManDoc: Name of the space.
		const char *n = "",
		//@ManDoc: Description of the space.
		const char *d = "",
		//@ManDoc: Background color of the space.
		const char *bg = "white"
	) {
		shape = s;
		centerX = x;
		centerY = y;
		name = n;
		descr = d;
		bgcolor = bg;
		UpdateRecord();
	}
	//@ManDoc: Copy constructor (from pointer).
	Space (const Space *that)
	{
		UpdateFromRecord((Record)(*that));
	}
	//@ManDoc: Copy constructor (from reference).
	Space (const Space &that)
	{
		UpdateFromRecord((Record)(that));
	}
	Space (const Record *rec)
	{
		UpdateFromRecord(*rec);
	}
	//@ManDoc: Destructor.
	~Space() {}
	//@ManDoc: Type conversion: convert to a Record.
	operator const Record  () const {return rawData;}
	//@ManDoc: Return this space's shape.
	SpaceShape Shape() const {return shape;}
	//@ManDoc: Set the shape.
	SpaceShape SetShape(SpaceShape newS)
	{
		shape = newS;
		UpdateRecord();
		return shape;
	}
	//@ManDoc: Return this space's center point X coordinate.
	double CenterX() const {return centerX;}
	//@ManDoc: Set the space's center point X coordinate.
	double SetCenterX(double newX)
	{
		centerX = newX;
		UpdateRecord();
		return centerX;
	}
	//@ManDoc: Return this space's center point Y coordinate.
	double CenterY() const {return centerY;}
	//@ManDoc: Set the space's center point Y coordinate.
	double SetCenterY(double newY)
	{
		centerY = newY;
		UpdateRecord();
		return centerY;
	}
	//@ManDoc: Return the nearest exit.
	const Exit* NearestExit(
		//@ManDoc: X coordinate.
		double x,
		//@ManDoc: Y coordinate.
		double y
	) const {return exitList(x,y);}
	//@ManDoc: Return the ith exit.
	const Exit* IndexedExit(
		//@ManDoc: Index.
		int index
	) const {return exitList[index];}
	//@ManDoc: Return the count of exits.
	int NumberOfExits() const {return exitList.ElementCount();}
	//@ManDoc: Insert an exit.
	void InsertExit (const Exit& source);
	//@ManDoc: Delete exit nearest to x,y.
	void DeleteExitNear(
		//@ManDoc: X coordinate.
		double x,
		//@ManDoc: Y coordinate.
		double y
	);
	//@ManDoc: Delete exit by index.
	void DeleteExitAtIndex(
		//@ManDoc: Element to delete.
		int index
	);
	//@ManDoc: Return the nearest item.
	const Item* Nearestitem(
		//@ManDoc: X coordinate.
		double x,
		//@ManDoc: Y coordinate.
		double y
	) const {return itemList(x,y);}
	//@ManDoc: Return the ith item.
	const Item* IndexedItem(
		//@ManDoc: Index.
		int index
	) const {return itemList[index];}
	//@ManDoc: Return the count of items.
	int NumberOfItems() const {return itemList.ElementCount();}
	//@ManDoc: Insert an item.
	void InsertItem (const Item& source);
	//@ManDoc: Delete item nearest to x,y.
	void DeleteItemNear(
		//@ManDoc: X coordinate.
		double x,
		//@ManDoc: Y coordinate.
		double y
	);
	//@ManDoc: Delete item by index.
	void DeleteItemAtIndex(
		//@ManDoc: Element to delete.
		int index
	);
	//@ManDoc: Return the name of the space.
	const char *Name() const {return name;}
	//@ManDoc: Set the name of the space.
	const char *SetName(const char *newN)
	{
		name = newN;
		UpdateRecord();
		return name;
	}
	//@ManDoc: Return the description of the space.
	const char *Description() const {return descr;}
	//@ManDoc: Set the description of the space.
	const char *SetDescription(const char *newD)
	{
		descr = newD;
		UpdateRecord();
		return descr;
	}
	//@ManDoc: Return background color of the space.
	const char *BackgroundColor() const {return bgcolor;}
	//@ManDoc: Set the background color of the space.
	const char *SetBackgroundColor(const char *newBG)
	{
		bgcolor = newBG;
		UpdateRecord();
		return bgcolor;
	}
};	
	




//@}


#endif // _SPACE_H_

