/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*- vi:set ts=8 sts=4 sw=4: */

/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.
    Copyright 2000-2010 the Rosegarden development team.

    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/

//!!! NOTE: all the NOTATION_DEBUG in this file were converted to std::cerr,
// because the messages weren't ever showing up.  I've been having this problem
// in many other contexts, and we really should sort that out one of these days.
// Something is busted somewhere.  Many different SOME_DEBUG streams go straight
// to /dev/null currently.  Not even all the plain RG_DEBUG ones are working,
// and I'm not sure what the common thread is.

#include "misc/Debug.h"
#include "misc/Strings.h"
#include "ChordMap.h"

#include <QFile>
#include <QTextStream>

namespace Rosegarden
{

namespace Guitar
{
    
ChordMap::ChordMap()
    : m_needSave(false)
{
}

void ChordMap::insert(const Chord& c)
{
    std::cerr << "ChordMap::insert - inserting chord..." << std::endl;
    m_map.insert(c);
    m_needSave = true;
}


ChordMap::chordarray
ChordMap::getChords(const QString& root, const QString& ext) const
{
    chordarray res;
    
    Chord tmp(root, ext);
    std::cerr << "ChordMap::getChords : chord = " /*<< tmp*/ << " - ext is empty : " << ext.isEmpty() << std::endl;
    
    for (chordset::const_iterator i = m_map.lower_bound(tmp); i != m_map.end(); ++i) {
        std::cerr << "ChordMap::getChords : checking chord "/* << *i*/ << std::endl;
        
        if (i->getRoot() != root)
            break;

        if (/* ext.isNull() || */ i->getExt() == ext) {
            std::cerr << "ChordMap::getChords : adding chord "/* << *i*/ << std::endl;
            res.push_back(*i);
        } else {
            break;
        }
    }
        
    return res;
}
    
QStringList
ChordMap::getRootList() const
{
    static QStringList rootNotes;
    
    if (rootNotes.count() == 0) {
        rootNotes = QString("A,A#/Bb,B,C,C#/Db,D,D#/Eb,E,F,F#/Gb,G,G#/Ab")
            .split(",", QString::SkipEmptyParts);
    }
    
    // extract roots from map itself - not a very good idea
    //
//    QString currentRoot;
//    
//    for(chordset::const_iterator i = m_map.begin(); i != m_map.end(); ++i) {
//        const Chord& chord = *i;
//        if (chord.getRoot() != currentRoot) {
//            rootNotes.push_back(chord.getRoot());
//            currentRoot = chord.getRoot();
//        } 
//    }

    return rootNotes;
}

QStringList
ChordMap::getExtList(const QString& root) const
{
    QStringList extList;
    QString currentExt = "ZZ";
    
    Chord tmp(root);
    
    for(chordset::const_iterator i = m_map.lower_bound(tmp); i != m_map.end(); ++i) {
        const Chord& chord = *i;
//        std::cerr << "ChordMap::getExtList : chord = " << chord << std::endl;
         
        if (chord.getRoot() != root)
            break;
            
        if (chord.getExt() != currentExt) {
//            std::cerr << "ChordMap::getExtList : adding ext " << chord.getExt() << " for root " << root << std::endl;
            extList.push_back(chord.getExt());
            currentExt = chord.getExt();
        }        
    }

    return extList;
}

void
ChordMap::substitute(const Chord& oldChord, const Chord& newChord)
{
    remove(oldChord);
    insert(newChord);
}

void
ChordMap::remove(const Chord& c)
{
    m_map.erase(c);
    m_needSave = true;    
}

bool ChordMap::saveDocument(const QString& filename, bool userChordsOnly, QString& errMsg)
{
    QFile file(filename);
    file.open(QIODevice::WriteOnly);
   
    QTextStream outStream(&file);
    
    outStream.setEncoding(QTextStream::UnicodeUTF8);
    
    outStream << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
    << "<!DOCTYPE rosegarden-chord-data>\n"
    << "<rosegarden-chord-data version=\"" << VERSION
    << "\" format-version-major=\"" << FILE_FORMAT_VERSION_MAJOR
    << "\" format-version-minor=\"" << FILE_FORMAT_VERSION_MINOR
    << "\" format-version-point=\"" << FILE_FORMAT_VERSION_POINT
    << "\">\n";
    
    outStream << "<chords>\n";
    
    QString currentExt, currentRoot;
    
    for(iterator i = begin(); i != end(); ++i) {
        const Chord& chord = *i;
    
        if (userChordsOnly && !chord.isUserChord())
            continue; // skip non-user chords
            
        if (chord.getRoot() != currentRoot) {

            currentRoot = chord.getRoot();
            
            // close current chordset (if there was one)
            if (i != begin())
                outStream << "\n</chordset>\n";

            // open new chordset            
            outStream << "<chordset root=\"" << chord.getRoot() << "\">\n";
            currentExt = "NEWEXT"; // to make sure we open a new chord right after that
        }
    
        if (chord.getExt() != currentExt) {
            
            currentExt = chord.getExt();
            
            // close current chord (if there was one)
            if (i != begin())
                outStream << "</chord>\n";

            // open new chord            
            outStream << "<chord";
            if (!chord.getExt().isEmpty())
                outStream << " ext=\"" << chord.getExt() << "\"";
            if (chord.isUserChord())
                outStream << " user=\"true\"";
                
            outStream << ">\n";
        }
        
        outStream << "<fingering>" << chord.getFingering().toString() << "</fingering>\n";
    }

    if (!m_map.empty())
        outStream << "</chord>\n"; // close last written chord
        
    outStream << "</chords>\n";    
    outStream << "</rosegarden-chord-data>\n";
  
    return outStream.device()->status() == QTextStream::Ok;
}

int ChordMap::FILE_FORMAT_VERSION_MAJOR = 1;
int ChordMap::FILE_FORMAT_VERSION_MINOR = 0;
int ChordMap::FILE_FORMAT_VERSION_POINT = 0;


void
ChordMap::debugDump() const
{
    for(chordset::const_iterator i = m_map.begin(); i != m_map.end(); ++i) {
        Chord chord = *i;
        std::cerr << "ChordMap::debugDump " /*<< chord*/ << std::endl;
    } 
}

}

}
