/*
 * Copyright Patrick Powell 1995
 * This code is based on code written by Patrick Powell (papowell@astart.com)
 * It may be used for any purpose as long as this notice remains intact
 * on all source code distributions
 */

/* IAN: Note, on his website he claims it's public domain.  I have no idea
 * which outweighs which.  In an email he says the whole thing has been traced
 * back and all authors agree (including Patrick Powell) that it is public
 * domain. */

/**************************************************************
 * Original:
 * Patrick Powell Tue Apr 11 09:48:21 PDT 1995
 * A bombproof version of doprnt (dopr) included.
 * Sigh.  This sort of thing is always nasty do deal with.  Note that
 * the version here does not include floating point...
 *
 * snprintf() is used instead of sprintf() as it does limit checks
 * for string length.  This covers a nasty loophole.
 *
 * The other functions are there to prevent NULL pointers from
 * causing nast effects.
 *
 * More Recently:
 *  Brandon Long <blong@fiction.net> 9/15/96 for mutt 0.43
 *  This was ugly.  It is still ugly.  I opted out of floating point
 *  numbers, but the formatter understands just about everything
 *  from the normal C string format, at least as far as I can tell from
 *  the Solaris 2.5 printf(3S) man page.
 *
 *  Brandon Long <blong@fiction.net> 10/22/97 for mutt 0.87.1
 *    Ok, added some minimal floating point support, which means this
 *    probably requires libm on most operating systems.  Don't yet
 *    support the exponent (e,E) and sigfig (g,G).  Also, fmtint()
 *    was pretty badly broken, it just wasn't being exercised in ways
 *    which showed it, so that's been fixed.  Also, formated the code
 *    to mutt conventions, and removed dead code left over from the
 *    original.  Also, there is now a builtin-test, just compile with:
 *           gcc -DTEST_SNPRINTF -o snprintf snprintf.c -lm
 *    and run snprintf for results.
 * 
 *  Thomas Roessler <roessler@guug.de> 01/27/98 for mutt 0.89i
 *    The PGP code was using unsigned hexadecimal formats. 
 *    Unfortunately, unsigned formats simply didn't work.
 *
 *  Michael Elkins <me@cs.hmc.edu> 03/05/98 for mutt 0.90.8
 *    The original code assumed that both snprintf() and vsnprintf() were
 *    missing.  Some systems only have snprintf() but not vsnprintf(), so
 *    the code is now broken down under HAVE_SNPRINTF and HAVE_VSNPRINTF.
 *
 *  Andrew Tridgell (tridge@samba.org) Oct 1998
 *    fixed handling of %.0f
 *    added test for HAVE_LONG_DOUBLE
 *
 *  Russ Allbery <rra@stanford.edu> 2000-08-26
 *    fixed return value to comply with C99
 *    fixed handling of snprintf(NULL, ...)
 *
 * AND:
 *
 *  Ian Main <imain@gtk.org> 27/01/01 for entity2
 *    Heavily modified for use with rbufs.
 *
 *  Ian Main <imain@gtk.org> 04/09/01 for entity2
 *    Fixed formatting of floating points that have
 *    0's right after the decimal point.
 *
 *  This has been tested against libc versions on Linux,
 *  FreeBSD, NetBSD (sparc64 and intel), OpenBSD, Solaris (32 and
 *  64 bit), and MSVC++ on win2k.
 *
 **************************************************************/


#ifdef HAVE_STDARG_H
#include <stdarg.h>
#endif

#include <roy.h>

static void 
fmtbuf (RBuf *buffer, RBuf *value, int flags, int min);

static void
fmtstr (RBuf *buf, char *value, int flags, int min);

static void
fmtint (RBuf *buf, long value, int base, int min, int max, int flags);

static void
fmtfp (RBuf *buf, long double fvalue, int min, int max, int flags);

static void
rbuf_append_vsprintf_real (RBuf *buffer, const char *format, va_list args);

/* format read states */
#define DP_S_DEFAULT 0
#define DP_S_FLAGS   1
#define DP_S_MIN     2
#define DP_S_DOT     3
#define DP_S_MAX     4
#define DP_S_MOD     5
#define DP_S_CONV    6
#define DP_S_DONE    7

/* format flags - Bits */
#define DP_F_MINUS 	(1 << 0)
#define DP_F_PLUS  	(1 << 1)
#define DP_F_SPACE 	(1 << 2)
#define DP_F_NUM   	(1 << 3)
#define DP_F_ZERO  	(1 << 4)
#define DP_F_UP    	(1 << 5)
#define DP_F_UNSIGNED 	(1 << 6)

/* Conversion Flags */
#define DP_C_SHORT   1
#define DP_C_LONG    2
#define DP_C_LDOUBLE 3

#define char_to_int(p) (p - '0')


static void
rbuf_append_vsprintf_real (RBuf *buffer, const char *format, va_list args)
{
    char ch;
    long value;
    long double fvalue;
    char *strvalue;
    RBuf *bufvalue;
    int min;
    int max;
    int state;
    int flags;
    int cflags;
    unsigned int currlen;

    state = DP_S_DEFAULT;
    currlen = flags = cflags = min = 0;
    max = -1;
    ch = *format++;

    while (state != DP_S_DONE) {
        if (ch == '\0') 
            state = DP_S_DONE;

        switch (state) {
            case DP_S_DEFAULT:
                if (ch == '%') 
                    state = DP_S_FLAGS;
                else 
                    rbuf_append_char (buffer, ch);
                ch = *format++;
                break;
            case DP_S_FLAGS:
                switch (ch) {
                    case '-':
                        flags |= DP_F_MINUS;
                        ch = *format++;
                        break;
                    case '+':
                        flags |= DP_F_PLUS;
                        ch = *format++;
                        break;
                    case ' ':
                        flags |= DP_F_SPACE;
                        ch = *format++;
                        break;
                    case '#':
                        flags |= DP_F_NUM;
                        ch = *format++;
                        break;
                    case '0':
                        flags |= DP_F_ZERO;
                        ch = *format++;
                        break;
                    default:
                        state = DP_S_MIN;
                        break;
                }
                break;
            case DP_S_MIN:
                if (isdigit ((unsigned char) ch)) {
                    min = 10 * min + char_to_int (ch);
                    ch = *format++;
                } else if (ch == '*') {
                    min = va_arg (args, int);
                    ch = *format++;
                    state = DP_S_DOT;
                } 
                else 
                    state = DP_S_DOT;
                break;
            case DP_S_DOT:
                if (ch == '.') {
                    state = DP_S_MAX;
                    ch = *format++;
                } else {
                    state = DP_S_MOD;
                }
                break;
            case DP_S_MAX:
                if (isdigit ((unsigned char) ch)) {
                    if (max < 0)
                        max = 0;
                    max = 10*max + char_to_int (ch);
                    ch = *format++;
                } else if (ch == '*') {
                    max = va_arg (args, int);
                    ch = *format++;
                    state = DP_S_MOD;
                } else {
                    state = DP_S_MOD;
                }
                break;
            case DP_S_MOD:
                /* Currently, we don't support Long Long, bummer */
                switch (ch) {
                    case 'h':
                        cflags = DP_C_SHORT;
                        ch = *format++;
                        break;
                    case 'l':
                        cflags = DP_C_LONG;
                        ch = *format++;
                        break;
                    case 'L':
                        cflags = DP_C_LDOUBLE;
                        ch = *format++;
                        break;
                    default:
                        break;
                }
                state = DP_S_CONV;
                break;
            case DP_S_CONV:
                switch (ch) 
                {
                    case 'd':
                    case 'i':
                        if (cflags == DP_C_SHORT)
                            value = (short int) va_arg (args, int);
                        else if (cflags == DP_C_LONG)
                            value = va_arg (args, long int);
                        else
                            value = va_arg (args, int);
                        fmtint (buffer, value, 10, min, max, flags);
                        break;
                        
                    case 'o':
                        flags |= DP_F_UNSIGNED;
                        if (cflags == DP_C_SHORT)
                            value = (unsigned short int) va_arg (args, int);
                        else if (cflags == DP_C_LONG)
                            value = va_arg (args, unsigned long int);
                        else
                            value = va_arg (args, unsigned int);
                        fmtint (buffer, value, 8, min, max, flags);
                        break;
                        
                    case 'u':
                        flags |= DP_F_UNSIGNED;
                        if (cflags == DP_C_SHORT)
                            value = (unsigned short int) va_arg (args, int);
                        else if (cflags == DP_C_LONG)
                            value = va_arg (args, unsigned long int);
                        else
                            value = va_arg (args, unsigned int);
                        fmtint (buffer, value, 10, min, max, flags);
                        break;

                    case 'X':
                        flags |= DP_F_UP;

                    case 'x':
                        flags |= DP_F_UNSIGNED;
                        if (cflags == DP_C_SHORT)
                            value = (unsigned short int) va_arg (args, int);
                        else if (cflags == DP_C_LONG)
                            value = va_arg (args, unsigned long int);
                        else
                            value = va_arg (args, unsigned int);
                        fmtint (buffer, value, 16, min, max, flags);
                        break;

                    case 'f':
                        if (cflags == DP_C_LDOUBLE)
                            fvalue = va_arg (args, long double);
                        else
                            fvalue = va_arg (args, double);
                        fmtfp (buffer, fvalue, min, max, flags);
                        break;

                    case 'E':
                        flags |= DP_F_UP;

                    case 'e':
                        if (cflags == DP_C_LDOUBLE)
                            fvalue = va_arg (args, long double);
                        else
                            fvalue = va_arg (args, double);
                        break;

                    case 'G':
                        flags |= DP_F_UP;

                    case 'g':
                        if (cflags == DP_C_LDOUBLE)
                            fvalue = va_arg (args, long double);
                        else
                            fvalue = va_arg (args, double);
                        break;

                    case 'c':
                        rbuf_append_char (buffer, (char) va_arg (args, int));
                        break;

                    case 'b':
                        bufvalue = va_arg (args, RBuf *);
                        if (max < 0) 
                        fmtbuf (buffer, bufvalue, flags, min);
                        break;

                    case 's':
                        strvalue = va_arg (args, char *);
                        if (max < 0) 
                        fmtstr (buffer, strvalue, flags, min);
                        break;

                    case 'p':
                        strvalue = va_arg (args, void *);
                        fmtint (buffer, (long) strvalue, 16, min, max, flags);
                        break;

                    case 'n':
                        if (cflags == DP_C_SHORT) {
                            short int *num;
                            num = va_arg (args, short int *);
                            *num = currlen;
                        } else if (cflags == DP_C_LONG) {
                            long int *num;
                            num = va_arg (args, long int *);
                            *num = currlen;
                        } else {
                            int *num;
                            num = va_arg (args, int *);
                            *num = currlen;
                        }
                        break;

                    case '%':
                        rbuf_append_char (buffer, ch);
                        break;

                    case 'w':
                        /* not supported yet, treat as next char */
                        ch = *format++;
                        break;

                    default:
                        /* Unknown, skip */
                        break;

                }
                
                ch = *format++;
                state = DP_S_DEFAULT;
                flags = cflags = min = 0;
                max = -1;
                break;

            case DP_S_DONE:
                break;
                
            default:
                break; /* some picky compilers need this */
        }
    }
}

static void 
fmtbuf (RBuf *buffer, RBuf *value, int flags, int min)
{
    int padlen, strln;     /* amount to pad */

    if (value == 0) {
        value = rbuf_auto ("<NULL>");
    }

    strln = rbuf_len (value);

    padlen = min - strln;
    if (padlen < 0) 
        padlen = 0;
    if (flags & DP_F_MINUS) 
        padlen = -padlen; /* Left Justify */

    while (padlen > 0) {
        rbuf_append_char (buffer, ' ');
        --padlen;
    }

    rbuf_append_rbuf (buffer, value);
    
    while (padlen < 0) {
        rbuf_append_char (buffer, ' ');
        ++padlen;
    }
}

static void 
fmtstr (RBuf *buffer, char *value, int flags, int min)
{
    int padlen, strln;     /* amount to pad */

    if (value == 0)
    {
        value = "<NULL>";
    }

    for (strln = 0; value[strln]; ++strln); /* strlen */

    padlen = min - strln;
    if (padlen < 0) 
        padlen = 0;
    
    if (flags & DP_F_MINUS) 
        padlen = -padlen; /* Left Justify */

    while (padlen > 0) {
        rbuf_append_char (buffer, ' ');
        --padlen;
    }

    rbuf_append_str (buffer, value);
    
    while (padlen < 0) {
        rbuf_append_char (buffer, ' ');
        ++padlen;
    }
}

/* Have to handle DP_F_NUM (ie 0x and 0 alternates) */

static void
fmtint (RBuf *buf, long value, int base, int min, int max, int flags)
{
    int signvalue = 0;
    unsigned long uvalue;
    char convert[20];
    int place = 0;
    int spadlen = 0; /* amount to space pad */
    int zpadlen = 0; /* amount to zero pad */
    int caps = 0;

    if (max < 0)
        max = 0;

    uvalue = value;

    if ( !(flags & DP_F_UNSIGNED)) {
        if (value < 0) {
            signvalue = '-';
            uvalue = -value;
        } else {
            if (flags & DP_F_PLUS)  {
                /* Do a sign (+/i) */
                signvalue = '+';
            } else {
                if (flags & DP_F_SPACE)
                    signvalue = ' ';
            }
        }
    }

    if (flags & DP_F_UP) 
        caps = 1; /* Should characters be upper case? */

    do {
        convert[place++] = (caps ? "0123456789ABCDEF":"0123456789abcdef") [uvalue % (unsigned) base];
        uvalue = (uvalue / (unsigned) base);

    } while (uvalue && (place < 20));
    
    if (place == 20) 
        place--;

    convert[place] = 0;

    zpadlen = max - place;
    spadlen = min - RMAX (max, place) - (signvalue ? 1 : 0);
    
    if (zpadlen < 0)
        zpadlen = 0;
    
    if (spadlen < 0) 
        spadlen = 0;
    
    if (flags & DP_F_ZERO) {
        zpadlen = RMAX(zpadlen, spadlen);
        spadlen = 0;
    }
    
    if (flags & DP_F_MINUS) 
        spadlen = -spadlen; /* Left Justifty */

#ifdef DEBUG_SNPRINTF
    printf ("zpad: %d, spad: %d, min: %d, max: %d, place: %d\n",
            zpadlen, spadlen, min, max, place);
#endif

    /* Spaces */
    while (spadlen > 0) {
        rbuf_append_char (buf, ' ');
        --spadlen;
    }

    /* Sign */
    if (signvalue)
        rbuf_append_char (buf, (char) signvalue);

    /* Zeros */
    if (zpadlen > 0) {
        while (zpadlen > 0) {
            rbuf_append_char (buf, '0');
            --zpadlen;
        }
    }

    /* Digits */
    while (place > 0)
        rbuf_append_char (buf, convert[--place]);

    /* Left Justified spaces */
    while (spadlen < 0) {
        rbuf_append_char (buf, ' ');
        ++spadlen;
    }
}

static long double
abs_val (long double value)
{
    long double result = value;

    if (value < 0)
        result = -value;

    return result;
}

static long double
pow10 (int exp)
{
    long double result = 1;

    while (exp) {
        result *= 10;
        exp--;
    }

    return result;
}

static long
round (long double value)
{
    long intpart;

    intpart = (long) value;
    value = value - intpart;
    if (value >= 0.5)
        intpart++;

    return (intpart);
}

static int
get_num_zeros (long double value)
{
    int num_zeros = 0;

    if (value == 0.0)
        return (0);

    value *= 10.0;

    while (value < 1.0) {
        value *= 10.0;
        num_zeros++;
    }

    return (num_zeros);
}

static void
fmtfp (RBuf *buffer, long double fvalue, int min, int max, int flags)
{
    int signvalue = 0;
    long double abs_value;
    char iconvert[20];
    char fconvert[20];
    int iplace = 0;
    int fplace = 0;
    int padlen = 0; /* amount to pad */
    int zpadlen = 0; 
    int caps = 0;
    int num_zeros = 0;
    long intpart;
    long fracpart;

    /* 
     * AIX manpage says the default is 0, but Solaris says the default
     * is 6, and sprintf on AIX defaults to 6
     */
    if (max < 0)
        max = 6;

    abs_value = abs_val (fvalue);

    if (fvalue < 0) {
        signvalue = '-';
    } else {
        if (flags & DP_F_PLUS) {
            /* Do a sign (+/i) */
            signvalue = '+';
        } else {
            if (flags & DP_F_SPACE) {
                signvalue = ' ';
            }
        }
    }

   
#if 0
    if (flags & DP_F_UP) 
        caps = 1; /* Should characters be upper case? */
#endif


    intpart = (long) abs_value;

    /* 
     * Sorry, we only support 9 digits past the decimal because of our 
     * conversion method
     */
    if (max > 9)
        max = 9;

    /* We "cheat" by converting the fractional part to integer by
     * multiplying by a factor of 10
     */
    fracpart = round ((pow10 (max)) * (abs_value - (long double) intpart));
  
    num_zeros = get_num_zeros (abs_value - (long double) intpart);

    if (fracpart >= pow10 (max)) {
        intpart++;
        fracpart -= (long) pow10 (max);
    }


#ifdef DEBUG_SNPRINTF
    printf ("fmtfp: %Lf =? int part: %ld  frac part: %ld\n", fvalue, intpart, fracpart);
#endif

    /* Convert integer part */
    do {
        iconvert[iplace++] = (caps ? "0123456789ABCDEF":"0123456789abcdef")[intpart % 10];
        intpart /= 10;
    } while ((intpart > 0) && (iplace < 20));
    
    if (iplace == 20) {
        iplace--;
    }
    iconvert[iplace] = '\0';

   
    
    /* Convert fractional part */
    do {
        fconvert[fplace++] = (caps? "0123456789ABCDEF":"0123456789abcdef")[fracpart % 10];
        fracpart /= 10;
    } while ((fracpart > 0) && (fplace < (20 - num_zeros)) && (fplace < (max - num_zeros)));
    
    /* Insert the appropriate number of 0's */
    while ((num_zeros > 0) && (fplace < 20) && (fplace < max)) {
        fconvert[fplace++] = '0';
        num_zeros--;
    }

    if (fplace == 20) {
        fplace--;
    }
    fconvert[fplace] = '\0';

    /* -1 for decimal point, another -1 if we are printing a sign */
    padlen = min - iplace - max - 1 - ((signvalue) ? 1 : 0); 
    zpadlen = max - fplace;
    
    if (zpadlen < 0)
        zpadlen = 0;
    
    if (padlen < 0) 
        padlen = 0;
    
    if (flags & DP_F_MINUS) 
        padlen = -padlen; /* Left Justifty */

    if ((flags & DP_F_ZERO) && (padlen > 0)) {
        if (signvalue) {
            rbuf_append_char (buffer, (char) signvalue);
            --padlen;
            signvalue = 0;
        }
        while (padlen > 0) {
            rbuf_append_char (buffer, '0');
            --padlen;
        }
    }
    while (padlen > 0) {
        rbuf_append_char (buffer, ' ');
        --padlen;
    }
    if (signvalue) 
        rbuf_append_char (buffer, (char) signvalue);

    while (iplace > 0) 
        rbuf_append_char (buffer, iconvert[--iplace]);

    /*
     * Decimal point.  This should probably use locale to find the correct
     * char to print out.
     */
    if (max > 0) {
        rbuf_append_char (buffer, '.');

        while (fplace > 0)
            rbuf_append_char (buffer, fconvert[--fplace]);
    }

    while (zpadlen > 0) {
        rbuf_append_char (buffer, '0');
        --zpadlen;
    }

    while (padlen < 0) {
        rbuf_append_char (buffer, ' ');
        ++padlen;
    }
}

void
rbuf_append_vsprintf (RBuf *buffer, const char *format, va_list args)
{
    RBuf *newbuf;

    newbuf = rbuf_new ();

    rbuf_append_vsprintf_real (newbuf, format, args);

    rbuf_append_rbuf (buffer, newbuf);
    rbuf_free (newbuf);
}


void
rbuf_append_sprintf (RBuf *buffer, const char *fmt, ...)
{
    va_list ap;
    RBuf *newbuf;

    newbuf = rbuf_new ();

    va_start (ap, fmt);
    rbuf_append_vsprintf_real (newbuf, fmt, ap);
    va_end (ap);

    rbuf_append_rbuf (buffer, newbuf);
    rbuf_free (newbuf);
}

void
rbuf_sprintf (RBuf *buffer, const char *fmt, ...)
{
    RBuf tmp;

    va_list ap;
    RBuf *newbuf;

    newbuf = rbuf_new ();

    va_start (ap, fmt);
    rbuf_append_vsprintf_real (newbuf, fmt, ap);
    va_end (ap);

    /* Here we just do a simple swap of RBuf struct
     * contents instead of doing a copy */
    tmp = *buffer;
    *buffer = *newbuf;
    *newbuf = tmp;

    /* Now freeing this buffer is really freeing the
     * newly created struct, and the passed in data */
    rbuf_free (newbuf);
}

RBuf *
rbuf_new_with_sprintf (const char *fmt, ...)
{
    RBuf *buffer;
    va_list ap;

    buffer = rbuf_new ();
    
    va_start (ap, fmt);
    rbuf_append_vsprintf_real (buffer, fmt, ap);
    va_end (ap);

    return (buffer);
}



