/*				       	-*- c-file-style: "bsd" -*-
 * rproxy -- dynamic caching and delta update in HTTP
 * $Id: socklib.c,v 1.19 2000/08/25 07:08:56 mbp Exp $
 * 
 * Copyright (C) 1999, 2000 by Martin Pool
 * Copyright (C) 1999 by tridge@samba.org
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

					/*
                                         | Nothing motivates a man
					 | more than to see his boss
					 | put in an honest day's
					 | work.
                                         */

#include "config.h"
#include "sysheaders.h"

#include <wait.h>
#include <sys/socket.h>
#include <netdb.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>

#include "rproxy.h"
#include "util.h"
#include "socklib.h"
#include "trace.h"
#include "myname.h"
#include "sockopt.h"
#include "error.h"

/*
 * Open a socket of the specified type, port and address for incoming
 * data.
 */
int
open_socket_in(int type, int port, uint32_t socket_addr,
	       char const *sock_opt_str)
{
    struct hostent *hp;
    struct sockaddr_in sock;
    char            host_name[1000];
    int             res;
    int             one = 1;

    /* get my host name */
    if (gethostname(host_name, sizeof(host_name)) == -1) {
	rp_log(LOGAREA_PROCESS, LOG_ERR, "can't work out my own hostname: %s: check /etc/hosts?",
               strerror(errno));
	return -1;
    }

    /* get host info */
    if ((hp = gethostbyname(host_name)) == 0) {
	rp_log(LOGAREA_PROCESS, LOG_ERR, "can't get hostent for %s: %s: check /etc/hosts?",
               host_name, strerror(errno));
	return -1;
    }

    bzero((char *) &sock, sizeof(sock));
    memcpy((char *) &sock.sin_addr, (char *) hp->h_addr, hp->h_length);

#ifdef HAVE_SOCK_SIN_LEN
    sock.sin_len = sizeof(sock);
#endif
    sock.sin_port = htons(port);
    sock.sin_family = hp->h_addrtype;
    sock.sin_addr.s_addr = socket_addr;
    res = socket(hp->h_addrtype, type, 0);
    if (res == -1) {
	rp_log(LOGAREA_PROCESS, LOG_ERR, "can't create a new server socket "
               "of addrtype %d, type %d: %s", hp->h_addrtype,
               type, strerror(errno));
	return -1;
    }

    setsockopt(res, SOL_SOCKET, SO_REUSEADDR, (char *) &one, sizeof(one));

    if (sock_opt_str)
	set_socket_options(res, sock_opt_str);

    /* now we've got a socket - we need to bind it */
    if (bind(res, (struct sockaddr *) &sock, sizeof(sock)) < 0) {
        rp_log(LOGAREA_PROCESS, LOG_ERR, "can't bind socket to port %d: %s",
               port, strerror(errno));
	return (-1);
    }

    return res;
}


/* Split a string into host and port parts.  arg is modified.  If there * is
   no colon, then the port is left unchanged.  */
void
_rp_split_host_port(char *arg, char **host, int *port)
{
    char           *p;

    assert(arg);
    p = strchr(arg, ':');
    if (p) {
        p++;
        *port = atoi(p);
        if (!*port) {
            struct servent *s;
            s = getservbyname(p, "tcp");
            if (!s) {
                rp_log(LOGAREA_PROCESS, LOG_ERR, "can't find tcp service %s: %s",
                       p, strerror(errno));
                exit(EXIT_STARTUP_ERR);
            }
            *port = ntohs(s->s_port);
        }
    }
    *host = xstrndup(arg, strcspn(arg, ":"));
}



/* open a socket to a tcp remote host with the specified port based on code
   from Warren */
int
rp_open_socket_out(request_t *req, char const *host, int port, int *sock)
{
    int             type = SOCK_STREAM;
    struct sockaddr_in sock_out;
    int             res;
    struct hostent *hp;

    res = socket(PF_INET, type, 0);
    if (res == -1) {
	return -1;
    }

    if (!host || !*host)
	host = "localhost";

    hp = gethostbyname(host);
    if (!hp) {
        rp_log(LOGAREA_PROTO, LOG_ERR,
                     "unknown host: %s: %s", host,
                     hstrerror(h_errno));
	rp_request_failed(req, HTTP_SERVICE_UNAVAILABLE,
                          "couldn't find host entry for %s: %s",
                          host, hstrerror(h_errno));
    }

    memcpy(&sock_out.sin_addr, hp->h_addr, hp->h_length);
    sock_out.sin_port = htons(port);
    sock_out.sin_family = PF_INET;

    if (connect(res, (struct sockaddr *) &sock_out, sizeof(sock_out))) {
	close(res);
        /* See http://cr.yp.to/docs/connect.html for notes on handling
         * errors from connect */
        rp_log(LOGAREA_PROTO, LOG_ERR,
                     "failed to connect to %s:%d: %s",
                     host, port, strerror(errno));
	return HTTP_SERVICE_UNAVAILABLE;
    }

    *sock = res;
    return DONE;
}



/*
 * SIGCHLD handler, invoked when a child terminates.  Actually, several
 * children may die while the signal is still pending, and signals don't
 * queue up.  So we have to keep looking until we've got all of them.
 *
 * Based on Stevens, ``Advanced Programming in the UNIX Environment'',
 * program 18.11.
 */

static void
rp_reap_children(void)
{
    int             status, errno_save;
    pid_t           pid;

    errno_save = errno;

    while ((pid = waitpid(-1, &status, WNOHANG)) > 0) {
	if (WIFSIGNALED(status)) {
	    char const     *signame = strsignal(WTERMSIG(status));

	    rp_log(LOGAREA_PROCESS, LOG_ERR,
                   "child %d exited abnormally: signal %d: %s %s\n",
                   pid, WTERMSIG(status),
                   signame ? signame : "(unknown signal)",
                   WCOREDUMP(status) ? ": core dumped" : "");
	} else if (WIFEXITED(status)) {
	    int             rc = WEXITSTATUS(status);

	    if (rc)
		trace(LOGAREA_PROCESS,
		      "child %d exited with exit code %d", pid, rc);
	    else
		trace(LOGAREA_PROCESS, "child %d died happy", pid);
	}
    }

    /* ECHILD just means that we have no more children alive or dead; we
       don't worry about that. */

    if (pid == -1 && errno != ECHILD) {
	rp_log(LOGAREA_PROCESS, LOG_ERR, "waitpid error: %s(%d)\n", strerror(errno), errno);
    }

    errno = errno_save;
}


#if 0
static void
hook_sigchld(void)
{
    struct sigaction act;

    bzero(&act, sizeof act);
    act.sa_handler = rp_sigchld_handler;
    act.sa_flags = SA_NOCLDSTOP;
    sigaction(SIGCHLD, &act, NULL);
}
#endif



/*
 * Serve a single request on stdin/stdout.  This is useful for running under
 * inetd, if you want to do that.  It's also good for debugging -- you can
 * just pipe in the test data without the hassle of binding ports.
 * 
 * Call this instead of start_accept_loop.
 */
void
serve_inetd(int (*fn) (int, int, struct sockaddr *))
{
    struct sockaddr addr, *paddr;
    size_t len;

    len = sizeof addr;
    if (getpeername(STDIN_FILENO, &addr, &len) == 0)
        paddr = &addr;
    else
        paddr = 0;
    /* if this fails, it's probably not a socket, so ignore it */
    
    rp_log(LOGAREA_NET, LOG_INFO, "accepted inetd connection");

    fn(STDIN_FILENO, STDOUT_FILENO, paddr);
}


void
start_accept_loop(int port,
		  int (*fn) (int, int, struct sockaddr *),
		  int dont_fork, char const *sock_opts)
{
    int             s;

    /* open an incoming socket */
    s = open_socket_in(SOCK_STREAM, port, 0, sock_opts);
    if (s == -1) {
	exit(1);
    }

    /* ready to listen */
    if (listen(s, 5) == -1) {
	close(s);
	exit(1);
    }
    
    trace(LOGAREA_HTTP, "listening for connections on :%d", port);
    rp_process_name("listening on :%d", port);

/*      hook_sigchld(); */

    /* Now accept incoming connections - forking a new process for
     * each incoming connection */
    while (1) {
	int             fd;
	struct sockaddr client_addr;
	int             in_addrlen = sizeof(client_addr);

        rp_reap_children();

	fd = accept(s, &client_addr, &in_addrlen);

	if (fd == -1) {
            rp_log(LOGAREA_NET, LOG_ERR, "accept on port %d failed: %s",
                   port, strerror(errno));
	    continue;
        }

	if (dont_fork) {
	    /* For debugging, just execute in the same process.  This
             * means we can serve only one request at a time. */
	    fn(fd, fd, &client_addr);
	} else {
	    if (fork() == 0) {
		close(s);
		exit(fn(fd, fd, &client_addr));
	    }
	}

	close(fd);
    }
}
