/*				       	-*- c-file-style: "bsd" -*-
 * rproxy -- dynamic caching and delta update in HTTP
 * $Id: urllib.c,v 1.12 2000/08/13 10:40:32 mbp Exp $
 * 
 * Copyright (C) 1999, 2000 by Martin Pool
 * Copyright (C) 1999 by Andrew Tridgell
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include "sysheaders.h"
#include "rproxy.h"

#include "urllib.h"
#include "util.h"
#include "msgpage.h"
#include "error.h"


/* Return true if the status code is reasonable */
int
reasonable_status(int code)
{
    return code >= 100 && code < 600;
}


/* 
 * Return true if we'd like to handle a request with HTTP method
 * req->method_id
 */
int
request_check_method(request_t * req)
{
    /* XXX: What should we do if we don't recognize the method at all?
     * I reckon we should probably fail it.  What does the RFC say? */
    
    if (req->method_id == METHOD_CONNECT) {
	/* FIXME: We MUST (rfc2616/s10.4.6) send an Allow header
         * saying which methods we will accept. Godnose what the
         * client's meant to do then, though: it can hardly rewrite
         * one type of request into another.  I guess it's just for
         * diagnostic purposes. */
	rp_request_failed(req, HTTP_METHOD_NOT_ALLOWED,
		       "rproxy doesn't do tunnelled connections at "
		       "the moment.<p>"
		       "This means it can't act as an https proxy.");
	return 0;
    }

    /* hope it's OK to accept every other method */
    return 1;
}


/* Return true if we're happy to use the HTTP protocol of REQ */
int
request_check_protocol(request_t * UNUSED(req))
{
    /* TODO: Something useful!  We don't necessarily comply to *any*
     * HTTP spec exactly at the moment, so we don't check it. */
    return 1;
}


/* 
 * Find the upstream hostname and port from the request.  This will work only
 * if the second word of the request line is of the form
 * 
 * "METHOD URL PROTOCOL"
 * 
 * XXX: This is a bad name for this function.
 * 
 * READ: req->url WRITE: req->path
 */
void
request_parse_url(request_t * req)
{
    char const     *start;
    char           *end;
    size_t          len;

    assert(req->url);

    /* FIXME: This doesn't work in accelerator mode at the moment -- fix
       that.  We must be able to cope with getting just a path -- how are we
       supposed to tell?  Perhaps if it doesn't begin with a protocol then we 
       assume it's a path? */

    /* RFC2616: ``Comparisons of scheme names MUST be case-insensitive'' */

    /* RFC2616 also says ``Comparisons of host names MUST be
       case-insensitive'' -- so perhaps we should squash it before computing
       the cache name?  Would that be a bit naughty? */

    if (!config.upstream_host && (strncasecmp(req->url, "http://", 7))) {
	rp_request_failed(req,
		       HTTP_BAD_REQUEST,
		       "can't handle the URL %s without an "
		       "upstream proxy, because it "
		       "doesn't begin with `http://'", req->url);
	abort();
    }

    start = req->url + 7;	/* "http://" */
    len = strcspn(start, ":/");
    if (!len) {
	rp_request_failed(req,
		       HTTP_BAD_REQUEST,
		       "can't find the hostname in this url");
	abort();
    }

    req->url_hostname = xstrndup(start, len);

    if (start[len] == ':') {
	req->url_port = strtol(start + len + 1, &end, 10);
	assert(*end = '/');
	req->path = xstrdup(end);
    } else {
	req->url_port = DEFAULT_HTTP_PORT;
	req->path = xstrdup(start + len);
    }
}
