;;;

(define-method line-vector ((self <line>))
  (values (- (x (to self)) (x (from self)))
	  (- (y (to self)) (y (from self)))))

(define-method point-on ((l <line>) (t <real>))
  (bind ((dx dy (line-vector l)))
    (make-point (+ (x (from l)) (* dx t))
		(+ (y (from l)) (* dy t)))))

(define-method distance^2 ((a <point>) (b <point>))
  (let ((dx (- (x a) (x b)))
	(dy (- (y a) (y b))))
    (+ (* dx dx) (* dy dy))))

;;  distance: <geometric> <point> => <real> <point>

(define-method distance ((a <point>) (b <point>))
  (values (sqrt (distance^2 a b)) a))

(define-method intersection-parameter ((l <line>) (p <point>))
  (bind ((dx dy (line-vector l))
	 (p-ax (- (x p) (x (from l))))
	 (p-ay (- (y p) (y (from l)))))
    (/ (+ (* p-ax dx) (* p-ay dy))
       (+ (* dx dx) (* dy dy)))))

;; returns the distance to a line segment
;; in particular, this may involve the distance to
;; one of the end points

(define-method distance ((l <line>) (p <point>))
  (let* ((t (intersection-parameter l p))
	 (a (point-on l (min (max t 0) 1))))
    (distance a p)))

(define-class <bezier-curve> (<curve>)
  (start-point type: <point>)
  (first-handle type: <point>)
  (second-handle type: <point>)
  (end-point type: <point>)
  (coeff-cache type: <vector> init-value: '#()))

(define-method clear-coefficient-cache! ((self <bezier-curve>))
  (set-coeff-cache! self '#())
  (values))

(define (curv x0 y0 x1 y1 x2 y2 x3 y3)
  (make <bezier-curve>
	start-point: (make-point x0 y0)
	first-handle: (make-point x1 y1)
	second-handle: (make-point x2 y2)
	end-point: (make-point x3 y3)))

#|
 Solve[ { p1 == p0 + c/3,
	  p2 == p1 + (c + b) / 3,
	  p3 == p0 + c + b + a },
        {ax,ay,bx,by,cx,cy,dx,dy} ]
 {{ax -> -x0 + 3 x1 - 3 x2 + x3, 
   ay -> -y0 + 3 y1 - 3 y2 + y3, 
   bx -> 3 x0 - 6 x1 + 3 x2, by -> 3 y0 - 6 y1 + 3 y2, 
   cx -> -3 x0 + 3 x1, cy -> -3 y0 + 3 y1}}
|#

(define-method coeffs ((self <bezier-curve>))
  (if (eq? (vector-length (coeff-cache self)) 0)
      (let (((p0 <point>) (start-point self))
	    ((p1 <point>) (first-handle self))
	    ((p2 <point>) (second-handle self))
	    ((p3 <point>) (end-point self)))
	(set-coeff-cache! 
	 self
	 (vector (+ (- (x p0)) (* 3 (x p1)) (* -3 (x p2)) (x p3))
		 (+ (- (y p0)) (* 3 (y p1)) (* -3 (y p2)) (y p3))
		 (+ (* 3 (x p0)) (* -6 (x p1)) (* 3 (x p2)))
		 (+ (* 3 (y p0)) (* -6 (y p1)) (* 3 (y p2)))
		 (+ (* -3 (x p0)) (* 3 (x p1)))
		 (+ (* -3 (y p0)) (* 3 (y p1)))
		 (x p0)
		 (y p0)))
	(coeff-cache self))
      (coeff-cache self)))

(define-method point-on ((c <bezier-curve>) (t <real>))
  (let* ((t^2 (* t t))
	 (t^3 (* t^2 t))
	 ((v <vector>) (coeffs c)))
    (make-point (+ (* t^3 (vector-ref v 0))
		   (* t^2 (vector-ref v 2))
		   (* t (vector-ref v 4))
		   (vector-ref v 6))
		(+ (* t^3 (vector-ref v 1))
		   (* t^2 (vector-ref v 3))
		   (* t (vector-ref v 5))
		   (vector-ref v 7)))))

(define-syntax (point-avg a b)
  (make-point (/ (+ (x a) (x b)) 2)
	      (/ (+ (y a) (y b)) 2)))

(define-method subdivide ((c <bezier-curve>))
  ;; page 508
  ;; [p1 p2 p3 p4] -> [p1 l2 l3 j] + [j r2 r3 p4]
  (let* (((p1 <point>) (start-point c))
	 ((p2 <point>) (first-handle c))
	 ((p3 <point>) (second-handle c))
	 ((p4 <point>) (end-point c))
	 ((l2 <point>) (point-avg p1 p2))
	 ((h <point>)  (point-avg p2 p3))
	 ((l3 <point>) (point-avg l2 h))
	 ((r3 <point>) (point-avg p3 p4))
	 ((r2 <point>) (point-avg h r3))
	 ((j <point>)  (point-avg r3 r2)))
    (values (make <bezier-curve>
		  start-point: p1
		  first-handle: l2
		  second-handle: l3
		  end-point: j)
	    (make <bezier-curve>
		  start-point: j
		  first-handle: r2
		  second-handle: r3
		  end-point: p4))))
