/*
*  program name: Rubrica <http://digilander.iol.it/nfragale/download.html>
*  file: card.c
*  
*  Copyright (C) 2000-2002 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <config.h>

#include <gnome.h>
#include <gconf/gconf-client.h>

#include "main.h"
#include "callbacks.h"
#include "card.h"
#include "form_int.h"
#include "view.h"
#include "rubrica.h"
#include "ui.h"
#include "rubrica_type.h"
#include "dialog.h"
#include "prefer.h"
#include "prefer_configure.h"
#include "sort.h"
#include "browse.h"

GtkWidget *tree_view;
GtkWidget *notebook;
gboolean confirm_remove = TRUE;
gint pages = 0;


static RubricaMenu columns[] = {
  RUBRICA_MENU_CHECK(N_("Show card column"), on_pop_check_item_clicked, 
                     RUBRICA_KEY_SHOW_CARD_COLUMN_CARD),
  RUBRICA_MENU_CHECK(N_("Show first name column"), on_pop_check_item_clicked,
		     RUBRICA_KEY_SHOW_CARD_COLUMN_FIRST_NAME),
  RUBRICA_MENU_CHECK(N_("Show last name column"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_CARD_COLUMN_LAST_NAME),
  RUBRICA_MENU_CHECK(N_("Show profession column"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_CARD_COLUMN_PROFESSION),
  RUBRICA_MENU_CHECK(N_("Show city column"), on_pop_check_item_clicked,
		     RUBRICA_KEY_SHOW_CARD_COLUMN_CITY),
  RUBRICA_MENU_CHECK(N_("Show country column"), on_pop_check_item_clicked,
		     RUBRICA_KEY_SHOW_CARD_COLUMN_COUNTRY),
  RUBRICA_MENU_CHECK(N_("Show web column"), on_pop_check_item_clicked,
		     RUBRICA_KEY_SHOW_CARD_COLUMN_WEB),
  RUBRICA_MENU_CHECK(N_("Show email column"), on_pop_check_item_clicked,
		     RUBRICA_KEY_SHOW_CARD_COLUMN_EMAIL),
  RUBRICA_MENU_CHECK(N_("Show telephone column"), on_pop_check_item_clicked,
		     RUBRICA_KEY_SHOW_CARD_COLUMN_TELEPHONE),
  RUBRICA_MENU_CHECK(N_("Show company column"), on_pop_check_item_clicked,
		     RUBRICA_KEY_SHOW_CARD_COLUMN_COMPANY),
  RUBRICA_MENU_CHECK(N_("Show assigment column"), on_pop_check_item_clicked,
		     RUBRICA_KEY_SHOW_CARD_COLUMN_ASSIGMENT),
  RUBRICA_MENU_CHECK(N_("Show categories column"), on_pop_check_item_clicked,
		     RUBRICA_KEY_SHOW_CARD_COLUMN_CATEGORIES),
  RUBRICA_MENU_CHECK(N_("Show categories pixmap column"), 
		     on_pop_check_item_clicked,
		     RUBRICA_KEY_SHOW_CARD_COLUMN_PIXMAP),
  RUBRICA_MENU_END
};

static RubricaMenu personal[] = {  
  RUBRICA_MENU_CHECK(N_("Show first name"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_FIRST_NAME),
  RUBRICA_MENU_CHECK(N_("Show middle name"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_MIDDLE_NAME),
  RUBRICA_MENU_CHECK(N_("Show last name"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_LAST_NAME),
  RUBRICA_MENU_CHECK(N_("Show profession"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_PROFESSION),
  RUBRICA_MENU_CHECK(N_("Show name prefix"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_NAME_PREFIX),
  RUBRICA_MENU_CHECK(N_("Show title"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_TITLE),
  RUBRICA_MENU_CHECK(N_("Show birthday"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_BIRTHDAY),
  RUBRICA_MENU_END
};

static RubricaMenu address[] ={
  RUBRICA_MENU_CHECK(N_("Show street"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_STREET),
  RUBRICA_MENU_CHECK(N_("Show street number"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_STREET_NUM),
  RUBRICA_MENU_CHECK(N_("Show zip code"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_ZIP_CODE),
  RUBRICA_MENU_CHECK(N_("Show city"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_CITY),
  RUBRICA_MENU_CHECK(N_("Show province"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_PROVINCE),
  RUBRICA_MENU_CHECK(N_("Show state"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_STATE),
  RUBRICA_MENU_CHECK(N_("Show country"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_COUNTRY),
  RUBRICA_MENU_END
};


static RubricaMenu communication[] = {
  RUBRICA_MENU_CHECK(N_("Show web urls"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_WEB),
  RUBRICA_MENU_CHECK(N_("Show email addresses"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_EMAIL),
  RUBRICA_MENU_CHECK(N_("Show telephone numbers"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_TELEPHONE),
  RUBRICA_MENU_END
};

static RubricaMenu company[] = {
  RUBRICA_MENU_CHECK(N_("Show company"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_COMPANY),
  RUBRICA_MENU_CHECK(N_("Show street"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_JOB_STREET),
  RUBRICA_MENU_CHECK(N_("Show street number"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_JOB_STREET_NUM),
  RUBRICA_MENU_CHECK(N_("Show zip code"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_JOB_ZIP_CODE),
  RUBRICA_MENU_CHECK(N_("Show city"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_JOB_CITY),
  RUBRICA_MENU_CHECK(N_("Show province"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_JOB_PROVINCE),
  RUBRICA_MENU_CHECK(N_("Show country"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_JOB_COUNTRY),
  RUBRICA_MENU_CHECK(N_("Show web url"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_JOB_WEB),
  RUBRICA_MENU_CHECK(N_("Show email"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_JOB_EMAIL),
  RUBRICA_MENU_CHECK(N_("Show operator"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_OPERATOR),
  RUBRICA_MENU_CHECK(N_("Show fax"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_FAX),
  RUBRICA_MENU_CHECK(N_("Show green"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_GREEN),
  RUBRICA_MENU_CHECK(N_("Show customer care"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_CUSTOMER_CARE),
  RUBRICA_MENU_CHECK(N_("Show notes"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_JOB_NOTE),
  RUBRICA_MENU_END
};


static RubricaMenu assigment[] = {
  RUBRICA_MENU_CHECK(N_("Show assigment"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_ASSIGMENT),
  RUBRICA_MENU_CHECK(N_("Show organization"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_ORG),
  RUBRICA_MENU_CHECK(N_("Show department"), on_pop_check_item_clicked,
		     RUBRICA_KEY_SHOW_DATA_DEP),
  RUBRICA_MENU_CHECK(N_("Show sub department"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_SUB_DEP),
  RUBRICA_MENU_CHECK(N_("Show secretary name"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_SECRETARY),
  RUBRICA_MENU_CHECK(N_("Show secretary phone"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_SECR_PHONE),
  RUBRICA_MENU_END
};


static  RubricaMenu note[] = {
  RUBRICA_MENU_CHECK(N_("Show if contact is married"), 
		     on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_MARRIED),
  RUBRICA_MENU_CHECK(N_("Show spouse name"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_SPOUSE),
  RUBRICA_MENU_CHECK(N_("Show spouse birthday"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_SPOUSE_BIRTH),
  RUBRICA_MENU_CHECK(N_("Show anniversary"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_ANNIVERSARY),
  RUBRICA_MENU_CHECK(N_("Show if contact has children"), 
		     on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_HAS_CHILDREN),
  RUBRICA_MENU_CHECK(N_("Show children notes"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_CHILDREN),
  RUBRICA_MENU_CHECK(N_("Show contact hobbies"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_HOBBIES),
  RUBRICA_MENU_CHECK(N_("Show notes"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_NOTE),
  RUBRICA_MENU_CHECK(N_("Show public key"), on_pop_check_item_clicked, 
		     RUBRICA_KEY_SHOW_DATA_PUBLIC_KEY),     
  RUBRICA_MENU_END
};


static RubricaMenu popcardmenu[] = {
  {
    RM_ITEM, N_("Cut"), NULL, RMP_STOCK, GTK_STOCK_CUT, 
    NULL, on_pop_cut_activate, NULL
  },
  {
    RM_ITEM, N_("Copy"), NULL, RMP_STOCK, GTK_STOCK_COPY, 
    NULL, on_pop_copy_activate, NULL
  },
  {
    RM_ITEM, N_("Paste"), NULL, RMP_STOCK, GTK_STOCK_PASTE, 
    NULL, on_pop_paste_activate, NULL
  },
  RUBRICA_MENU_SEPARATOR,
  {
    RM_ITEM, N_("_Add"), NULL, RMP_FILE, "add-menu.png",
    NULL, on_pop_add_activate, NULL
  },
  {
    RM_ITEM, N_("_Remove"), NULL, RMP_FILE, "remove-menu.png",
    NULL, on_pop_remove_activate, NULL
  }, 
  {
    RM_ITEM, N_("_Modify"), NULL, RMP_FILE, "mod-menu.png",
    NULL, on_pop_modify_activate, NULL
  },
  RUBRICA_MENU_SEPARATOR,
  RUBRICA_MENU_SUBTREE(N_("Show columns"), columns),
  RUBRICA_MENU_SUBTREE(N_("Show personal data"), personal),
  RUBRICA_MENU_SUBTREE(N_("Show address data"), address),
  RUBRICA_MENU_SUBTREE(N_("Show telephone and net data"), communication),
  RUBRICA_MENU_SUBTREE(N_("Show company data"), company),
  RUBRICA_MENU_SUBTREE(N_("Show assigment data"), assigment),
  RUBRICA_MENU_SUBTREE(N_("Show notes data"), note),
  RUBRICA_MENU_END
};  



GtkWidget *rubrica_card_get_notebook(void)
{
  return (GtkWidget *) notebook;
}

gint rubrica_card_get_current_notebook_page(void)
{
  return gtk_notebook_get_current_page(GTK_NOTEBOOK (notebook));
}


GtkWidget *rubrica_card_get_nth_parent(gint page)
{
  return (GtkWidget *) gtk_notebook_get_nth_page(GTK_NOTEBOOK (notebook),
						 page);
}


GtkTreeView *rubrica_card_get_view(GtkWidget *child)
{
  g_return_val_if_fail(child != NULL, NULL);

  return GTK_TREE_VIEW(g_object_get_data(G_OBJECT(child), "tree_view"));
}


GtkTreeView *rubrica_card_get_current_view(void)
{
  GtkWidget *child;

  child = rubrica_card_get_current_child();

  return GTK_TREE_VIEW(g_object_get_data(G_OBJECT(child), "tree_view"));
}


void rubrica_card_set_view(GtkWidget *view)
{
  tree_view = view; 
} 


GtkWidget *rubrica_card_get_current_child(void)
{
  GtkWidget *child;
  gint page;

  page = rubrica_card_get_current_notebook_page();
  child = gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), page);

  return (GtkWidget *) child;
}


gint rubrica_card_get_pages(void)
{
  return (gint) pages;
}


void rubrica_card_update_tab_name(GtkWidget *child, gchar *str)
{
  gtk_notebook_set_tab_label_text(GTK_NOTEBOOK(notebook), child, str);
}


Rubrica *rubrica_card_get_rubrica_from_view(GtkTreeView *view)
{
  return (Rubrica *) g_object_get_data(G_OBJECT(view), "rubrica");
}


RubricaItem* rubrica_card_get_nth_item(GtkTreeView *view, gint n)
{
  RubricaItem *item;
  GtkTreeModel *model;
  GtkTreeIter iter; 
  gchar *string;
  gboolean bool;

  item   = g_malloc(sizeof(RubricaItem));
  model  = gtk_tree_view_get_model(GTK_TREE_VIEW(view));
  string = g_strdup_printf("%d", n);
  
  bool = gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(model), &iter, 
					     string);
  g_free(string);

  if (bool)
    gtk_tree_model_get (GTK_TREE_MODEL(model), &iter, RUBRICA_COLUMN, 
			&item, -1);
  else
    return NULL;

  return (RubricaItem *) item;
}


/**
 * rubrica_card_interface_creation
 *
 * create the interface (a notebook) for card's window. 
 * the notebook's child is a scrolled window, 
 * and a tree view is packed into scrolle. Do not delete this notebook
 *
 * Return: notebook  
 */ 
GtkWidget *rubrica_card_interface_creation(RubricaInfoChild *info)
{
  notebook = gtk_notebook_new();

  gtk_notebook_set_show_border (GTK_NOTEBOOK (notebook), FALSE);
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_BOTTOM);
  gtk_notebook_set_scrollable(GTK_NOTEBOOK (notebook), TRUE);
  gtk_widget_show(notebook);

  info->child = notebook;

  return (GtkWidget *) notebook;
}



GtkTreeModel *rubrica_card_create_model(void)
{
  GtkListStore *model;
  gint i;

  /*      create list store 
   */
  model = gtk_list_store_new(COLUMNS, 
			     GDK_TYPE_PIXBUF,     /* pixmap     */
			     G_TYPE_STRING,       /* card       */
			     G_TYPE_STRING,       /* first name */ 
			     G_TYPE_STRING,       /* last name  */ 
			     G_TYPE_STRING,       /* profession */  
			     G_TYPE_STRING,       /* city       */
			     G_TYPE_STRING,       /* country    */
			     G_TYPE_STRING,       /* web        */
			     G_TYPE_STRING,       /* email      */ 
			     G_TYPE_STRING,       /* telephone  */ 
			     G_TYPE_STRING,       /* company    */  
			     G_TYPE_STRING,       /* assigment  */
			     G_TYPE_STRING,       /* categories */
			     G_TYPE_POINTER);     /* rubrica's data */      

  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(model), CARD_COLUMN, 
				       GTK_SORT_ASCENDING);

  gtk_tree_sortable_set_default_sort_func(GTK_TREE_SORTABLE(model), 
					  (GtkTreeIterCompareFunc) 
					  rubrica_sort_alfa,
					  GINT_TO_POINTER(CARD_COLUMN), NULL);
  
  for (i= FIRST_NAME_COLUMN; i < RUBRICA_COLUMN; i++)
    gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(model), i, 
				    (GtkTreeIterCompareFunc) 
				    rubrica_sort_alfa, 
				    GINT_TO_POINTER(i), NULL);

  gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(model), RUBRICA_COLUMN,
				  (GtkTreeIterCompareFunc) rubrica_sort_date, 
				  GINT_TO_POINTER(RUBRICA_COLUMN), NULL);

  return GTK_TREE_MODEL(model);
}


GtkWidget *rubrica_card_new_view(GtkTreeModel *model)
{
  GtkWidget *scrollwin;
  GtkWidget *view;
  GtkTreeSelection *select;

  scrollwin = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrollwin), 
				      GTK_SHADOW_ETCHED_IN);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin),
				 GTK_POLICY_AUTOMATIC, 
				 GTK_POLICY_AUTOMATIC);

  /*     create a new view using the model and connect the select
	 widget to changed signal
  */
  view = gtk_tree_view_new_with_model(model);
  select = gtk_tree_view_get_selection(GTK_TREE_VIEW(view));
  rubrica_card_set_view(view);

  g_object_unref(G_OBJECT(model));

  /*    set view's properties
  */
  gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
  gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(view), TRUE);
  gtk_tree_view_set_reorderable (GTK_TREE_VIEW (view), TRUE);
  gtk_tree_view_set_headers_clickable(GTK_TREE_VIEW (view), TRUE);

  rubrica_card_add_columns(GTK_TREE_VIEW(view));

  gtk_container_add(GTK_CONTAINER(scrollwin), view);
  g_object_set_data(G_OBJECT(scrollwin), "tree_view", view);
  gtk_widget_show(view);

  g_signal_connect(G_OBJECT(select), "changed", 
		   G_CALLBACK(rubrica_card_row_selected), view);
  g_signal_connect(G_OBJECT(view), "row_activated", 
		   G_CALLBACK(rubrica_card_row_activated), view);
  
  return scrollwin;
}

void rubrica_card_set_column_attributes(GConfClient *client, 
					GtkTreeView *view, 
					gchar *key, gint col)
{
  GtkTreeViewColumn *column;
  
  column = gtk_tree_view_get_column(view, col);
//  gtk_tree_view_column_set_resizable(column, TRUE);
//  gtk_tree_view_column_set_clickable(column, TRUE);
  gtk_tree_view_column_set_reorderable(column, TRUE);

  rubrica_preferences_column_config(client, column, key);
}


void rubrica_card_add_columns(GtkTreeView *view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;
  gint col;
  GConfClient *client;

  client = gconf_client_get_default(); 
  
  /* create list view
   */
  renderer = gtk_cell_renderer_pixbuf_new();
  col = gtk_tree_view_insert_column_with_attributes(view,  /* tree view */
						    -1,    /* column position
							      (-1 = last) */
						    NULL,     // column title 
						    renderer, // column cell 
						    "pixbuf", /* column 
								 attribute */
						    PIXMAP_COLUMN, /* column */
						    NULL);     /* no more
								  attribute */
  column = gtk_tree_view_get_column(view, col-1);
  gtk_tree_view_column_set_sizing(GTK_TREE_VIEW_COLUMN(column),
				  GTK_TREE_VIEW_COLUMN_FIXED);
  gtk_tree_view_column_set_fixed_width(GTK_TREE_VIEW_COLUMN(column), 25);  
  rubrica_card_set_column_attributes(client, view, 
				     RUBRICA_KEY_SHOW_CARD_COLUMN_PIXMAP, 
				     col-1);

  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes (view, -1, _("Categories"),
						     renderer,
						     "text", 
						     CATEGORIES_COLUMN, NULL);
  column = gtk_tree_view_get_column(view, col-1);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column), 
					  CATEGORIES_COLUMN);
  rubrica_card_set_column_attributes(client, view, 
				     RUBRICA_KEY_SHOW_CARD_COLUMN_CATEGORIES, 
				     col-1);

  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes(view, -1, _("Card"), 
						    renderer,    
						    "text", CARD_COLUMN, NULL);
  column = gtk_tree_view_get_column(view, col-1);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column), 
					  CARD_COLUMN);
  rubrica_card_set_column_attributes(client, view, 
				     RUBRICA_KEY_SHOW_CARD_COLUMN_CARD, col-1);

  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes(view, -1, _("First Name"),
						    renderer,
						    "text", FIRST_NAME_COLUMN,
						    NULL);
  column = gtk_tree_view_get_column(view, col-1);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column), 
					  FIRST_NAME_COLUMN);
  rubrica_card_set_column_attributes(client, view, 
				     RUBRICA_KEY_SHOW_CARD_COLUMN_FIRST_NAME,
				     col-1);
  
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes (view, -1, _("Last Name"),
						     renderer,
						     "text", LAST_NAME_COLUMN,
						     NULL);
  column = gtk_tree_view_get_column(view, col-1);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column), 
					  LAST_NAME_COLUMN);
  rubrica_card_set_column_attributes(client, view, 
				     RUBRICA_KEY_SHOW_CARD_COLUMN_LAST_NAME, 
				     col-1);

  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes (view, -1, _("Profession"),
						     renderer,
						     "text",
						     PROFESSION_COLUMN, NULL);
  column = gtk_tree_view_get_column(view, col-1);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column), 
					  PROFESSION_COLUMN);
  rubrica_card_set_column_attributes(client, view, 
				     RUBRICA_KEY_SHOW_CARD_COLUMN_PROFESSION, 
				     col-1);

  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes (view, -1, _("City"), 
						     renderer,
						     "text", 
						     CITY_COLUMN, NULL);
  column = gtk_tree_view_get_column(view, col-1);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column), 
					  CITY_COLUMN);
  rubrica_card_set_column_attributes(client, view, 
				     RUBRICA_KEY_SHOW_CARD_COLUMN_CITY, 
				     col-1);
  
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes (view, -1, _("Country"), 
						     renderer,
						     "text", COUNTRY_COLUMN,
						     NULL);
  column = gtk_tree_view_get_column(view, col-1);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column),
					  COUNTRY_COLUMN);
  rubrica_card_set_column_attributes(client, view, 
				     RUBRICA_KEY_SHOW_CARD_COLUMN_COUNTRY, 
				     col-1);
  
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes (view, -1, _("Web"),
						     renderer,
						     "text", WEB_CARD_COLUMN,
						     NULL);
  column = gtk_tree_view_get_column(view, col-1);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column), 
					  WEB_CARD_COLUMN);
  rubrica_card_set_column_attributes(client, view,
				     RUBRICA_KEY_SHOW_CARD_COLUMN_WEB, col-1);
  
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes (view, -1, _("Email"),
						     renderer,
						     "text",
						     EMAIL_CARD_COLUMN, NULL);
  column = gtk_tree_view_get_column(view, col-1);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column), 
					  EMAIL_CARD_COLUMN);
  rubrica_card_set_column_attributes(client, view,
				     RUBRICA_KEY_SHOW_CARD_COLUMN_EMAIL, 
				     col-1);
  
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes (view, -1, _("Telephone"),
						     renderer,
						     "text", TELEPHONE_COLUMN,
						     NULL);
  column = gtk_tree_view_get_column(view, col-1);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column),
					  TELEPHONE_COLUMN);
  rubrica_card_set_column_attributes(client, view, 
				     RUBRICA_KEY_SHOW_CARD_COLUMN_TELEPHONE, 
				     col-1);
  
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes (view, -1, _("Company"), 
						     renderer,
						     "text", COMPANY_COLUMN,
						     NULL);
  column = gtk_tree_view_get_column(view, col-1);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column),
					  COMPANY_COLUMN);
  rubrica_card_set_column_attributes(client, view, 
				     RUBRICA_KEY_SHOW_CARD_COLUMN_COMPANY, 
				     col-1);
  
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes (view, -1, _("Assigment"),
						     renderer,
						     "text",
						     ASSIGMENT_COLUMN, NULL);
  column = gtk_tree_view_get_column(view, col-1);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column),
					  ASSIGMENT_COLUMN);
  rubrica_card_set_column_attributes(client, view, 
				     RUBRICA_KEY_SHOW_CARD_COLUMN_ASSIGMENT, 
				     col-1); 
}


/*  add into addressbook's notebook a new page with the 
    content of file "filename"

    return the number of the new page
*/

gint 
rubrica_card_add_page_view(gchar *filename)
{
  Rubrica   *rubrica;       /* the new addressbook item */

  GtkWidget *popup;
  GtkWidget *tab_label;
  GtkWidget *tab_box;
  GtkWidget *green;
  GtkWidget *red;

  GtkTreeModel *model;
  GtkWidget *child;      /* notebook child, this will contain the tree_view */
  GtkTreeView *view;     /* the cards list */
  gint   page;           /* the # of the new notebook's page */
  gchar *path; 
  gchar *fname = NULL;


  popup = rubrica_menu_popup_new(popcardmenu);
  
  if (filename)
    {
      path  = g_dirname(filename);
      fname = g_strdup(g_basename(filename));
    }
  else
    path = rubrica_preferences_get_home();

  /*      get the number of this new page 
   */
  page = rubrica_card_get_pages();

  /*    build new addressbook. 
	It will be related to new notebook page  
  */
  rubrica = rubrica_new();

  /*    make notebook page's child 
   */
  model = rubrica_card_create_model();
  child = rubrica_card_new_view(model);
  view  = rubrica_card_get_view(child); 
  gtk_widget_show(child);

  tab_box = rubrica_ui_default_box_new(HBOX);  

  tab_label = (fname ? gtk_label_new(fname) : gtk_label_new(_("no name")));  
  gtk_widget_show(tab_label);
  
  green = rubrica_ui_image_new("ledgreen.png");
  red   = rubrica_ui_image_new("ledred.png");
  gtk_widget_hide(red);
  
  gtk_box_pack_start(GTK_BOX(tab_box), green, FALSE, FALSE, 1);
  gtk_box_pack_start(GTK_BOX(tab_box), red,   FALSE, FALSE, 1);
  gtk_box_pack_start(GTK_BOX(tab_box), tab_label, TRUE, TRUE, 1);

  /*      store infos about this addressbook/page 
   */
  rubrica->name   = (fname ? g_strdup(fname) : NULL);
  rubrica->path   = (path  ? g_strdup(path)  : NULL);
  rubrica->parent = child;
  rubrica->view   = view;
  rubrica->page   = page;

  g_object_set_data(G_OBJECT(child), "rubrica", rubrica);
  g_object_set_data(G_OBJECT(view), "rubrica", rubrica);
  g_object_set_data(G_OBJECT(tab_box), "tab_label", tab_label);
  g_object_set_data(G_OBJECT(tab_box), "green", green);
  g_object_set_data(G_OBJECT(tab_box), "red", red);

  /*      append a new addressbook view 
	  and increase the pages number 
   */
  rubrica_card_append_page(child, tab_box); 
  rubrica_menu_attach_popup(GTK_WIDGET(view), GTK_MENU(popup));

  /*      return the number of the new page 
   */
  return (gint) page;  
}

void rubrica_card_set_tab_name(gchar *name)
{
  gint page;
  GtkWidget *tab_box;
  GtkWidget *tab_label;

  page = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
  tab_box = gtk_notebook_get_tab_label(GTK_NOTEBOOK(notebook), 
				       rubrica_card_get_nth_parent(page));
  tab_label = g_object_get_data(G_OBJECT(tab_box), "tab_label");

  gtk_label_set_text(GTK_LABEL(tab_label), name);
}

void rubrica_card_append_page(GtkWidget *child, GtkWidget *tabbox)
{
  pages++;
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), child, tabbox);  
  gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), -1);  
}


void rubrica_card_remove_page(gint page)
{
  gtk_notebook_remove_page(GTK_NOTEBOOK(notebook), page);

  pages--;
}


void rubrica_card_append_item(Rubrica *rubrica, RubricaItem *item)
{
  GtkTreeView *view;
  GtkTreeIter iter;
  GtkTreeModel *model;
  GdkPixbuf *image = NULL;
  gchar *web, *email, *number, *pixmap;

  g_return_if_fail(rubrica != NULL);
  g_return_if_fail(item != NULL);

  if (!item)
    g_print("\n\aitem NULL");
  g_return_if_fail(item != NULL);

  view = GTK_TREE_VIEW(rubrica_get_view(rubrica));
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(view));

  pixmap = rubrica_form_get_option_pixmap(item->group.type);
  if (pixmap)
    image = rubrica_ui_pixbuf_new(pixmap);

  if (item->net.web && item->net.web->data)
    web = g_strdup(item->net.web->data);
  else
    web = "";

  if (item->net.email && item->net.email->data)
    email = g_strdup(item->net.email->data);
  else
    email = "";

  if (item->phone.telephone && item->phone.telephone->data)
    {
      TelNum *tn;
      
      tn = item->phone.telephone->data;
      number = g_strdup(tn->number);
    }
  else
    number = "";
  
  gtk_list_store_append(GTK_LIST_STORE(model), &iter);
  gtk_list_store_set(GTK_LIST_STORE(model), &iter,
		     PIXMAP_COLUMN,     image,
		     CARD_COLUMN,       item->card, 
		     FIRST_NAME_COLUMN, item->personal.first, 
		     LAST_NAME_COLUMN,  item->personal.last,
		     PROFESSION_COLUMN, item->personal.profession,
		     CITY_COLUMN,       item->address.city,
		     COUNTRY_COLUMN,    item->address.country,
		     WEB_CARD_COLUMN,   web,
		     EMAIL_CARD_COLUMN, email,
		     TELEPHONE_COLUMN,  number,
		     COMPANY_COLUMN,    item->company.name,
		     ASSIGMENT_COLUMN,  item->work.assigment,
		     CATEGORIES_COLUMN, item->group.label, 
		     RUBRICA_COLUMN,    item,
		     -1);  
}


void rubrica_card_insert_item(Rubrica *rubrica, RubricaItem *item, gint index)
{
  GtkTreeView *view;
  GtkTreeIter iter;
  GtkTreeModel *model;
  GdkPixbuf *image = NULL;
  gchar *web, *email, *number, *pixmap;

  g_return_if_fail(rubrica != NULL);
  g_return_if_fail(item != NULL);
  
  if (!item)
    g_print("\n\aitem NULL");
  g_return_if_fail(item != NULL);

  view = GTK_TREE_VIEW(rubrica_get_view(rubrica));
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree_view));

  pixmap = rubrica_form_get_option_pixmap(item->group.type);
  if (pixmap)
    image = rubrica_ui_pixbuf_new(pixmap);

  if (item->net.web && item->net.web->data)
    web = g_strdup(item->net.web->data);
  else
    web = "";

  if (item->net.email && item->net.email->data)
    email = g_strdup(item->net.email->data);
  else
    email = "";

  if (item->phone.telephone && item->phone.telephone->data)
    {
      TelNum *tn;
      
      tn = item->phone.telephone->data;
      number = g_strdup(tn->number);
    }
  else
    number = ""; 

  gtk_list_store_insert(GTK_LIST_STORE(model), &iter, index);
  gtk_list_store_set(GTK_LIST_STORE(model), &iter,
		     PIXMAP_COLUMN,     image,
		     CARD_COLUMN,       item->card, 
		     FIRST_NAME_COLUMN, item->personal.first, 
		     LAST_NAME_COLUMN,  item->personal.last,
		     PROFESSION_COLUMN, item->personal.profession,
		     CITY_COLUMN,       item->address.city,
		     COUNTRY_COLUMN,    item->address.country,
		     WEB_CARD_COLUMN,   web,
		     EMAIL_CARD_COLUMN, email,
		     TELEPHONE_COLUMN,  number,
		     COMPANY_COLUMN,    item->company.name,
		     ASSIGMENT_COLUMN,  item->work.assigment,
		     CATEGORIES_COLUMN, item->group.label, 
		     RUBRICA_COLUMN,    item,
		     -1);  
}


void rubrica_card_modify_item(Rubrica *rubrica, RubricaItem *r, gint index)
{
  confirm_remove = FALSE;
  
  if (rubrica_card_remove_selected_item(rubrica))
    rubrica_card_insert_item(rubrica, r, index);  

  confirm_remove = TRUE;  
}


gboolean rubrica_card_remove_selected_item(Rubrica *rubrica)
{
  GtkTreeIter iter;
  GtkTreeView *view;
  GtkTreeModel *model;
  GtkTreeSelection *selection;  
  RubricaItem *item;

  view  = GTK_TREE_VIEW(rubrica_get_view(rubrica));
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(view));
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(view));

  if (gtk_tree_selection_get_selected(selection, NULL, &iter))
    {      
      gchar *card;

      if (confirm_remove)
	{
	  GtkWidget *dialog;
	  gchar *str;
	  gint choose;
	  
	  gtk_tree_model_get (model, &iter, 
			      CARD_COLUMN, &card, 
			      RUBRICA_COLUMN, &item,
			      -1);
	  str = g_strdup_printf(_("Do you want delete the %s card?"), card);

	  dialog = rubrica_dialog_query(str, GTK_BUTTONS_YES_NO);
	  gtk_widget_show(dialog);
	  g_free(str);
	  
	  choose = gtk_dialog_run(GTK_DIALOG(dialog));
	  gtk_widget_destroy(dialog);

	  switch(choose)
	    {
	    case GTK_RESPONSE_YES:
	      break;
	      
	    case GTK_RESPONSE_NO:
	    default:
	      return FALSE;
	      break;
	    }	

	  if (item->delete)
	    {
	      gtk_list_store_remove(GTK_LIST_STORE(model), &iter);      
	      
	      return TRUE;
	    }
	  else
	    {
	      gchar *msg;
	      
	      msg = g_strdup_printf(_("The %s card is marked undeleteble.\n"
				      "Please modify the card status and "
				      "then delete it"), card);
	      rubrica_dialog_error(msg);
	      
	      g_free(msg);

	      return FALSE;
	    }    	
	}
      else
	{
	  gtk_list_store_remove(GTK_LIST_STORE(model), &iter);

	  return TRUE;
	}      
    }
  else
    {
      rubrica_dialog_message(_("I can't continue if you don't select a card"),
			     GTK_MESSAGE_ERROR, GTK_BUTTONS_OK);
    }
  
  return FALSE;    
}

gboolean 
rubrica_card_remove_last_item(Rubrica *rubrica)
{
  GtkTreeIter iter;
  GtkTreeView *view;
  GtkTreeModel *model;
  gint n;

  view  = GTK_TREE_VIEW(rubrica_get_view(rubrica));
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(view));
  
  n = gtk_tree_model_iter_n_children(model, NULL); 
  if (gtk_tree_model_iter_nth_child(model, &iter, NULL, n-1))
    {
      gtk_list_store_remove(GTK_LIST_STORE(model), &iter);      
      
      return TRUE;
    }

  return FALSE;
}

gboolean 
rubrica_card_remove_nth_item(Rubrica *rubrica, gint n)
{
  GtkTreeIter iter;
  GtkTreeView *view;
  GtkTreeModel *model;
  gchar *str;
  
  view  = GTK_TREE_VIEW(rubrica_get_view(rubrica));
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(view));
  str = g_strdup_printf("%d", n);
  
  if (gtk_tree_model_get_iter_from_string(model, &iter, str))
    {
      gtk_list_store_remove(GTK_LIST_STORE(model), &iter);  
      g_free(str);
      
      return TRUE;
    }

  g_free(str);
  return FALSE;  
}


gboolean 
rubrica_card_free_memory(GtkTreeModel *model, GtkTreePath *path, 
			 GtkTreeIter *iter, gpointer data)
{
  RubricaItem *item;
  Rubrica *rubrica = data;
  
  if (rubrica_is_empty(rubrica))
    return TRUE;
  
  gtk_tree_model_get(model, iter, RUBRICA_COLUMN, &item, -1);

  return FALSE;
}


RubricaItem *rubrica_card_item_cut(Rubrica *rubrica)
{
  RubricaItem *item = NULL;
  GtkTreeIter iter;
  GtkTreeView *view;
  GtkTreeModel *model;
  GtkTreeSelection *selection;
  
  view      = GTK_TREE_VIEW(rubrica_get_view(rubrica));
  model     = gtk_tree_view_get_model(view);
  selection = gtk_tree_view_get_selection(view);
  
  if (gtk_tree_selection_get_selected(selection, NULL, &iter))    
    {
      gtk_tree_model_get (model, &iter, RUBRICA_COLUMN, &item, -1);
      gtk_list_store_remove(GTK_LIST_STORE(model), &iter);      
    }
  else
    rubrica_dialog_message(_("I can't continue if you don't select a card"), 
			   GTK_MESSAGE_ERROR, GTK_BUTTONS_OK);       
    
  return item;
}


RubricaItem *rubrica_card_item_copy(Rubrica *rubrica)
{
  RubricaItem *item = NULL;
  GtkTreeIter iter;
  GtkTreeView *view;
  GtkTreeModel *model;
  GtkTreeSelection *selection;
  
  view      = GTK_TREE_VIEW(rubrica_get_view(rubrica));
  model     = gtk_tree_view_get_model(view);
  selection = gtk_tree_view_get_selection(view);
  
  if (gtk_tree_selection_get_selected(selection, NULL, &iter))    
    {
      gtk_tree_model_get (model, &iter, RUBRICA_COLUMN, &item, -1);

      return rubrica_card_clone_item(item);
    }
  else
    rubrica_dialog_message(_("I can't continue if you don't select a card"), 
			   GTK_MESSAGE_ERROR, GTK_BUTTONS_OK);       

  return NULL;
}


RubricaItem *rubrica_card_clone_item(RubricaItem *item)
{
  RubricaItem *retval;

  g_return_val_if_fail(item != NULL, NULL);

  retval = g_malloc(sizeof(RubricaItem));

  if (!retval)
    {
      rubrica_dialog_message(_("I have not enough memory to perform "
			       "this task.\n"
			       "I cant copy the selected item"), 
			     GTK_MESSAGE_ERROR, GTK_BUTTONS_OK);     
      
      return NULL;
    }
  
  retval->net.web = retval->net.email = retval->phone.telephone = NULL;

  retval->delete = item->delete;

  retval->group.label  = g_strdup(item->group.label);
  retval->group.type   = item->group.type;
  
  retval->card = (item->card ? g_strdup(item->card) : "");
  
  retval->personal.first  = (item->personal.first  ? 
			     g_strdup(item->personal.first) : "");
  retval->personal.middle = (item->personal.middle ? 
			     g_strdup(item->personal.middle): "");
  retval->personal.last   = (item->personal.last   ? 
			     g_strdup(item->personal.last): "");
  retval->personal.profession = (item->personal.profession ? 
				 g_strdup(item->personal.profession): "");
  retval->personal.prefix = (item->personal.prefix ? 
			     g_strdup(item->personal.prefix): "");
  retval->personal.title  = (item->personal.title ? 
			     g_strdup(item->personal.title): "");
  retval->personal.birthknow = item->personal.birthknow;
  retval->personal.birthdate = item->personal.birthdate;
  
  retval->address.street   = (item->address.street   ? 
			      g_strdup(item->address.street): "");
  retval->address.number   = (item->address.number   ? 
			      g_strdup(item->address.number): "");
  retval->address.zip      = (item->address.zip      ? 
			      g_strdup(item->address.zip): "");
  retval->address.city     = (item->address.city     ? 
			      g_strdup(item->address.city): "");
  retval->address.province = (item->address.province ? 
			      g_strdup(item->address.province): "");
  retval->address.state    = (item->address.state    ? 
			      g_strdup(item->address.state): "");
  retval->address.country  = (item->address.country  ? 
			      g_strdup(item->address.country): "");

  while (item->net.web)
    {
      gchar *web;

      web = g_strdup(item->net.web->data);
      retval->net.web = g_list_append(retval->net.web, web);

      item->net.web = g_list_next(item->net.web);
    }

  while (item->net.email)    
    {
      gchar *email;

      email = g_strdup(item->net.email->data);
      retval->net.email = g_list_append(retval->net.email, email);

      item->net.email = g_list_next(item->net.email);
    }  

  while (item->phone.telephone)
    {
      TelNum *tn;
      TelNum *tel;

      tn = (TelNum *) g_malloc(sizeof(TelNum));
      tel = item->phone.telephone->data;
      
      tn->number = g_strdup(tel->number);
      tn->type   = g_strdup(tel->type);
      retval->phone.telephone = g_list_append(retval->phone.telephone, tn);
      
      item->phone.telephone = g_list_next(item->phone.telephone);
    }
  
  retval->work.assigment    = (item->work.assigment  ?
			       g_strdup(item->work.assigment): "");
  retval->work.organization = (item->work.organization ? 
			       g_strdup(item->work.organization):"");
  retval->work.department   = (item->work.department ? 
			       g_strdup(item->work.department): "");
  retval->work.subdep       = (item->work.subdep ? 
			       g_strdup(item->work.subdep): "");
  retval->work.secretary    = (item->work.secretary ? 
			       g_strdup(item->work.secretary): "");
  retval->work.telephone    = (item->work.telephone ? 
			       g_strdup(item->work.telephone): "");
  
  retval->company.name     = (item->company.name     ? 
			      g_strdup(item->company.name): "");
  retval->company.street   = (item->company.street   ? 
			      g_strdup(item->company.street): "");
  retval->company.number   = (item->company.number   ? 
			      g_strdup(item->company.number): "");
  retval->company.zip      = (item->company.zip      ? 
			      g_strdup(item->company.zip): "");
  retval->company.city     = (item->company.city     ? 
			      g_strdup(item->company.city): "");
  retval->company.province = (item->company.province ? 
			      g_strdup(item->company.province): "");
  retval->company.country  = (item->company.country  ? 
			      g_strdup(item->company.country): "");
  retval->company.web      = (item->company.web      ? 
			      g_strdup(item->company.web): "");
  retval->company.email    = (item->company.email    ? 
			      g_strdup(item->company.email): "");
  retval->company.operator = (item->company.operator ? 
			      g_strdup(item->company.operator): "");
  retval->company.fax      = (item->company.fax      ? 
			      g_strdup(item->company.fax): "");
  retval->company.green    = (item->company.green    ? 
			      g_strdup(item->company.green): "");
  retval->company.customer_care = (item->company.customer_care ? 
				  g_strdup(item->company.customer_care): "");
  retval-> company.notes = (item->company.notes ? 
			    g_strdup(item->company.notes): "");
  
  retval->notes.is_married   = item->notes.is_married;
  retval->notes.has_children = item->notes.has_children;
  retval->notes.birthknow    = item->notes.birthknow;  
  retval->notes.anniverknow  = item->notes.anniverknow;  

  retval->notes.birthdate   = item->notes.birthdate;  
  retval->notes.anniverdate = item->notes.anniverdate;  

  retval->notes.spouse    = (item->notes.spouse ? 
			     g_strdup(item->notes.spouse): "");
  retval->notes.children  = (item->notes.children ? 
			     g_strdup(item->notes.children): "");
  retval->notes.hobbies   = (item->notes.hobbies ? 
			     g_strdup(item->notes.hobbies): "");
  retval->notes.notes     = (item->notes.notes ? 
			     g_strdup(item->notes.notes): "");
  retval->notes.pubkey    = (item->notes.pubkey ? 
			     g_strdup(item->notes.pubkey): "");
  
  retval->created     = item->created;
  retval->last_change = item->last_change;

  return retval;
}



/***
 *               CALLBACKS
 ***/

void rubrica_card_row_selected(GtkTreeSelection *select, gpointer data)
{
  GtkTreeModel *model = gtk_tree_view_get_model(data);
  Rubrica *rubrica;
  RubricaItem *item;
  GtkTreePath *path;
  GtkTreeIter iter;
  gint index;
  
  rubrica = rubrica_get_current_rubrica();
  
  if (gtk_tree_selection_get_selected(select, &model, &iter))
    {
      gtk_tree_model_get(model, &iter, RUBRICA_COLUMN, &item, -1);
      path = gtk_tree_model_get_path(model, &iter);
      rubrica_browse_save_path(path);

      if (gtk_tree_selection_path_is_selected(select, path))
	{
	  index = gtk_tree_path_get_indices(path)[0];
	  rubrica_set_index(rubrica, index);
	}      
      
      rubrica_view_show_card_item(item);
    }
}

void rubrica_card_row_activated(GtkTreeView *tree_view, GtkTreePath *path,
				GtkTreeViewColumn *column)
{
  GtkTreeIter iter;
  GtkTreeModel *model;
  RubricaItem *item;
  gint index;

  model = gtk_tree_view_get_model (tree_view);
  
  gtk_tree_model_get_iter (model, &iter, path);
  index = gtk_tree_path_get_indices (path)[0];
  gtk_tree_model_get (model, &iter, RUBRICA_COLUMN, &item, -1);
  rubrica_browse_save_path(path);

  rubrica_modify_card_item(item, index);
}

