/*
*  Rubrica
*  file: dictionary.c
*
*  
*  Copyright (C) 2000-2001 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <gconf/gconf-client.h>
#include <libxml/tree.h>
#include <libxml/parser.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>


#include "dictionary.h"
#include "prefer.h"
#include "prefer_int.h"
#include "main.h"
#include "log.h"
#include "dialog.h"
#include "utils.h"
#include "rubrica_type.h"

RubricaDictionary *rubrica_dict[53][2] = {
  {"Scheda",              "Card"             },       // 0
  {"Dati",                "Data"             },
  {"Indirizzo",           "Address"          },
  {"Compagnia",           "Company"          },
  {"Lavoro",              "Work"             },
  {"Nome",                "First name"       }, 
  {"Secondo nome",        "Middle name"      },
  {"Cognome",             "Last name"        },
  {"Professione",         "Profession"       },
  {"Prefisso",            "Name prefix"      },      // 9
  {"Titolo",              "Title"            },      // 10
  {"Via",                 "Street"           },
  {"Numero",              "Street number"    },
  {"Cap",                 "Zip code"         },
  {"Citt",               "City"             },
  {"Provincia",           "Province"         },
  {"Stato",               "State"            },
  {"Nazione",             "Country"          },
  {"Indirizzi web",       "Web addresses"    },
  {"Web",                 "Web"              },      // 19
  {"Indirizzi email",     "Email addresses"  },      // 20
  {"EMail",               "EMail"            },
  {"Numeri telefonici",   "Telephone numbers"},
  {"Telefono",            "Telephone"        },
  {"tipo",                "type"             },
  {"Lavoro",              "Work"             },
  {"Casa",                "Home"             },
  {"Fax",                 "Fax"              },
  {"Cellulare",           "Cellphone"        },
  {"Altro",               "Other"            },      // 29
  {"Nome compagnia",      "Company name"     },      // 30
  {"Centralino",          "Operator"         },
  {"Numero verde",        "Green"            },
  {"Assistenza clienti",  "Customer care"    },
  {"Ruolo",               "Assigment"        },
  {"Organizzazione",      "Organization"     },
  {"Dipartimento",        "Department"       },
  {"Sotto dipartimento",  "Sub department"   },
  {"Segretaria",          "Secretary"        },
  {"Telefono segretaria", "Secreatary phone" },      // 39
  {"Coniuge",             "Spouse"           },      // 40
  {"Coniugato",           "Married"          },
  {"Anniversario",        "Anniversary"      },
  {"Figli",               "Children"         },
  {"Note figli",          "Childen notes"    },
  {"Notes",               "Notes"            },
  {"Passatempi",          "Hobbies"          },
  {"Chiave publica",      "Public key"       },
  {"Data nascita",        "Birthday"         },
  {"Nome coniuge",        "Spouse name"      },      // 49
  {"Altre note",          "Other notes"      },      // 50
  {"Categoria",           "Categories"       },
  {NULL, NULL}
};


void rubrica_dictionary_init(RubricaLanguageType language)
{ 
  rubrica_dictionary_get(language);
}


void rubrica_dictionary_get_italian(void)
{
  rubrica_dictionary_get(ITALIAN);
}

void rubrica_dictionary_get_english(void)
{
  rubrica_dictionary_get(ENGLISH);
}

void rubrica_dictionary_get_user_dictionary(void)
{
  gboolean err;

  err = rubrica_dictionary_get(USER);

  if (err)
    rubrica_dictionary_get(ITALIAN);
}


void rubrica_dictionary_compile(GConfClient *client,
				RubricaLanguageType language, 
				RubricaDictionaryCode code,
				gchar *key)
{
  gchar *str = NULL;
  
  str = g_locale_to_utf8(rubrica_dict[code][language], -1, NULL, NULL, NULL);
  
  if (str && g_utf8_validate (str, -1, NULL))
    {
      gconf_client_set_string(client, key, str, NULL); 	
    }
  else
    {
      gchar *error;
      
      error = str ? 
	g_strdup_printf("\ncan't read dictionary key: %s for %s", key, str) : 
	g_strdup_printf("\ncan't validate this dictionary value: %s", str);

      rubrica_log_write_log(error);

      g_free(error);
    }
}

gboolean rubrica_dictionary_get(RubricaLanguageType lang)
{
  GConfClient *client;
  gboolean err;

  client = gconf_client_get_default(); 

  if (lang == USER)
    {      
      err = rubrica_dictionary_load_user_dictionary(client);
      
      return err;
    }
  
  rubrica_dictionary_compile(client, lang, RDC_CARD, RUBRICA_KEY_DICT_CARD);  
  rubrica_dictionary_compile(client, lang, RDC_DATA, RUBRICA_KEY_DICT_DATA);
  rubrica_dictionary_compile(client, lang, RDC_ADDRESS, 
			     RUBRICA_KEY_DICT_ADDRESS);
  rubrica_dictionary_compile(client, lang, RDC_COMPANY, 
			     RUBRICA_KEY_DICT_COMPANY);
  rubrica_dictionary_compile(client, lang, RDC_WORK, 
			     RUBRICA_KEY_DICT_WORK);
  rubrica_dictionary_compile(client, lang, RDC_FIRST_NAME, 
			     RUBRICA_KEY_DICT_FIRST_NAME);
  rubrica_dictionary_compile(client, lang, RDC_MIDDLE_NAME, 
			     RUBRICA_KEY_DICT_MIDDLE_NAME);
  rubrica_dictionary_compile(client, lang, RDC_LAST_NAME, 
			     RUBRICA_KEY_DICT_LAST_NAME);
  rubrica_dictionary_compile(client, lang, RDC_PROFESSION, 
			     RUBRICA_KEY_DICT_PROFESSION);
  rubrica_dictionary_compile(client, lang, RDC_NAME_PREFIX, 
			     RUBRICA_KEY_DICT_NAME_PREFIX);
  rubrica_dictionary_compile(client, lang, RDC_TITLE,
			     RUBRICA_KEY_DICT_TITLE);
  rubrica_dictionary_compile(client, lang, RDC_STREET, 
			     RUBRICA_KEY_DICT_STREET);
  rubrica_dictionary_compile(client, lang, RDC_STREET_NUMBER, 
			     RUBRICA_KEY_DICT_STREET_NUMBER);
  rubrica_dictionary_compile(client, lang, RDC_ZIP_CODE, 
			     RUBRICA_KEY_DICT_ZIP_CODE);
  rubrica_dictionary_compile(client, lang, RDC_CITY,
			     RUBRICA_KEY_DICT_CITY);
  rubrica_dictionary_compile(client, lang, RDC_PROVINCE,
			     RUBRICA_KEY_DICT_PROVINCE);
  rubrica_dictionary_compile(client, lang, RDC_STATE, 
			     RUBRICA_KEY_DICT_STATE);
  rubrica_dictionary_compile(client, lang, RDC_COUNTRY, 
			     RUBRICA_KEY_DICT_COUNTRY);
  rubrica_dictionary_compile(client, lang, RDC_WEB_ADDRESS, 
			     RUBRICA_KEY_DICT_WEB_ADDRESS);
  rubrica_dictionary_compile(client, lang, RDC_WEB, 
			     RUBRICA_KEY_DICT_WEB);
  rubrica_dictionary_compile(client, lang, RDC_EMAIL_ADDRESS, 
			     RUBRICA_KEY_DICT_EMAIL_ADDRESS);
  rubrica_dictionary_compile(client, lang, RDC_EMAIL, 
			     RUBRICA_KEY_DICT_EMAIL);
  rubrica_dictionary_compile(client, lang, RDC_TELEPHONE_NUMBERS, 
			     RUBRICA_KEY_DICT_TELEPHONE_LIST);
  rubrica_dictionary_compile(client, lang, RDC_TELEPHONE, 
			     RUBRICA_KEY_DICT_TELEPHONE_NUMBER);
  rubrica_dictionary_compile(client, lang, RDC_TYPE, 
			     RUBRICA_KEY_DICT_TELEPHONE_TYPE);
  rubrica_dictionary_compile(client, lang, RDC_TELEPHONE_WORK, 
			     RUBRICA_KEY_DICT_TELEPHONE_WORK);
  rubrica_dictionary_compile(client, lang, RDC_TELEPHONE_HOME, 
			     RUBRICA_KEY_DICT_TELEPHONE_HOME);
  rubrica_dictionary_compile(client, lang, RDC_TELEPHONE_FAX, 
			     RUBRICA_KEY_DICT_TELEPHONE_FAX);
  rubrica_dictionary_compile(client, lang, RDC_TELEPHONE_CELL, 
			     RUBRICA_KEY_DICT_TELEPHONE_CELL);
  rubrica_dictionary_compile(client, lang, RDC_TELEPHONE_OTHER, 
			     RUBRICA_KEY_DICT_TELEPHONE_OTHER);
  rubrica_dictionary_compile(client, lang, RDC_COMPANY_NAME, 
			     RUBRICA_KEY_DICT_COMPANY_NAME);
  rubrica_dictionary_compile(client, lang, RDC_OPERATOR, 
			     RUBRICA_KEY_DICT_OPERATOR);
  rubrica_dictionary_compile(client, lang, RDC_GREEN,
			     RUBRICA_KEY_DICT_GREEN);
  rubrica_dictionary_compile(client, lang, RDC_CUSTOMER_CARE, 
			     RUBRICA_KEY_DICT_CUSTOMER_CARE);
  rubrica_dictionary_compile(client, lang, RDC_ASSIGMENT, 
			     RUBRICA_KEY_DICT_ASSIGMENT);
  rubrica_dictionary_compile(client, lang, RDC_ORGANIZATION,
			     RUBRICA_KEY_DICT_ORGANIZATION);
  rubrica_dictionary_compile(client, lang, RDC_DEPARTMENT,
			     RUBRICA_KEY_DICT_DEPARTMENT);
  rubrica_dictionary_compile(client, lang, RDC_SUB_DEPARTMENT, 
			     RUBRICA_KEY_DICT_SUB_DEPARTMENT);
  rubrica_dictionary_compile(client, lang, RDC_SECRETARY, 
			     RUBRICA_KEY_DICT_SECRETARY);
  rubrica_dictionary_compile(client, lang, RDC_SECRETARY_PHONE, 
			     RUBRICA_KEY_DICT_SECRETARY_PHONE);
  rubrica_dictionary_compile(client, lang, RDC_SPOUSE, 
			     RUBRICA_KEY_DICT_SPOUSE_NAME);
  rubrica_dictionary_compile(client, lang, RDC_MARRIED, 
			     RUBRICA_KEY_DICT_MARRIED);
  rubrica_dictionary_compile(client, lang, RDC_ANNIVERSARY, 
			     RUBRICA_KEY_DICT_ANNIVERSARY);
  rubrica_dictionary_compile(client, lang, RDC_CHILDREN, 
			     RUBRICA_KEY_DICT_CHILDREN);
  rubrica_dictionary_compile(client, lang, RDC_CHILDREN_NOTES, 
			     RUBRICA_KEY_DICT_CHILDREN_NOTES);
  rubrica_dictionary_compile(client, lang, RDC_HOBBIES, 
			     RUBRICA_KEY_DICT_HOBBY);
  rubrica_dictionary_compile(client, lang, RDC_NOTE, 
			     RUBRICA_KEY_DICT_NOTES);
  rubrica_dictionary_compile(client, lang, RDC_PUBLIC_KEY, 
			     RUBRICA_KEY_DICT_PUBLIC_KEY);
  rubrica_dictionary_compile(client, lang, RDC_BIRTHDAY, 
			     RUBRICA_KEY_DICT_BIRTHDAY);
  rubrica_dictionary_compile(client, lang, RDC_OTHER_NOTE, 
			     RUBRICA_KEY_DICT_OTHER_NOTES);
  rubrica_dictionary_compile(client, lang, RDC_CARD_GROUP,
			     RUBRICA_KEY_DICT_CARD_GROUP);
  
  return FALSE;
}

void rubrica_dictionary_display_string(GtkWidget *prefbox, GConfClient* client,
				       gchar *obj_key, gchar *gconf_key)
{
  GtkEntry *entry;
  gchar *str;

  entry = gtk_object_get_data(GTK_OBJECT(prefbox), obj_key);
  str   = gconf_client_get_string(client, gconf_key, NULL);

  gtk_entry_set_text(GTK_ENTRY(entry), str);
}


void rubrica_dictionary_display_dictionary(GtkWidget *prefbox, 
					   GConfClient* client)
{
  rubrica_dictionary_display_string(prefbox, client, "entry_card", 
				    RUBRICA_KEY_DICT_CARD);
  rubrica_dictionary_display_string(prefbox, client, "entry_categories", 
				    RUBRICA_KEY_DICT_CARD_GROUP);
  rubrica_dictionary_display_string(prefbox, client, "entry_data", 
				    RUBRICA_KEY_DICT_DATA);
  rubrica_dictionary_display_string(prefbox, client, "entry_address", 
				    RUBRICA_KEY_DICT_ADDRESS);
  rubrica_dictionary_display_string(prefbox, client, "entry_company", 
				    RUBRICA_KEY_DICT_COMPANY);
  rubrica_dictionary_display_string(prefbox, client, "entry_work", 
				    RUBRICA_KEY_DICT_WORK);
  rubrica_dictionary_display_string(prefbox, client, "entry_first_name", 
				    RUBRICA_KEY_DICT_FIRST_NAME);
  rubrica_dictionary_display_string(prefbox, client, "entry_middle_name", 
				    RUBRICA_KEY_DICT_MIDDLE_NAME);
  rubrica_dictionary_display_string(prefbox, client, "entry_last_name", 
				    RUBRICA_KEY_DICT_LAST_NAME);
  rubrica_dictionary_display_string(prefbox, client, "entry_profession", 
				    RUBRICA_KEY_DICT_PROFESSION);
  rubrica_dictionary_display_string(prefbox, client, "entry_name_prefix", 
				    RUBRICA_KEY_DICT_NAME_PREFIX);
  rubrica_dictionary_display_string(prefbox, client, "entry_title", 
				    RUBRICA_KEY_DICT_TITLE);
  rubrica_dictionary_display_string(prefbox, client, "entry_street",
				    RUBRICA_KEY_DICT_STREET);
  rubrica_dictionary_display_string(prefbox, client, "entry_street_number", 
				    RUBRICA_KEY_DICT_STREET_NUMBER);
  rubrica_dictionary_display_string(prefbox, client, "entry_zip_code", 
				    RUBRICA_KEY_DICT_ZIP_CODE);
  rubrica_dictionary_display_string(prefbox, client, "entry_city", 
				    RUBRICA_KEY_DICT_CITY);
  rubrica_dictionary_display_string(prefbox, client, "entry_province", 
				    RUBRICA_KEY_DICT_PROVINCE);
  rubrica_dictionary_display_string(prefbox, client, "entry_state", 
				    RUBRICA_KEY_DICT_STATE);
  rubrica_dictionary_display_string(prefbox, client, "entry_country", 
				    RUBRICA_KEY_DICT_COUNTRY);
  rubrica_dictionary_display_string(prefbox, client, "entry_web_address", 
				    RUBRICA_KEY_DICT_WEB_ADDRESS);
  rubrica_dictionary_display_string(prefbox, client, "entry_web", 
				    RUBRICA_KEY_DICT_WEB);
  rubrica_dictionary_display_string(prefbox, client, "entry_email_address", 
				    RUBRICA_KEY_DICT_EMAIL_ADDRESS);
  rubrica_dictionary_display_string(prefbox, client, "entry_email", 
				    RUBRICA_KEY_DICT_EMAIL);
  rubrica_dictionary_display_string(prefbox, client, "entry_tel_list", 
				    RUBRICA_KEY_DICT_TELEPHONE_LIST);
  rubrica_dictionary_display_string(prefbox, client, "entry_telephone", 
				    RUBRICA_KEY_DICT_TELEPHONE_NUMBER);
  rubrica_dictionary_display_string(prefbox, client, "entry_type", 
				    RUBRICA_KEY_DICT_TELEPHONE_TYPE);

  rubrica_dictionary_display_string(prefbox, client, "entry_tel_work", 
				    RUBRICA_KEY_DICT_TELEPHONE_WORK);
  rubrica_dictionary_display_string(prefbox, client, "entry_tel_home", 
				    RUBRICA_KEY_DICT_TELEPHONE_HOME);
  rubrica_dictionary_display_string(prefbox, client, "entry_tel_fax", 
				    RUBRICA_KEY_DICT_TELEPHONE_FAX);
  rubrica_dictionary_display_string(prefbox, client, "entry_tel_cell", 
				    RUBRICA_KEY_DICT_TELEPHONE_CELL);
  rubrica_dictionary_display_string(prefbox, client, "entry_tel_other", 
				    RUBRICA_KEY_DICT_TELEPHONE_OTHER);

  rubrica_dictionary_display_string(prefbox, client, "entry_company_name", 
				    RUBRICA_KEY_DICT_COMPANY_NAME);
  rubrica_dictionary_display_string(prefbox, client, "entry_operator", 
				    RUBRICA_KEY_DICT_OPERATOR);
  rubrica_dictionary_display_string(prefbox, client, "entry_green", 
				    RUBRICA_KEY_DICT_GREEN);
  rubrica_dictionary_display_string(prefbox, client, "entry_customer_care", 
				    RUBRICA_KEY_DICT_CUSTOMER_CARE);
  rubrica_dictionary_display_string(prefbox, client, "entry_assigment", 
				    RUBRICA_KEY_DICT_ASSIGMENT);
  rubrica_dictionary_display_string(prefbox, client, "entry_organization", 
				    RUBRICA_KEY_DICT_ORGANIZATION);
  rubrica_dictionary_display_string(prefbox, client, "entry_department", 
				    RUBRICA_KEY_DICT_DEPARTMENT);
  rubrica_dictionary_display_string(prefbox, client, "entry_sub_dep", 
				    RUBRICA_KEY_DICT_SUB_DEPARTMENT);
  rubrica_dictionary_display_string(prefbox, client, "entry_secretary", 
				    RUBRICA_KEY_DICT_SECRETARY);
  rubrica_dictionary_display_string(prefbox, client, "entry_secretary_phone", 
				    RUBRICA_KEY_DICT_SECRETARY_PHONE);
  rubrica_dictionary_display_string(prefbox, client, "entry_spouse_name", 
				    RUBRICA_KEY_DICT_SPOUSE_NAME);
  rubrica_dictionary_display_string(prefbox, client, "entry_married", 
				    RUBRICA_KEY_DICT_MARRIED);
  rubrica_dictionary_display_string(prefbox, client, "entry_anniversary",
				    RUBRICA_KEY_DICT_ANNIVERSARY);
  rubrica_dictionary_display_string(prefbox, client, "entry_children", 
				    RUBRICA_KEY_DICT_CHILDREN);
  rubrica_dictionary_display_string(prefbox, client, "entry_child_notes", 
				    RUBRICA_KEY_DICT_CHILDREN_NOTES);
  rubrica_dictionary_display_string(prefbox, client, "entry_hobby", 
				    RUBRICA_KEY_DICT_HOBBY);
  rubrica_dictionary_display_string(prefbox, client, "entry_notes", 
				    RUBRICA_KEY_DICT_NOTES);
  rubrica_dictionary_display_string(prefbox, client, "entry_public_key", 
				    RUBRICA_KEY_DICT_PUBLIC_KEY);
  rubrica_dictionary_display_string(prefbox, client, "entry_birthday", 
				    RUBRICA_KEY_DICT_BIRTHDAY);
  rubrica_dictionary_display_string(prefbox, client, "entry_other_notes", 
				    RUBRICA_KEY_DICT_OTHER_NOTES);
}


void rubrica_dictionary_load_dictionary(gchar *dictionary)
{
  GtkWidget *prefbox;
  GConfClient *client;
  gboolean err;

  client  = gconf_client_get_default();

  err = rubrica_dictionary_real_dictionary_load(dictionary);
  
  if (!err)
    {
      gchar *name;

      prefbox = rubrica_preferences_get_preferencesbox();

      name = g_strdup(g_basename(dictionary));
      rubrica_preferences_set_user_dictionary(name);
      rubrica_dictionary_display_dictionary(prefbox, client);
      g_free(name);
    }
  else
    rubrica_preferences_set_user_dictionary(""); 
}

/*
  return TRUE if an error has occurred
  return FALSE if error hasn't occurred
 */
gboolean rubrica_dictionary_load_user_dictionary(GConfClient* client)
{
  gchar *dictionary;
  gchar *language;
  gchar *home;
  gboolean err;
  
  home       = rubrica_preferences_get_rubrica_home();
  language   = rubrica_preferences_get_user_dictionary();  
  if (!language)
    {
      g_print("\nI don't have dictionary name");

      return FALSE;
    }
  dictionary = g_strdup_printf("%s/dictionary/%s", home, language); 

  err = rubrica_dictionary_real_dictionary_load(dictionary);

  g_free(dictionary);

  return err;
}


gboolean rubrica_dictionary_real_dictionary_load(gchar *dictionary)
{
  GConfClient *client;
  xmlDocPtr doc;
  xmlNodePtr node;
  gchar *lang, *home;

  if (!g_file_test(dictionary, G_FILE_TEST_EXISTS))
    {
      gchar *str;
      
      str = g_strdup_printf(_("The dictionary %s doesn't exist"), dictionary);

      rubrica_log_write_log(str);
      rubrica_dialog_error(str);

      g_free(str);

      return TRUE;
    }

  client = gconf_client_get_default();
  home = g_strdup(g_get_home_dir());

  rubrica_utils_check_dictionary_home(home);
  g_free(home);
  
  doc = xmlParseFile(dictionary);
  if (!doc)
    {
      gchar *str;
      
      str = g_strdup_printf(_("I can't read the dictionary\n%s"), dictionary); 

      rubrica_log_write_log(str);
      rubrica_dialog_error(str);
      
      g_free(str);

      return TRUE;
    }

   if (!doc->children || !doc->children->name || 
      g_strcasecmp(doc->children->name, "Rubrica_dictionary") != 0)
    {
      gchar *str;

      str = g_strdup_printf(_("%s\ndon't seem to be a Rubrica dictionary"
			      "\nHeader doesn't match with "
			      "\"Rubrica_dictionary\"\nCan't "
			      "read this file."), dictionary);

      rubrica_log_write_log(str);     
      rubrica_dialog_error(str);
      xmlFreeDoc(doc);
      g_free(str);
      
      return TRUE;
    } 

   lang = xmlNodeGetLang(doc->children); 
   if (g_strcasecmp(lang, g_basename(dictionary)) != 0)
     {
       gchar str[] = N_("Language stored in file and requested "
			"language don't match");

       rubrica_dialog_error(str);
       rubrica_log_write_log(str);            

       return TRUE;
     }

   node = doc->children->children;
   if (xmlIsBlankNode(node))
     node = node->next;
   
   while (node)
     {
       rubrica_dictionary_parse_node(client, node);

       node = node->next;
       if (node && xmlIsBlankNode(node))
	 node = node->next;
     }   

   g_object_unref(client);
   xmlFreeDoc(doc);

   return FALSE;
}



void rubrica_dictionary_save_dictionary(gchar *dictionary)
{
  xmlDocPtr doc;
  xmlNodePtr generale;
  xmlNodePtr personale;
  xmlNodePtr net;
  xmlNodePtr tel;
  xmlNodePtr work;
  xmlNodePtr misc;
  GConfClient *client;
  
  client = gconf_client_get_default();
  
  doc = xmlNewDoc("1.0");

  doc->children = xmlNewDocNode(doc, NULL, "Rubrica_dictionary", NULL);
  xmlNodeSetLang(doc->children, g_basename(dictionary));
  xmlSetProp(doc->children, "version", VERSION);
  
  generale = xmlNewChild(doc->children, NULL, "General", "");
  xmlSetProp(generale, "Card", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_CARD, NULL)); 
  xmlSetProp(generale, "Categories", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_CARD_GROUP, 
				     NULL));   
  xmlSetProp(generale, "Data", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_DATA, NULL));
  xmlSetProp(generale, "Address", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_ADDRESS, NULL));
  xmlSetProp(generale, "Company", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_COMPANY, NULL));
  xmlSetProp(generale, "Work", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_WORK, NULL));

  personale = xmlNewChild(doc->children, NULL, "Personal", "");
  xmlSetProp(personale, "FirstName", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_FIRST_NAME, 
				     NULL));
  xmlSetProp(personale, "MiddleName", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_MIDDLE_NAME, 
				     NULL));
  xmlSetProp(personale, "LastName", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_LAST_NAME, 
				     NULL));
  xmlSetProp(personale, "Profession", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_PROFESSION, 
				     NULL));
  xmlSetProp(personale, "NamePrefix", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_NAME_PREFIX, 
				     NULL));
  xmlSetProp(personale, "Title", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_TITLE, NULL));

  xmlSetProp(personale,"Street", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_STREET, NULL));  
  xmlSetProp(personale, "StreetNumber", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_STREET_NUMBER, 
				     NULL));
  xmlSetProp(personale,"ZipCode", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_ZIP_CODE, NULL));
  xmlSetProp(personale,"City", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_CITY, NULL));
  xmlSetProp(personale,"Province", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_PROVINCE, NULL));
  xmlSetProp(personale, "State", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_STATE, NULL));
  xmlSetProp(personale,"Country", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_COUNTRY, NULL));

  net = xmlNewChild(doc->children, NULL, "Net", "");
  xmlSetProp(net, "WebAddress", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_WEB_ADDRESS, 
				     NULL));
  xmlSetProp(net,"Web", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_WEB, NULL));   
  xmlSetProp(net, "EMailAddress", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_EMAIL_ADDRESS, 
				     NULL));
  xmlSetProp(net,"Email", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_EMAIL, NULL));

  tel = xmlNewChild(doc->children, NULL, "Telephone", "");
  xmlSetProp(tel, "TelephoneNumbers", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_TELEPHONE_LIST ,
				     NULL));
  xmlSetProp(tel, "Telephone", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_TELEPHONE_NUMBER,
				     NULL));
  xmlSetProp(tel,"TelephoneType", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_TELEPHONE_TYPE, 
				     NULL));
  xmlSetProp(tel,"TelephoneWork", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_TELEPHONE_WORK, 
				     NULL));
  xmlSetProp(tel,"TelephoneHome", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_TELEPHONE_HOME, 
				     NULL));

  xmlSetProp(tel,"TelephoneFax", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_TELEPHONE_FAX, 
				     NULL));
  xmlSetProp(tel,"TelephoneCell", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_TELEPHONE_CELL, 
				     NULL));

  xmlSetProp(tel,"TelephoneOther", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_TELEPHONE_OTHER, 
				     NULL));


  work = xmlNewChild(doc->children, NULL, "Work", "");
  xmlSetProp(work, "CompanyName", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_COMPANY_NAME, 
				     NULL));
  xmlSetProp(work, "Operator", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_OPERATOR, NULL));
  xmlSetProp(work, "Green", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_GREEN, NULL));
  xmlSetProp(work, "CustomerCare", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_CUSTOMER_CARE, 
				     NULL));
  xmlSetProp(work, "Assigment", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_ASSIGMENT, 
				     NULL));
  xmlSetProp(work, "Organization", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_ORGANIZATION, 
				     NULL));
  xmlSetProp(work, "Department", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_DEPARTMENT, 
				     NULL));
  xmlSetProp(work, "SubDepartment", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_SUB_DEPARTMENT, 
				     NULL));
  xmlSetProp(work, "Secretary", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_SECRETARY, 
				     NULL));
  xmlSetProp(work, "SecretaryPhone", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_SECRETARY_PHONE,
				     NULL));

  misc = xmlNewChild(doc->children, NULL, "Misc", "");
  xmlSetProp(misc, "SpouseName", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_SPOUSE_NAME, 
				     NULL));
  xmlSetProp(misc, "Married", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_MARRIED, NULL));
  xmlSetProp(misc, "Anniversary", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_ANNIVERSARY, 
				     NULL));
  xmlSetProp(misc, "Children", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_CHILDREN, NULL));
  xmlSetProp(misc, "ChildrenNotes", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_CHILDREN_NOTES, 
				     NULL));
  xmlSetProp(misc, "Hobbies", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_HOBBY, NULL));
  xmlSetProp(misc, "Notes", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_NOTES, NULL));
  xmlSetProp(misc,"PublicKey",
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_PUBLIC_KEY, 
				     NULL));
  xmlSetProp(misc, "Birthday", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_BIRTHDAY, NULL));

  xmlSetProp(misc, "OtherNotes", 
	     gconf_client_get_string(client, RUBRICA_KEY_DICT_OTHER_NOTES, 
				     NULL));

  if( xmlSaveFile(dictionary, doc) == -1)
    {
      gchar *str;

      str = g_strdup_printf(_("Can't save dictionary %s"), dictionary);

      g_warning(str);
      rubrica_log_write_log(str);
      rubrica_dialog_error(str);
      
      g_free(str);

      return;
    }
  else
    {
      gchar *str;

      str = g_strdup_printf(_("\n%s\ndictionary has been saved in:\n%s"), 
			    dictionary,
			    rubrica_preferences_get_dictionary_home());

      rubrica_log_write_log(str);
      rubrica_dialog_simple_message(str);
      
      g_free(str);
    }

  xmlFreeDoc(doc);
}


void rubrica_dictionary_parse_node(GConfClient *client, xmlNodePtr node)
{
  gchar *buffer;

  if (strcmp(node->name, "General") == 0)
    {
      buffer = xmlGetProp(node, "Card");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_CARD, 
			      buffer ? buffer : "", NULL); 

      buffer = xmlGetProp(node, "Categories");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_CARD_GROUP, 
			      buffer ? buffer : "", NULL); 

      buffer = xmlGetProp(node, "Data");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_DATA, 
			      buffer ? buffer : "", NULL);   

      buffer = xmlGetProp(node, "Address");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_ADDRESS, 
			      buffer ? buffer : "", NULL);     

      buffer = xmlGetProp(node, "Company");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_COMPANY, 
			      buffer ? buffer : "", NULL);

      buffer = xmlGetProp(node, "Work");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_WORK, 
			      buffer ? buffer : "", NULL);
    }
  else if (strcmp(node->name, "Personal") == 0)
    {
      buffer = xmlGetProp(node, "FirstName");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_FIRST_NAME, 
			      buffer ? buffer : "", NULL);      

      buffer = xmlGetProp(node, "MiddleName");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_MIDDLE_NAME, 
			      buffer ? buffer : "", NULL);   

      buffer = xmlGetProp(node, "LastName");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_LAST_NAME, 
			      buffer ? buffer : "", NULL);   

      buffer = xmlGetProp(node, "Profession");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_PROFESSION, 
			      buffer ? buffer : "", NULL);   

      buffer = xmlGetProp(node, "NamePrefix");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_NAME_PREFIX, 
			      buffer ? buffer : "", NULL);   

      buffer = xmlGetProp(node, "Title");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_TITLE, 
			      buffer ? buffer : "", NULL);   

      buffer = xmlGetProp(node, "Street");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_STREET, 
			      buffer ? buffer : "", NULL);   

      buffer = xmlGetProp(node, "StreetNumber");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_STREET_NUMBER, 
			      buffer ? buffer : "", NULL);   

      buffer = xmlGetProp(node, "ZipCode");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_ZIP_CODE, 
			      buffer ? buffer : "", NULL);   

      buffer = xmlGetProp(node, "City");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_CITY, 
			      buffer ? buffer : "", NULL);   

      buffer = xmlGetProp(node, "Province");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_PROVINCE, 
			      buffer ? buffer : "", NULL);   

      buffer = xmlGetProp(node, "State");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_STATE, 
			      buffer ? buffer : "", NULL);   

      buffer = xmlGetProp(node, "Country");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_COUNTRY, 
			      buffer ? buffer : "", NULL);   
    }

  else if (strcmp(node->name, "Net") == 0)
    {
      buffer = xmlGetProp(node, "WebAddress");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_WEB_ADDRESS, 
			      buffer ? buffer : "", NULL);   

      buffer = xmlGetProp(node, "Web");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_WEB, 
			      buffer ? buffer : "", NULL);   

      buffer = xmlGetProp(node, "EMailAddress");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_EMAIL_ADDRESS, 
			      buffer ? buffer : "", NULL);   

      buffer = xmlGetProp(node, "Email");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_EMAIL, 
			      buffer ? buffer : "", NULL);   
    }
  else if (strcmp(node->name, "Telephone") == 0)
    {
      buffer = xmlGetProp(node, "TelephoneNumbers");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_TELEPHONE_LIST, 
			      buffer ? buffer : "", NULL);   

      buffer = xmlGetProp(node, "Telephone");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_TELEPHONE_NUMBER, 
			      buffer ? buffer : "", NULL);   

      buffer = xmlGetProp(node, "TelephoneType");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_TELEPHONE_TYPE, 
			      buffer ? buffer : "", NULL);         

      buffer = xmlGetProp(node, "TelephoneWork");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_TELEPHONE_WORK, 
			      buffer ? buffer : "", NULL);         

      buffer = xmlGetProp(node, "TelephoneHome");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_TELEPHONE_HOME, 
			      buffer ? buffer : "", NULL);         

      buffer = xmlGetProp(node, "TelephoneFax");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_TELEPHONE_FAX, 
			      buffer ? buffer : "", NULL);         

      buffer = xmlGetProp(node, "TelephoneCell");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_TELEPHONE_CELL, 
			      buffer ? buffer : "", NULL);         

      buffer = xmlGetProp(node, "TelephoneOther");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_TELEPHONE_OTHER, 
			      buffer ? buffer : "", NULL);         

    }
  else if (strcmp(node->name, "Work") == 0)
    {
      buffer = xmlGetProp(node, "CompanyName");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_COMPANY_NAME, 
			      buffer ? buffer : "", NULL); 

      buffer = xmlGetProp(node, "Operator");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_OPERATOR, 
			      buffer ? buffer : "", NULL); 

      buffer = xmlGetProp(node, "Green");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_GREEN, 
			      buffer ? buffer : "", NULL); 

      buffer = xmlGetProp(node, "CustomerCare");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_CUSTOMER_CARE, 
			      buffer ? buffer : "", NULL); 

      buffer = xmlGetProp(node, "Assigment");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_ASSIGMENT, 
			      buffer ? buffer : "", NULL); 

      buffer = xmlGetProp(node, "Organization");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_ORGANIZATION, 
			      buffer ? buffer : "", NULL); 

      buffer = xmlGetProp(node, "Department");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_DEPARTMENT, 
			      buffer ? buffer : "", NULL); 

      buffer = xmlGetProp(node, "SubDepartment");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_SUB_DEPARTMENT, 
			      buffer ? buffer : "", NULL); 

      buffer = xmlGetProp(node, "Secretary");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_SECRETARY, 
			      buffer ? buffer : "", NULL); 

      buffer = xmlGetProp(node, "SecretaryPhone");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_SECRETARY_PHONE, 
			      buffer ? buffer : "", NULL); 
      }
  else if (strcmp(node->name, "Misc") == 0)
    {
      buffer = xmlGetProp(node, "SpouseName");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_SPOUSE_NAME, 
			      buffer ? buffer : "", NULL);   

      buffer = xmlGetProp(node, "Married");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_MARRIED, 
			      buffer ? buffer : "", NULL); 

      buffer = xmlGetProp(node, "Anniversary");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_ANNIVERSARY, 
			      buffer ? buffer : "", NULL); 

      buffer = xmlGetProp(node, "Children");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_CHILDREN, 
			      buffer ? buffer : "", NULL); 

      buffer = xmlGetProp(node, "ChildrenNotes");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_CHILDREN_NOTES, 
			      buffer ? buffer : "", NULL); 

      buffer = xmlGetProp(node, "Hobbies");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_HOBBY, 
			      buffer ? buffer : "", NULL); 

      buffer = xmlGetProp(node, "Notes");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_NOTES, 
			      buffer ? buffer : "", NULL); 

      buffer = xmlGetProp(node, "PublicKey");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_PUBLIC_KEY, 
			      buffer ? buffer : "", NULL);

      buffer = xmlGetProp(node, "Birthday");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_BIRTHDAY, 
			      buffer ? buffer : "", NULL); 

      buffer = xmlGetProp(node, "OtherNotes");
      gconf_client_set_string(client, RUBRICA_KEY_DICT_OTHER_NOTES, 
			      buffer ? buffer : "", NULL);
    }
  else 
    return; 
}
