module Fog
  module AWS
    class RDS
      class Real
        require 'fog/aws/parsers/rds/create_event_subscription'

        # Subscribes a db instance to an SNS queue
        #
        # @see http://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateEventSubscription.html
        # === Parameters
        # * Enabled <~Boolean> - set to true to activate the subscription, set to false to create the subscription but not active it
        # * EventCategories <~Array> - A list of event categories for a SourceType that you want to subscribe to
        # * SnsTopicArn <~String> - The Amazon Resource Name of the SNS topic created for event notification
        # * SourceIds <~Array> - The list of identifiers of the event sources for which events will be returned
        # * SourceType <~String> - The type of source that will be generating the events. For example, if you want to be notified of events generated by a DB instance, you would set this parameter to db-instance. if this value is not specified, all events are returned
        # * SubscriptionName <~String> - The name of the subscription
        # * Tags <~Array> - A list of tags
        def create_event_subscription(options={})
          if event_categories = options.delete("EventCategories")
            options.merge!(Fog::AWS.indexed_param('EventCategories.member.%d', [*event_categories]))
          end
          if source_ids = options.delete("SourceIds")
            options.merge!(Fog::AWS.indexed_param('SourceIds.member.%d', [*source_ids]))
          end
          if tags = options.delete("tags")
            options.merge!(Fog::AWS.indexed_param('Tags.member.%d', [*tags]))
          end

          request({
            "Action" => "CreateEventSubscription",
            :parser  => Fog::Parsers::AWS::RDS::CreateEventSubscription.new,
          }.merge(options))
        end
      end

      class Mock
        def create_event_subscription(options={})
          response = Excon::Response.new
          name     = options.delete('SubscriptionName')
          arn      = options.delete('SnsTopicArn')

          if self.data[:event_subscriptions][name]
            raise Fog::AWS::RDS::IdentifierTaken.new("SubscriptionAlreadyExist => Subscription already exists")
          end

          subscription = {
            'CustSubscriptionId' => name,
            'EventCategories'    => options['EventCategories'] || [],
            'SourceType'         => options['SourceType'],
            'Enabled'            => options.fetch(:enabled, "true"),
            'Status'             => 'creating',
            'CreationTime'       => Time.now,
            'SnsTopicArn'        => arn,
          }

          self.data[:event_subscriptions][name] = subscription

          response.body = {
            "ResponseMetaData" => {"RequestId" => Fog::AWS::Mock.request_id},
            "CreateEventSubscriptionResult" => { "EventSubscription" => subscription }
          }
          response
        end
      end
    end
  end
end
