#include <rumba/manifold.h>
#include <rumba/manifoldFile.h>
#include <rumba/arghandler.h>
#include <rumba/parse.h>
#include <rumba/util.hpp>
#include <iostream>
#include <cstdlib>
#include <vector>

using std::cerr;
using std::endl;
using RUMBA::stream_cast;
using RUMBA::intPoint;

void error()
{
	cerr << "Usage: getslice -i infile -o outfile x y z t dx dy dz dt" << endl;
}

int main(int argc, char** argv)
{
	RUMBA::ArgHandler::setRequiredDefaultArg("infile");
	RUMBA::ArgHandler::setRequiredDefaultArg("outfile");

	try 
	{
		RUMBA::ArgHandler argh(argc,argv);
		std::string infile, outfile;
		intPoint origin,dims;
		bool debug = true;
		if (argh.arg("help"))
		{
			error();
			exit(0);
		}

		argh.arg("infile", infile );
		argh.arg("outfile", outfile );
		std::vector<string> v = argh.loose();


		if ( v.size () < 8 )
		{
			error();
			exit(EXIT_FAILURE);
		}

		try 
		{
			dims.x() = stream_cast<int> ( v[4] );
			dims.y() = stream_cast<int> ( v[5] );
			dims.z() = stream_cast<int> ( v[6] );
			dims.t() = stream_cast<int> ( v[7] );
			origin.x() = stream_cast<int> ( v[0] );
			origin.y() = stream_cast<int> ( v[1] );
			origin.z() = stream_cast<int> ( v[2] );
			origin.t() = stream_cast<int> ( v[3] );
		}
		catch ( RUMBA::Exception & e )
		{
			cerr << "Numeric argument expected" << std::endl;
			if (debug)
				cerr << "Debug: " << e.error() << endl;
		}



		RUMBA::ManifoldFile* M = RUMBA::ManifoldFile::construct(infile.c_str());
		if (!M)
			throw RUMBA::Exception (
					std::string ("Couldn't open input file ") + infile 
					);

		if (
				dims.x() + origin.x() > M->extent().x() ||
				dims.y() + origin.y() > M->extent().y() ||
				dims.z() + origin.z() > M->extent().z() ||
				dims.t() + origin.t() > M->extent().t() ||
				origin.x() < 0 || 
				origin.y() < 0 || 
				origin.z() < 0 || 
				origin.t() < 0 ||
				dims.x() <= 0 ||
				dims.y() <= 0 ||
				dims.z() <= 0 ||
				dims.t() <= 0
				)
			throw RUMBA::Exception ( "Extents out of range" );

		RUMBA::BaseManifold* N = M->get ( origin, dims );

		N->save ( outfile.c_str() );
		std::ofstream f("outmanifold2.txt");
		M->print(f);
		delete M;
		delete N;

	
	}
    catch (RUMBA::MissingArgumentException& s)
    {
		cerr << "Error, missing required argument: " << s.error() << endl;
		error();
	}
    catch (RUMBA::DuplicateArgumentException& s)
    {
		cerr << "Duplicate argument: " << s.error() << endl;
		error();
	}
	catch (RUMBA::ArgHandlerException& s)
	{
		cerr << "Error: " << s.error() << endl;
		error();
	}                   
	catch ( RUMBA::Exception & e)
	{
		cerr << "Fatal exception " << e.error() << endl;	
	}

	return 0;
}
