
#include <AnalyzeFile.h>

#include <string>
#include <fstream>

#include <rumba/binary_header.h>
#include <rumba/rumba_system.h>
#include <rumba/exception.h>
#include <rumba/parse.h>
#include <rumba/parsecsv.h>
#include <rumba/factory.h>
#include <rumba/fileIO.h>

#include <sys/stat.h>

#include <rumba/iohandler_base.h>

using RUMBA::AnalyzeFile;
using RUMBA::ManifoldFile;
using RUMBA::header_request;

using std::endl;
using std::list;
using std::map;
using std::string;

/**
  *	Register AnalyzeFile and create a prototype
  *
  */

RUMBA::AnalyzeFile AnalyzeFileInstance(Exemplar("AnalyzeFile"));
RUMBA::AnalyzeFile* AnalyzeFileExemplar = &AnalyzeFileInstance;


AnalyzeFile::~AnalyzeFile()
{
	if (Data && Data->writeable())
	{
		saveHeader(HeaderFile.c_str());
	}
}

// specific hack to determine byte order.
// if the header_request includes LittleEndian, leave the 
// body of this function empty
void AnalyzeFile::endian_hack(const char* filename, ifstream& in)
{
	LittleEndian = false;
	struct stat statBuf;
	stat(filename,&statBuf);
	int tmp = RUMBA::getInt(in, LittleEndian);
	if (statBuf.st_size != tmp ) LittleEndian = true;
}


AnalyzeFile::AnalyzeFile()
	: ManifoldFile("AnalyzeFile")
{
	loadRc();
}

ManifoldFile* AnalyzeFile::getNew()
{
	return new AnalyzeFile;
}



AnalyzeFile::AnalyzeFile( Exemplar a )
:ManifoldFile(a, "AnalyzeFile")
{
	log.logName() << "loading rc ... ";
	loadRc(); // important !! we use this in isMine()
	log.logName() << "done ";

	log.logName() << "Calling " << classname << "(Exemplar)\n";
}

bool AnalyzeFile::isMine( std::string filename )
{
	log.logName() << "Calling " << classname << "::isMine(" << filename << ")\n";
	string ext;
	std::string::size_type p = filename.find_last_of(".");
	if (p==std::string::npos)
		return false;
	ext.assign( filename, p, filename.length()); 
	bool tmp = ( ext == data_extension || ext == header_extension );
	log.logName() << "Returning " << tmp;
	return tmp;
}









void AnalyzeFile::saveHeader(const char* filename) 
{
	log.logName() << "Calling " << classname << "::saveHeader(" << filename << ")\n";

	std::ofstream fin(filename);

	if (!fin){
		#ifdef USE_EXCEPTIONS
		throw RUMBA::BadFile ( "loadHeader : Cannot open file");
		#endif
	}

	fin.seekp( stream_cast<int>(header_size)  - 1 );
	fin.put(0);

	loadCsv();
	copyHeaderDataFromManifold();
	HeaderData["datatype"] = stream_cast<int> ( types [ HeaderData["normalized_datatype"].asString() ] );
	log.logName() << "In saveHeader(), Header datatype : "<<HeaderData["datatype"] << "\n";
	log.logName() << "In saveHeader(), normalized datatype : "<<HeaderData["normalized_datatype"] << "\n";

	fillRequiredHeaders();
	RUMBA::putData ( HeaderData, HeaderRequests, fin, LittleEndian );
	fin.close();


}



void AnalyzeFile::loadHeader(const char* filename)
{	
	log.logName() << "Calling " << classname << "::loadHeader(" << filename << ")\n";

	std::ifstream fin(filename);

	if (!fin){
		#ifdef USE_EXCEPTIONS
		throw RUMBA::BadFile ( "loadHeader : Cannot open file");
		#endif
	}

	endian_hack(filename,fin);

	loadCsv();
	log.logName() << "calling GetData()" << "\n";
	HeaderData = RUMBA::getData ( HeaderRequests, fin, LittleEndian );
	log.logName() << "succesfully got out of getData()" << "\n";

	for ( std::map<std::string, RUMBA::Splodge>::const_iterator it = HeaderData.begin(); it!=HeaderData.end(); ++it)
	{
		log.logName() << "HeaderData : " << it->first << " " << it->second << "\n";
	}
	fin.close();
	log.logName() << "succesfully got out of getData()" << "\n";
	copyHeaderDataToManifold();
	Skip = skip();
	log.logName() << "skipX " << Skip.x() << "\n";

	Orient = RUMBA::Orientation 
		(
		 RUMBA::patient_right(), RUMBA::patient_back(), RUMBA::patient_foot(),
		 extent()
		);

}

void AnalyzeFile::loadCsv()
{
	std::ifstream conf_in;

	std::string basename = classname + std::string(".csv");
	std::string conf = RUMBA::find_file_modules() + string("/") + basename; 

	conf_in.open(conf.c_str());
	if ( !conf_in )
	{
		log.logName() << "Couldn't open file: " << conf << "\n";
		throw RUMBA::BadFile(conf);
	}
	parseCsv(HeaderRequests, conf_in);
	log.logName() << "Succesfully loaded csv file" << "\n";
}

void AnalyzeFile::loadRc()
{
	std::ifstream conf_in;

	std::string basename = classname + std::string(".rc");
	std::string conf = RUMBA::find_file_modules() + string("/") + basename; 
	log.logName() << "Opening rc file: " << conf << "\n";

	conf_in.open(conf.c_str());
	if ( !conf_in )
	{
		log.logName() << "Couldn't open file: " << conf << "\n";
		throw RUMBA::BadFile(conf);
	}
	
	RUMBA::rcFind(conf_in, "data extension", data_extension );
	RUMBA::rcFind(conf_in, "header extension", header_extension );
	RUMBA::rcFind(conf_in, "header size", header_size );
	



	types["char"] = "";
	types["int16"] = "";
	types["int32"] = "";
	types["float32"] = "";
	types["float64"] = "";
	types["defaulttype"] = "";

	RUMBA::rcFind(conf_in, "char", types["char"] );
	RUMBA::rcFind(conf_in, "int16", types["int16"] );
	RUMBA::rcFind(conf_in, "int32", types["int32"] );
	RUMBA::rcFind(conf_in, "float32", types["float32"] );
	RUMBA::rcFind(conf_in, "float64", types["float64"] );
	RUMBA::rcFind(conf_in, "defaulttype", types["defaulttype"] );

}



RUMBA::Factory* AnalyzeFile::getFactoryFromDataType(std::string t)
{
	log.logName() << "In getFactoryFromDataType() : t is " << t << "\n";
	log.logName() << "types[float64] " << types["float64"] << "\n";
	if ( t == types["char"] || t == "char" )
		return CharFactory::get();
	else if ( t == types["int16"] || t == "int16" )
		return ShortFactory::get();
	else if ( t == types["int32"] || t == "int32" )
		return IntFactory::get();
	else if ( t == types["float32"] || t == "float32" )
		return FloatFactory::get();
	else if ( t == types["float64"] || t == "float64" )
		return DoubleFactory::get();

	throw RUMBA::Exception("getFactoryFromDataType failed");
}

/*
std::string AnalyzeFile::normalizeDataType(std::string t)
{
	log.logName() << "In normalizeDataType() : t is " << t << "\n";
	if ( t == types["char"] || t == "char" )
		return "char";
	else if ( t == types["int16"] || t == "int16" )
		return "int16";
	else if ( t == types["int32"] || t == "int32" )
		return "int32";
	else if ( t == types["float32"] || t == "float32" )
		return "float32";
	else if ( t == types["float64"] || t == "float64" )
		return "float64";

	throw RUMBA::Exception("normalizeDataType failed");

}
*/

void AnalyzeFile::fillRequiredHeaders()
{
	list<header_request>::const_iterator lit;
	map<string, RUMBA::Splodge >::const_iterator mit;

	if ( HeaderRequests.empty() ) 
	{
		return;
	}

	for (lit = HeaderRequests.begin(); lit != HeaderRequests.end(); ++lit )
		if ( lit->required )
		{
			mit = HeaderData.find(lit->name);
			if ( mit == HeaderData.end() )
			{
				HeaderData.insert ( std::make_pair( lit->name, lit->default_value ));
				log.logName() << "Adding required header " << lit->name << "=" << lit->default_value << "\n";
			}
			else if ( mit->second.asString() == "0" || mit->second.asString() == "" )
			{
				HeaderData[ lit->name ] =  lit->default_value;
				log.logName() << "Adding required header " << lit->name << "=" << lit->default_value << "\n";
			}
		}
}
