// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "framergb.h"
#include "colorscalergb.h"
#include "fitsimage.h"
#include "ps.h"
#include "util.h"
#include "nan.h"
#include "contour.h"

// Frame Member Functions

FrameRGB::FrameRGB(Tcl_Interp* i, Tk_Canvas c, Tk_Item* item)
  : FrameBase(i, c, item)
{
  channel = 0;
  rgbSystem = WCS;

  for (int ii=0; ii<3; ii++) {
    view[ii] = 1;

    fits[ii] = NULL;
    cfits[ii] = NULL;
    fitsCount[ii] = 0;

    bias[ii] = .5;
    contrast[ii] = 1.0;

    colorScale[ii] = NULL;
    
    displayMode[ii] = SINGLE;
    mosaicType[ii] = NOMOSAIC;
    mosaicSystem[ii] = WCS;

    WCSrot[ii] = 0;
    WCSorientation[ii] = NORMAL;

    contour[ii] = NULL;
  }

  colorCount = 0;
  colorCells = NULL;

  setCurrentFits(NULL);
  keyFits = NULL;
  channelFits = &fits[channel];
  currentCount = &fitsCount[channel];
  currentScale = &frScale[channel];
  currentMode = &displayMode[channel];
  currentMosaic = &mosaicType[channel];
  currentMosaicSystem = &mosaicSystem[channel];

  currentWCScdelt = &WCScdelt[channel];
  currentWCSrot = &WCSrot[channel];
  currentWCSorientation = &WCSorientation[channel];
  currentWCSbb = &WCSbb[channel];
  currentWCSmatrix = &WCSmatrix[channel];

  currentcontour = &contour[channel];

  isRGBCube = 0;
}

FrameRGB::~FrameRGB()
{
  for (int ii=0; ii<3; ii++) {
    if (colorScale[ii])
      delete colorScale[ii];
  }

  if (colorCells)
    delete [] colorCells;
}

void FrameRGB::reset()
{
  for (int ii=0; ii<3; ii++) {
    bias[ii] = 0.5;
    contrast[ii] = 1.0;
  }

  isRGBCube = 0;
  FrameBase::reset();
}

void FrameRGB::setCurrentFits(FitsImage* f)
{
  currentFits = f;
  cfits[channel] = f;
}

void FrameRGB::setBinCursor(const Vector& v)
{
  for (int ii=0; ii<3; ii++)
    if (fits[ii])
      fits[ii]->setBinCursor(cursor);
}

void FrameRGB::updateBin(const Matrix& m)
{
  // Note: cursor is in USER coords, imageCenter() in IMAGE coords
  cursor = imageCenter();

  if (hasKeyFits() && (*keyFits == currentFits)) {
    // update markers,crosshair,aux contours
    crosshair *= m;
    markerbg();
    updateMarkerCoords(m);
    markerfg();
    updateMarkerCoords(m);

    // aux contours
    {
      Contour* ptr=auxcontours.head();
      while (ptr) {
	ptr->updateCoords(m);
	ptr=ptr->next();
      }
    }
  }

  FitsImage* currentSave = currentFits;

  for (int ii=0; ii<3; ii++) 
    if (fits[ii] && fits[ii]->isHist()) {
      // update bin

      if (fits[ii] == currentSave && contour[ii])
	contour[ii]->updateCoords(m);

      currentFits = fits[ii];
      cfits[ii] = fits[ii];
      currentScale = &frScale[ii];
      channelFits = &fits[ii];
      currentCount = &fitsCount[ii];
      currentMode = &displayMode[ii];
      currentMosaic = &mosaicType[ii];
      currentMosaicSystem = &mosaicSystem[ii];

      // delete any previous slices
      {
	FitsImage* ptr = (*channelFits)->next();
	(*channelFits)->setNext(NULL);
	while (ptr) {
	  FitsImage* tmp = ptr->next();
	  delete ptr;
	  ptr = tmp;
	}
      }

      // finish bin
      *currentCount = 1;
      *currentMode = SINGLE;

      // bin data cube
      int bd = (*channelFits)->binDepth();
      if (bd > 1) {
	FitsImage* ptr = *channelFits;
	for (int i=1; i<bd; i++) {
	  FitsImage* next = NULL;
	  next = 
	    new FitsImageFitsNextHist(this,*channelFits, ptr->getImgFile());
	  if (next && next->isValid()) {
	    ptr->setNext(next);
	    ptr = next;
	    (*currentCount)++;
	  }
	  else {
	    if (next)
	      delete next;
	    break;
	  }
	}

      }

      updateBinFileNames();
      resetClip(0);
    }

  slice = 1;
  currentFits = fits[channel];
  cfits[channel] = fits[channel];
  currentScale = &frScale[channel];

  channelFits = &fits[channel];
  currentCount = &fitsCount[channel];
  currentMode = &displayMode[channel];
  currentMosaic = &mosaicType[channel];
  currentMosaicSystem = &mosaicSystem[channel];

  currentWCScdelt = &WCScdelt[channel];
  currentWCSrot = &WCSrot[channel];
  currentWCSorientation = &WCSorientation[channel];
  currentWCSbb = &WCSbb[channel];
  currentWCSmatrix = &WCSmatrix[channel];

  currentcontour = &contour[channel];

  align();
  update(MATRIX); // because we have changed zoom
}

BBox FrameRGB::imageBBox()
{
  // returns imageBBox in IMAGE coords

  // make sure we have the rgb matrices defined
  align();

  BBox r;
  int first=1;
  for (int ii=0; ii<3; ii++) {
    if (fits[ii]) {

      int* params = fits[ii]->getDataParams(frScale[ii].scanMode());
      int& xmin = params[1];
      int& xmax = params[2];
      int& ymin = params[3];
      int& ymax = params[4];

      if (first) {
	switch (mosaicType[ii]) {
	case NOMOSAIC:
	  {
	    Vector aa = Vector(xmin,ymin) * Translate(.5,.5) * 
	      rgb[ii] * dataToImage;
	    r = BBox(aa,aa);
	    r.bound(Vector(xmax-1,ymin) * Translate(.5,.5) * 
		    rgb[ii] * dataToImage);
	    r.bound(Vector(xmax-1,ymax-1) * Translate(.5,.5) * 
		    rgb[ii] * dataToImage);
	    r.bound(Vector(xmin,ymax-1) * Translate(.5,.5) * 
		    rgb[ii] * dataToImage);
	  }
	  break;
	case IRAF:
	  r = fits[ii]->getIRAFbb();
	  break;
	case WCSMOSAIC:
	case WFPC2:
	  r = (WCSbb[ii]);
	  break;
	}
	first = 0;
      } 
      else {
	switch (mosaicType[ii]) {
	case NOMOSAIC:
	  r.bound(Vector(xmin,ymin) * Translate(.5,.5) * 
		  rgb[ii]* dataToImage);
	  r.bound(Vector(xmax-1,ymin) * Translate(.5,.5) * 
		  rgb[ii] * dataToImage);
	  r.bound(Vector(xmax-1,ymax-1) * Translate(.5,.5) * 
		  rgb[ii] * dataToImage);
	  r.bound(Vector(xmin,ymax-1) * Translate(.5,.5) * 
		  rgb[ii] * dataToImage);
	  break;
	case IRAF:
	  r.bound(fits[ii]->getIRAFbb());
	  break;
	case WCSMOSAIC:
	case WFPC2:
	  r.bound(WCSbb[ii]);
	  break;
	}
      }
    }
  }

  if (DebugMosaic)
    cerr << "imageBBox() RGB " << r << ' ' 
	 << r.size() << ' ' 
	 << r.center() << endl;
  return r;
}

void FrameRGB::unloadFits()
{
  if (DebugPerf)
    cerr << "FrameRGB::unloadFits" << endl;

  while (fits[channel]) {
    FitsImage* tmp = fits[channel]->next();
    delete fits[channel];
    fits[channel] = tmp;
  }
  fits[channel] = NULL;
  cfits[channel] = NULL;

  if (contour[channel])
    delete contour[channel];
  contour[channel] = NULL;

  rgb[channel].identity();

  keyFits = NULL;
  slice = 1;
  setCurrentFits(NULL);
  *currentCount = 0;
  *currentMode = SINGLE;
  *currentMosaic = NOMOSAIC;

  // reset user mode
  // if (frScale[channel].clipMode() == FrScale::USERCLIP)
  // frScale[channel].setClipMode(FrScale::MINMAX);

  resetClip(0);
}

void FrameRGB::unloadAllFits()
{
  if (DebugPerf)
    cerr << "Frame::RGBunloadAllFits" << endl;

  for (int ii=0; ii<3; ii++) {
    while (fits[ii]) {
      FitsImage* tmp = fits[ii]->next();
      delete fits[ii];
      fits[ii] = tmp;
    }
    fits[ii] = NULL;
    cfits[ii] = NULL;
    fitsCount[ii] = 0;
    displayMode[ii] = SINGLE;
    mosaicType[ii] = NOMOSAIC;
    mosaicSystem[ii] = WCS;

    WCScdelt[ii] = Vector();
    WCSrot[ii] = 0;
    WCSorientation[ii] = NORMAL;
    WCSbb[ii] = BBox();
    WCSmatrix[ii] = Matrix();

    if (contour[ii])
      delete contour[ii];
    contour[ii] = NULL;

    rgb[ii].identity();

    // reset user mode
    // if (frScale[ii].clipMode() == FrScale::USERCLIP)
    // frScale[ii].setClipMode(FrScale::MINMAX);

    frScale[ii].deleteHistogramX();
    frScale[ii].deleteHistogramY();
    frScale[ii].deleteHistequ();

    if (frScale[ii].colorScaleType() == FrScale::HISTEQUSCALE)
      updateColorScale();
  }

  keyFits = NULL;
  FrameBase::unloadFits();
}

void FrameRGB::updateMatrices()
{
  // very important! do this first
  FrameBase::updateMatrices();

  for (int ii=0; ii<3; ii++) {
    FitsImage* ptr = fits[ii];
    while (ptr) {
      ptr->updateMatrices(mosaicType[ii], rgb[ii], refToUser, userToWidget,
			  widgetToCanvas, userToPanner);
      ptr = ptr->next();
    }
  }
}

void FrameRGB::updateMagnifierMatrices(Matrix& refToMagnifier)
{
  for (int ii=0; ii<3; ii++) {
    FitsImage* ptr = fits[ii];
    while (ptr) {
      ptr->updateMagnifierMatrices(refToMagnifier);
      ptr = ptr->next();
    }
  }
}

void FrameRGB::updateColorCells(unsigned char* cells, int cnt)
{
  if (DebugRGB) 
    cerr << "updateColorCells" << endl;

  // the colorbar widget will pass us a pointer to the indexCells

  colorCount = cnt;

  // copy the rgb vales to the colorCells array (for postscript printing)

  if (colorCells)
    delete [] colorCells;
  colorCells = new unsigned char[cnt*3];
  if (!colorCells) {
    cerr << "Frame Internal Error: Unable to Alloc colorCells" << endl;
    exit(1);
  }

  memcpy(colorCells, cells, cnt*3);
}

void FrameRGB::updateColorScale()
{
  // we need colors before we can construct a scale
  if (!colorCells)
    return;

  if (DebugRGB) 
    cerr << "updateColorScale" << endl;

  // since we need to know about bytes per pixel and byte order, 
  // we may need to wait til we have an XImage available and try again
  //  if (!baseXImage)
  //    return;

  for (int ii=0; ii<3; ii++) {
    if (colorScale[ii])
      delete colorScale[ii];

    switch (frScale[ii].colorScaleType()) {
    case FrScale::LINEARSCALE:
      colorScale[ii] = 
	new LinearScaleRGB(ii, colorCount, colorCells, colorCount);
      break;
    case FrScale::LOGSCALE:
      colorScale[ii] =
	new LogScaleRGB(ii, SCALESIZE, colorCells, colorCount);
      break;
    case FrScale::SQUAREDSCALE:
      colorScale[ii] =
	new SquaredScaleRGB(ii, SCALESIZE, colorCells, colorCount);
      break;
    case FrScale::SQRTSCALE:
      colorScale[ii] =
	new SqrtScaleRGB(ii, SCALESIZE, colorCells, colorCount);
      break;
    case FrScale::HISTEQUSCALE:
      calcHistEqu();
      colorScale[ii] = 
	new HistEquScaleRGB(ii, SCALESIZE, colorCells, colorCount, 
			    frScale[ii].histequ(), HISTEQUSIZE);
      break;
    }
  }
}

void FrameRGB::align()
{
  // set keyFits
  keyFits = NULL;
  int keyFitsID = 0;
  for (int ii=0; ii<3; ii++)
    if (fits[ii]) {
	keyFits = &fits[ii]; // first available becomes the key
	keyFitsID = ii;
	break;
    }
  
  if (!wcsAlignment || !hasKeyFits() || !(*keyFits)->hasWCS(wcsSystem)) {
    wcsRotation = 0;
    wcsOrientation = NORMAL;
    wcsOrientationMatrix.identity();
  }
  else {
    // wcs align
    if ((*keyFits)->hasWCS(wcsSystem)) {
      // this causes problems because imageBBox() call align()
      wcsRotation = (*keyFits)->getWCSRotation(wcsSystem,wcsSky);
      wcsOrientation = (*keyFits)->getWCSOrientation(wcsSystem,wcsSky);
    }
    else {
      wcsRotation = 0;
      wcsOrientation = NORMAL;
    }

    switch (wcsOrientation) {
    case NORMAL:
      wcsOrientationMatrix.identity();
      break;
    case XX:
      wcsOrientationMatrix = FlipX();
      break;
    case YY:
      wcsOrientationMatrix = FlipY();
      break;
    case XY:
      wcsOrientationMatrix = FlipXY();
      break;
    }

    // take into account any internal mosaic rotation/flip
    wcsRotation += (*keyFits)->rotation();
    switch ((*keyFits)->orientation()) {
    case NORMAL:
      break;
    case XX:
      wcsOrientationMatrix *= FlipX();
      break;
    case YY:
      wcsOrientationMatrix *= FlipY();
      break;
    case XY:
      wcsOrientationMatrix *= FlipXY();
      break;
    }
  }

  // image,pysical,amplifier,detector are ok, check for wcs
  if (rgbSystem >= WCS) {
    for (int ii=0; ii<3; ii++) {
      if (fits[ii] && !fits[ii]->hasWCS(rgbSystem)) {
	// ok, don't have requested coordinate system
	// down grade to image
	rgbSystem = IMAGE;
	break;
      }
    }
  }

  // rgb align
  for (int ii=0; ii<3; ii++) {
    rgb[ii].identity();

    if (fits[ii] && hasKeyFits()) {
      switch (rgbSystem) {
      case IMAGE:
	// nothing to do here
	break;
      case PHYSICAL:
	if (fits[ii] != *keyFits) 
	  rgb[ii] = 
	    fits[ii]->getImageToPhysical()*(*keyFits)->getPhysicalToImage();
	break;
      case AMPLIFIER:
	if (fits[ii] != *keyFits) 
	  rgb[ii] = fits[ii]->getImageToAmplifier() * 
	    (*keyFits)->getAmplifierToImage();
	break;
      case DETECTOR:
	if (fits[ii] != *keyFits) 
	  rgb[ii] = fits[ii]->getImageToDetector() * 
	    (*keyFits)->getDetectorToImage();
	break;
      default:
	rgbAlignWCS(ii,keyFitsID);
	break;
      }
    }

    if (DebugRGB) 
      cerr << "rgb[" << ii << "] " << rgb[ii] << endl;
  }
}   

void FrameRGB::rgbAlignWCS(int ii, int key)
{
  if (hasKeyFits() && (*keyFits)->hasWCS(rgbSystem)) {
    if (ii == key) {

      // orientation
      Matrix flip;
      Orientation orientation = NORMAL;
      
      // rotation
      double rotation = 0;

      // zoom
      double zoom = 1;

      // origin
      Vector origin = Vector();

      // rgb matrix
      Vector c = fits[ii]->getWCScrpix(rgbSystem) * imageToData;
      rgb[ii] =
	Translate(-c) *
	Rotate(rotation) *
	flip *
	Scale(zoom) *
	Translate(c) *
	Translate(origin);

      if (DebugRGB)
	cerr << "rgbAlignWCS " << ii << endl
	     << "  center " << c << endl
	     << "  orientation " << orientation << endl
	     << "  rotation " << radToDeg(rotation) << endl
	     << "  zoom " << zoom << endl
	     << "  origin " << origin << endl
	     << "  matrix " << rgb[ii] << endl;
    }
    else {
      // orientation
      Orientation o1 = fits[ii]->getWCSOrientation(rgbSystem,FK5);
      Orientation o2 = (*keyFits)->getWCSOrientation(rgbSystem,FK5);
      Orientation orientation;
      Matrix flip;
      switch (o1) {
      case NORMAL:
	{
	  switch (o2) {
	  case NORMAL:
	    orientation = NORMAL;
	    break;
	  case XX:
	    orientation = XX;
	    flip = FlipX();
	    break;
	  }
	}
	break;
      case XX:
	{
	  switch (o2) {
	  case NORMAL:
	    orientation = XX;
	    flip = FlipX();
	    break;
	  case XX:
	    orientation = NORMAL;
	    break;
	  }
	}
	break;
      }

      // rotation
      double r1 = fits[ii]->getWCSRotation(rgbSystem,FK5);
      double r2 = (*keyFits)->getWCSRotation(rgbSystem,FK5);
      double rotation;
      switch (o1) {
      case NORMAL:
	{
	  switch (o2) {
	  case NORMAL:
	    rotation = r2-r1;
	    break;
	  case XX:
	    rotation = r2+r1;
	    break;
	  }
	}
	break;
      case XX:
	{
	  switch (o2) {
	  case NORMAL:
	    rotation = -r2-r1;
	    break;
	  case XX:
	    rotation = -r2+r1;
	    break;
	  }
	}
	break;
      }

      // zoom
      double zoom = fabs(fits[ii]->getWCScdelt(rgbSystem)[0] / 
		       (*keyFits)->getWCScdelt(rgbSystem)[0]);

      // origin
      Vector aa = fits[ii]->getWCScrpix(rgbSystem);
      Vector bb = fits[ii]->pix2wcs(aa,rgbSystem,FK5);
      Vector org = (*keyFits)->wcs2pix(bb,rgbSystem,FK5) * imageToData;

      Vector c = fits[ii]->getWCScrpix(rgbSystem) * imageToData;

      Matrix mm =
	Translate(-c) *
	Rotate(rotation) *
	flip *
	Scale(zoom) *
	Translate(c);
      Vector origin = org*rgb[key] - c*mm;

      // rgb matrix
      rgb[ii] =
	Translate(-c) *
	Rotate(rotation) *
	flip *
	Scale(zoom) *
	Translate(c) *
	Translate(origin);

      if (DebugRGB)
	cerr << "rgbAlignWCS " << ii << endl
	     << "  center " << c << endl
	     << "  rotation " << radToDeg(rotation) << endl
	     << "  orientation " << orientation << endl
	     << "  zoom " << zoom << endl
	     << "  origin " << origin << endl
	     << "  matrix " << rgb[ii] << endl;
    }
  }
}

int FrameRGB::hasFitsCube()
{
  if (isRGBCube)
    return 0;
    
  if (fits[channel] && displayMode[channel]==SINGLE && fitsCount[channel]>1)
    return 1;
  else
    return 0;
}

// Fits Commands

void FrameRGB::getFitsSizeCmd()
{
  if (hasKeyFits())
    printVector(Vector((*keyFits)->width(),(*keyFits)->height()),DEFAULT);
}

void FrameRGB::getFitsSizeCmd(CoordSystem sys, SkyFormat format)
{
  if (hasKeyFits())
    printVector((*keyFits)->mapLenFromRef(imageSize(), sys, format), FIXED);
}

void FrameRGB::zoomToFitCmd(double s)
{
  if (hasKeyFits()) {
    centerImage();
    zoom = s*calcZoom(options->width,options->height,imageSize());
    update(MATRIX);
  }
}

void FrameRGB::sliceCmd(int s)
{

  for (int ii=0; ii<3; ii++) {
    cfits[ii] = fits[ii];
    if (fits[ii] && displayMode[ii]==SINGLE && (s>0) && (s<=fitsCount[ii])) {
      for (int i=1; i<s; i++)
	cfits[ii] = cfits[ii]->next();
    }
  }

  currentFits = cfits[channel];
  slice = s;

  // execute any update callbacks
  updateCBMarkers();

  update(BASE);
}

// Colormap Commands

#if __GNUC__ >= 3
void FrameRGB::getColormapCmd()
{
  ostringstream str;

  str << setiosflags(ios::fixed);
  for (int ii=0; ii<3; ii++)
    str << bias[ii] << ' ';
  for (int ii=0; ii<3; ii++)
    str << contrast[ii] << ' ';
  str << ends;

  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}
#else
void FrameRGB::getColormapCmd()
{
  char buf[128];
  ostrstream str(buf,128);

  str << setiosflags(ios::fixed);
  for (int ii=0; ii<3; ii++)
    str << bias[ii] << ' ';
  for (int ii=0; ii<3; ii++)
    str << contrast[ii] << ' ';
  str << ends;

  Tcl_AppendResult(interp, buf, NULL);
}
#endif

void FrameRGB::getTypeCmd()
{
  Tcl_AppendResult(interp, "rgb", NULL);
}

// Pan Zoom Rotate Orient Commands

void FrameRGB::getWCSZoomCmd(CoordSystem sys, Precision p)
{
  if (hasKeyFits() && (*keyFits)->hasWCS(wcsSystem)) {
    Vector v = (*keyFits)->getWCScdelt(sys);
    printDouble(v[0], p);
  }
  else
    Tcl_AppendResult(interp, "1", NULL);
}

void FrameRGB::wcsZoomCmd(CoordSystem sys, double a)
{
  if (hasKeyFits() && (*keyFits)->hasWCS(wcsSystem)) {
    Vector b = (*keyFits)->getWCScdelt(sys);
    zoom *= fabs(b[0]/a);
  }

  update(MATRIX);
}

// RGB Commands

void FrameRGB::getRGBChannelCmd()
{
  switch (channel) {
  case 0:
    Tcl_AppendResult(interp, "red", NULL);
    return;
  case 1:
    Tcl_AppendResult(interp, "green", NULL);
    return;
  case 2:
    Tcl_AppendResult(interp, "blue", NULL);
    return;
  }
}

void FrameRGB::setRGBChannelCmd(const char* c)
{
  if (!strncmp(c,"red",3))
    channel = 0;
  else if (!strncmp(c,"gre",3))
    channel = 1;
  else if (!strncmp(c,"blu",3))
    channel = 2;
  else
    channel = 0;

  channelFits = &fits[channel];
  currentFits = cfits[channel];
  currentCount = &fitsCount[channel];
  currentScale = &frScale[channel];
  currentMode = &displayMode[channel];
  currentMosaic = &mosaicType[channel];
  currentMosaicSystem = &mosaicSystem[channel];

  currentWCScdelt = &WCScdelt[channel];
  currentWCSrot = &WCSrot[channel];
  currentWCSorientation = &WCSorientation[channel];
  currentWCSbb = &WCSbb[channel];
  currentWCSmatrix = &WCSmatrix[channel];

  currentcontour = &contour[channel];

  // execute any update callbacks
  updateCBMarkers();

 // always update
  update(BASE);
}

void FrameRGB::getRGBViewCmd()
{
  for (int ii=0; ii<3; ii++)
    Tcl_AppendElement(interp, view[ii] ? "1" : "0");
}

void FrameRGB::setRGBViewCmd(int r, int g, int b)
{
  view[0] = r ? 1 : 0;
  view[1] = g ? 1 : 0;
  view[2] = b ? 1 : 0;

  update(BASE); // always update
}

void FrameRGB::getRGBSystemCmd()
{
  printCoordSystem(rgbSystem);
}

void FrameRGB::setRGBSystemCmd(CoordSystem sys)
{
  rgbSystem = sys;

  // save current matrix
  Matrix old[3];
  for (int ii=0; ii<3; ii++)
    old[ii] = rgb[ii];

  align();

  // fix any contours
  for(int ii=0; ii<3; ii++)
    if (contour[ii]) {
      Matrix mm = old[ii].invert() * rgb[ii];
      contour[ii]->updateCoords(mm);
    }

  update(MATRIX);
}

// Load Commands
// RGB Cube

void FrameRGB::loadRGBCubeAllocCmd(const char* fn)
{
  unloadAllFits();
  FitsImage* img = new FitsImageFitsAlloc(this, fn, FitsFile::NOFLUSH);
  loadRGBCube(ALLOC,fn,img);
}

void FrameRGB::loadRGBCubeAllocGZCmd(const char* fn)
{
  unloadAllFits();
  FitsImage* img = new FitsImageFitsAllocGZ(this, fn, FitsFile::NOFLUSH);
  loadRGBCube(ALLOCGZ,fn,img);
}

void FrameRGB::loadRGBCubeChannelCmd(const char* ch, const char* fn)
{
  unloadAllFits();
  FitsImage* img = new FitsImageFitsChannel(this, interp, ch, fn, 
					    FitsFile::NOFLUSH);
  loadRGBCube(CHANNEL,fn,img);
}

void FrameRGB::loadRGBCubeMMapCmd(const char* fn, LoadMethod lm)
{
  unloadAllFits();
  FitsImage* img = new FitsImageFitsMMap(this, fn);
  setScanModeIncr(lm);
  loadRGBCube(MMAP,fn,img);
}

void FrameRGB::loadRGBCubeSMMapCmd(const char* hdr, const char* fn, 
				   LoadMethod lm)
{
  unloadAllFits();
  FitsImage* img = new FitsImageFitsSMMap(this, hdr, fn);
  setScanModeIncr(lm);
  loadRGBCube(SMMAP,fn,img);
}

void FrameRGB::loadRGBCubeMMapIncrCmd(const char* fn, LoadMethod lm)
{
  unloadAllFits();
  FitsImage* img = new FitsImageFitsMMapIncr(this, fn);
  setScanModeIncr(lm);
  loadRGBCube(MMAPINCR,fn,img);
}

void FrameRGB::loadRGBCubeShareCmd(ShmType type, int id, const char* fn,
				   LoadMethod lm)
{
  unloadAllFits();
  FitsImage* img = new FitsImageFitsShare(this, type, id, fn);
  setScanModeIncr(lm);
  loadRGBCube(SHARE,fn,img);
}

void FrameRGB::loadRGBCubeSShareCmd(ShmType type, int hdr, int id,
				    const char* fn, LoadMethod lm)
{
  unloadAllFits();
  FitsImage* img = new FitsImageFitsSShare(this, type, hdr, id, fn);
  setScanModeIncr(lm);
  loadRGBCube(SSHARE,fn,img);
}

void FrameRGB::loadRGBCubeSocketCmd(int s, const char* fn)
{
  unloadAllFits();
  FitsImage* img = new FitsImageFitsSocket(this, s, fn, FitsFile::FLUSH);
  loadRGBCube(SOCKET,fn,img);
}

void FrameRGB::loadRGBCubeSocketGZCmd(int s, const char* fn)
{
  unloadAllFits();
  FitsImage* img = new FitsImageFitsSocketGZ(this, s, fn, FitsFile::FLUSH);
  loadRGBCube(SOCKETGZ,fn,img);
}

void FrameRGB::loadRGBCubeVarCmd(const char* ch, const char* fn,
				     LoadMethod lm)
{
  unloadAllFits();
  FitsImage* img = new FitsImageFitsVar(this, interp, ch, fn);
  setScanModeIncr(lm);
  loadRGBCube(VAR,fn,img);
}

// RGB Slave

void FrameRGB::loadRGBCubeSlaveCmd(const char* fn, FitsFile* f)
{
  unloadAllFits();
  FitsImage* img = new FitsImageSlave(this,fn,f);
  loadRGBCube(MMAPINCR,fn,img);
}

// RGB Image

void FrameRGB::loadRGBImageAllocCmd(const char* fn)
{
  unloadAllFits();
  FitsImage* img = new FitsImageMosaicAlloc(this, fn, FitsFile::NOFLUSH);
  loadRGBImage(ALLOC,fn,img);
}

void FrameRGB::loadRGBImageAllocGZCmd(const char* fn)
{
  unloadAllFits();
  FitsImage* img = new FitsImageMosaicAllocGZ(this, fn, FitsFile::NOFLUSH);
  loadRGBImage(ALLOCGZ,fn,img);
}

void FrameRGB::loadRGBImageChannelCmd(const char* ch, const char* fn)
{
  unloadAllFits();
  FitsImage* img = new FitsImageMosaicChannel(this, interp, ch, fn, 
					      FitsFile::NOFLUSH);
  loadRGBCube(CHANNEL,fn,img);
}

void FrameRGB::loadRGBImageMMapCmd(const char* fn, LoadMethod lm)
{
  unloadAllFits();
  FitsImage* img = new FitsImageMosaicMMap(this, fn);
  setScanModeIncr(lm);
  loadRGBImage(MMAP,fn,img);
}

void FrameRGB::loadRGBImageMMapIncrCmd(const char* fn, LoadMethod lm)
{
  unloadAllFits();
  FitsImage* img = new FitsImageMosaicMMapIncr(this, fn);
  setScanModeIncr(lm);
  loadRGBImage(MMAPINCR,fn,img);
}

void FrameRGB::loadRGBImageShareCmd(ShmType type, int id, const char* fn,
				    LoadMethod lm)
{
  unloadAllFits();
  FitsImage* img = new FitsImageMosaicShare(this, type, id, fn);
  setScanModeIncr(lm);
  loadRGBImage(SHARE,fn,img);
}

void FrameRGB::loadRGBImageSocketCmd(int s, const char* fn)
{
  unloadAllFits();
  FitsImage* img = new FitsImageMosaicSocket(this, s, fn, FitsFile::FLUSH);
  loadRGBImage(SOCKET,fn,img);
}

void FrameRGB::loadRGBImageSocketGZCmd(int s, const char* fn)
{
  unloadAllFits();
  FitsImage* img = new FitsImageMosaicSocketGZ(this, s, fn, FitsFile::FLUSH);
  loadRGBImage(SOCKETGZ,fn,img);
}

void FrameRGB::loadRGBImageVarCmd(const char* ch, const char* fn,
				      LoadMethod lm)
{
  unloadAllFits();
  FitsImage* img = new FitsImageMosaicVar(this, interp, ch, fn);
  setScanModeIncr(lm);
  loadRGBImage(VAR,fn,img);
}

// Array RGB Cube

void FrameRGB::loadArrRGBCubeAllocCmd(const char* fn)
{
  unloadAllFits();
  FitsImage* img = new FitsImageArrAlloc(this, fn, FitsFile::NOFLUSH);
  loadRGBCube(ALLOC,fn,img);
}

void FrameRGB::loadArrRGBCubeAllocGZCmd(const char* fn)
{
  unloadAllFits();
  FitsImage* img = new FitsImageArrAllocGZ(this, fn, FitsFile::NOFLUSH);
  loadRGBCube(ALLOCGZ,fn,img);
}

void FrameRGB::loadArrRGBCubeChannelCmd(const char* ch, const char* fn)
{
  unloadAllFits();
  FitsImage* img = new FitsImageArrChannel(this, interp, ch, fn, 
					   FitsFile::NOFLUSH);
  loadRGBCube(CHANNEL,fn,img);
}

void FrameRGB::loadArrRGBCubeMMapCmd(const char* fn)
{
  unloadAllFits();
  FitsImage* img = new FitsImageArrMMap(this, fn);
  loadRGBCube(MMAP,fn,img);
}

void FrameRGB::loadArrRGBCubeMMapIncrCmd(const char* fn)
{
  unloadAllFits();
  FitsImage* img = new FitsImageArrMMapIncr(this, fn);
  loadRGBCube(MMAPINCR,fn,img);
}

void FrameRGB::loadArrRGBCubeShareCmd(ShmType type, int id, const char* fn)
{
  unloadAllFits();
  FitsImage* img = new FitsImageArrShare(this, type, id, fn);
  loadRGBCube(SHARE,fn,img);
}

void FrameRGB::loadArrRGBCubeSocketCmd(int s, const char* fn)
{
  unloadAllFits();
  FitsImage* img = new FitsImageArrSocket(this, s, fn, FitsFile::FLUSH);
  loadRGBCube(SOCKET,fn,img);
}

void FrameRGB::loadArrRGBCubeSocketGZCmd(int s, const char* fn)
{
  unloadAllFits();
  FitsImage* img = new FitsImageArrSocketGZ(this, s, fn, FitsFile::FLUSH);
  loadRGBCube(SOCKETGZ,fn,img);
}

void FrameRGB::loadArrRGBCubeVarCmd(const char* ch, const char* fn)
{
  unloadAllFits();
  FitsImage* img = new FitsImageArrVar(this, interp, ch, fn);
  loadRGBCube(VAR,fn,img);
}

// loadRGB

void FrameRGB::loadRGBCube(MemType which, const char* fn, FitsImage* img)
{
  if (!img || !img->isValid() || !img->isImage() || (img->depth() != 3))
    goto error;

  fits[0] = img;

  switch (which) {
  case ALLOC:
    if (fits[0] && fits[0]->isValid())
      fits[1] = new FitsImageFitsNextAlloc(this, fn, fits[0]->getFitsFile());
    if (fits[1] && fits[1]->isValid())
      fits[2] = new FitsImageFitsNextAlloc(this, fn, fits[1]->getFitsFile());
    break;
  case ALLOCGZ:
    if (fits[0] && fits[0]->isValid())
      fits[1] = new FitsImageFitsNextAllocGZ(this, fn, fits[0]->getFitsFile());
    if (fits[1] && fits[1]->isValid())
      fits[2] = new FitsImageFitsNextAllocGZ(this, fn, fits[1]->getFitsFile());
    break;
  case CHANNEL:
    if (fits[0] && fits[0]->isValid())
      fits[1] = new FitsImageFitsNextChannel(this, fn, fits[0]->getFitsFile());
    if (fits[1] && fits[1]->isValid())
      fits[2] = new FitsImageFitsNextChannel(this, fn, fits[1]->getFitsFile());
    break;
  case MMAP:
    if (fits[0] && fits[0]->isValid())
      fits[1] = new FitsImageFitsNextMMap(this, fn, fits[0]->getFitsFile());
    if (fits[1] && fits[1]->isValid())
      fits[2] = new FitsImageFitsNextMMap(this, fn, fits[1]->getFitsFile());
    break;
  case SMMAP:
    if (fits[0] && fits[0]->isValid())
      fits[1] = new FitsImageFitsNextSMMap(this, fn, fits[0]->getFitsFile());
    if (fits[1] && fits[1]->isValid())
      fits[2] = new FitsImageFitsNextSMMap(this, fn, fits[1]->getFitsFile());
    break;
  case MMAPINCR:
    if (fits[0] && fits[0]->isValid())
      fits[1] = new FitsImageFitsNextMMapIncr(this, fn,fits[0]->getFitsFile());
    if (fits[1] && fits[1]->isValid())
      fits[2] = new FitsImageFitsNextMMapIncr(this, fn,fits[1]->getFitsFile());
    break;
  case SHARE:
    if (fits[0] && fits[0]->isValid())
      fits[1] = new FitsImageFitsNextShare(this, fn, fits[0]->getFitsFile());
    if (fits[1] && fits[1]->isValid())
      fits[2] = new FitsImageFitsNextShare(this, fn, fits[1]->getFitsFile());
    break;
  case SSHARE:
    if (fits[0] && fits[0]->isValid())
      fits[1] = new FitsImageFitsNextSShare(this, fn, fits[0]->getFitsFile());
    if (fits[1] && fits[1]->isValid())
      fits[2] = new FitsImageFitsNextSShare(this, fn, fits[1]->getFitsFile());
    break;
  case SOCKET:
    if (fits[0] && fits[0]->isValid())
      fits[1] = new FitsImageFitsNextSocket(this, fn, fits[0]->getFitsFile());
    if (fits[1] && fits[1]->isValid())
      fits[2] = new FitsImageFitsNextSocket(this, fn, fits[1]->getFitsFile());
    break;
  case SOCKETGZ:
    if (fits[0] && fits[0]->isValid())
      fits[1] = new FitsImageFitsNextSocketGZ(this, fn,fits[0]->getFitsFile());
    if (fits[1] && fits[1]->isValid())
      fits[2] = new FitsImageFitsNextSocketGZ(this, fn,fits[1]->getFitsFile());
    break;
  case VAR:
    if (fits[0] && fits[0]->isValid())
      fits[1] = new FitsImageFitsNextVar(this, fn, fits[0]->getFitsFile());
    if (fits[1] && fits[1]->isValid())
      fits[2] = new FitsImageFitsNextVar(this, fn, fits[1]->getFitsFile());
    break;
  }

  // is everything ok?
  if (fits[0] && fits[0]->isValid() && fits[0]->isImage() &&
      fits[1] && fits[1]->isValid() && fits[1]->isImage() &&
      fits[2] && fits[2]->isValid() && fits[2]->isImage()) {

    isRGBCube = 1;
    loadRGBFinish();
    return;
  }

 error:
  unloadAllFits();
  reset();
  resetClip(0);
  update(MATRIX);
    
  Tcl_AppendResult(interp, "Unable to load rgb cube file", NULL);
  result = TCL_ERROR;
  return;
}

void FrameRGB::loadRGBImage(MemType which, const char* fn, FitsImage* img)
{
  FitsImage* r = img;
  FitsImage* g = NULL;
  FitsImage* b = NULL;

  if (!img || !img->isValid() || !img->isImage())
    goto error;

  switch (which) {
  case ALLOC:
    if (r && r->isValid())
      g = new FitsImageMosaicNextAlloc(this, fn, r->getFitsFile(), 
				       FitsFile::NOFLUSH);
    if (g && g->isValid())
      b = new FitsImageMosaicNextAlloc(this, fn, g->getFitsFile(), 
				       FitsFile::NOFLUSH);
    break;
  case ALLOCGZ:
    if (r && r->isValid())
      g = new FitsImageMosaicNextAllocGZ(this,fn,r->getFitsFile(), 
					 FitsFile::NOFLUSH);
    if (g && g->isValid())
      b = new FitsImageMosaicNextAllocGZ(this,fn,g->getFitsFile(), 
					 FitsFile::NOFLUSH);
    break;
  case CHANNEL:
    if (r && r->isValid())
      g = new FitsImageMosaicNextChannel(this,fn,r->getFitsFile(), 
					 FitsFile::NOFLUSH);
    if (g && g->isValid())
      b = new FitsImageMosaicNextChannel(this,fn,g->getFitsFile(), 
					 FitsFile::NOFLUSH);
    break;
  case MMAP:
    if (r && r->isValid())
      g = new FitsImageMosaicNextMMap(this, fn, r->getFitsFile());
    if (g && g->isValid())
      b = new FitsImageMosaicNextMMap(this, fn, g->getFitsFile());
    break;
  case MMAPINCR:
    if (r && r->isValid())
      g = new FitsImageMosaicNextMMapIncr(this, fn, r->getFitsFile());
    if (g && g->isValid())
      b = new FitsImageMosaicNextMMapIncr(this, fn, g->getFitsFile());
    break;
  case SHARE:
    if (r && r->isValid())
      g = new FitsImageMosaicNextShare(this,fn, r->getFitsFile());
    if (g && g->isValid())
      b = new FitsImageMosaicNextShare(this,fn, g->getFitsFile());
    break;
  case SOCKET:
    if (r && r->isValid())
      g = new FitsImageMosaicNextSocket(this,fn,r->getFitsFile(), 
					FitsFile::FLUSH);
    if (g && g->isValid())
      b = new FitsImageMosaicNextSocket(this,fn,g->getFitsFile(), 
					FitsFile::FLUSH);
    break;
  case SOCKETGZ:
    if (r && r->isValid())
      g = new FitsImageMosaicNextSocketGZ(this,fn,r->getFitsFile(), 
					  FitsFile::FLUSH);
    if (g && g->isValid())
      b = new FitsImageMosaicNextSocketGZ(this,fn,g->getFitsFile(), 
					  FitsFile::FLUSH);
    break;
  case VAR:
    if (r && r->isValid())
      g = new FitsImageMosaicNextVar(this, fn, r->getFitsFile());
    if (g && g->isValid())
      b = new FitsImageMosaicNextVar(this, fn, g->getFitsFile());
    break;
  }

  // ok, figure out which is which channel
  fits[0] = fits[1] = fits[2] = NULL;

  if (r->findKeyword("EXTNAME")) {
    char* c = r->getKeyword("EXTNAME");
    if (!strncmp(c,"RED",3))
      fits[0] = r;
    else if (!strncmp(c,"GREEN",3))
      fits[1] = r;
    else if (!strncmp(c,"BLUE",3))
      fits[2] = r;

    if (c)
      delete [] c;
  }
  else
    fits[0] = r;

  if (g->findKeyword("EXTNAME")) {
    char* c = g->getKeyword("EXTNAME");
    if (!strncmp(c,"RED",3))
      fits[0] = g;
    else if (!strncmp(c,"GREEN",3))
      fits[1] = g;
    else if (!strncmp(c,"BLUE",3))
      fits[2] = g;

    if (c)
      delete [] c;
  }
  else
    fits[1] = g;

  if (b->findKeyword("EXTNAME")) {
    char* c = b->getKeyword("EXTNAME");
    if (!strncmp(c,"RED",3))
      fits[0] = b;
    else if (!strncmp(c,"GREEN",3))
      fits[1] = b;
    else if (!strncmp(c,"BLUE",3))
      fits[2] = b;

    if (c)
      delete [] c;
  }
  else
    fits[2] = b;

  // is everything ok?
  if (fits[0] && fits[0]->isValid() && fits[0]->isImage() &&
      fits[1] && fits[1]->isValid() && fits[1]->isImage() &&
      fits[2] && fits[2]->isValid() && fits[2]->isImage()) {

    loadRGBFinish();
    return;
  }

 error:
  unloadAllFits();
  reset();
  resetClip(0);
  update(MATRIX);
    
  Tcl_AppendResult(interp, "Unable to load rgb image file", NULL);
  result = TCL_ERROR;
  return;
}

void FrameRGB::loadRGBFinish()
{
  for (int ii=0; ii<3; ii++) {
    cfits[ii] = fits[ii];
    fitsCount[ii] = 1;
    displayMode[ii] = SINGLE;
    mosaicType[ii] = NOMOSAIC;
    mosaicSystem[ii] = WCS;
  }

  // update clip values
  channel = 0;
  slice = 1;
  for (int ii=0; ii<3; ii++) {
    currentFits = fits[ii];
    currentScale = &frScale[ii];
    channelFits = &fits[ii];
    currentCount = &fitsCount[ii];
    currentMode = &displayMode[ii];
    currentMosaic = &mosaicType[ii];
    currentMosaicSystem = &mosaicSystem[ii];

    if (!currentFits->hasDATAMIN()&&currentScale->mmMode()==FrScale::DATAMIN)
      currentScale->setMMMode(FrScale::SCAN);
    if (!currentFits->hasIRAFMIN()&&currentScale->mmMode()==FrScale::IRAFMIN)
      currentScale->setMMMode(FrScale::SCAN);
    
    resetClip(0);
  }

  channel = 0;
  slice = 1;
  currentFits = fits[channel];
  currentScale = &frScale[channel];

  channelFits = &fits[channel];
  currentCount = &fitsCount[channel];
  currentMode = &displayMode[channel];
  currentMosaic = &mosaicType[channel];
  currentMosaicSystem = &mosaicSystem[channel];

  currentWCScdelt = &WCScdelt[channel];
  currentWCSrot = &WCSrot[channel];
  currentWCSorientation = &WCSorientation[channel];
  currentWCSbb = &WCSbb[channel];
  currentWCSmatrix = &WCSmatrix[channel];

  currentcontour = &contour[channel];

  resetValues();

  update(MATRIX);
}

void FrameRGB::getInfoSingle(const Vector& v, InternalSystem ref,
			     Matrix& (FitsImage::*matrixToData)(),
			     Matrix& (FitsImage::*matrixToImage)(),
			     SkyFrame sky, SkyFormat format, char* var)
{
  FrameBase::getInfoSingle(v,ref,matrixToData,matrixToImage,sky,format,var);

  // set RGB values
  char* array[3] = {"value,red","value,green","value,blue"};
  for (int ii=0; ii<3; ii++) {
    if (cfits[ii]) {
      Vector img = v * (cfits[ii]->*matrixToData)();
      int* params = cfits[ii]->getDataParams(currentScale->scanMode());
      int& xmin = params[1];
      int& xmax = params[2];
      int& ymin = params[3];
      int& ymax = params[4];
	
      if (img[0]>=xmin && img[0]<xmax && img[1]>=ymin && img[1]<ymax)
	Tcl_SetVar2(interp,var,array[ii],(char*)cfits[ii]->getValue(img),0);
    }
    else
      Tcl_SetVar2(interp,var,array[ii],"",0);
  }
}

void FrameRGB::getInfoMosaic(const Vector& v, InternalSystem ref,
			     Matrix& (FitsImage::*matrixToData)(),
			     Matrix& (FitsImage::*matrixToImage)(),
			     SkyFrame sky, SkyFormat format, char* var)
{
  FrameBase::getInfoMosaic(v,ref,matrixToData,matrixToImage,sky,format,var);

  // set RGB values

  char* array[3] = {"value,red","value,green","value,blue"};
  for (int ii=0; ii<3; ii++) {
    FitsImage* ptr = fits[ii];
    while (ptr) {
      Vector img = v * (ptr->*matrixToData)();
      int* params = ptr->getDataParams(currentScale->scanMode());
      int& xmin = params[1];
      int& xmax = params[2];
      int& ymin = params[3];
      int& ymax = params[4];

      if (img[0]>=xmin && img[0]<xmax && img[1]>=ymin && img[1]<ymax) {
	Tcl_SetVar2(interp,var,array[ii],(char*)ptr->getValue(img),0);
	break;
      }
      ptr = ptr->next();
    }
  }
}

