#  Copyright (C) 1999-2005
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

# catreg -- convert catalog table into region string
proc CATReg {varname interactive} {
    upvar #0 $varname var
    global $varname
    global $var(tbldb)
    global $var(symdb)

    # How to process each field of a filter
    # -------------------------------------
    # condition: expr
    # shape: constant
    # color: constant
    # text: subst
    # size1: expr
    # size2: expr
    # angle: expr

    # constants
    set shapes {circle ellipse box vector text point \
		    {circle point} {box point} {diamond point} \
		    {cross point} {x point} {arrow point} {boxcircle point}}
    set colors {green red blue yellow cyan magenta white black}

    # init last error
    set errors {}

    # find RA and DEC
    set racol [starbase_colnum $var(tbldb) "_RAJ2000"]
    set deccol [starbase_colnum $var(tbldb) "_DEJ2000"]

    # do we have formats for RA and DEC?
    if [catch {starbase_hdrget $var(tbldb) UFMT} ff] {
	 set ff {}
     }
    set raformat  [lindex $ff 0]
    set decformat [lindex $ff 1]

    # init result
    set result {}

    # process prologue
    append result "# Region file format: DS9 version 3.0\n"

    # for each row in the catalog table ...
    for {set CRROW 1} {$CRROW <= [starbase_nrows $var(tbldb)]} {incr CRROW} {

	# define each colunm variable
	foreach col [starbase_columns $var(tbldb)] {
	    set val [starbase_get $var(tbldb) $CRROW \
			 [starbase_colnum $var(tbldb) $col]]
	    # here's a tough one-- what to do if the col is blank
	    # for now, just set it to '0'
	    if {[string trim "$val"] == {}} {
		set val 0
	    }
	    eval "set \{$col\} \{$val\}"
	}

	# look through each filter
	for {set jj 1} {$jj <= [starbase_nrows $var(symdb)]} {incr jj} {

	    # eval condition
	    set cond [starbase_get $var(symdb) $jj \
			  [starbase_colnum $var(symdb) condition]]

	    if {$cond != {}} {
		# subst any column vars
		if [catch {subst $cond} cc] {
		    set errors "Unable to evaluate condition $cc"
		    break
		}
		# evaluate filter
		if [catch {expr $cc} found] {
		    set errors "Unable to evaluate condition $cc"
		    break
		}
	    } else {
		set found 1
	    }

	    # if not true, goto the next filter
	    if {!$found} {
		continue
	    }

	    # SYSTEM
	    if [catch {starbase_hdrget $var(tbldb) equinox} CRSYSTEM] {
		set CRSYSTEM fk5
	    }
	    switch -- $CRSYSTEM {
		1950.0 -
		B1950 {set CRSYSTEM fk4}
		2000.0 -
		J2000 -
		{} {set CRSYSTEM fk5}
	    }

	    # SHAPE
	    set CRSHAPE [starbase_get $var(symdb) $jj \
			   [starbase_colnum $var(symdb) shape]]
	    if {$CRSHAPE == {}} {
		set CRSHAPE circle
	    }
	    if {[string first $CRSHAPE $shapes] == -1} {
		set errors "Unknown shape $CRSHAPE"
		break
	    }

	    # RA
	    set CRRA [starbase_get $var(tbldb) $CRROW $racol]
	    if {$raformat != {}} {
		set CRRA [uformat $raformat d $CRRA]
	    }

	    # DEC
	    set CRDEC [starbase_get $var(tbldb) $CRROW $deccol]
	    if {$decformat != {}} {
		set CRDEC [uformat $decformat d $CRDEC]
	    }

	    # SIZE
	    set units [starbase_get $var(symdb) $jj \
			    [starbase_colnum $var(symdb) units]]
	    switch -- $units {
		image {set units i}
		physical {set units p}
		degrees {set units d}
		arcmin {set units {'}}
		arcsec {set units {"}}
		{} {set units p}
	    }

	    switch -- $CRSHAPE {
		text -
		point -
		{circle point} -
		{box point} -
		{diamond point} -
		{cross point} -
		{x point} -
		{arrow point} -
		{boxcircle point} {set CRSIZE {}}

		circle {
		    set size [starbase_get $var(symdb) $jj \
				    [starbase_colnum $var(symdb) size]]
		    if {$size != {}} {
			if [catch {expr $size} ss] {
			    set errors "Unable to evaluate size $size"
			    break
			} else {
			    set size $ss
			}
		    } else {
			set size 5
		    }
		    set CRSIZE "${size}${units}"
		}

		vector {
		    set size [starbase_get $var(symdb) $jj \
				    [starbase_colnum $var(symdb) size]]
		    set angle [starbase_get $var(symdb) $jj \
				     [starbase_colnum $var(symdb) angle]]
		    if {$size != {}} {
			if [catch {expr $size} ss] {
			    set errors "Unable to evaluate size $size"
			    break
			} else {
			    set size $ss
			}
		    } else {
			set size 5
		    }
		    if {$angle != {}} {
			if [catch {expr $angle} aa] {
			    set errors "Unable to evaluate angle $angle"
			    break
			} else {
			    set angle $aa
			}
		    } else {
			set angle 0
		    }

		    set CRSIZE "${size}${units} ${angle}"
		}

		ellipse -
		box {
		    set size [starbase_get $var(symdb) $jj \
				    [starbase_colnum $var(symdb) size]]
		    set size2 [starbase_get $var(symdb) $jj \
				     [starbase_colnum $var(symdb) size2]]
		    set angle [starbase_get $var(symdb) $jj \
				     [starbase_colnum $var(symdb) angle]]

		    if {$size != {}} {
			if [catch {expr $size} ss] {
			    set errors "Unable to evaluate size $size"
			    break
			} else {
			    set size $ss
			}
		    } else {
			set size 5
		    }
		    if {$size2 != {}} {
			if [catch {expr $size2} ss] {
			    set errors "Unable to evaluate size $size2"
			    break
			} else {
			    set size2 $ss
			}
		    } else {
			set size2 5
		    }
		    if {$angle != {}} {
			if [catch {expr $angle} aa] {
			    set errors "Unable to evaluate angle $angle"
			    break
			} else {
			    set angle $aa
			}
		    } else {
			set angle 0
		    }

		    set CRSIZE "${size}${units} ${size2}${units} ${angle}"
		}
	    }

	    # COLOR
	    set CRCOLOR [starbase_get $var(symdb) $jj \
			   [starbase_colnum $var(symdb) color]]
	    if {$CRCOLOR == {}} {
		set CRCOLOR green
	    }
	    if {[string first $CRCOLOR $colors] == -1} {
		set errors "Unknown color $CRCOLOR"
		break
	    }

	    # TEXT
	    set CRTEXT [starbase_get $var(symdb) $jj \
			  [starbase_colnum $var(symdb) text]]
	    if {$CRTEXT != {}} {
		if [catch {subst $CRTEXT} tt] {
		    set errors "Unable to evaluate text $CRTEXT"
		    break
		} else {
		    set CRTEXT $tt
		}
	    }
	    if {$CRSHAPE == {text} && $CRTEXT == {}} {
		set CRTEXT "$CRROW"
	    }

	    # final substitution and append result
	    # init result for substitutions
	    if {$interactive} {
		set template "\${CRSYSTEM};\${CRSHAPE}(\${CRRA} \${CRDEC} \${CRSIZE}) # color=\${CRCOLOR} text=\{\${CRTEXT}\} tag=$varname tag={${varname}.\${CRROW}} callback=highlite CATHighliteCB {${varname}.\${CRROW}} callback=unhighlite CATUnhighliteCB {${varname}.\${CRROW}}\n"
	    } else {
		set template "\${CRSYSTEM};\${CRSHAPE}(\${CRRA} \${CRDEC} \${CRSIZE}) # color=\${CRCOLOR} text=\{\${CRTEXT}\} tag=$varname\n"
	    }
	    append result [subst $template]
	}
    } 

    # any errors?
    if {$errors != {}} {
	Error "$errors"
	return {}
    }

    return $result
}
