// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

%{
#define YYPARSE_PARAM fr
#define YYDEBUG 1

#define FR ((FrameBase*)fr)
#define FITSPTR (FR->findFits(1))

#include <math.h>
#include <string.h>
#include <iostream.h>

#include "framebase.h"
#include "fitsimage.h"
#include "util.h"
#include "vector.h"
#include "list.h"
#include "basemarker.h"
#include "coord.h"

extern int ciaolex(void);
extern void ciaoerror(const char*);

int ciaosign;
int ciaosign2;

static char *color = "green";
static char *font = "helvetica 10 normal";
static char *text = "";
static unsigned short props;

static List<Vertex> polylist;
static List<Tag> taglist;
static List<CallBack> cblist;

static void setProps(unsigned short* props, unsigned short prop, int value);

%}

%union {
  double real;
  int integer;
  char str[2048];
  double vector[3];
}

%type <real> numeric

%type <real> angle
%type <real> value
%type <vector> vvalue
%type <real> sexagesimal
%type <vector> coord

%token <integer> INT
%token <real> REAL

%token <integer> HOUR
%token <integer> MINUTE
%token <real> SECOND
%token <real> ARCMINUTE
%token <real> ARCSECOND

%token <str> SEXSTR

%token EOF_

%token ANNULUS_
%token BOX_
%token CIRCLE_
%token DEBUG_
%token ELLIPSE_
%token OFF_
%token ON_
%token PIE_
%token POINT_
%token POLYGON_
%token ROTBOX_
%token VERSION_

%%

commands: commands command terminator
	| command terminator
	;

command : /* empty */
	| DEBUG_ debug
	| VERSION_ {cerr << "CIAO Regions File 1.0" << endl;}
	| init include shape
	;

terminator: '\n'
	| ';'
	| EOF_ {YYACCEPT;}
	;

numeric	: REAL {$$=$1;}
	| INT {$$=$1;}
	;

debug	: ON_ {yydebug=1;}
	| OFF_ {yydebug=0;}
	;

sp	: /* empty */
	| ','
	;

bp	: /* empty */
	| '('
	;

ep	: /* emtpy */
	| ')'
	;

init:	{
	  props =
	    Marker::SELECT | Marker::EDIT | Marker::MOVE |
	    Marker::ROTATE | Marker::DELETE | Marker::HIGHLITE |
	    Marker::INCLUDE | Marker::SOURCE;
	}
	;

angle	: numeric {$$ = degToRad($1);}
	;

value	: numeric {$$ = FITSPTR->mapLenToRef($1, PHYSICAL);}
	| ARCMINUTE {$$ = FITSPTR->mapLenToRef($1, WCS, ARCMIN);}
	| ARCSECOND {$$ = FITSPTR->mapLenToRef($1, WCS, ARCSEC);}
	;

vvalue	: numeric sp numeric 
	{
	  Vector r = FITSPTR->mapLenToRef(Vector($1,$3), PHYSICAL);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| ARCMINUTE sp ARCMINUTE 
	{
	  Vector r = FITSPTR->mapLenToRef(Vector($1,$3), WCS, ARCMIN);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| ARCSECOND sp ARCSECOND 
	{
	  Vector r = FITSPTR->mapLenToRef(Vector($1,$3), WCS, ARCSEC);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	;

sexagesimal: SEXSTR {$$ = parseDMS($1);}
	;

coord	: sexagesimal sp sexagesimal
	{
	  Vector r = FITSPTR->mapToRef(Vector($1*360./24.,$3),WCS,FK5);

	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| HOUR MINUTE SECOND {ciaosign2 = ciaosign;} sp INT ARCMINUTE ARCSECOND
        {
	  Vector r = FITSPTR->mapToRef(
	    Vector(hmsToDegree(ciaosign2,$1,$2,$3), 
	    dmsToDegree(ciaosign,$6,$7,$8)),
	    WCS, FK5);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| INT ARCMINUTE ARCSECOND {ciaosign2 = ciaosign;} sp 
	  INT ARCMINUTE ARCSECOND
        {
	  Vector r = FITSPTR->mapToRef(
	    Vector(dmsToDegree(ciaosign2,$1,$2,$3),
	    dmsToDegree(ciaosign,$6,$7,$8)),
	    WCS, FK5);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| numeric sp numeric 
	{
	  Vector r = FITSPTR->mapToRef(Vector($1,$3), PHYSICAL);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	;

include	: /* empty */
	| '+' {setProps(&props, Marker::INCLUDE, 1);}
	| '-' {setProps(&props, Marker::INCLUDE, 0);}
	;

shape	: CIRCLE_ bp coord sp value ep
	  {FR->createCircleCmd(Vector($3),
	    $5,
	    color,1,font,text,props,NULL,taglist,cblist);}

	| ELLIPSE_ bp coord sp vvalue sp angle ep
	  {FR->createEllipseCmd(Vector($3),
	    Vector($5),$7,
	    color,1,font,text,props,NULL,taglist,cblist);}

	| ANNULUS_ bp coord sp value sp value ep
	  {FR->createAnnulusCmd(Vector($3),
	    $5,$7,1,
	    color,1,font,text,props,NULL,taglist,cblist);}

	| PIE_ bp coord sp value sp value sp angle sp angle ep
	  {FR->createCpandaCmd(Vector($3),
	    $9,$11,1,
	    $5,$7,1,
	    color,1,font,text,props,NULL,taglist,cblist);}

	| BOX_ bp coord sp vvalue ep
	  {FR->createBoxCmd(Vector($3),
	    Vector($5),
	    0,
	    color,1,font,text,props,NULL,taglist,cblist);}

	| ROTBOX_ bp coord sp vvalue sp angle ep
	  {FR->createBoxCmd(Vector($3),
	     Vector($5),
	     $7,
	     color,1,font,text,props,NULL,taglist,cblist);}

	| POINT_ bp coord ep 
	  {FR->createBoxCirclePointCmd(Vector($3),
	    color,1,font,text,props,NULL,taglist,cblist);}

	| POLYGON_ {polylist.deleteAll();} bp polyNodes ep
	  {FR->createPolygonCmd(polylist,
	    color,1,font,text,props,NULL,taglist,cblist);} 
	;

polyNodes : polyNodes sp polyNode
	| polyNode
	;

polyNode : coord {polylist.append(new Vertex($1));}
	;

%%

static void setProps(unsigned short* props, unsigned short prop, int value)
{
  if (value)
    *props |= prop;
  else
    *props &= ~prop;
}
