// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "frametrue24.h"
#include "colorscaletrue24.h"
#include "colorscaletrue32.h"
#include "fitsimage.h"
#include "ps.h"
#include "util.h"

// Tk Canvas Widget Function Declarations

int FrameTrueColor24CreateProc(Tcl_Interp*, Tk_Canvas, Tk_Item*, int, Tcl_Obj *const []);

// FrameTrueColor24 Specs

static Tk_CustomOption tagsOption = {
  Tk_CanvasTagsParseProc, Tk_CanvasTagsPrintProc, NULL
};

static Tk_ConfigSpec frameTrueColor24Specs[] = {

  {TK_CONFIG_STRING, "-command", NULL, NULL, "frame",
   Tk_Offset(WidgetOptions, cmdName), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-x", NULL, NULL, "1",
   Tk_Offset(WidgetOptions, x), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-y", NULL, NULL, "1",
   Tk_Offset(WidgetOptions, y), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-width", NULL, NULL, "512",
   Tk_Offset(WidgetOptions, width), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-height", NULL, NULL, "512",
   Tk_Offset(WidgetOptions, height), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_ANCHOR, "-anchor", NULL, NULL, "nw",
   Tk_Offset(WidgetOptions, anchor), 0, NULL},
  {TK_CONFIG_CUSTOM, "-tags", NULL, NULL, NULL,
   0, TK_CONFIG_NULL_OK, &tagsOption},

  {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0, NULL},
};

// Tk Static Structure

static Tk_ItemType frameTrueColor24Type = {
  "frametruecolor24",           // name
  sizeof(WidgetOptions),        // item size
  FrameTrueColor24CreateProc,   // configProc
  frameTrueColor24Specs,        // configSpecs
  WidgetConfigProc,             // configProc
  WidgetCoordProc,              // coordProc
  WidgetDeleteProc,             // deleteProc
  WidgetDisplayProc,            // displayProc
  0,                            // alwaysRedraw
  WidgetPointProc,              // pointProc
  WidgetAreaProc,               // areaProc
  WidgetPostscriptProc,         // postscriptProc
  WidgetScaleProc,              // scaleProc
  WidgetTranslateProc,          // translateProc
  (Tk_ItemIndexProc*)NULL,      // indexProc
  WidgetICursorProc,            // icursorProc
  (Tk_ItemSelectionProc*)NULL,  // selectionProc
  (Tk_ItemInsertProc*)NULL,     // insertProc
  (Tk_ItemDCharsProc*)NULL,     // dCharsProc
  (Tk_ItemType*)NULL            // nextPtr
};

// Non-Member Functions

int FrameTrueColor24_Init(Tcl_Interp* interp)
{
  Tk_CreateItemType(&frameTrueColor24Type);
  return TCL_OK;
}

int FrameTrueColor24CreateProc(Tcl_Interp* interp, Tk_Canvas canvas, 
			       Tk_Item* item, int argc, Tcl_Obj *const argv[])
{
  FrameTrueColor24* frame = new FrameTrueColor24(interp, canvas, item);

  // and set default configuration

  if (frame->configure(argc, (const char**)argv, 0) != TCL_OK) {
    delete frame;
    Tcl_AppendResult(interp, " error occured while creating frame.", NULL);
    return TCL_ERROR;
  }

  return TCL_OK;
}

// FrameTrueColor24 Member Functions

FrameTrueColor24::FrameTrueColor24(Tcl_Interp* i, Tk_Canvas c, Tk_Item* item)
  : FrameTrueColor(i, c, item)
{
  configSpecs = frameTrueColor24Specs;  // frame configure options
}

FrameTrueColor24::~FrameTrueColor24()
{
  // we must do this at this level, because updateColorScale is called
  unloadAllFits();
}

void FrameTrueColor24::encodeTrueColor(XColor* src, char* dest)
{
  if (!baseXImage)
    return;

  switch (baseXImage->bits_per_pixel) {
  case 24:
    encodeTrueColor24(src,dest);
    break;
  case 32:
    encodeTrueColor32(src,dest);
    break;
  }
}

void FrameTrueColor24::encodeTrueColor24(XColor* src, char* dest)
{
  int msb = baseXImage->byte_order;
#ifndef _WIN32
  int rs = decodeMask((unsigned long)visual->red_mask);
  int gs = decodeMask((unsigned long)visual->green_mask);
  int bs = decodeMask((unsigned long)visual->blue_mask);
#else
  int rs = decodeMask((unsigned long)0x00FF0000);
  int gs = decodeMask((unsigned long)0x0000FF00);
  int bs = decodeMask((unsigned long)0x000000FF);
#endif
      
  // we need to check to byteswap when we have cross platforms

  unsigned int r = (unsigned char)src->red;
  unsigned int g = (unsigned char)src->green;
  unsigned int b = (unsigned char)src->blue;
  unsigned int a = 0;
  a |= r << rs;
  a |= g << gs;
  a |= b << bs;

  if ((!msb && lsb()) || (msb && !lsb()))
    memcpy(dest, &a, 3);
  else {
    unsigned char* rr = (unsigned char*)(&a);
    *(dest) = *(rr+3);
    *(dest+1) = *(rr+2);
    *(dest+2) = *(rr+1);
  }
}

void FrameTrueColor24::encodeTrueColor32(XColor* src, char* dest)
{
  int msb = baseXImage->byte_order;
#ifndef _WIN32
  int rs = decodeMask((unsigned long)visual->red_mask);
  int gs = decodeMask((unsigned long)visual->green_mask);
  int bs = decodeMask((unsigned long)visual->blue_mask);
#else
  int rs = decodeMask((unsigned long)0x00FF0000);
  int gs = decodeMask((unsigned long)0x0000FF00);
  int bs = decodeMask((unsigned long)0x000000FF);
#endif
      
  unsigned int r = (unsigned char)src->red;
  unsigned int g = (unsigned char)src->green;
  unsigned int b = (unsigned char)src->blue;
  unsigned int a = 0;
  a |= r << rs;
  a |= g << gs;
  a |= b << bs;

  if ((!msb && lsb()) || (msb && !lsb()))
    memcpy(dest, &a, 4);
  else {
    unsigned char* rr = (unsigned char*)(&a);
    *(dest) = *(rr+3);
    *(dest+1) = *(rr+2);
    *(dest+2) = *(rr+1);
    *(dest+3) = *(rr);
  }
}

void FrameTrueColor24::updateColorScale()
{
  // we need colors before we can construct a scale

  if (!indexCells || !colorCells)
    return;

  // since we need to know about bytes per pixel and byte order, 
  // we may need to wait til we have an XImage available and try again

  if (!baseXImage)
    return;

  if (colorScale)
    delete colorScale;

  // determine if we have 3 bytes or 4 bytes per pixel

  switch (baseXImage->bits_per_pixel) {
  case 24:
    updateColorScale24();
    encodeTrueColor24(bgColor,bgTrueColor_);
    encodeTrueColor24(nanColor,nanTrueColor_);
    break;
  case 32:
    updateColorScale32();
    encodeTrueColor32(bgColor,bgTrueColor_);
    encodeTrueColor32(nanColor,nanTrueColor_);
    break;
  }
}

void FrameTrueColor24::updateColorScale24()
{
  switch (currentScale->colorScaleType()) {
  case FrScale::LINEARSCALE:
    colorScale = 
      new LinearScaleTrueColor24(colorCount, indexCells, colorCells, 
				 colorCount, visual, baseXImage->byte_order);
    break;
  case FrScale::LOGSCALE:
    colorScale =
      new LogScaleTrueColor24(SCALESIZE, indexCells, colorCells, 
			      colorCount, visual, baseXImage->byte_order);
    break;
  case FrScale::SQUAREDSCALE:
    colorScale =
      new SquaredScaleTrueColor24(SCALESIZE, indexCells, colorCells, 
				  colorCount, visual, baseXImage->byte_order);
    break;
  case FrScale::SQRTSCALE:
    colorScale =
      new SqrtScaleTrueColor24(SCALESIZE, indexCells, colorCells, colorCount, 
			       visual, baseXImage->byte_order);
    break;
  case FrScale::IISSCALE:
    colorScale = 
      new IISScaleTrueColor24(indexCells, colorCells, colorCount, 
			      visual, baseXImage->byte_order);
    break;
  case FrScale::HISTEQUSCALE:
    calcHistEqu();
    colorScale = 
      new HistEquScaleTrueColor24(SCALESIZE, indexCells, colorCells, 
				  colorCount, currentScale->histequ(),
				  HISTEQUSIZE,
				  visual, baseXImage->byte_order);
    break;
  }
}

void FrameTrueColor24::updateColorScale32()
{
  switch (currentScale->colorScaleType()) {
  case FrScale::LINEARSCALE:
    colorScale = 
      new LinearScaleTrueColor32(colorCount, indexCells, colorCells, 
				 colorCount, visual, baseXImage->byte_order);
    break;
  case FrScale::LOGSCALE:
    colorScale =
      new LogScaleTrueColor32(SCALESIZE, indexCells, colorCells, 
			      colorCount, visual, baseXImage->byte_order);
    break;
  case FrScale::SQUAREDSCALE:
    colorScale =
      new SquaredScaleTrueColor32(SCALESIZE, indexCells, colorCells,
				  colorCount, visual, baseXImage->byte_order);
    break;
  case FrScale::SQRTSCALE:
    colorScale =
      new SqrtScaleTrueColor32(SCALESIZE, indexCells, colorCells, colorCount, 
			       visual, baseXImage->byte_order);
    break;
  case FrScale::IISSCALE:
    colorScale = 
      new IISScaleTrueColor32(indexCells, colorCells, colorCount, 
			      visual, baseXImage->byte_order);
    break;
  case FrScale::HISTEQUSCALE:
    calcHistEqu();
    colorScale = 
      new HistEquScaleTrueColor32(SCALESIZE, indexCells, colorCells, 
				  colorCount, currentScale->histequ(),
				  HISTEQUSIZE,
				  visual, baseXImage->byte_order);
    break;
  }
}


