
// Copyright (c) 1996-2000 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu
//          Radharamanan Radhakrishnan  ramanan@ece.uc.edu
//	    Magnus Danielson	cfmd@swipnet.se

//---------------------------------------------------------------------------

#include "IIRScram_SubprogramDeclaration.hh"
#include "IIR_InterfaceDeclaration.hh"
#include "IIR_TypeDefinition.hh"
#include "IIR_SubtypeDeclaration.hh"
#include "IIR_ReturnStatement.hh"
#include "IIR_StringLiteral.hh"
#include "IIR_AttributeSpecificationList.hh"
#include "IIR_AttributeSpecification.hh"
#include "IIR_InterfaceList.hh"
#include "IIR_InterfaceDeclaration.hh"
#include "IIR_SequentialStatementList.hh"
#include "IIR_AliasDeclaration.hh"
#include "IIR_ProcessStatement.hh"
#include "IIR_Label.hh"
#include "IIR_FunctionDeclaration.hh"
#include "set.hh"
#include "resolution_func.hh"
#include "symbol_table.hh"
#include "savant.hh"
#include "published_file.hh"

IIRScram_SubprogramDeclaration::IIRScram_SubprogramDeclaration(){
  _my_contains_body = FALSE;
}

IIRScram_SubprogramDeclaration::~IIRScram_SubprogramDeclaration() {}

void 
IIRScram_SubprogramDeclaration::_publish_cc_include( published_file &_cc_out ){
  // First the default includes.
  if(_contains_body() == TRUE){

    // The types used in the interface declarations.
    IIR_Declaration *current_decl = interface_declarations.first();
    while (current_decl != NULL) {
      switch(current_decl->get_kind()) {
      case IIR_SIGNAL_INTERFACE_DECLARATION:
      case IIR_VARIABLE_INTERFACE_DECLARATION:
      case IIR_CONSTANT_INTERFACE_DECLARATION:
	current_decl->_publish_cc_include( _cc_out );
	break;
      case IIR_FILE_INTERFACE_DECLARATION:
	// Do Nothing!!
	break;
      default:
	cerr << "IIRScram_SubprogramDeclaration.cc_publish_cc_headers( published_file &_cc_out ):"
	     << " Unknown declaration type: " 
	     << current_decl->get_kind_text() << endl;
	break;
      }
      current_decl = interface_declarations.successor((IIR_InterfaceDeclaration *)current_decl);
    }
    
    // The types in the subprogram declarations.
    current_decl = subprogram_declarations.first();
    while (current_decl != NULL) {
      switch(current_decl->get_kind()) {
      case IIR_TYPE_DECLARATION:
      case IIR_SUBTYPE_DECLARATION:
	current_decl->_publish_cc_include( _cc_out );
 	break;
      default:
	break;
      }
      current_decl = subprogram_declarations.successor(current_decl);
    }
  }
}


void 
IIRScram_SubprogramDeclaration::_publish_cc_declarations( published_file &_cc_out ) {
  IIR_Declaration *decl = subprogram_declarations.first();
  
  SCRAM_CC_REF( _cc_out, "IIRScram_SubprogramDeclaration::_publish_cc_declarations" );

  _cc_out.insert_comment("initialize necessary type_info structures...");
  subprogram_declarations._publish_cc_type_info( _cc_out );
  while (decl != NULL) {
    // Some publishing changes the output file, so to return to this
    // file, we have to save it and restore it here.
    SCRAM_CC_REF( _cc_out, "IIRScram_SubprogramDeclaration::_publish_cc_declarations" );
    switch(decl->get_kind()) {
    case IIR_TYPE_DECLARATION:
    case IIR_SUBTYPE_DECLARATION:
      // These have been published earlier.  Nothing to be done here.
      break;
    case IIR_SIGNAL_DECLARATION:
    case IIR_VARIABLE_DECLARATION:
    case IIR_CONSTANT_DECLARATION:
      if(decl->get_subtype()->_is_access_type()) {
	decl->_publish_cc_decl( _cc_out );
      } else {
	_cc_out << decl->get_subtype()->_get_cc_type_name();
	_cc_out << " ";
	decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
	decl->_publish_cc_constructor_args( _cc_out );
	_cc_out << ";" << NL();
      }
      break;
    case IIR_ALIAS_DECLARATION:
      _cc_out << decl->get_subtype()->_get_cc_type_name();
      _cc_out << " ";
      ((IIR_AliasDeclaration*)decl)->_publish_cc_alias_object_init( _cc_out );
      _cc_out << ";" << NL();
      break;
    case IIR_ATTRIBUTE_SPECIFICATION:
      ((IIR_AttributeSpecification *) decl)->_publish_cc_decl_with_constructor( _cc_out );
      break;
    case IIR_ATTRIBUTE_DECLARATION:
      break;			// Nothing to be done.
    case IIR_FILE_DECLARATION:
      decl->_publish_cc_decl( _cc_out );
      break;
      
    case IIR_FILE_INTERFACE_DECLARATION:
      // Do Nothing Yet.  Possibly wrong -- SK!!
    case IIR_FUNCTION_DECLARATION: 
    case IIR_PROCEDURE_DECLARATION:
      // Has been taken care of before this subprogram publishing was
      // started.  Look at _publish_cc_decl( _cc_out )
      break;
    default:
      cerr << "IIRScram_SubprogramDeclaration.cc_publish_cc_declarations( published_file &_cc_out ):"
	   << " Unknown declaration type: " << decl->get_kind_text() << endl;
      break;
    }
    decl = subprogram_declarations.successor(decl);
  }

  // After the declarations are over, we publish the identifiers necessary
  // for th for loop statements in the body.
  subprogram_body._publish_cc_decl_for_loop_iterator( _cc_out );
}
  
void 
IIRScram_SubprogramDeclaration::_publish_cc_prototype( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_SubprogramDeclaration::_publish_cc_prototype" );

  if(_contains_body() == TRUE){
    _publish_cc_savantnow_process_ptr( _cc_out );
    
    if(_get_return_type() != NULL) {
      _cc_out << _get_return_type()->_get_cc_type_name();
    }
    else {
      _cc_out << "int";
    }
    _cc_out << NL();

    _get_declarator()->_publish_cc_lvalue( _cc_out );

    _cc_out << OS("(VHDLProcess *processPtr");
    if(interface_declarations.num_elements() != 0) {
      _cc_out << "," << NL();
      interface_declarations._publish_cc_decl_subprogram_args( _cc_out, FALSE );
    }
    _cc_out << CS(");");
    
    
    if(_is_possible_resolution_function() == TRUE) {
      // Publish the prototype of the function, as a resolution function.
      _cc_out << "VHDLType *" << NL();
      _publish_cc_resolution_function_name( _cc_out );
      _cc_out << "( VHDLProcess *, int, VHDLType** );" << NL();
    } // if
    
    if(_is_possible_type_conversion_function() == TRUE) {
      // Publish the prototype of the function, as a type conversion
      // function.
      _cc_out << "VHDLType *" << NL();
      _publish_cc_type_conversion_function_name( _cc_out );
      _cc_out << "( VHDLProcess *, VHDLType*);" << NL();
    }
    
    _publish_cc_savantnow_no_process_ptr( _cc_out );

    subprogram_declarations._publish_cc_prototype( _cc_out );
  }
}

void
IIRScram_SubprogramDeclaration::_publish_cc_resolution_function_name( published_file & ){
  _report_undefined_scram_fn("_publish_cc_resolution_function_name( _cc_out )");
}

void
IIRScram_SubprogramDeclaration::_publish_cc_type_conversion_function_name( published_file & ){
  _report_undefined_scram_fn("_publish_cc_type_conversion_function_name( _cc_out )");
}

// This method publishes file_close (which is implicit) for files declared
// in this subprogram

void
IIRScram_SubprogramDeclaration::_publish_cc_implicit_file_close( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_SubprogramDeclaration::_publish_cc_implicit_file_close" );

  IIR_Declaration *decl = subprogram_declarations.first();
  while (decl != NULL) {
    if (decl->get_kind() == IIR_FILE_DECLARATION) {
      _cc_out << "savantfile_close(processPtr, ";
      decl->_publish_cc_lvalue( _cc_out );
      _cc_out << ");" << NL();
    }
    decl = subprogram_declarations.successor(decl);
  }
}

void
IIRScram_SubprogramDeclaration::_publish_cc_define_current_state(published_file &_cc_out){
  // We will redefine state only if this subprogram has been
  // declared within the declarative region of a process statement..

  ASSERT ( _get_declarative_region() != NULL );
  
  if (_get_declarative_region()->get_kind() == IIR_PROCESS_STATEMENT) {
    IIR_ProcessStatement *ps = (IIR_ProcessStatement *) _get_declarative_region();
    _cc_out << "  " << *(ps->_get_label()) << "_state *currentState = static_cast<"
	    << *(ps->_get_label()) << "_state *>(processPtr->getState())";
    _cc_out.end_statement();
  }
}

IIR_Boolean
IIRScram_SubprogramDeclaration::_is_implicit_file_operation(){
  IIR_Boolean retval = false;
  if( _is_implicit_declaration() &&
      (( IIR_TextLiteral::_cmp( get_declarator(),
				"file_open" ) == 0 ) ||
       (IIR_TextLiteral::_cmp( get_declarator(),
			       "file_close" ) == 0 ) ||
       (IIR_TextLiteral::_cmp( get_declarator(),
			       "read" ) == 0 ) ||
       (IIR_TextLiteral::_cmp( get_declarator(),
			       "write" ) == 0 ) ||
       (IIR_TextLiteral::_cmp( get_declarator(),
			       "endfile" ) == 0 ))){
    retval = true;
  }
      
  return retval;
}

void
IIRScram_SubprogramDeclaration::_publish_cc_subprogram_call_name( published_file &_cc_out ){
  if( _is_implicit_file_operation() ){
    _publish_cc_implicit_file_operation_call_name( _cc_out );
  }
  else{
    _publish_cc_lvalue( _cc_out );
  }
}

void
IIRScram_SubprogramDeclaration::_publish_cc_implicit_file_operation_call_name( published_file &_cc_out ){
  _cc_out << "savant" << *get_declarator();
}

IIR_TypeDefinition* 
IIRScram_SubprogramDeclaration::_get_return_type() {
  return NULL;
}

bool 
IIRScram_SubprogramDeclaration::_check_param( IIR_TypeDefinition *type_def,
					      int param_num ){

  ASSERT( param_num >= 0 );
  ASSERT( type_def != NULL );

  IIR_TypeDefinition *param_type = _get_type_of_param( param_num );
  if( param_type == NULL ){
    return false;
  }

  // Now current is on the right argument...
  if( param_type->_is_compatible( type_def ) != NULL ){
    return true;
  }
  else{
    return false;
  }

}

IIR_Int32
IIRScram_SubprogramDeclaration::_num_subprogram_args(){
  return interface_declarations.num_elements();
}

IIR_Int32
IIRScram_SubprogramDeclaration::_num_required_args(){
  int retval = 0;

  IIR_InterfaceDeclaration *current_decl;
  current_decl =  interface_declarations.first();
  while ( current_decl != NULL ){
    if( current_decl->get_value() == NULL ){
      retval++;
    }
    current_decl =  interface_declarations.successor( current_decl );
  }

  return retval;
}

void 
IIRScram_SubprogramDeclaration::_type_check(){
  subprogram_declarations._type_check_attribute_specifications( subprogram_body );
}

IIR_Boolean 
IIRScram_SubprogramDeclaration::_type_check_return_statements( IIR_TypeDefinition *my_rval ){
  IIR_Boolean retval = FALSE;
  
  set<IIR_TypeDefinition> *context_set = NULL;
  if( my_rval != NULL ){
    context_set = new set<IIR_TypeDefinition>( my_rval );
  }

  IIR_SequentialStatement *stmt;

  IIR_Boolean found_one = FALSE;  
  stmt = subprogram_body.first();
  while( stmt != NULL ){
    if( stmt->get_kind() == IIR_RETURN_STATEMENT ){
      IIR_ReturnStatement *return_stmt = (IIR_ReturnStatement *)stmt;
      return_stmt->set_enclosing_subprogram((IIR_SubprogramDeclaration *)this);
      return_stmt->_type_check( context_set );

      ASSERT( return_stmt->_is_resolved() == TRUE );

      retval = TRUE;
    }
    else if( stmt->_has_statement_list() == TRUE ){
      found_one = 
	stmt->_type_check_return_statements( context_set, (IIR_SubprogramDeclaration *)this );
      if( found_one == TRUE ){
	retval = TRUE;
      }      
    }

    stmt = subprogram_body.successor( stmt );
  }
  
  delete context_set;

  return retval;
}


ostream &
IIRScram_SubprogramDeclaration::_print( ostream &os ){
  os << *get_declarator();
  os << "(";

  IIR_InterfaceDeclaration *current_param = interface_declarations.first();
  while( current_param != NULL ){
    os << *current_param;
    current_param = interface_declarations.successor( current_param );
    if( current_param != NULL ){
      os << ", ";
    }
  }
  
  os << ")";

  return os;
}

IIR_TypeDefinition *
IIRScram_SubprogramDeclaration::_get_type_of_param( int param_number ){

  IIR_InterfaceDeclaration *current = interface_declarations.first();
  if( current == NULL ){
    return NULL;
  }

  int i;
  for( i = 0; i < param_number; i++ ){
    current = interface_declarations.successor( current );

    if( current == NULL ){
      return NULL;
    }  
  }

  return current->get_subtype();
}

IIR_Boolean 
IIRScram_SubprogramDeclaration::_compare_signatures( IIR_SubprogramDeclaration *a,
						     IIR_SubprogramDeclaration *b ){
  ASSERT( a != NULL );
  ASSERT( b != NULL );
  
  if( a->get_subtype()->_is_compatible( b->get_subtype() ) == NULL ){
    return FALSE;
  }

  int a_num_params =  a->interface_declarations.num_elements();
  int b_num_params =  b->interface_declarations.num_elements();
  if( a_num_params != b_num_params ){
    return FALSE;
  }

  int i;
  for( i = 0 ; i < a_num_params; i++ ){
    if( a->_get_type_of_param( i )->_is_compatible( b->_get_type_of_param( i ) ) == NULL ){
      return FALSE;
    }
  }

  // If we made it here, everything matched up...
  return TRUE;
}

void
IIRScram_SubprogramDeclaration::_make_interface_visible( symbol_table *sym_tab ){
  sym_tab->make_visible( (IIR_DeclarationList *)&interface_declarations );
}

IIR_Boolean 
IIRScram_SubprogramDeclaration::_contains_body(){
  return _my_contains_body;
}

void
IIRScram_SubprogramDeclaration::_set_contains_body( IIR_Boolean new_contains_body ){
  _my_contains_body = new_contains_body;
}

IIR_Boolean 
IIRScram_SubprogramDeclaration::_is_homograph_of( IIR_Declaration *compare_to ){
  ASSERT( compare_to != NULL );
  ASSERT( IIR_TextLiteral::_cmp( compare_to->get_declarator(), get_declarator() ) == 0 );

  // If they're not the same kind of node, they're not.
  if( get_kind() != compare_to->get_kind() ){
    return FALSE;
  }

  ASSERT( compare_to->_is_iir_subprogram_declaration() == TRUE );
  IIR_SubprogramDeclaration *compare_as_subprog;
  compare_as_subprog = (IIR_SubprogramDeclaration *)compare_to;
  
  if( _compare_signatures( (IIR_SubprogramDeclaration *)this, compare_as_subprog ) == FALSE ){
    return FALSE;
  }
  
  return TRUE;
}

IIR_Boolean 
IIRScram_SubprogramDeclaration::_can_be_in_same_region( IIR_Declaration *to_check ){
  if(( _contains_body() == TRUE && to_check->_contains_body() == FALSE ) ||
     ( _contains_body() == FALSE && to_check->_contains_body() == TRUE )){
    return TRUE;
  }
  else if(( _is_implicit_operator() == TRUE && to_check->_is_implicit_operator() == FALSE )||
	  ( _is_implicit_operator() == FALSE && to_check->_is_implicit_operator() == TRUE )){
    return TRUE;
  }
  else{
    return FALSE;
  }
}

void 
IIRScram_SubprogramDeclaration::_add_declaration(){
  _get_symbol_table()->add_subprogram_declaration( (IIR_SubprogramDeclaration *)this );
}

IIR_Boolean 
IIRScram_SubprogramDeclaration::_is_operator(){
  IIR_TextLiteral *declarator = get_declarator();
  if( declarator->operator[]( 0 ) == '"' &&
      declarator->operator[]( declarator->get_text_length() ) == '"' ){
    ASSERT( interface_declarations.num_elements() == 1 ||
	    interface_declarations.num_elements() == 2 );
    
    return TRUE;
  }
  else{
    return FALSE;
  }
}

IIR_Boolean 
IIRScram_SubprogramDeclaration::_is_implicit_operator(){
  if( _is_implicit_declaration() == TRUE && _is_operator() == TRUE ){
    return TRUE;
  }
  else{
    return FALSE;
  }
}

visitor_return_type *
IIRScram_SubprogramDeclaration::_accept_visitor( node_visitor *visitor, 
						 visitor_argument_type *arg){
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_SubprogramDeclaration(this, arg);
}
