/* ---------------------------------------------------------------------
*
*  -- ScaLAPACK routine (version 1.0) --
*     University of Tennessee, Knoxville, Oak Ridge National Laboratory,
*     and University of California, Berkeley.
*     November 17, 1996
*
*  ---------------------------------------------------------------------
*/
/*
*  Include files
*/
#include "pblas.h"

void pdnrm2_( n, norm2, X, ix, jx, desc_X, incx )
/*
*  .. Scalar Arguments ..
*/
   int         * incx, * ix, * jx, * n;
   double      * norm2;
/* ..
*  .. Array Arguments ..
*/
   int         desc_X[];
   double      X[];
{
/*
*  Purpose
*  =======
*
*  PDNRM2 returns the 2-norm of a distributed vector sub( X ),
*
*  where sub( X ) denotes X(IX,JX:JX+N-1) if INCX = M_X,
*                         X(IX:IX+N-1,JX) if INCX = 1 and INCX <> M_X.
*
*  Notes
*  =====
*
*  Each global data object is described by an associated description
*  vector.  This vector stores the information required to establish
*  the mapping between an object element and its corresponding process
*  and memory location.
*
*  Let A be a generic term for any 2D block cyclicly distributed array.
*  Such a global array has an associated description vector descA.
*  In the following comments, the character _ should be read as
*  "of the global array".
*
*  NOTATION        STORED IN      EXPLANATION
*  --------------- -------------- --------------------------------------
*  DT_A   (global) descA[ DT_ ]   The descriptor type.  In this case,
*                                 DT_A = 1.
*  CTXT_A (global) descA[ CTXT_ ] The BLACS context handle, indicating
*                                 the BLACS process grid A is distribu-
*                                 ted over. The context itself is glo-
*                                 bal, but the handle (the integer
*                                 value) may vary.
*  M_A    (global) descA[ M_ ]    The number of rows in the global
*                                 array A.
*  N_A    (global) descA[ N_ ]    The number of columns in the global
*                                 array A.
*  MB_A   (global) descA[ MB_ ]   The blocking factor used to distribu-
*                                 te the rows of the array.
*  NB_A   (global) descA[ NB_ ]   The blocking factor used to distribu-
*                                 te the columns of the array.
*  RSRC_A (global) descA[ RSRC_ ] The process row over which the first
*                                 row of the array A is distributed.
*  CSRC_A (global) descA[ CSRC_ ] The process column over which the
*                                 first column of the array A is
*                                 distributed.
*  LLD_A  (local)  descA[ LLD_ ]  The leading dimension of the local
*                                 array.  LLD_A >= MAX(1,LOCr(M_A)).
*
*  Let K be the number of rows or columns of a distributed matrix,
*  and assume that its process grid has dimension p x q.
*  LOCr( K ) denotes the number of elements of K that a process
*  would receive if K were distributed over the p processes of its
*  process column.
*  Similarly, LOCc( K ) denotes the number of elements of K that a
*  process would receive if K were distributed over the q processes of
*  its process row.
*  The values of LOCr() and LOCc() may be determined via a call to the
*  ScaLAPACK tool function, NUMROC:
*          LOCr( M ) = NUMROC( M, MB_A, MYROW, RSRC_A, NPROW ),
*          LOCc( N ) = NUMROC( N, NB_A, MYCOL, CSRC_A, NPCOL ).
*  An upper bound for these quantities may be computed by:
*          LOCr( M ) <= ceil( ceil(M/MB_A)/NPROW )*MB_A
*          LOCc( N ) <= ceil( ceil(N/NB_A)/NPCOL )*NB_A
*
*  Because vectors may be seen as particular matrices, a distributed
*  vector is considered to be a distributed matrix.
*
*
*  Parameters
*  ==========
*
*  N       (global input) pointer to INTEGER
*          The number of components of the distributed vector sub( X ).
*          N >= 0.
*
*  NORM2   (local output) pointer to DOUBLE PRECISION
*          The 2-norm of the distributed vector sub( X ) only in its
*          scope.
*
*  X       (local input) DOUBLE PRECISION array containing the local
*          pieces of a distributed matrix of dimension of at least
*              ( (JX-1)*M_X + IX + ( N - 1 )*abs( INCX ) )
*          This array contains the entries of the distributed vector
*          sub( X ).
*
*  IX      (global input) pointer to INTEGER
*          The global row index of the submatrix of the distributed
*          matrix X to operate on.
*
*  JX      (global input) pointer to INTEGER
*          The global column index of the submatrix of the distributed
*          matrix X to operate on.
*
*  DESCX   (global and local input) INTEGER array of dimension 8.
*          The array descriptor of the distributed matrix X.
*
*  INCX    (global input) pointer to INTEGER
*          The global increment for the elements of X. Only two values
*          of INCX are supported in this version, namely 1 and M_X.
*
*  =====================================================================
*
*  .. Local Scalars ..
*/
   int         ictxt, iix, info, ixcol, ixrow, jjx, mone=-1, mycol,
               myrow, nn, np, nprow, npcol, nq, nz, one=1;
/* ..
*  .. External Functions ..
*/
   void        blacs_gridinfo_();
   void        pbchkvect();
   void        pberror_();
   F_VOID_FCT  pdtreecomb_();
   F_VOID_FCT  dcombnrm2_();
   F_INTG_FCT  numroc_();
   F_DBLE_FCT  dnrm2_();
/* ..
*  .. Executable Statements ..
*
*  Get grid parameters
*/
   ictxt = desc_X[CTXT_];
   blacs_gridinfo_( &ictxt, &nprow, &npcol, &myrow, &mycol );
/*
*  Test the input parameters
*/
   info = 0;
   if( nprow == -1 )
      info = -(600+CTXT_+1);
   else
      pbchkvect( *n, 1, *ix, *jx, desc_X, *incx, 6, &iix, &jjx, &ixrow,
                 &ixcol, nprow, npcol, myrow, mycol, &info );
   if( info )
   {
   if( info )
      pberror_( &ictxt, "PDNRM2", &info );
      return;
   }
/*
*  Quick return if possible.
*/
   *norm2 = ZERO;
   if( *n == 0 ) return;
/*
*  norm2 <- || x ||
*/
   if( ( *incx == 1 ) && ( desc_X[M_] == 1 ) && ( *n == 1 ) )
   {
      if( ( myrow == ixrow ) && ( mycol == ixcol ) )
      {
      *norm2 = ABS( X[iix-1+(jjx-1)*desc_X[LLD_]] );
      }
      return;
   }

   if( *incx == desc_X[M_] )   /* X is distributed over a process row */
   {
      if( myrow == ixrow )
      {
         nz = (*jx-1) % desc_X[NB_];
         nn = *n + nz;
         nq = numroc_( &nn, &desc_X[NB_], &mycol, &ixcol, &npcol );
         if( mycol == ixcol )
            nq -= nz;
         if( nq > 0 )
         {
            *norm2 = dnrm2_( &nq, &X[iix-1+(jjx-1)*desc_X[LLD_]],
                             &desc_X[LLD_] );
         }
         pdtreecomb_( &ictxt, C2F_CHAR( ROW ), &one, norm2, &mone, &mycol,
                      dcombnrm2_ );
      }
   }
   else
   {
      if( mycol == ixcol )
      {
         nz = (*ix-1) % desc_X[MB_];
         nn = *n + nz;
         np = numroc_( &nn, &desc_X[MB_], &myrow, &ixrow, &nprow );
         if( myrow == ixrow )
            np -= nz;
         if( np > 0 )
         {
            *norm2 = dnrm2_( &np, &X[iix-1+(jjx-1)*desc_X[LLD_]],
                             &one );
         }
         pdtreecomb_( &ictxt, C2F_CHAR( COLUMN ), &one, norm2, &mone,
                      &mycol, dcombnrm2_ );
      }
   }
}
