//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [temp,pres,dens] = CL_mod_atmUS76(alt)
// US 76 Atmospheric model
//
// Calling Sequence
// [temp,pres,dens] = CL_mod_atmUS76(alt)
//
// Description
// <itemizedlist><listitem>
// Computes temperature (<emphasis role="bold">temp</emphasis>), pressure (<emphasis role="bold">pres</emphasis>) and density (<emphasis role="bold">dens</emphasis>) at a given altitude (<emphasis role="bold">alt</emphasis>),
// using US76 atmospheric model.
// </listitem>
// <listitem>
// Notes: 
// <para> - The US76 atmospheric model is only to envisaged for altitudes up to 1000 km. </para>
// <para> - If the imput altitude is negative, altitude is enforced to 0.</para>
// <para/><inlinemediaobject><imageobject><imagedata fileref="US76.gif"/></imageobject></inlinemediaobject></listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// alt : Altitude [m] (1xN)
// temp : Kinetic temperature [K] (1xN)
// pres : Pressure [Pa] (1xN)
// dens : Density [kg/m^3] (1xN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1 CNES - MSLIB FORTRAN 90, Volume P (mp_atm_US76)
//
// Examples
// alt = 700.e3;
// [temp,pres,dens] = CL_mod_atmUS76(alt)
//

// Declarations:


// Code:

N = size(alt,2)
temp = zeros(1,N)
rhpot = zeros(1,N)
tmol = zeros(1,N)
xmol = zeros(1,N)
coef = zeros(1,N)
pres = zeros(1,N)
dens = zeros(1,N)

//PARAMETERS DEFINITION
nmax = 430        // dimensions de tableaux

//   * alts geopotentielles de debut et de fin des differents gradients de temp
htab = [0.,11e3,20e3,32e3,47e3,51e3,71e3,84.852e3,0.,0.,0.,0.,0.]

//   * alts geometriques
ztab = [ 0.,0.,0.,0.,0.,0.,0.,86e3,91e3,110e3,120e3,500e3,1000e3 ]

//   * coeficients correcteurs pour le calcul de la temp  entre 80 et 86 km d'alt
xmm0tb = [ 1.,0.999996,0.999989,0.999971,0.999941,0.999909,0.999870,0.999829,0.999786,0.999741,0.999694,0.999641,0.999579 ]

//   * derivees de la temp par rapport a rhpot ou rzgeo
dttab = [ -6.5e-3,0.,1e-3,2.8e-3,0.,-2.8e-3,-2.0e-3,0.,0.,12e-3,0.,0.,0. ]

//   * press pour htab
ptab = [ 1.01325e5,2.2632e4,5.4748e3,8.6801e2,1.1090e2,6.6938e1,3.9564,3.7338e-1 ]

//   * temps correspondant a htab ou ztab
ttab = [0.,0.,0.,0.,0.,0.,0.,186.8673,186.8673,240.,360.,999.24,1000. ]

tmtab = [ 288.15,216.65,216.65,228.65,270.65,270.65,214.65,186.946 ]

//   * masses molaires des elements N2,O,O2,Ar,He et H
rmoltb =[ 0.,28.0134,15.9994,31.9988,39.948, 4.0026,1.0080 ]

r0=6356.766e3  // rayon de la Terre utilise pour le calcul des alts geopotentielles
g0=9.80665     // acceleration due a la gravite au niveau de la mer

xmol0=28.9644     //   masse moleculaire de l'atmosphere au sol (kg/kmol)
boltz=1.380622e-23//   constante de Boltzmann (n.m/k)
rstar=8.31432e3   //   constante des gaz parfaits (n.m/(kmol.k))

rappor=g0*xmol0/rstar

rlambd = 1.875e-05// = dttab(1+1+9)/(ttab(1+12)-ttab(1+10))   (constante)
                                                       // attention: valeur pour des alts en metres
epspre = 1e-06 // epsilon de test du gradient de temp pour le calcul de la pres

tc = 263.1905
gda_ = -76.3232
pta = -19.9429e3

//   * composition de l'atmosphere en fonction des differents elements
//   * qui la composent ... N2,O,O2,Ar,He et H et en fonction de
//   * l'alt a partir de 86 km
// le nombre de tableaux a initialiser depend de nmax, etant donne un nombre de cartes de continuations limite a 39
// par le langage (ici on a pris 30 par commodite)

rntab01 = [ ...
86.0e3,1.130e20,8.600e16,3.031e19,1.351e18,7.582e14,0.; ...
86.5e3,1.034e20,9.939e16,2.772e19,1.236e18,6.976e14,0.; ...
87.0e3,9.456e19,1.147e17,2.535e19,1.130e18,6.422e14,0.; ...
87.5e3,8.651e19,1.320e17,2.319e19,1.033e18,5.915e14,0.; ...
88.0e3,7.915e19,1.513e17,2.120e19,9.437e17,5.453e14,0.; ...
88.5e3,7.242e19,1.724e17,1.938e19,8.624e17,5.031e14,0.; ...
89.0e3,6.626e19,1.952e17,1.772e19,7.880e17,4.647e14,0.; ...
89.5e3,6.062e19,2.193e17,1.619e19,7.198e17,4.296e14,0.; ...
90.0e3,5.547e19,2.443e17,1.479e19,6.574e17,3.976e14,0.; ...
90.5e3,5.075e19,2.699e17,1.351e19,6.002e17,3.685e14,0.; ...
91.0e3,4.643e19,2.953e17,1.234e19,5.478e17,3.419e14,0.; ...
91.5e3,4.248e19,3.200e17,1.126e19,4.998e17,3.177e14,0.; ...
92.0e3,3.886e19,3.434e17,1.027e19,4.557e17,2.956e14,0.; ...
92.5e3,3.553e19,3.651e17,9.361e18,4.152e17,2.753e14,0.; ...
93.0e3,3.249e19,3.846e17,8.527e18,3.781e17,2.568e14,0.; ...
93.5e3,2.970e19,4.016e17,7.761e18,3.441e17,2.399e14,0.; ...
94.0e3,2.715e19,4.159e17,7.060e18,3.129e17,2.244e14,0.; ...
94.5e3,2.481e19,4.275e17,6.418e18,2.844e17,2.103e14,0.; ...
95.0e3,2.268e19,4.365e17,5.830e18,2.583e17,1.973e14,0.; ...
95.5e3,2.072e19,4.429e17,5.293e18,2.345e17,1.854e14,0.; ...
96.0e3,1.894e19,4.471e17,4.801e18,2.127e17,1.745e14,0.; ...
96.5e3,1.730e19,4.493e17,4.353e18,1.928e17,1.645e14,0.; ...
97.0e3,1.581e19,4.500e17,3.943e18,1.746e17,1.553e14,0.; ...
97.5e3,1.445e19,4.494e17,3.570e18,1.581e17,1.468e14,0.; ...
98.0e3,1.320e19,4.476e17,3.230e18,1.430e17,1.390e14,0.; ...
98.5e3,1.206e19,4.447e17,2.920e18,1.292e17,1.317e14,0.; ...
99.0e3,1.102e19,4.408e17,2.639e18,1.167e17,1.251e14,0.; ...
99.5e3,1.008e19,4.358e17,2.383e18,1.053e17,1.190e14,0.; ...
100.0e3,9.210e18,4.298e17,2.151e18,9.501e16,1.133e14,0.; ...
101.0e3,7.740e18,4.168e17,1.756e18,7.735e16,1.034e14,0.];

rntab02 = [ ...
102.0e3,6.508e18,4.007e17,1.430e18,6.279e16,9.497e13,0.; ...
103.0e3,5.475e18,3.821e17,1.163e18,5.082e16,8.776e13,0.; ...
104.0e3,4.609e18,3.619e17,9.434e17,4.101e16,8.160e13,0.; ...
105.0e3,3.883e18,3.406e17,7.645e17,3.299e16,7.633e13,0.; ...
106.0e3,3.273e18,3.188e17,6.189e17,2.645e16,7.181e13,0.; ...
107.0e3,2.760e18,2.968e17,5.005e17,2.113e16,6.789e13,0.; ...
108.0e3,2.327e18,2.748e17,4.045e17,1.681e16,6.443e13,0.; ...
109.0e3,1.959e18,2.528e17,3.263e17,1.331e16,6.128e13,0.; ...
110.0e3,1.641e18,2.303e17,2.621e17,1.046e16,5.821e13,0.; ...
111.0e3,1.373e18,2.083e17,2.104e17,8.200e15,5.526e13,0.; ...
112.0e3,1.158e18,1.889e17,1.706e17,6.481e15,5.271e13,0.; ...
113.0e3,9.841e17,1.718e17,1.398e17,5.169e15,5.044e13,0.; ...
114.0e3,8.422e17,1.565e17,1.156e17,4.162e15,4.838e13,0.; ...
115.0e3,7.254e17,1.428e17,9.646e16,3.386e15,4.648e13,0.; ...
116.0e3,6.285e17,1.305e17,8.120e16,2.779e15,4.473e13,0.; ...
117.0e3,5.475e17,1.194e17,6.891e16,2.301e15,4.310e13,0.; ...
118.0e3,4.794e17,1.096e17,5.892e16,1.920e15,4.160e13,0.; ...
119.0e3,4.217e17,1.007e17,5.072e16,1.614e15,4.019e13,0.; ...
120.0e3,3.726e17,9.275e16,4.395e16,1.366e15,3.888e13,0.; ...
121.0e3,3.306e17,8.562e16,3.832e16,1.164e15,3.766e13,0.; ...
122.0e3,2.947e17,7.925e16,3.360e16,9.979e14,3.652e13,0.; ...
123.0e3,2.637e17,7.354e16,2.963e16,8.606e14,3.547e13,0.; ...
124.0e3,2.368e17,6.840e16,2.625e16,7.460e14,3.448e13,0.; ...
125.0e3,2.135e17,6.376e16,2.336e16,6.498e14,3.356e13,0.; ...
126.0e3,1.930e17,5.956e16,2.087e16,5.685e14,3.270e13,0.; ...
127.0e3,1.750e17,5.576e16,1.871e16,4.994e14,3.189e13,0.; ...
128.0e3,1.592e17,5.229e16,1.683e16,4.403e14,3.112e13,0.; ...
129.0e3,1.451e17,4.914e16,1.519e16,3.896e14,3.040e13,0.; ...
130.0e3,1.326e17,4.625e16,1.375e16,3.458e14,2.972e13,0.; ...
131.0e3,1.215e17,4.361e16,1.247e16,3.078e14,2.907e13,0.];

rntab03 = [ ...
132.0e3,1.116e17,4.118e16,1.134e16,2.748e14,2.846e13,0.; ...
133.0e3,1.026e17,3.894e16,1.034e16,2.460e14,2.787e13,0.; ...
134.0e3,9.460e16,3.688e16,9.444e15,2.207e14,2.732e13,0.; ...
135.0e3,8.735e16,3.497e16,8.645e15,1.985e14,2.679e13,0.; ...
136.0e3,8.080e16,3.320e16,7.927e15,1.789e14,2.629e13,0.; ...
137.0e3,7.487e16,3.156e16,7.283e15,1.616e14,2.581e13,0.; ...
138.0e3,6.947e16,3.004e16,6.702e15,1.463e14,2.535e13,0.; ...
139.0e3,6.456e16,2.862e16,6.177e15,1.326e14,2.491e13,0.; ...
140.0e3,6.009e16,2.729e16,5.702e15,1.205e14,2.449e13,0.; ...
141.0e3,5.600e16,2.605e16,5.272e15,1.096e14,2.408e13,0.; ...
142.0e3,5.225e16,2.489e16,4.881e15,9.989e13,2.369e13,0.; ...
143.0e3,4.881e16,2.380e16,4.524e15,9.118e13,2.332e13,0.; ...
144.0e3,4.565e16,2.278e16,4.199e15,8.335e13,2.296e13,0.; ...
145.0e3,4.275e16,2.183e16,3.903e15,7.630e13,2.261e13,0.; ...
146.0e3,4.007e16,2.092e16,3.631e15,6.994e13,2.228e13,0.; ...
147.0e3,3.760e16,2.007e16,3.382e15,6.420e13,2.196e13,0.; ...
148.0e3,3.531e16,1.927e16,3.153e15,5.900e13,2.165e13,0.; ...
149.0e3,3.320e16,1.852e16,2.943e15,5.428e13,2.135e13,0.; ...
150e3,3.124e16,1.780e16,2.750e15,5.000e13,2.106e13,3.767e11; ...
151e3,2.942e16,1.712e16,2.572e15,4.611e13,2.078e13,3.659e11; ...
152e3,2.773e16,1.648e16,2.407e15,4.256e13,2.051e13,3.557e11; ...
153e3,2.616e16,1.587e16,2.255e15,3.933e13,2.024e13,3.461e11; ...
154e3,2.469e16,1.530e16,2.114e15,3.638e13,1.999e13,3.369e11; ...
155e3,2.333e16,1.475e16,1.984e15,3.368e13,1.974e13,3.283e11; ...
156e3,2.206e16,1.423e16,1.863e15,3.121e13,1.950e13,3.201e11; ...
157e3,2.087e16,1.373e16,1.751e15,2.895e13,1.927e13,3.123e11; ...
158e3,1.975e16,1.326e16,1.647e15,2.687e13,1.905e13,3.049e11; ...
159e3,1.871e16,1.281e16,1.550e15,2.496e13,1.883e13,2.978e11; ...
160e3,1.774e16,1.238e16,1.460e15,2.321e13,1.861e13,2.911e11; ...
161e3,1.682e16,1.197e16,1.376e15,2.159e13,1.841e13,2.847e11];

rntab04 = [ ...
162e3,1.596e16,1.158e16,1.297e15,2.011e13,1.820e13,2.786e11; ...
163e3,1.516e16,1.120e16,1.224e15,1.874e13,1.801e13,2.728e11; ...
164e3,1.440e16,1.085e16,1.156e15,1.747e13,1.782e13,2.672e11; ...
165e3,1.369e16,1.050e16,1.092e15,1.630e13,1.763e13,2.619e11; ...
166e3,1.302e16,1.018e16,1.032e15,1.522e13,1.745e13,2.568e11; ...
167e3,1.239e16,9.863e15,9.757e14,1.422e13,1.727e13,2.520e11; ...
168e3,1.179e16,9.562e15,9.232e14,1.329e13,1.710e13,2.473e11; ...
169e3,1.123e16,9.273e15,8.739e14,1.243e13,1.693e13,2.429e11; ...
170e3,1.070e16,8.996e15,8.277e14,1.163e13,1.676e13,2.386e11; ...
171e3,1.020e16,8.730e15,7.843e14,1.089e13,1.660e13,2.345e11; ...
172e3,9.724e15,8.474e15,7.435e14,1.020e13,1.644e13,2.306e11; ...
173e3,9.277e15,8.228e15,7.051e14,9.565e12,1.629e13,2.268e11; ...
174e3,8.853e15,7.992e15,6.690e14,8.970e12,1.614e13,2.232e11; ...
175e3,8.452e15,7.765e15,6.350e14,8.417e12,1.599e13,2.197e11; ...
176e3,8.072e15,7.546e15,6.030e14,7.901e12,1.585e13,2.163e11; ...
177e3,7.712e15,7.335e15,5.728e14,7.420e12,1.571e13,2.131e11; ...
178e3,7.371e15,7.132e15,5.443e14,6.971e12,1.557e13,2.100e11; ...
179e3,7.047e15,6.936e15,5.174e14,6.553e12,1.543e13,2.070e11; ...
180e3,6.740e15,6.747e15,4.921e14,6.162e12,1.530e13,2.041e11; ...
181e3,6.448e15,6.565e15,4.681e14,5.797e12,1.517e13,2.013e11; ...
182e3,6.170e15,6.389e15,4.455e14,5.456e12,1.504e13,1.987e11; ...
183e3,5.907e15,6.220e15,4.241e14,5.136e12,1.492e13,1.961e11; ...
184e3,5.656e15,6.056e15,4.039e14,4.838e12,1.479e13,1.936e11; ...
185e3,5.417e15,5.897e15,3.847e14,4.558e12,1.467e13,1.911e11; ...
186e3,5.190e15,5.744e15,3.666e14,4.296e12,1.456e13,1.888e11; ...
187e3,4.974e15,5.596e15,3.494e14,4.050e12,1.444e13,1.866e11; ...
188e3,4.768e15,5.453e15,3.331e14,3.820e12,1.433e13,1.844e11; ...
189e3,4.572e15,5.315e15,3.177e14,3.604e12,1.421e13,1.823e11; ...
190e3,4.385e15,5.181e15,3.031e14,3.401e12,1.410e13,1.802e11; ...
191e3,4.207e15,5.051e15,2.892e14,3.211e12,1.400e13,1.782e11];

rntab05 = [ ...
192e3,4.037e15,4.926e15,2.760e14,3.033e12,1.389e13,1.763e11; ...
193e3,3.875e15,4.804e15,2.635e14,2.865e12,1.379e13,1.745e11; ...
194e3,3.720e15,4.686e15,2.517e14,2.707e12,1.368e13,1.727e11; ...
195e3,3.572e15,4.572e15,2.404e14,2.558e12,1.358e13,1.709e11; ...
196e3,3.430e15,4.461e15,2.297e14,2.419e12,1.348e13,1.692e11; ...
197e3,3.295e15,4.354e15,2.195e14,2.288e12,1.339e13,1.676e11; ...
198e3,3.166e15,4.249e15,2.098e14,2.164e12,1.329e13,1.660e11; ...
199e3,3.043e15,4.148e15,2.006e14,2.047e12,1.319e13,1.645e11; ...
200e3,2.925e15,4.050e15,1.918e14,1.938e12,1.310e13,1.630e11; ...
201e3,2.812e15,3.955e15,1.834e14,1.834e12,1.301e13,1.615e11; ...
202e3,2.704e15,3.862e15,1.755e14,1.737e12,1.292e13,1.601e11; ...
203e3,2.601e15,3.773e15,1.679e14,1.645e12,1.283e13,1.587e11; ...
204e3,2.502e15,3.685e15,1.607e14,1.558e12,1.274e13,1.574e11; ...
205e3,2.407e15,3.600e15,1.538e14,1.477e12,1.266e13,1.561e11; ...
206e3,2.316e15,3.518e15,1.473e14,1.399e12,1.257e13,1.548e11; ...
207e3,2.229e15,3.438e15,1.410e14,1.327e12,1.249e13,1.536e11; ...
208e3,2.146e15,3.360e15,1.351e14,1.258e12,1.240e13,1.524e11; ...
209e3,2.066e15,3.284e15,1.294e14,1.193e12,1.232e13,1.512e11; ...
210e3,1.989e15,3.211e15,1.239e14,1.131e12,1.224e13,1.501e11; ...
211e3,1.915e15,3.139e15,1.188e14,1.073e12,1.216e13,1.490e11; ...
212e3,1.845e15,3.069e15,1.138e14,1.019e12,1.208e13,1.479e11; ...
213e3,1.777e15,3.001e15,1.091e14,9.666e11, 1.201e13,1.468e11; ...
214e3,1.712e15,2.935e15,1.046e14,9.176e11, 1.193e13,1.458e11; ...
215e3,1.650e15,2.871e15,1.003e14,8.711e11, 1.185e13,1.448e11; ...
216e3,1.590e15,2.808e15,9.617e13,8.272e11, 1.178e13,1.439e11; ...
217e3,1.533e15,2.747e15,9.224e13,7.856e11, 1.171e13,1.429e11; ...
218e3,1.477e15,2.688e15,8.848e13,7.463e11, 1.163e13,1.420e11; ...
219e3,1.424e15,2.630e15,8.489e13,7.090e11, 1.156e13,1.411e11; ...
220e3,1.373e15,2.573e15,8.145e13,6.737e11, 1.149e13,1.402e11; ...
221e3,1.324e15,2.518e15,7.816e13,6.402e11, 1.142e13,1.393e11];

rntab06 = [ ...
222e3,1.277e15,2.465e15,7.502e13,6.085e11, 1.135e13,1.385e11; ...
223e3,1.232e15,2.412e15,7.201e13,5.785e11, 1.128e13,1.377e11; ...
224e3,1.188e15,2.361e15,6.913e13,5.500e11, 1.122e13,1.369e11; ...
225e3,1.147e15,2.312e15,6.637e13,5.230e11, 1.115e13,1.361e11; ...
226e3,1.106e15,2.263e15,6.373e13,4.974e11, 1.108e13,1.353e11; ...
227e3,1.068e15,2.216e15,6.121e13,4.731e11, 1.102e13,1.345e11; ...
228e3,1.030e15,2.170e15,5.879e13,4.501e11, 1.095e13,1.338e11; ...
229e3,9.945e14,2.125e15,5.647e13,4.282e11, 1.089e13,1.331e11; ...
230e3,9.600e14,2.081e15,5.425e13,4.075e11, 1.083e13,1.324e11; ...
231e3,9.268e14,2.038e15,5.212e13,3.878e11, 1.076e13,1.317e11; ...
232e3,8.948e14,1.996e15,5.009e13,3.691e11, 1.070e13,1.310e11; ...
233e3,8.640e14,1.955e15,4.813e13,3.514e11, 1.064e13,1.304e11; ...
234e3,8.343e14,1.915e15,4.626e13,3.345e11, 1.058e13,1.297e11; ...
235e3,8.058e14,1.876e15,4.446e13,3.185e11, 1.052e13,1.291e11; ...
236e3,7.782e14,1.838e15,4.274e13,3.033e11, 1.046e13,1.285e11; ...
237e3,7.517e14,1.801e15,4.109e13,2.888e11, 1.040e13,1.279e11; ...
238e3,7.262e14,1.765e15,3.951e13,2.751e11, 1.034e13,1.273e11; ...
239e3,7.016e14,1.729e15,3.799e13,2.621e11, 1.029e13,1.267e11; ...
240e3,6.778e14,1.695e15,3.653e13,2.497e11, 1.023e13,1.261e11; ...
241e3,6.550e14,1.661e15,3.513e13,2.379e11, 1.017e13,1.256e11; ...
242e3,6.329e14,1.628e15,3.379e13,2.267e11, 1.012e13,1.250e11; ...
243e3,6.117e14,1.595e15,3.251e13,2.160e11, 1.006e13,1.245e11; ...
244e3,5.912e14,1.564e15,3.127e13,2.059e11, 1.001e13,1.240e11; ...
245e3,5.714e14,1.533e15,3.008e13,1.962e11, 9.953e12,1.234e11; ...
246e3,5.523e14,1.503e15,2.895e13,1.871e11, 9.899e12,1.229e11; ...
247e3,5.339e14,1.473e15,2.785e13,1.783e11, 9.846e12,1.224e11; ...
248e3,5.162e14,1.444e15,2.680e13,1.700e11, 9.794e12,1.219e11; ...
249e3,4.991e14,1.416e15,2.579e13,1.621e11, 9.741e12,1.215e11; ...
250e3,4.826e14,1.388e15,2.482e13,1.546e11, 9.690e12,1.210e11; ...
251e3,4.666e14,1.361e15,2.389e13,1.474e11, 9.638e12,1.205e11];

rntab07 = [ ...
252e3,4.512e14,1.335e15,2.300e13,1.406e11, 9.587e12,1.201e11; ...
253e3,4.364e14,1.309e15,2.214e13,1.341e11, 9.537e12,1.196e11; ...
254e3,4.221e14,1.284e15,2.132e13,1.280e11, 9.487e12,1.192e11; ...
255e3,4.082e14,1.259e15,2.052e13,1.221e11, 9.438e12,1.188e11; ...
256e3,3.949e14,1.235e15,1.976e13,1.165e11, 9.389e12,1.183e11; ...
257e3,3.820e14,1.211e15,1.903e13,1.111e11, 9.340e12,1.179e11; ...
258e3,3.695e14,1.188e15,1.832e13,1.060e11, 9.292e12,1.175e11; ...
259e3,3.575e14,1.165e15,1.765e13,1.012e11, 9.244e12,1.171e11; ...
260e3,3.459e14,1.143e15,1.700e13,9.658e10,9.196e12,1.167e11; ...
261e3,3.347e14,1.121e15,1.637e13,9.218e10,9.149e12,1.163e11; ...
262e3,3.238e14,1.100e15,1.577e13,8.799e10,9.103e12,1.159e11; ...
263e3,3.134e14,1.079e15,1.519e13,8.399e10,9.056e12,1.156e11; ...
264e3,3.033e14,1.059e15,1.463e13,8.019e10,9.010e12,1.152e11; ...
265e3,2.935e14,1.039e15,1.410e13,7.655e10,8.965e12,1.148e11; ...
266e3,2.841e14,1.019e15,1.358e13,7.309e10,8.920e12,1.145e11; ...
267e3,2.749e14,9.998e14,1.309e13,6.979e10,8.875e12,1.141e11; ...
268e3,2.661e14,9.811e14,1.261e13,6.665e10,8.830e12,1.138e11; ...
269e3,2.576e14,9.627e14,1.215e13,6.365e10,8.786e12,1.134e11; ...
270e3,2.494e14,9.447e14,1.171e13,6.078e10,8.743e12,1.131e11; ...
271e3,2.414e14,9.270e14,1.128e13,5.805e10,8.699e12,1.127e11; ...
272e3,2.337e14,9.097e14,1.088e13,5.545e10,8.656e12,1.124e11; ...
273e3,2.263e14,8.928e14,1.048e13,5.297e10,8.613e12,1.121e11; ...
274e3,2.191e14,8.762e14,1.010e13,5.060e10,8.571e12,1.118e11; ...
275e3,2.121e14,8.599e14,9.739e12,4.834e10,8.529e12,1.115e11; ...
276e3,2.054e14,8.440e14,9.388e12,4.618e10,8.487e12,1.112e11; ...
277e3,1.989e14,8.284e14,9.050e12,4.412e10,8.445e12,1.109e11; ...
278e3,1.926e14,8.131e14,8.725e12,4.216e10,8.404e12,1.106e11; ...
279e3,1.865e14,7.981e14,8.412e12,4.029e10,8.363e12,1.103e11; ...
280e3,1.806e14,7.834e14,8.110e12,3.850e10,8.322e12,1.100e11; ...
281e3,1.750e14,7.691e14,7.820e12,3.679e10,8.282e12,1.097e11];

rntab08 = [ ...
282e3,1.695e14,7.549e14,7.540e12,3.516e10,8.242e12,1.094e11; ...
283e3,1.641e14,7.411e14,7.271e12,3.360e10,8.202e12,1.091e11; ...
284e3,1.590e14,7.276e14,7.011e12,3.212e10,8.163e12,1.088e11; ...
285e3,1.540e14,7.143e14,6.761e12,3.070e10,8.124e12,1.086e11; ...
286e3,1.492e14,7.012e14,6.521e12,2.935e10,8.085e12,1.083e11; ...
287e3,1.445e14,6.885e14,6.289e12,2.805e10,8.046e12,1.080e11; ...
288e3,1.400e14,6.759e14,6.065e12,2.682e10,8.008e12,1.078e11; ...
289e3,1.356e14,6.637e14,5.850e12,2.564e10,7.969e12,1.075e11; ...
290e3,1.314e14,6.516e14,5.643e12,2.451e10,7.931e12,1.073e11; ...
291e3,1.273e14,6.398e14,5.443e12,2.344e10,7.894e12,1.070e11; ...
292e3,1.234e14,6.282e14,5.251e12,2.241e10,7.856e12,1.067e11; ...
293e3,1.195e14,6.169e14,5.065e12,2.143e10,7.819e12,1.065e11; ...
294e3,1.158e14,6.058e14,4.886e12,2.049e10,7.782e12,1.063e11; ...
295e3,1.122e14,5.948e14,4.714e12,1.960e10,7.746e12,1.060e11; ...
296e3,1.088e14,5.841e14,4.548e12,1.874e10,7.709e12,1.058e11; ...
297e3,1.054e14,5.736e14,4.388e12,1.792e10,7.673e12,1.055e11; ...
298e3,1.021e14,5.633e14,4.234e12,1.714e10,7.637e12,1.053e11; ...
299e3,9.898e13,5.532e14,4.085e12,1.639e10,7.602e12,1.051e11; ...
300e3,9.593e13,5.433e14,3.942e12,1.568e10,7.566e12,1.049e11; ...
302e3,9.011e13,5.241e14,3.670e12,1.435e10,7.496e12,1.044e11; ...
304e3,8.466e13,5.055e14,3.418e12,1.313e10,7.427e12,1.040e11; ...
306e3,7.954e13,4.877e14,3.184e12,1.202e10,7.358e12,1.035e11; ...
308e3,7.474e13,4.705e14,2.966e12,1.100e10,7.290e12,1.031e11; ...
310e3,7.024e13,4.540e14,2.763e12,1.007e10,7.224e12,1.027e11; ...
312e3,6.602e13,4.380e14,2.574e12,9.223e09,7.157e12,1.023e11; ...
314e3,6.206e13,4.227e14,2.399e12,8.447e09,7.092e12,1.019e11; ...
316e3,5.834e13,4.079e14,2.236e12,7.737e09,7.028e12,1.015e11; ...
318e3,5.485e13,3.937e14,2.084e12,7.087e09,6.964e12,1.012e11; ...
320e3,5.158e13,3.800e14,1.942e12,6.493e09,6.901e12,1.008e11; ...
322e3,4.850e13,3.668e14,1.811e12,5.950e09,6.839e12,1.004e11];

rntab09 = [ ...
324e3,4.561e13,3.541e14,1.688e12,5.452e09,6.777e12,1.001e11; ...
326e3,4.290e13,3.418e14,1.574e12,4.997e09,6.717e12,9.971e10; ...
328e3,4.035e13,3.300e14,1.468e12,4.580e09,6.657e12,9.937e10; ...
330e3,3.796e13,3.186e14,1.369e12,4.199e09,6.597e12,9.903e10; ...
332e3,3.571e13,3.076e14,1.277e12,3.850e09,6.538e12,9.869e10; ...
334e3,3.360e13,2.970e14,1.191e12,3.530e09,6.480e12,9.836e10; ...
336e3,3.162e13,2.868e14,1.111e12,3.237e09,6.423e12,9.804e10; ...
338e3,2.975e13,2.770e14,1.037e12,2.969e09,6.366e12,9.772e10; ...
340e3,2.800e13,2.675e14,9.674e11, 2.723e09,6.310e12,9.741e10; ...
342e3,2.635e13,2.583e14,9.027e11, 2.498e09,6.254e12,9.710e10; ...
344e3,2.480e13,2.495e14,8.424e11, 2.292e09,6.199e12,9.680e10; ...
346e3,2.335e13,2.410e14,7.862e11, 2.103e09,6.145e12,9.650e10; ...
348e3,2.198e13,2.328e14,7.338e11, 1.929e09,6.091e12,9.620e10; ...
350e3,2.069e13,2.249e14,6.850e11, 1.771e09,6.038e12,9.591e10; ...
352e3,1.948e13,2.172e14,6.394e11, 1.625e09,5.985e12,9.562e10; ...
354e3,1.834e13,2.099e14,5.969e11, 1.491e09,5.933e12,9.534e10; ...
356e3,1.727e13,2.027e14,5.573e11, 1.369e09,5.881e12,9.505e10; ...
358e3,1.627e13,1.959e14,5.204e11, 1.257e09,5.830e12,9.478e10; ...
360e3,1.532e13,1.893e14,4.859e11, 1.154e09,5.779e12,9.450e10; ...
362e3,1.443e13,1.829e14,4.538e11, 1.059e09,5.729e12,9.423e10; ...
364e3,1.359e13,1.767e14,4.238e11, 9.728e08,5.680e12,9.397e10; ...
366e3,1.280e13,1.707e14,3.958e11, 8.934e08,5.631e12,9.370e10; ...
368e3,1.206e13,1.650e14,3.697e11, 8.205e08,5.582e12,9.344e10; ...
370e3,1.136e13,1.594e14,3.454e11, 7.536e08,5.534e12,9.318e10; ...
372e3,1.070e13,1.541e14,3.226e11, 6.922e08,5.487e12,9.293e10; ...
374e3,1.008e13,1.489e14,3.014e11, 6.359e08,5.439e12,9.268e10; ...
376e3,9.498e12,1.439e14,2.816e11, 5.842e08,5.393e12,9.243e10; ...
378e3,8.950e12,1.391e14,2.631e11, 5.367e08,5.347e12,9.218e10; ...
380e3,8.434e12,1.344e14,2.459e11, 4.932e08,5.301e12,9.193e10; ...
382e3,7.948e12,1.300e14,2.297e11, 4.532e08,5.256e12,9.169e10];

rntab10 = [ ...
384e3,7.490e12,1.256e14,2.147e11, 4.165e08,5.211e12,9.145e10; ...
386e3,7.059e12,1.214e14,2.006e11, 3.827e08,5.167e12,9.121e10; ...
388e3,6.653e12,1.174e14,1.875e11, 3.518e08,5.123e12,9.098e10; ...
390e3,6.271e12,1.135e14,1.753e11, 3.234e08,5.079e12,9.074e10; ...
392e3,5.911e12,1.097e14,1.638e11, 2.972e08,5.036e12,9.051e10; ...
394e3,5.572e12,1.061e14,1.532e11, 2.733e08,4.993e12,9.028e10; ...
396e3,5.253e12,1.025e14,1.432e11, 2.512e08,4.951e12,9.005e10; ...
398e3,4.952e12,9.913e13,1.339e11, 2.310e08,4.909e12,8.983e10; ...
400e3,4.669e12,9.584e13,1.252e11, 2.124e08,4.868e12,8.960e10; ...
402e3,4.402e12,9.267e13,1.170e11, 1.953e08,4.827e12,8.938e10; ...
404e3,4.151e12,8.960e13,1.094e11, 1.796e08,4.786e12,8.916e10; ...
406e3,3.914e12,8.664e13,1.023e11, 1.652e08,4.746e12,8.894e10; ...
408e3,3.691e12,8.378e13,9.568e10, 1.519e08,4.706e12,8.872e10; ...
410e3,3.480e12,8.101e13,8.948e10, 1.397e08,4.666e12,8.851e10; ...
412e3,3.282e12,7.834e13,8.369e10, 1.285e08,4.627e12,8.829e10; ...
414e3,3.095e12,7.576e13,7.827e10, 1.182e08,4.588e12,8.808e10; ...
416e3,2.919e12,7.327e13,7.321e10, 1.088e08,4.550e12,8.787e10; ...
418e3,2.754e12,7.086e13,6.848e10, 1.001e08,4.512e12,8.766e10; ...
420e3,2.597e12,6.853e13,6.406e10, 9.207e07,4.474e12,8.745e10; ...
422e3,2.450e12,6.628e13,5.993e10, 8.472e07,4.437e12,8.725e10; ...
424e3,2.311e12,6.410e13,5.606e10, 7.796e07,4.399e12,8.704e10; ...
426e3,2.180e12,6.200e13,5.245e10, 7.174e07,4.363e12,8.684e10; ...
428e3,2.057e12,5.997e13,4.907e10, 6.602e07,4.326e12,8.663e10; ...
430e3,1.940e12,5.800e13,4.592e10, 6.076e07,4.290e12,8.643e10; ...
432e3,1.831e12,5.611e13,4.297e10, 5.593e07,4.255e12,8.623e10; ...
434e3,1.727e12,5.427e13,4.020e10, 5.148e07,4.219e12,8.603e10; ...
436e3,1.630e12,5.250e13,3.762e10, 4.739e07,4.184e12,8.583e10; ...
438e3,1.538e12,5.079e13,3.521e10, 4.362e07,4.150e12,8.564e10; ...
440e3,1.451e12,4.913e13,3.295e10, 4.016e07,4.115e12,8.544e10; ...
442e3,1.369e12,4.753e13,3.084e10, 3.698e07,4.081e12,8.525e10];

rntab11 = [ ...
444e3,1.292e12,4.598e13,2.887e10, 3.404e07,4.047e12,8.505e10; ...
446e3,1.220e12,4.448e13,2.702e10, 3.135e07,4.014e12,8.486e10; ...
448e3,1.151e12,4.303e13,2.529e10, 2.887e07,3.981e12,8.467e10; ...
450e3,1.086e12,4.164e13,2.368e10, 2.658e07,3.948e12,8.448e10; ...
452e3,1.025e12,4.028e13,2.216e10, 2.448e07,3.915e12,8.429e10; ...
454e3,9.679e11, 3.898e13,2.075e10, 2.255e07,3.883e12,8.410e10; ...
456e3,9.136e11, 3.771e13,1.943e10, 2.077e07,3.851e12,8.391e10; ...
458e3,8.625e11, 3.649e13,1.819e10, 1.913e07,3.819e12,8.373e10; ...
460e3,8.142e11, 3.531e13,1.703e10, 1.762e07,3.788e12,8.354e10; ...
462e3,7.686e11, 3.416e13,1.595e10, 1.623e07,3.757e12,8.336e10; ...
464e3,7.256e11, 3.306e13,1.493e10, 1.495e07,3.726e12,8.317e10; ...
466e3,6.851e11, 3.199e13,1.398e10, 1.377e07,3.695e12,8.299e10; ...
468e3,6.468e11, 3.096e13,1.309e10, 1.269e07,3.665e12,8.281e10; ...
470e3,6.107e11, 2.996e13,1.226e10, 1.169e07,3.635e12,8.263e10; ...
472e3,5.766e11, 2.899e13,1.148e10, 1.077e07,3.605e12,8.245e10; ...
474e3,5.445e11, 2.806e13,1.076e10, 9.929e06,3.576e12,8.227e10; ...
476e3,5.142e11, 2.715e13,1.007e10, 9.149e06,3.547e12,8.209e10; ...
478e3,4.855e11, 2.628e13,9.436e09,8.432e06,3.518e12,8.191e10; ...
480e3,4.585e11, 2.543e13,8.839e09,7.771e06,3.489e12,8.173e10; ...
482e3,4.330e11, 2.461e13,8.280e09,7.162e06,3.461e12,8.155e10; ...
484e3,4.090e11, 2.382e13,7.757e09,6.602e06,3.432e12,8.138e10; ...
486e3,3.863e11, 2.306e13,7.267e09,6.085e06,3.404e12,8.120e10; ...
488e3,3.648e11, 2.232e13,6.808e09,5.609e06,3.377e12,8.103e10; ...
490e3,3.446e11, 2.160e13,6.378e09,5.171e06,3.349e12,8.085e10; ...
492e3,3.255e11, 2.091e13,5.976e09,4.767e06,3.322e12,8.068e10; ...
494e3,3.075e11, 2.024e13,5.599e09,4.395e06,3.295e12,8.051e10; ...
496e3,2.904e11, 1.959e13,5.247e09,4.052e06,3.268e12,8.034e10; ...
498e3,2.744e11, 1.896e13,4.917e09,3.737e06,3.242e12,8.017e10; ...
500e3,2.592e11, 1.836e13,4.607e09,3.445e06,3.215e12,8.000e10; ...
505e3,2.249e11, 1.693e13,3.917e09,2.814e06,3.151e12,7.959e10];

rntab12 = [ ...
510e3,1.951e11, 1.561e13,3.331e09,2.299e06,3.087e12,7.918e10; ...
515e3,1.694e11, 1.440e13,2.834e09,1.878e06,3.026e12,7.878e10; ...
520e3,1.470e11, 1.328e13,2.411e09,1.535e06,2.965e12,7.838e10; ...
525e3,1.277e11, 1.225e13,2.052e09,1.255e06,2.906e12,7.798e10; ...
530e3,1.109e11, 1.130e13,1.747e09,1.027e06,2.848e12,7.758e10; ...
535e3,9.633e10, 1.043e13,1.487e09,8.400e05,2.791e12,7.719e10; ...
540e3,8.370e10, 9.624e12,1.267e09,6.875e05,2.735e12,7.680e10; ...
545e3,7.274e10, 8.883e12,1.079e09,5.628e05,2.681e12,7.641e10; ...
550e3,6.323e10, 8.200e12,9.196e08,4.609e05,2.628e12,7.602e10; ...
555e3,5.497e10, 7.570e12,7.838e08,3.775e05,2.576e12,7.564e10; ...
560e3,4.781e10, 6.989e12,6.682e08,3.093e05,2.525e12,7.526e10; ...
565e3,4.158e10, 6.454e12,5.697e08,2.535e05,2.475e12,7.488e10; ...
570e3,3.617e10, 5.960e12,4.859e08,2.079e05,2.426e12,7.451e10; ...
575e3,3.148e10, 5.505e12,4.146e08,1.705e05,2.379e12,7.413e10; ...
580e3,2.740e10, 5.085e12,3.537e08,1.398e05,2.332e12,7.376e10; ...
585e3,2.385e10, 4.698e12,3.019e08,1.147e05,2.286e12,7.339e10; ...
590e3,2.076e10, 4.341e12,2.578e08,9.419e04,2.241e12,7.303e10; ...
595e3,1.808e10, 4.011e12,2.201e08,7.733e04,2.197e12,7.267e10; ...
600e3,1.575e10, 3.707e12,1.880e08,6.351e04,2.154e12,7.231e10; ...
605e3,1.372e10, 3.426e12,1.606e08,5.217e04,2.112e12,7.195e10; ...
610e3,1.196e10, 3.167e12,1.372e08,4.287e04,2.071e12,7.159e10; ...
615e3,1.042e10, 2.928e12,1.173e08,3.524e04,2.031e12,7.124e10; ...
620e3,9.085e09,2.707e12,1.003e08,2.898e04,1.991e12,7.089e10; ...
625e3,7.921e09,2.503e12,8.573e07,2.383e04,1.953e12,7.054e10; ...
630e3,6.908e09,2.315e12,7.332e07,1.961e04,1.915e12,7.019e10; ...
635e3,6.025e09,2.141e12,6.272e07,1.612e04,1.878e12,6.985e10; ...
640e3,5.257e09,1.981e12,5.367e07,1.328e04,1.842e12,6.950e10; ...
645e3,4.587e09,1.832e12,4.593e07,1.094e04,1.806e12,6.916e10; ...
650e3,4.003e09,1.695e12,3.932e07,9.006e03,1.771e12,6.883e10; ...
655e3,3.495e09,1.569e12,3.367e07,7.420e03,1.737e12,6.849e10 ];

rntab13 = [ ...
660e3,3.051e09,1.452e12,2.883e07,6.114e03,1.704e12,6.816e10; ...
665e3,2.665e09,1.344e12,2.470e07,5.040e03,1.671e12,6.782e10; ...
670e3,2.327e09,1.244e12,2.116e07,4.155e03,1.639e12,6.749e10; ...
675e3,2.033e09,1.151e12,1.813e07,3.427e03,1.608e12,6.717e10; ...
680e3,1.777e09,1.066e12,1.554e07,2.827e03,1.577e12,6.684e10; ...
685e3,1.553e09,9.870e11, 1.333e07,2.333e03,1.547e12,6.652e10; ...
690e3,1.357e09,9.140e11, 1.143e07,1.926e03,1.518e12,6.620e10; ...
695e3,1.187e09,8.465e11, 9.802e06,1.590e03,1.489e12,6.588e10; ...
700e3,1.038e09,7.840e11, 8.410e06,1.313e03,1.461e12,6.556e10; ...
705e3,9.075e08,7.263e11, 7.216e06,1.085e03,1.433e12,6.524e10; ...
710e3,7.939e08,6.728e11, 6.194e06,8.964e02,1.406e12,6.493e10; ...
715e3,6.946e08,6.234e11, 5.317e06,7.409e02,1.379e12,6.462e10; ...
720e3,6.078e08,5.777e11, 4.566e06,6.126e02,1.353e12,6.431e10; ...
725e3,5.320e08,5.354e11, 3.921e06,5.066e02,1.328e12,6.400e10; ...
730e3,4.658e08,4.962e11, 3.368e06,4.191e02,1.303e12,6.370e10; ...
735e3,4.078e08,4.599e11, 2.894e06,3.467e02,1.278e12,6.339e10; ...
740e3,3.572e08,4.264e11, 2.487e06,2.870e02,1.254e12,6.309e10; ...
745e3,3.129e08,3.953e11, 2.138e06,2.376e02,1.231e12,6.279e10; ...
750e3,2.741e08,3.666e11, 1.838e06,1.967e02,1.208e12,6.249e10; ...
755e3,2.402e08,3.399e11, 1.581e06,1.630e02,1.185e12,6.220e10; ...
760e3,2.105e08,3.153e11, 1.360e06,1.350e02,1.163e12,6.190e10; ...
765e3,1.845e08,2.924e11, 1.170e06,1.119e02,1.141e12,6.161e10; ...
770e3,1.618e08,2.712e11, 1.007e06,9.276e01,1.120e12,6.132e10; ...
775e3,1.419e08,2.516e11, 8.664e05,7.692e01,1.099e12,6.103e10; ...
780e3,1.244e08,2.335e11, 7.458e05,6.380e01,1.079e12,6.074e10; ...
785e3,1.092e08,2.166e11, 6.422e05,5.293e01,1.059e12,6.046e10; ...
790e3,9.577e07,2.011e11, 5.531e05,4.392e01,1.039e12,6.017e10; ...
795e3,8.404e07,1.866e11, 4.764e05,3.646e01,1.020e12,5.989e10; ...
800e3,7.377e07,1.732e11, 4.105e05,3.027e01,1.001e12,5.961e10; ...
805e3,6.476e07,1.608e11, 3.537e05,2.514e01,9.826e11, 5.933e10];

rntab14 = [ ...
810e3,5.686e07,1.493e11, 3.049e05,2.088e01,9.645e11, 5.905e10; ...
815e3,4.993e07,1.386e11, 2.628e05,1.735e01,9.468e11, 5.878e10; ...
820e3,4.386e07,1.287e11, 2.267e05,1.442e01,9.294e11, 5.851e10; ...
825e3,3.853e07,1.195e11, 1.955e05,1.199e01,9.124e11, 5.823e10; ...
830e3,3.386e07,1.110e11, 1.686e05,9.970e00,8.957e11, 5.796e10; ...
835e3,2.975e07,1.031e11, 1.455e05,8.293e00,8.793e11, 5.769e10; ...
840e3,2.615e07,9.580e10, 1.256e05,6.900e00,8.632e11, 5.743e10; ...
845e3,2.299e07,8.901e10, 1.084e05,5.742e00,8.475e11, 5.716e10; ...
850e3,2.022e07,8.270e10, 9.358e04,4.780e00,8.320e11, 5.690e10; ...
855e3,1.778e07,7.685e10, 8.081e04,3.980e00,8.169e11, 5.664e10; ...
860e3,1.564e07,7.142e10, 6.979e04,3.314e00,8.021e11, 5.637e10; ...
865e3,1.376e07,6.638e10, 6.029e04,2.761e00,7.875e11, 5.612e10; ...
870e3,1.211e07,6.171e10, 5.210e04,2.301e00,7.733e11, 5.586e10; ...
875e3,1.066e07,5.737e10, 4.503e04,1.918e00,7.593e11, 5.560e10; ...
880e3,9.380e06,5.334e10, 3.892e04,1.599e00,7.456e11, 5.535e10; ...
885e3,8.258e06,4.959e10, 3.365e04,1.333e00,7.321e11, 5.509e10; ...
890e3,7.271e06,4.612e10, 2.910e04,1.112e00,7.189e11, 5.484e10; ...
895e3,6.404e06,4.289e10, 2.517e04,9.277e-1,7.060e11, 5.459e10; ...
900e3,5.641e06,3.989e10, 2.177e04,7.742e-1,6.933e11, 5.434e10; ...
905e3,4.970e06,3.711e10, 1.884e04,6.462e-1,6.809e11, 5.410e10; ...
910e3,4.379e06,3.452e10, 1.631e04,5.396e-1,6.687e11, 5.385e10; ...
915e3,3.859e06,3.212e10, 1.411e04,4.506e-1,6.567e11, 5.361e10; ...
920e3,3.402e06,2.989e10, 1.222e04,3.764e-1,6.450e11, 5.336e10; ...
925e3,2.999e06,2.781e10, 1.058e04,3.145e-1,6.335e11, 5.312e10; ...
930e3,2.645e06,2.588e10, 9.165e03,2.629e-1,6.222e11, 5.288e10; ...
935e3,2.332e06,2.409e10, 7.940e03,2.197e-1,6.111e11, 5.264e10; ...
940e3,2.057e06,2.242e10, 6.880e03,1.837e-1,6.003e11, 5.241e10; ...
945e3,1.815e06,2.088e10, 5.962e03,1.537e-1,5.896e11, 5.217e10; ...
950e3,1.602e06,1.944e10, 5.168e03,1.286e-1,5.792e11, 5.194e10; ...
955e3,1.414e06,1.810e10, 4.481e03,1.076e-1,5.689e11, 5.170e10];

rntab15 = [ ...
960e3,1.248e06,1.685e10, 3.886e03,9.004e-2,5.589e11, 5.147e10; ...
965e3,1.102e06,1.569e10, 3.370e03,7.538e-2,5.490e11, 5.124e10; ...
970e3,9.726e05,1.462e10, 2.924e03,6.312e-2,5.393e11, 5.101e10; ...
975e3,8.590e05,1.362e10, 2.537e03,5.287e-2,5.298e11, 5.078e10; ...
980e3,7.587e05,1.268e10, 2.201e03,4.430e-2,5.205e11, 5.056e10; ...
985e3,6.703e05,1.182e10, 1.911e03,3.712e-2,5.114e11, 5.033e10; ...
990e3,5.922e05,1.101e10, 1.659e03,3.111e-2,5.024e11, 5.011e10; ...
995e3,5.234e05,1.026e10, 1.440e03,2.609e-2,4.936e11, 4.989e10; ...
100e4,4.626e05,9.562e09,1.251e03,2.188e-2,4.850e11, 4.967e10; ...
100e4,0.    ,0.    ,0.    ,0.     ,0.    ,0.];

rntab = [rntab01;rntab02;rntab03;rntab04;rntab05; ...
          rntab06;rntab07;rntab08;rntab09;rntab10; ...
          rntab11;rntab12;rntab13;rntab14;rntab15];

//************************************************************************

//COMPUTATION

//data control
if find(alt>1000e3)~=[] then CL__error('altitude bigger than 1000 km'); end //error if alt bigger than 1000e3
i_zero = find(alt < 0)  //alt set to zero if negative
alt(i_zero) = 0;


i_alt1 = find(alt <= 86e3)  //get indexs of altitudes lower or equal than 86e3
i_alt2 = find(alt > 86e3)   //get indexs of altitudes bigger than 86e3

if i_alt1~=[]
  //calcul de temp (t), pres (p) et xmol (m) pour alt < ou = 86 km
  rhpot(i_alt1) = r0 .* alt(i_alt1) ./ (r0+alt(i_alt1))    //calcul de l'alt geopotentielle h [equation 18]

  //recherche de l'intervalle d'alt encadrant rhpot tel que: htab(ii) <= rhpot < htab(ii+1)
  ii = 7 * ones(1,N)
  i = 2 * ones(1,N);
  kk = find( (rhpot>=htab(i)) & (alt <= 86e3));
  while (kk~=[])&(find(i>7)==[])
    i(kk) = i(kk)+1;
    kk = find( (rhpot>=htab(i)) & (alt <= 86e3));
  end
  ii = i-1;

  tmol(i_alt1) = tmtab(ii(i_alt1)) + dttab(ii(i_alt1)) .* (rhpot(i_alt1)-htab(ii(i_alt1)))  //calcul de la temp moleculaire tm [equation 23]

  // calcul de la temp cinetique t et de la masse molaire m:
  i1 = find( (alt < 80e3) & (alt <= 86e3) ) // pour des alts z < 80 km
  i2 = find( (alt >= 80e3) & (alt <= 86e3) ) // pour des alts z > ou = 80 km

  if i1~=[]
    temp(i1) = tmol(i1)
    xmol(i1) = xmol0
  end //end i1

  if i2~=[]
    //recherche de l'intervalle d'altitude encadrant alt tel que:80km + 500m*(j-1) < alt <= 80km + 500m*j
    jj = 11 * ones(1,N)
    j = 1 * ones(1,N);
    zj = 80000 + j .* 500;
    kk = find( (alt > zj) & (alt >= 80e3) & (alt <= 86e3) );
    //j(kk) = j(kk)+1;
    while (kk~=[])&(find(j>11)==[])
      zj = 80000 + j .* 500;
      kk = find( (alt > zj) & (alt >= 80e3) & (alt <= 86e3) );
      j(kk) = j(kk)+1;
    end
    jj = j-1;

    // interpolation (en fonction de l'alt) du rapport m/m0:
    coef(i2) = xmm0tb(1+jj(i2)) + ( xmm0tb(1+jj(i2)+1)-xmm0tb(1+jj(i2)) ) ./ 500 .* ( alt(i2)-zj(i2)+500 )
    temp(i2) = tmol(i2) .* coef(i2)  //calcul de la temp cinetique t (avec m/m0 interpole) [equation 22]
    xmol(i2) = xmol0 .* coef(i2) //calcul de la masse molaire m (interpolee)
  end //end i2

  //  calcul de la pres p:
  //  test sur le gradient de temp
  i3 = find( ( abs(dttab(ii)) < epspre ) & (alt <= 86e3))
  i4 = find( ( abs(dttab(ii)) >= epspre ) & (alt <= 86e3))

  //if i3
    pres(i3) = ptab(ii(i3)) .* exp( -rappor .* ( rhpot(i3) - htab(ii(i3)) )./tmtab(ii(i3)) ) //[equation 33b]
  //else if i4
    pres(i4) = ptab(ii(i4)) .* ( tmtab(ii(i4))./tmol(i4) ).^( rappor./dttab(ii(i4)) )        //[equation 33a]
  //end i3 i4

end //end i_alt1

if i_alt2~=[]
  //calcul de temp (t), pres (p) et xmol (m) pour alt > 86 km
  //calcul de la temp cinetique t:
  i1 = find( (alt < ztab(1+8)) & (alt > 86e3) )
  i2 = find( (alt < ztab(1+9)) & (~(alt < ztab(1+8))) & (alt > 86e3) )
  i3 = find( (alt < ztab(1+10)) & (~(alt < ztab(1+8))) & (~(alt < ztab(1+9))) & (alt > 86e3) )
  i4 = find( (~(alt < ztab(1+8))) & (~(alt < ztab(1+9))) & (~(alt < ztab(1+10))) & (alt > 86e3) )

  //if i1
    temp(i1) = ttab(1+7)  //[equation 25]
  //elseif i2
    rcar = sqrt( 1-( ( alt(i2)-ztab(1+8) )./pta ).^2 )  //[equation 27]
    temp(i2) = tc + gda_ .* rcar
  //elseif i3
    temp(i3) = ttab(1+9) + dttab(1+9) .* (alt(i3)-ztab(1+9))
  //else (i4)
    xi = (alt(i4)-ztab(1+10)) .* (r0+ztab(1+10)) ./ (r0+alt(i4))  //[equation 31]
    temp(i4) = ttab(1+12) - (ttab(1+12)-ttab(1+10)) .* exp(-rlambd*xi)
  //end i1 i2 i3 i4

  //calcul des number densities "Ni" pour chaque element N2,O,O2,Ar,He et H:

  //recherche de l'intervalle d'alt encadrant alt tel que
  //      rntab(kk,1) < alt <= rntab(kk+1,1)
  //On effectue cette recherche par dichotomie
  kmoins = 1 * ones(1,N)
  k_dicho = nmax/2 * ones(1,N)
  pas_dicho = k_dicho/2
  mm = find( (pas_dicho>0) & (alt > 86e3) )
  while mm~=[]
    mm1 = find( (pas_dicho>0) & (alt<=rntab(k_dicho,1)') & (alt > 86e3) )
    mm2 = find( (pas_dicho>0) & (alt>rntab(k_dicho,1)') & (alt > 86e3) )
    //if mm1
      k_dicho(mm1) = k_dicho(mm1) - pas_dicho(mm1)
    //else (mm2)
      kmoins(mm2) = k_dicho(mm2)
      k_dicho(mm2) = k_dicho(mm2) + pas_dicho(mm2)
    //end mm1 mm2
    pas_dicho(mm) = pas_dicho(mm)/2
    mm = find( (pas_dicho>0) & (alt > 86e3) )
  end

  kk = floor(kmoins) //round to the immediate lower integer because accessing matrixs: rntab(4.999) is the same as rntab(4)
  mm = find( (rntab(kk+1,1)'<alt) & (alt > 86e3) )
  while mm~=[]
    kk(mm) = kk(mm)+1
    mm = find( (rntab(kk+1,1)'<alt) & (alt > 86e3) )
  end

  pen = zeros(7,N)
  rnn = zeros(7,N)
  pen(2:7,i_alt2) = ((rntab(kk(i_alt2)+1,2:7)-rntab(kk(i_alt2),2:7)) ./ ((rntab(kk(i_alt2)+1,1)-rntab(kk(i_alt2),1)).*.ones(1,6)))'
  rnn(2:7,i_alt2) = (rntab(kk(i_alt2),2:7) + pen(2:7,i_alt2)' .* ((alt(i_alt2)'-rntab(kk(i_alt2),1)).*.ones(1,6)))'   //interpolation suivant l'alt pour chaque "ni"
  xmol(i_alt2) = CL_dot(rnn(2:7,i_alt2),(rmoltb(2:7)').*.ones(1,size(i_alt2,2)))                  //contribution a la somme totale de chaque "ni"
  rn = sum(rnn(2:7,i_alt2),1)

  xmol(i_alt2) = xmol(i_alt2)./rn   //calcul de la masse molaire m [equation 20]

  pres(i_alt2) = rn.*boltz.*temp(i_alt2)           //calcul de la pres p [equation 33c]

end //end i_alt2

dens = (pres.*xmol) ./ (rstar.*temp)   //calcul de la dens rho  [equation 42]

endfunction
