/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <helper.h>
#include <charsets.h>
#include "smscoding.h"
#include <intincl.h>

#include <string.h>

char* sms_data_gsm_encode (ucs4char_t* input,
			   uint8_t headersize,
			   uint8_t* enc_len)
{
  char* retval;
  gsmchar_t* temp;
  unsigned int state, inc, outc;
  unsigned char first, end;

  if ((enc_len == NULL && ucs4len(input) > 0) ||
      headersize > 140) {
    return NULL;
  }

  temp = convert_to_gsm(input);
  if (enc_len != NULL) *enc_len = strlen(temp);

  retval = mem_alloc((140*2) + 1, 1);
  if (headersize) memset(retval,' ',headersize*2);

  if (ucs4len(input) > 0) {
    //7bit-GSM octet encoding
    state = (headersize*8)/7;
    if (headersize%7 != 0) ++state;
    outc = headersize*2;
    for (inc = 0; inc < *enc_len && outc <= (140-1)*2 ; ++inc) {
      if (state != 7) {
	state %= 8;
	//the character are 7bit
	//shifting dependent on state (0-7)
	first = ((temp[inc+1] & 0x7F) << (7-state)) & 0xFF;
	end = (temp[inc] & 0x7F) >> state;
	//storing as hexstring
	sprintf (retval + outc, "%02X", (first | end));
	outc += 2;
      }
      ++state;
    }

    //Return NULL to indicate error during encoding.
    //In this case, input string was too large to fit.
    if (inc != *enc_len) {
      mem_realloc(retval,0);
      return NULL;
    }
  }

  mem_realloc(temp,0);
  return retval;
}

char* sms_data_ucs2_encode (ucs4char_t* input,
			    uint8_t headersize,
			    uint8_t* enc_len)
{
  char* retval;

  if ((enc_len == NULL && ucs4len(input) > 0) ||
      headersize > 140) {
    return NULL;
  }

  retval = convert_to_ucs2_hexstring(input);
  if (enc_len != NULL) {
    *enc_len = strlen(retval)/2;
  
    //Return NULL to indicate error during encoding.
    //In this case, input string was too large to fit.
    if (*enc_len + headersize > 140) {
      mem_realloc(retval,0);
      return NULL;
    }
  }

  retval = mem_realloc(retval,(140*2)+1);

  if (headersize) {
    retval = memmove(&retval[headersize*2],retval,strlen(retval)+1);
    memset(retval,' ',headersize*2);
  }

  return retval;
}

char* sms_data_encode (enum sms_encoding charset,
		       ucs4char_t* input,
		       uint8_t* enc_len,
		       uint8_t headersize)
{
  char* retval = NULL;
  char hlen[3];

  switch (charset) {
  case SMS_CHARSET_GSM:
    retval = sms_data_gsm_encode(input,headersize,enc_len);
    break;
  case SMS_CHARSET_UCS2:
    retval = sms_data_ucs2_encode(input,headersize,enc_len);
    break;
  case SMS_CHARSET_8BIT:
    //no break
  default:
    return NULL; //not supported
  }
  
  if (retval != NULL && headersize > 0) {
    sprintf(hlen,"%02X",headersize);
    strncpy(retval,hlen,2);
  }
  return retval;
}
