/*  Screem:  screem-helper.c
 *
 *  Copyright (C) 2002 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>


#include <unistd.h>
#include <fcntl.h>

#include <ctype.h>

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <errno.h>

#include <glade/glade.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <glib.h>

#include "screem-window.h"
#include "screem-helper.h"
#include "screem-page.h"

/* needed for special case */
#include "screem-cvs.h"

#include "fileops.h"
#include "support.h"

#include "screemmarshal.h"


enum {
	PROP_BOGUS,
	PROP_PATHNAME,
	PROP_ICONPATH,
	PROP_HELPER_NAME,
	PROP_INPUT_MODE,
	PROP_OUTPUT_MODE,
	PROP_EXEC_ON_SAVE,
	PROP_EXEC_ON_SAVE_TYPE,
	PROP_UIPATH,
	PROP_FILE
};

enum {
	HELPER_FAILED,
	HELPER_ERROR,
	LAST_SIGNAL
};
static guint screem_helper_signals[LAST_SIGNAL] = { 0 };

typedef struct {
	ScreemHelper *helper;
	
	ScreemPage *page;

	gint std_out;
	gint std_err;
	gint watch;
	GPid pid;

	GIOChannel *out;
	GIOChannel *err;
	gint owatch;
	gint ewatch;
	gint pulse;

	GString *input;
	ScreemWindow *window;
	GtkWidget *dialog;
} ScreemHelperData;

struct ScreemHelperPrivate {
	/* a list of ScreemWindows */
	GSList *list;

	/* set to FALSE if the helper .desktop file contains
	   a TryExec field and the executable doesn't exist */
	gboolean exists;

	gchar *name;
	gchar *pathname;
	gchar *iconpath;

	HelperInputMode imode;
	HelperOutputMode omode;

	gboolean exec;
	gchar *exec_type;

	gchar *uipath;
		
	gboolean nonblocking;
	
	/* file that the helper was loaded from */
	gchar *filename;

	/* a list of ScreemHelperData for running instances
	 * of the helper */
	GSList *running;
};

static void screem_helper_verb_cb( GtkAction *action, gpointer user_data );

static void child_setup( gpointer data );
static void child_watch( GPid pid, gint status, gpointer data );
static gboolean child_stdout( GIOChannel *source,
		GIOCondition condition, gpointer data );
static gboolean child_stderr( GIOChannel *source,
		GIOCondition condition, gpointer data );
static gboolean screem_helper_pulse( ScreemHelperData *hdata );
static gboolean exec_exists( const char *exec );
static gchar *screem_helper_build_action_name( const gchar *name );



ScreemHelper* screem_helper_new( const gchar *name, 
				 const gchar *pathname,
				 const gchar *iconpath,
				 HelperInputMode imode,
				 HelperOutputMode omode )
{
	ScreemHelper *helper;
	GType type;
	
	type = screem_helper_get_type();

	helper = SCREEM_HELPER( g_object_new( type, 
					      "name", name,
					      "pathname", pathname,
					      "iconpath", iconpath,
					      "input", imode,
					      "output", omode,
					      NULL ) );

	return helper;
}

ScreemHelper *screem_helper_new_from_file( const gchar *file )
{
	ScreemHelper *helper;

	GKeyFile *entries;
	GError *error;
	
	gchar *name;
	gchar *pathname;
	gchar *icon;
	HelperInputMode imode;
	HelperOutputMode omode;
	gboolean exec;
	gchar *type;
	gchar *uipath;

	gchar *tryexec;
	gboolean exists;
	gchar *tmp;
	gint argc;
	gchar **argv;

	helper = NULL;
	error = NULL;
	entries = g_key_file_new();
	if( g_key_file_load_from_file( entries, file,
				G_KEY_FILE_NONE, &error ) ) {
		tryexec = g_key_file_get_string( entries,
				"Desktop Entry",
				"TryExec",
				&error );
		if( error ) {
			g_error_free( error );
		}
	
		error = NULL;
		name = g_key_file_get_string( entries,
				"Desktop Entry",
				"Name",
				&error );
		if( error ) {
			g_error_free( error );
		}

		error = NULL;
		pathname = g_key_file_get_string( entries,
				"Desktop Entry",
				"Exec",
				&error );
		if( error ) {
			g_error_free( error );
		}
		
		error = NULL;
		icon = g_key_file_get_string( entries,
				"Desktop Entry",
				"Icon",
				&error );
		if( error ) {
			g_error_free( error );
		}

		error = NULL;
		imode = g_key_file_get_integer( entries,
				"Desktop Entry",
				"X-SCREEM-imode",
				&error );
		if( error ) {
			g_error_free( error );
		}

		error = NULL;
		omode = g_key_file_get_integer( entries,
				"Desktop Entry",
				"X-SCREEM-omode",
				&error );
		if( error ) {
			g_error_free( error );
		}
		
		error = NULL;
		exec = g_key_file_get_boolean( entries,
				"Desktop Entry",
				"X-SCREEM-exec",
				&error );
		if( error ) {
			g_error_free( error );
		}

		error = NULL;
		type = g_key_file_get_string( entries,
				"Desktop Entry",
				"X-SCREEM-exectype",
				&error );
		if( error ) {
			g_error_free( error );
		}

		error = NULL;
		uipath = g_key_file_get_string( entries,
				"Desktop Entry",
				"X-SCREEM-uipath",
				&error );
		/* ignore missing X-SCREEM-uipath */
		if( error ) {
			g_error_free( error );
			error = NULL;
		}
		
		helper = screem_helper_new( name, pathname,
				icon, imode, omode );

		g_object_set( G_OBJECT( helper ),
				"exec_on_save", exec,
				"exec_type", type,
				"uipath", uipath,
				NULL );

		if( tryexec ) {
			exists = exec_exists( tryexec );
			g_free( tryexec );
		} else {
			/* problem: this prevents helpers that use a relative path from
			   being active */
			if( ! g_shell_parse_argv( pathname, &argc, &argv, NULL ) ) {
				exists = FALSE;	
			} else if( argc < 1 ) {
				g_strfreev( argv );
				exists = FALSE;
			} else {
				exists = exec_exists( argv[ 0 ] );
				g_strfreev( argv );
			}
		}
		helper->priv->exists = exists;

		error = NULL;
		tmp = g_key_file_get_string( entries,
				"Desktop Entry",
				"X-SCREEM-nonblocking",
				&error );
		/* ignore missing X-SCREEM-nonblocking */
		if( error ) {
			g_error_free( error );
			error = NULL;
		}
		helper->priv->nonblocking = ( tmp && *tmp == '1' );
		g_free( tmp );
		
		g_free( name );
		g_free( pathname );
		g_free( icon );
		g_free( type );	
		g_free( uipath );
		
	}	
	
	if( error ) {
		g_printerr( "Failed to load helper %s : %s\n",
				file, error->message );
		g_error_free( error );
	} else {
		helper->priv->filename = g_strdup( file );
	}
	
	g_key_file_free( entries );

	return helper;
}

gboolean screem_helper_save( ScreemHelper *helper )
{
	gchar *pathname;
	gchar *tmp;

	gchar *name;
	gchar *path;
	gchar *icon;
	HelperInputMode imode;
	HelperOutputMode omode;

	gboolean exec;
	gchar *type;
	
	GString *output;
	gboolean ret;
	
	tmp = screem_get_dot_dir();
	pathname = g_build_path( G_DIR_SEPARATOR_S, tmp,
			"helpers", NULL );
	g_free( tmp );

	ret = FALSE;
	
	if( mkdir_recursive( pathname, 
			GNOME_VFS_PERM_USER_ALL, NULL, NULL ) ) { 
		g_object_get( G_OBJECT( helper ), 
				"name", &name, 
				"pathname", &path,
				"iconpath", &icon, 
				"input", &imode, 
				"output", &omode,
				"exec_on_save", &exec,
				"exec_type", &type,
				NULL );

		tmp = g_strconcat( pathname, G_DIR_SEPARATOR_S,
				name, ".desktop", NULL );
	
		output = g_string_new( "[Desktop Entry]\n" );
		g_string_append_printf( output, "Name=%s\n", name );
		g_string_append_printf( output, "Exec=%s\n", path );
		g_string_append( output, "Comment=Screem Helper\n" );
		g_string_append_printf( output, "Icon=%s\n", icon );
		g_string_append( output, "Type=Application\n" );
		g_string_append( output, "Terminal=0\n" );
		g_string_append_printf( output,
				"X-SCREEM-imode=%i\n", imode );
		g_string_append_printf( output,
				"X-SCREEM-omode=%i\n", omode );
		g_string_append_printf( output,
				"X-SCREEM-exec=%i\n", exec );
		g_string_append_printf( output,
				"X-SCREEM-exectype=%s\n", type );

		ret = save_file( tmp, output->str,
				GNOME_VFS_PERM_USER_ALL, 
				FALSE, TRUE, NULL );
		g_string_free( output, TRUE );
		g_free( name );
		g_free( path );
		g_free( icon );
		g_free( type );
		
		g_free( tmp );
	} 	
	g_free( pathname );

	return ret;
}

void screem_helper_execute( ScreemHelper *helper, 
		ScreemPage *page, ScreemWindow *window )
{
	ScreemHelperPrivate *priv;
	const gchar *pathname;
	const gchar *pagepath;
	gchar *dirname;

	gint argc;
	gchar **argv;
	
	GPid pid;
	gint std_in;
	FILE *out;
	
	GError *error;
	const gchar *charset;
	
	GladeXML *xml;
	GtkWidget *widget;
	
	gchar *label;
	gchar *tmp;

	gint button;

	gchar *data;
	gint flags;

	
	const gchar *exec;
	GString *path;
	gunichar c;

	ScreemPage *wpage;
	
	ScreemSite *site;
	const gchar *sitepath;
	const gchar *remote;

	GSpawnFlags spawnflags;

	ScreemHelperData *newdata;
	
	priv = helper->priv;

	wpage = NULL;
	site = NULL;
	if( window ) {
		screem_window_clear_messages( window );
		screem_window_clear_errors( window );
		wpage = screem_window_get_document( window );
		site = screem_window_get_current( window );
	}
	
	if( wpage && ! page ) {
		page = wpage;
	}
	
	if( ( ! page ) &&  
	    ( priv->imode != SCREEM_HELPER_STDIN_NONE &&
	      priv->omode != SCREEM_HELPER_STDOUT_NONE ) ) {
		return;
	}

	sitepath = NULL;
	if( site ) {
		sitepath = screem_site_get_pathname( site );
		/* strip file: from local methods */
		if( sitepath ) {
			if( ! strncmp( "file://", sitepath,
				strlen( "file://" ) ) ) {
				sitepath += strlen( "file://" );	
			}
		}
	}
	
	/* get pathname, and strip file: from local methods */
	if( page ) {
		pathname = pagepath = screem_page_get_pathname( page );
	} else {
		pathname = pagepath = NULL;
	}
	dirname = NULL;
	if( pagepath ) {
		/* need local path for working directory, for
		 * remote files we stay in the current dir */
		if( ! strncmp( "file://", pagepath,
					strlen( "file://" ) ) ) {
			pagepath += strlen( "file://" );
		
			dirname = g_path_get_dirname( pagepath );
		}
	}

	/* special case CVS helper */
	if( SCREEM_IS_CVS( helper ) ) {
		g_free( dirname );
		dirname = NULL;	
	}
	
	remote = screem_site_get_http_url( site );
	
	/* replace pathnames in the pattern to exec */
	path = g_string_new( NULL );
	exec = priv->pathname;
	do {
		tmp = strchr( exec, '%' );
		if( tmp ) {
			g_string_append_len( path, exec,
					tmp - exec );

			tmp = g_utf8_next_char( tmp );
			c = g_utf8_get_char( tmp );
			switch( c ) {
				case 'S':
					if( sitepath ) {
						g_string_append_printf( path, "\"%s\"", 
							sitepath );
					}
					break;
				case 'U':
					if( pathname ) {
						g_string_append_printf( path, "\"%s\"",
							pathname );
					}
					break;
				case 'f':
					if( pagepath ) {
						g_string_append_printf( path, "\"%s\"",
								pagepath );
					}
					break;
				case 'd':
					if( dirname ) {
						g_string_append_printf( path, "\"%s\"",
								dirname );
					}
					break;
				case 'p':
					if( remote && pagepath && sitepath ) {
						g_string_append_printf( path, "\"%s%s\"",
							remote, pagepath + strlen( sitepath ) );
					}
					break;
				case '%':
					g_string_append_unichar( path,
							c );
					break;
				default:
					/* unknown escape, leave it in */
					g_string_append_unichar( path,
							c );
					break;
				
			}
			tmp = g_utf8_next_char( tmp );
		} else {
			g_string_append( path, exec );
		}
		exec = tmp;
	} while( exec );

	error = NULL;
	if( ! g_shell_parse_argv( path->str, &argc, &argv, &error ) ) {
		g_string_free( path, TRUE );
		if( error ) {
			if( window ) {
				screem_window_show_error( window, 
						error->message, TRUE );
			}
			g_error_free( error );
		} else if( window ) {
			screem_window_show_error( window,
					path->str, TRUE );
		}
		return;
	}
	g_string_free( path, TRUE );
	
	newdata = g_new0( ScreemHelperData, 1 );
	newdata->page = page;
	newdata->window = window;
	
	spawnflags = G_SPAWN_SEARCH_PATH;
	spawnflags |= G_SPAWN_DO_NOT_REAP_CHILD;
	
	if( ! g_spawn_async_with_pipes( dirname,
			argv, NULL, spawnflags,
			child_setup,
			helper,
			&pid,
			&std_in,
			&newdata->std_out,
			&newdata->std_err,
			&error ) ) {
		/* error starting helper */
		g_signal_emit( G_OBJECT( helper ),
			       screem_helper_signals[ HELPER_FAILED ],
			       0, error->message );
		g_strfreev( argv );
		if( error ) {
			if( window ) {
				screem_window_show_error( newdata->window, error->message,
					TRUE );
			}
			g_error_free( error );
		}

		/* free newdata up, it hasn't gone into the list
		 * yet */
		g_free( newdata );
		
		return;
	}
	g_strfreev( argv );
	
	newdata->pid = pid;
	
	flags = fcntl( newdata->std_out, F_GETFL, 0 );
	if( flags != -1 ) {
		fcntl( newdata->std_out, F_SETFL, flags | O_NONBLOCK );
	} else {
		/* free newdata up, it hasn't gone into the list
		 * yet */
		g_free( newdata );
		
		/* FIXME: close pipes */
		return;
	}
	flags = fcntl( newdata->std_err, F_GETFL, 0 );
	if( flags != -1 ) {
		fcntl( newdata->std_err, F_SETFL, flags | O_NONBLOCK );
	} else {
		/* free newdata up, it hasn't gone into the list
		 * yet */
		g_free( newdata );

		/* FIXME: close pipes */
		return;
	}
	
	switch( priv->imode ) {
		case SCREEM_HELPER_STDIN_SELECTION:
			out = fdopen( std_in, "w" );
			if( page == wpage ) {
				data = screem_window_get_selection( window );
			} else {
				data = screem_page_get_data( page );
			}
			/* FIXME: convert to system locale */

			fprintf( out, "%s", data );
			g_free( data );
			fclose( out );
			break;
		case SCREEM_HELPER_STDIN_USER:
			out = fdopen( std_in, "w" );
			xml = glade_xml_new( GLADE_PATH"/screem.glade", 
					"helper_user_input", NULL );
			widget = glade_xml_get_widget( xml, 
					"helper_name" );
			gtk_label_set_text( GTK_LABEL( widget ), 
					priv->name );
			widget = glade_xml_get_widget( xml, 
							"helper_user_input" );
			if( gtk_dialog_run( GTK_DIALOG( widget ) ) == GTK_RESPONSE_OK ) {
				GtkWidget *widg;

				widg = glade_xml_get_widget( xml, 
							     "use_selection" );
				if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widg ) ) ) {
					if( wpage == page ) {
						data = screem_window_get_selection( window );
					} else {
						data = screem_page_get_data( page );
					}
				} else {
					GtkTextBuffer *buffer;
					GtkTextIter it;
					GtkTextIter eit;
					
					widg = glade_xml_get_widget( xml, "user_data" );
					buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( widg ) );
					gtk_text_buffer_get_bounds( buffer, &it, &eit );
					data = gtk_text_buffer_get_text( buffer, &it, &eit, TRUE );
				}
				/* FIXME: convert to locale */

				fprintf( out, "%s", data );
				g_free( data );
			} else {
				/* canceled, stop the helper */
				kill( pid, SIGKILL );
			}
			fclose( out );
			gtk_widget_destroy( widget );
		
			g_object_unref( xml );

			break;
		case SCREEM_HELPER_STDIN_NONE:
			break;
	}
	close( std_in );

	newdata->input = g_string_new( NULL );
	
	/* we want to listen for child exits */
	newdata->watch = g_child_watch_add( pid, child_watch, newdata );

	/* listen to priv->std_out and priv->std_err */
	newdata->out = g_io_channel_unix_new( newdata->std_out );
	newdata->err = g_io_channel_unix_new( newdata->std_err );

	if( ! g_get_charset( &charset ) ) {
		g_io_channel_set_encoding( newdata->out, charset, &error );
		if( error ) {
			g_error_free( error );
			error = NULL;
		}
		g_io_channel_set_encoding( newdata->err, charset, &error );
		if( error ) {
			g_error_free( error );
			error = NULL;
		}
	}

	newdata->helper = helper;
	newdata->owatch = g_io_add_watch( newdata->out, 
			G_IO_IN | G_IO_ERR | G_IO_HUP, 
			child_stdout, newdata );
	newdata->ewatch = g_io_add_watch( newdata->err, 
			G_IO_IN | G_IO_ERR | G_IO_HUP, 
			child_stderr, newdata );
	
	priv->running = g_slist_prepend( priv->running, newdata );
	
	if( ! priv->nonblocking ) {
		/* display helper running dialog */
		xml = glade_xml_new( GLADE_PATH"/screem.glade", "helper_run", NULL );

		widget = glade_xml_get_widget( xml, "helperlabel" );
		tmp = g_strdup_printf( "Running %s", priv->name );
		label = g_strconcat( "<span weight=\"bold\" size=\"large\">",
				tmp, "</span>", NULL );
		g_free( tmp );
		gtk_label_set_markup( GTK_LABEL( widget ), label );
		g_free( label );
	
		widget = glade_xml_get_widget( xml, "helperprogress" );
		newdata->pulse = g_timeout_add( 250, (GSourceFunc)screem_helper_pulse,
			newdata );
	
		widget = glade_xml_get_widget( xml, "helper_run" );

		newdata->dialog = widget;

		do {
			button = gtk_dialog_run( GTK_DIALOG( widget ) );
			switch( button ) {
				case GTK_RESPONSE_CANCEL:
					kill( pid, SIGKILL );
					break;
				default:
					break;
			}
		} while( button != GTK_RESPONSE_CLOSE );
	
		gtk_widget_destroy( widget );
	}
	g_free( dirname );
}

void screem_helper_add_to_window( ScreemHelper *helper, ScreemWindow *window )
{
	ScreemHelperPrivate *priv;
	gchar *name;
	gchar *pathname;
	gchar *iconpath;

	gchar *uiname;
	GtkAction *action;
	guint merge_id;
	gchar *tmp;

	GString *ui;
	const gchar *uipath;
	gchar **paths;
	gchar **tpaths;
	GtkActionGroup *group;
	gchar *accel_path;

	GtkAction *subaction;
	
	g_return_if_fail( window != NULL );
	
	priv = helper->priv;
	
	priv->list = g_slist_append( priv->list, window );
	
	g_object_get( G_OBJECT( helper ),
		      "name", &name, "pathname", &pathname, 
		      "iconpath", &iconpath, NULL );

	uiname = screem_helper_build_action_name( name );
	
	tmp = screem_escape_underlines( name );
	action = gtk_action_new( uiname, tmp, pathname, 
			GTK_STOCK_EXECUTE );
	gtk_action_set_sensitive( action, priv->exists );
	
	g_free( tmp );
	g_signal_connect( G_OBJECT( action ), "activate",
			G_CALLBACK( screem_helper_verb_cb ), helper );

	accel_path = g_strconcat( "<Actions>", "/ScreemHelperActions/",
			name, NULL );
	gtk_action_set_accel_path( action, accel_path );
	g_free( accel_path );
	
	group = GTK_ACTION_GROUP( window->action_group );
	gtk_action_group_add_action( group, action );


	/* build ui path */
	uipath = priv->uipath;
	if( ( ! uipath ) || *uipath == '\0'  ) {
		uipath = "Tools/Helpers";
	}
	ui = g_string_new( "<ui><menubar>" );
	paths = g_strsplit( uipath, "/", -1 );
	for( tpaths = paths; tpaths && *tpaths; tpaths ++ ) {
		if( **tpaths ) {
			g_string_append_printf( ui,
					"<menu action=\"%s\">",
					*tpaths );

			/* do we have an action *tpaths ? if
			 * not create one */
			subaction = gtk_action_group_get_action( group,
					*tpaths );
			if( ! subaction ) {
				tmp = screem_escape_underlines( *tpaths );
				subaction = gtk_action_new( *tpaths,
						tmp, "", "" );
				gtk_action_group_add_action( group, 
						subaction );
			}
		}
	}
	g_string_append_printf( ui, "<menuitem action=\"%s\" />",
			uiname );
	for( tpaths = paths; tpaths && *tpaths; tpaths ++ ) {
		if( **tpaths ) {
			g_string_append( ui, "</menu>" );
		}
	}
	g_strfreev( paths );
	g_string_append( ui, "</menubar></ui>" );
	
	merge_id = gtk_ui_manager_add_ui_from_string( GTK_UI_MANAGER( window->merge ),
							ui->str, 
							ui->len,
							NULL );
	g_object_set_data( G_OBJECT( action ), "merge_id",
				GUINT_TO_POINTER( merge_id ) );

	g_object_set_data( G_OBJECT( action ), "window", window );

	g_free( uiname );
	g_string_free( ui, TRUE );

	g_free( pathname );
	g_free( name );
	g_free( iconpath );
	
}

void screem_helper_remove_from_window( ScreemHelper *helper,
				       ScreemWindow *window )
{
	gchar *name;
	gchar *uiname;
	GtkAction *action;
	guint merge_id;
	
	g_return_if_fail( SCREEM_IS_HELPER( helper ) );
	
	helper->priv->list = g_slist_remove( helper->priv->list, 
						window );

	g_object_get( G_OBJECT( helper ), "name", &name, NULL );

	uiname = screem_helper_build_action_name( name );

	g_free( name );
	
	action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ),
						uiname );
	/* action may have already been removed when closing the
	 * window */
	if( action ) {
		merge_id = GPOINTER_TO_UINT( g_object_get_data( G_OBJECT( action ),
						"merge_id" ) );
		gtk_ui_manager_remove_ui( GTK_UI_MANAGER( window->merge ), merge_id );
		gtk_action_group_remove_action( GTK_ACTION_GROUP( window->action_group ),
						action );
	}
	g_free( uiname );

}

gchar *screem_helper_get_uiname( ScreemHelper *helper )
{
	gchar *ret;
	
	g_return_val_if_fail( SCREEM_IS_HELPER( helper ), NULL );

	ret = screem_helper_build_action_name( helper->priv->name );

	return ret;
}

/* static stuff */
static void screem_helper_verb_cb( GtkAction *action, gpointer user_data )
{
	ScreemHelper *helper;
	ScreemWindow *window;

	helper = SCREEM_HELPER( user_data );
	window = g_object_get_data( G_OBJECT( action ), "window" );

	screem_helper_execute( helper, NULL, window );
}

static void child_setup( gpointer data )
{


}
static void child_watch( GPid pid, gint status, gpointer data )
{
	ScreemHelperData *hdata;
	ScreemHelper *helper;
	ScreemHelperPrivate *priv;
	ScreemPage *wpage;
	gchar *tmp;
	GSList *temp;
	ScreemView *view;
	
	hdata = (ScreemHelperData*)data;
	
	helper = hdata->helper;
	priv = helper->priv;

	if( pid != hdata->pid ) {
		return;
	}
		
	wpage = screem_window_get_document( hdata->window );
	view = screem_window_get_view( hdata->window );
	
	g_spawn_close_pid( pid );

	g_source_remove( hdata->watch );
	
	g_source_remove( hdata->owatch );
	g_source_remove( hdata->ewatch );

	if( hdata->pulse ) {
		g_source_remove( hdata->pulse );
		hdata->pulse = 0;
	}
	
	/* read any remaining data */
	child_stdout( hdata->out, G_IO_IN, data );
	child_stderr( hdata->err, G_IO_IN, data );
		
	g_io_channel_shutdown( hdata->out, FALSE, NULL );
	g_io_channel_shutdown( hdata->err, FALSE, NULL );
	g_io_channel_unref( hdata->out );
	g_io_channel_unref( hdata->err );
	
	hdata->out = hdata->err = NULL;
	
	gdk_threads_enter();
	
	if( WIFEXITED( status ) != 0 ) {
		if( status == 0 ) {
			/* success */
			switch( priv->omode ) {
				case SCREEM_HELPER_STDOUT_SELECTION:
					if( wpage == hdata->page ) {
						screem_window_replace_selection( hdata->window,
							hdata->input->str );
					} else {
						screem_page_set_data( hdata->page, hdata->input->str );
					}
					break;
				case SCREEM_HELPER_STDOUT_INSERT:
					screem_view_insert( view, -1,
							hdata->input->str );
					break;
				case SCREEM_HELPER_STDOUT_NONE:
					/* will have been displayed in
					   the windows message display
					   by the callback */
					break;
			}
		} else {
			/* helper returned an error */
			g_signal_emit( G_OBJECT( helper ),
				       screem_helper_signals[ HELPER_ERROR ], 0, status );
		}
	} else {
		/* helper didn't exit normally */
		tmp = g_strdup_printf( _( "%s finished due to an error" ),
				priv->name );
		g_signal_emit( G_OBJECT( helper ),
			       screem_helper_signals[ HELPER_FAILED ],
			       0, tmp );
		g_free( tmp );
	}
	
	g_string_free( hdata->input, TRUE );
	if( hdata->dialog ) {
		gtk_dialog_response( GTK_DIALOG( hdata->dialog ), GTK_RESPONSE_CLOSE );
		hdata->dialog = NULL;
	}

	/* free up newdata, removing it from the list */
	temp = g_slist_find( priv->running, hdata );
	if( temp ) {
		priv->running = g_slist_remove( priv->running, hdata );
		g_free( hdata );
	} else {
		g_warning( "Orphan ScreemHelperData\n" );
	}
	
	gdk_threads_leave();
}

static gboolean child_stdout( GIOChannel *source,
		GIOCondition condition, gpointer data )
{
	ScreemHelperData *hdata;
	ScreemHelper *helper;
	ScreemHelperPrivate *priv;
	GIOStatus status;
	gchar buf[ BUFSIZ + 1 ];
	GError *error;
	gsize read;

	hdata = (ScreemHelperData*)data;
	helper = hdata->helper;
	priv = helper->priv;

	error = NULL;
	status = G_IO_STATUS_AGAIN;

	gdk_threads_enter();
	
	if( condition == G_IO_IN ) {
		do {
			read = 0;
			status = g_io_channel_read_chars( source,
					buf, BUFSIZ, &read, &error );
			if( status == G_IO_STATUS_NORMAL ||
				status == G_IO_STATUS_EOF ) {
	
				buf[ read ] = '\0';
				if( priv->omode == SCREEM_HELPER_STDOUT_NONE ) {
					screem_window_show_message( hdata->window,
							buf, TRUE );
				} else {	
					g_string_append_len( hdata->input,
							buf, read );
				}
			}
		} while( read > 0 );
		if( error  ) {
			g_error_free( error );
		}
	}
	if( condition == G_IO_ERR ) {
		if( hdata->dialog ) {
			gtk_dialog_response( GTK_DIALOG( hdata->dialog ),
					GTK_RESPONSE_CANCEL );
		}
				
	} else if( condition == G_IO_HUP || status == G_IO_STATUS_EOF ) {
		if( hdata->dialog ) {
			gtk_dialog_response( GTK_DIALOG( hdata->dialog ),
					GTK_RESPONSE_CLOSE );
		}
	}
	
	gdk_threads_leave();
	
	return TRUE;
}

static gboolean child_stderr( GIOChannel *source,
		GIOCondition condition, gpointer data )
{
	ScreemHelperData *hdata;
	ScreemHelper *helper;
	ScreemHelperPrivate *priv;
	GIOStatus status;
	gchar buf[ BUFSIZ + 1 ];
	GError *error;
	gsize read;

	hdata = (ScreemHelperData*)data;
	helper = hdata->helper;
	priv = helper->priv;

	error = NULL;
	status = G_IO_STATUS_AGAIN;
	
	gdk_threads_enter();
	
	if( condition == G_IO_IN ) {
		do {
			read = 0;
			status = g_io_channel_read_chars( source,
					buf, BUFSIZ, &read, &error );
			if( status == G_IO_STATUS_NORMAL ||
				status == G_IO_STATUS_EOF ) {
			
				buf[ read ] = '\0';
				screem_window_show_error( hdata->window, buf,
						TRUE );
			}
		} while( read > 0 );
		if( error  ) {
			g_error_free( error );
		}
	}
	if( condition == G_IO_ERR ) {
		/* we don't mind if this occurs on stderr */
		
	} else if( condition == G_IO_HUP || status == G_IO_STATUS_EOF ) {
		/* we don't mind if this occurs on stderr */
	}
	
	gdk_threads_leave();

	return TRUE;
}

static gboolean screem_helper_pulse( ScreemHelperData *hdata )
{
	GladeXML *xml;
	GtkWidget *widget;

	gdk_threads_enter();
	
	xml = glade_get_widget_tree( hdata->dialog );
	
	widget = glade_xml_get_widget( xml, "helperprogress" );
	gtk_progress_bar_pulse( GTK_PROGRESS_BAR( widget ) );

	gdk_threads_leave();
	
	return TRUE;
}

/* take from libgnome-desktop/gnome-desktop-item.c */
static gboolean exec_exists( const char *exec )
{
	if( g_path_is_absolute( exec ) ) {
		if( access( exec, X_OK ) == 0 )
			return TRUE;
		else
			return FALSE;
	} else {
		char *tryme;

		tryme = g_find_program_in_path( exec );
		if( tryme != NULL ) {
			g_free( tryme );
			return TRUE;
		}
		return FALSE;
	}
}

static gchar *screem_helper_build_action_name( const gchar *name )
{
	gchar *tmp;
	gchar *uiname;
	
	tmp = screem_escape_char( name, ' ' );
	uiname = g_strconcat( "Exec_Helper_", tmp, NULL );
	g_free( tmp );
	
	return uiname;
}

/* G Object stuff */

G_DEFINE_TYPE( ScreemHelper, screem_helper, G_TYPE_OBJECT )

static void screem_helper_finalize( GObject *object );
static void screem_helper_set_property(GObject *object, guint prop_id,
				       const GValue *value, GParamSpec *pspec);
static void screem_helper_get_property( GObject *object, guint prop_id,
					GValue *value, GParamSpec *pspec );

static void screem_helper_class_init( ScreemHelperClass *klass )
{
	GObjectClass *object_class;
	GParamSpec *spec;
		
	object_class = G_OBJECT_CLASS( klass );
	screem_helper_parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = screem_helper_finalize;

	object_class->set_property = screem_helper_set_property;
	object_class->get_property = screem_helper_get_property;

	spec = g_param_spec_string( "pathname", "Helper Path",
			"The Pathname to the Helper", "",
			G_PARAM_READWRITE );
	g_object_class_install_property( object_class,
					 PROP_PATHNAME, spec );

	spec = g_param_spec_string( "iconpath", "Icon Path",
			"The Pathname to the image file for the Helper icon",
			"", G_PARAM_READWRITE );
	g_object_class_install_property( object_class,
					 PROP_ICONPATH, spec );


	spec = g_param_spec_string( "name", "Helper name",
			"The name of the Helper", "",
			G_PARAM_READWRITE );
	g_object_class_install_property( object_class,
					 PROP_HELPER_NAME, spec );

	spec = g_param_spec_int( "input", "Input Mode",
			"The input mode of the Helper",
			SCREEM_HELPER_STDIN_NONE,
			SCREEM_HELPER_STDIN_USER,
			SCREEM_HELPER_STDIN_SELECTION,
			G_PARAM_READWRITE );
	g_object_class_install_property( object_class,
					 PROP_INPUT_MODE, spec );
							
	spec = g_param_spec_int( "output", "Output Mode", 
			"The output mode of the Helper",
			SCREEM_HELPER_STDOUT_NONE,
			SCREEM_HELPER_STDOUT_INSERT,
			SCREEM_HELPER_STDOUT_SELECTION,
			G_PARAM_READWRITE );
	g_object_class_install_property( object_class,
					 PROP_OUTPUT_MODE, spec );


	spec = g_param_spec_boolean( "exec_on_save", "Execute On Save",
			"Should the helper run automatically when documents are saved",
			FALSE,
			G_PARAM_READWRITE );
	g_object_class_install_property( object_class,
					 PROP_EXEC_ON_SAVE, spec );

	spec = g_param_spec_string( "exec_type", "Execute Type",
			"The type of document to execute on when it is saved", "",
			G_PARAM_READWRITE );
	g_object_class_install_property( object_class,
					 PROP_EXEC_ON_SAVE_TYPE, spec );

	spec = g_param_spec_string( "uipath", "UI menu path",
			"The menu path to use when adding the helper to a GtkUIManager", "",
			G_PARAM_READWRITE );
	g_object_class_install_property( object_class,
					 PROP_UIPATH, spec );
	
	spec = g_param_spec_string( "filename", "Helper File",
			"The .desktop file the helper was loaded from", "",
			G_PARAM_READWRITE );
	g_object_class_install_property( object_class,
					 PROP_FILE, spec );
	
	screem_helper_signals[ HELPER_FAILED ] = 
		g_signal_new( "helper_failed",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemHelperClass, 
					       helper_failed ),
			      NULL, NULL,
			      screem_marshal_VOID__STRING,
			      G_TYPE_NONE, 1,
			      G_TYPE_STRING );

	screem_helper_signals[ HELPER_ERROR ] = 
		g_signal_new( "helper_error",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemHelperClass, 
					       helper_error ),
			      NULL, NULL,
			      screem_marshal_VOID__INT,
			      G_TYPE_NONE, 1,
			      G_TYPE_INT );
}

static void screem_helper_init( ScreemHelper *helper )
{
	helper->priv = g_new0( ScreemHelperPrivate, 1 );
	helper->priv->exists = TRUE;
}

static void screem_helper_finalize( GObject *object )
{
	ScreemHelper *helper;

	helper = SCREEM_HELPER( object );

	/* we need to remove outselves from all windows */
	while( helper->priv->list ) {
		ScreemWindow *window = 
			SCREEM_WINDOW( helper->priv->list->data );

		screem_helper_remove_from_window( helper, window );
	}

	if( helper->priv->name ) {
		g_free( helper->priv->name );
	}
	if( helper->priv->pathname ) {
		g_free( helper->priv->pathname );
	}
	if( helper->priv->iconpath ) {
		g_free( helper->priv->iconpath );
	}
	if( helper->priv->exec_type ) {
		g_free( helper->priv->exec_type );
	}

	g_free( helper->priv );

	G_OBJECT_CLASS( screem_helper_parent_class )->finalize( object );
}


static void screem_helper_set_property( GObject *object, guint prop_id,
					const GValue *value, GParamSpec *pspec)
{
	ScreemHelper *helper;
	ScreemHelperPrivate *priv;
	const gchar *val;
	gint mode;
	gboolean exists;
	gint argc;
	gchar **argv;
	
	helper = SCREEM_HELPER( object );
	priv = helper->priv;

	switch( prop_id ) {
	case PROP_PATHNAME:
		val = g_value_get_string( value );
		if( priv->pathname )
			g_free( priv->pathname );
		if( val ) {
			priv->pathname = g_strdup( val );
			if( ! g_shell_parse_argv( priv->pathname, &argc, &argv, NULL ) ) {
			exists = FALSE;	
			} else if( argc < 1 ) {
				g_strfreev( argv );
				exists = FALSE;
			} else {
				exists = exec_exists( argv[ 0 ] );
				g_strfreev( argv );
			}
		} else {
			priv->pathname = NULL;
			exists = FALSE;
		}
		priv->exists = exists;
	
		break;
	case PROP_ICONPATH:
		val = g_value_get_string( value );
		if( priv->iconpath )
			g_free( priv->iconpath );
		if( val ) {
			priv->iconpath = g_strdup( val );
		} else {
			priv->iconpath = NULL;
		}
		break;
	case PROP_HELPER_NAME:
		val = g_value_get_string( value );
		if( priv->name ) {
			g_free( priv->name );
		}
		if( val ) {
			priv->name = g_strdup( val );
		} else {
			priv->name = NULL;
		}
		break;
	case PROP_INPUT_MODE:
		mode = g_value_get_int( value );
		if( mode < 0 || mode > SCREEM_HELPER_STDIN_USER ) {
			mode = SCREEM_HELPER_STDIN_NONE;
		}
		priv->imode = mode;
		break;
	case PROP_OUTPUT_MODE:
		mode = g_value_get_int( value );
		if( mode < 0 || mode > SCREEM_HELPER_STDOUT_INSERT ) {
			mode = SCREEM_HELPER_STDOUT_NONE;
		}
		priv->omode = mode;
		break;
	case PROP_EXEC_ON_SAVE:
		priv->exec = g_value_get_boolean( value );
		break;
	case PROP_EXEC_ON_SAVE_TYPE:
		val = g_value_get_string( value );
		g_free( priv->exec_type );
		priv->exec_type = g_strdup( val );
		break;
	case PROP_UIPATH:
		val = g_value_get_string( value );
		g_free( priv->uipath );
		priv->uipath = g_strdup( val );
		break;
	case PROP_FILE:
		val = g_value_get_string( value );
		g_free( priv->filename );
		priv->filename = g_strdup( val );
		break;
	default:
		break;
	}
}

static void screem_helper_get_property( GObject *object, guint prop_id,
					GValue *value, GParamSpec *pspec )
{
	ScreemHelper *helper;
	ScreemHelperPrivate *priv;

	helper = SCREEM_HELPER( object );
	priv = helper->priv;

	switch( prop_id ) {
	case PROP_PATHNAME:
		if( priv->pathname ) {
			g_value_set_string( value, priv->pathname );
		} else {
			g_value_set_string( value, "" );
		}
		break;
	case PROP_ICONPATH:
		if( priv->iconpath ) {
			g_value_set_string( value, priv->iconpath );
		} else {
			g_value_set_string( value, "" );
		}
		break;
	case PROP_HELPER_NAME:
		if( priv->name ) {
			g_value_set_string( value, priv->name );
		} else {
			g_value_set_string( value, "" );
		}
		break;
	case PROP_INPUT_MODE:
		g_value_set_int( value, priv->imode );
		break;
	case PROP_OUTPUT_MODE:
		g_value_set_int( value, priv->omode );
		break;
	case PROP_EXEC_ON_SAVE:
		g_value_set_boolean( value, priv->exec );
		break;
	case PROP_EXEC_ON_SAVE_TYPE:
		if( priv->exec_type ) {
			g_value_set_string( value, priv->exec_type );
		} else {
			g_value_set_string( value, "" );
		}
		break;
	case PROP_UIPATH:
		if( priv->uipath ) {
			g_value_set_string( value, priv->uipath );
		} else {
			g_value_set_string( value, "" );
		}
		break;
	case PROP_FILE:
		if( priv->filename ) {
			g_value_set_string( value, priv->filename );
		} else {
			g_value_set_string( value, "" );
		}
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID( object, prop_id, pspec );
		break;
	}
}

