/*  Screem:  screem-plugin.c
 *
 *  Copyright (C) 2004 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */
#include <config.h>

#include <gtk/gtk.h>

#include <string.h>

#include "screem-plugin.h"

#include "screem-window.h"
#include "screem-window-private.h"

#include "screem-editor.h"

struct ScreemPluginPrivate {
	gchar *name;
	
	GtkUIManager *merge;

	GSList *ids;
	
	GtkActionGroup *group;

	gboolean in_setup;

	gboolean has_actions;
	
	ScreemWindow *window;
};

enum {
	ARG_0,
	ARG_NAME,
	ARG_WINDOW
};

static void screem_plugin_class_init( ScreemPluginClass *klass );
static void screem_plugin_init( ScreemPlugin *plugin );
static void screem_plugin_finalize( GObject *object );
static void screem_plugin_set_prop( GObject *object, guint property_id,
		const GValue *value, GParamSpec *pspec );
static void screem_plugin_get_prop( GObject *object, guint property_id,
		GValue *value, GParamSpec *pspec );


static gboolean screem_plugin_add_ui( ScreemPlugin *plugin,
		const gchar *itemtype, const gchar *path,
		const gchar *action, GError **error );
static void screem_plugin_remove_ui( ScreemPlugin *plugin );
static void screem_plugin_remove_actions( ScreemPlugin *plugin );

/* G Object stuff */
#define PARENT_TYPE G_TYPE_OBJECT

static gpointer parent_class;

static void screem_plugin_class_init( ScreemPluginClass *klass )
{
	GObjectClass *object_class;

	GParamSpec *pspec;

	object_class = G_OBJECT_CLASS( klass );

	object_class->finalize = screem_plugin_finalize;
	parent_class = g_type_class_peek_parent( klass );

	object_class->get_property = screem_plugin_get_prop;
	object_class->set_property = screem_plugin_set_prop;

	pspec = g_param_spec_string( "name", "Plugin name",
			"The name of the plugin", "",
			G_PARAM_CONSTRUCT | G_PARAM_READWRITE );
	g_object_class_install_property( object_class, ARG_NAME, pspec );

	pspec = g_param_spec_object( "window", "Window",
			"The Screem Window this object is for",
			SCREEM_TYPE_WINDOW,
			G_PARAM_WRITABLE );
	g_object_class_install_property( object_class, ARG_WINDOW,
			pspec );
}

static void screem_plugin_init( ScreemPlugin *plugin )
{
	plugin->priv = g_new0( ScreemPluginPrivate, 1 );

}

static void screem_plugin_finalize( GObject *object )
{
	ScreemPlugin *plugin;
	ScreemPluginPrivate *priv;
	
	plugin = SCREEM_PLUGIN( object );
	priv = plugin->priv;

	screem_plugin_remove_actions( plugin );
	
	g_free( priv );
	
	G_OBJECT_CLASS( parent_class )->finalize( object );
}

static void screem_plugin_set_prop( GObject *object, guint property_id,
				    const GValue *value, GParamSpec *pspec )
{
	ScreemPlugin *plugin;
	ScreemPluginPrivate *priv;
	
	plugin = SCREEM_PLUGIN( object );
	priv = plugin->priv;
	
	switch( property_id ) {
		case ARG_NAME:
			g_free( priv->name );
			priv->name = g_value_dup_string( value );
			break;
		case ARG_WINDOW:
			g_return_if_fail( priv->window == NULL );
			priv->window = g_value_get_object( value );
			priv->group = gtk_action_group_new( priv->name );
			priv->merge = GTK_UI_MANAGER( priv->window->merge );
			gtk_ui_manager_insert_action_group( priv->merge,
					priv->group, 1 );
			break;
		default:
			break;
	}
}

static void screem_plugin_get_prop( GObject *object, guint property_id,
				    GValue *value, GParamSpec *pspec)
{
	ScreemPlugin *plugin;
	ScreemPluginPrivate *priv;
	
	plugin = SCREEM_PLUGIN( object );
	priv = plugin->priv;
	
	switch( property_id ) {
		case ARG_NAME:
			g_value_set_string( value, priv->name );
			break;
		default:
			break;
	}
}

GType screem_plugin_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemPluginClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_plugin_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemPlugin ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_plugin_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       "ScreemPlugin",
					       &info, 0 );
	}

	return type;
}

/* Private API */
static gboolean screem_plugin_add_ui( ScreemPlugin *plugin,
		const gchar *itemtype, const gchar *path,
		const gchar *action, GError **error )
{
	ScreemPluginPrivate *priv;
	GString *ui;
	gchar **items;
	gchar **tmp;
	gboolean menu;
	const gchar *pathtype;
	guint id;
	gboolean ret;
	
	if( error ) {
		*error = NULL;
	}
	
	g_return_val_if_fail( SCREEM_IS_PLUGIN( plugin ), FALSE );
	
	priv = plugin->priv;
	
	g_return_val_if_fail( priv->merge != NULL, FALSE );

	g_return_val_if_fail( priv->has_actions == TRUE, FALSE );
	
	ui = g_string_new( "<ui>" );

	items = g_strsplit( path, "/", 0 );
	
	g_return_val_if_fail( items != NULL, FALSE );
	
	menu = ( ! strcmp( "menu", itemtype ) );
	if( menu ) {
		g_string_append( ui, "<menubar>" );
		pathtype = itemtype;
	} else {
		pathtype = "toolbar";
	}
	
	for( tmp = items; tmp && *tmp; tmp ++ ) {
		if( **tmp ) {
			g_string_append_printf( ui, "<%s action=\"%s\">",
						pathtype, *tmp );
		}
	}
	
	g_string_append_printf( ui, "<%sitem action=\"%s\"/>",
				itemtype, action );
	
	for( tmp = items; tmp && *tmp; tmp ++ ) {
		if( **tmp ) {
			g_string_append_printf( ui, "</%s>", pathtype );
		}
	}
	
	if( menu ) {		
		g_string_append( ui, "</menubar>" );
	}
	
	g_string_append( ui, "</ui>" );

	id = gtk_ui_manager_add_ui_from_string( priv->merge,
			ui->str, ui->len, error  );
	
	g_string_free( ui, TRUE );

	ret = FALSE;
	if( id != 0 ) {
		ret = TRUE;
	
		priv->ids = g_slist_prepend( priv->ids,
				GUINT_TO_POINTER( id ) );
	}
	
	gtk_ui_manager_ensure_update( priv->merge );
	
	return ret;
}

static void screem_plugin_remove_ui( ScreemPlugin *plugin )
{
	ScreemPluginPrivate *priv;
	GSList *tmp;
	guint id;
	
	g_return_if_fail( SCREEM_IS_PLUGIN( plugin ) );
	
	priv = plugin->priv;
	
	g_return_if_fail( GTK_IS_UI_MANAGER( priv->merge ) );
	
	for( tmp = priv->ids; tmp; tmp = tmp->next ) {
		id = GPOINTER_TO_UINT( tmp->data );
		gtk_ui_manager_remove_ui( priv->merge, id );	
	}

	gtk_ui_manager_ensure_update( priv->merge );
}

static void screem_plugin_remove_actions( ScreemPlugin *plugin )
{
	ScreemPluginPrivate *priv;
	
	g_return_if_fail( SCREEM_IS_PLUGIN( plugin ) );
	
	priv = plugin->priv;
	
	g_return_if_fail( GTK_IS_UI_MANAGER( priv->merge ) );
	g_return_if_fail( GTK_IS_ACTION_GROUP( priv->group ) );
	
	if( priv->ids ) {
		screem_plugin_remove_ui( plugin );
	}
	
	gtk_ui_manager_remove_action_group( priv->merge, priv->group );

	gtk_ui_manager_ensure_update( priv->merge );
}

/* Public API */

ScreemPlugin *screem_plugin_new( const gchar *name )
{
	ScreemPlugin *plugin;

	plugin = SCREEM_PLUGIN( g_object_new( SCREEM_TYPE_PLUGIN, 
				"name", name,
				NULL ) );

	return plugin;
}

gboolean screem_plugin_setup( ScreemPlugin *plugin )
{
	gboolean ret;
	
	g_return_val_if_fail( SCREEM_IS_PLUGIN( plugin ), FALSE );
	g_return_val_if_fail( plugin->setup != NULL, FALSE );
	g_return_val_if_fail( plugin->priv->window != NULL, FALSE );

	g_return_val_if_fail( plugin->priv->in_setup == FALSE, FALSE );
	
	plugin->priv->in_setup = TRUE;

	ret = plugin->setup( plugin );
	
	plugin->priv->in_setup = FALSE;

	return ret;
}

gboolean screem_plugin_add_menu( ScreemPlugin *plugin,
		const gchar *path, const gchar *action,
		GError **error )
{
	g_return_val_if_fail( SCREEM_IS_PLUGIN( plugin ), FALSE );
	
	return screem_plugin_add_ui( plugin, "menu", 
			path, action, error );
}

gboolean screem_plugin_add_toolbar( ScreemPlugin *plugin,
		const gchar *path, const gchar *action,
		GError **error )
{
	g_return_val_if_fail( SCREEM_IS_PLUGIN( plugin ), FALSE );
	
	return screem_plugin_add_ui( plugin, "tool", 
			path, action, error );
}

gboolean screem_plugin_add_action( ScreemPlugin *plugin,
		const gchar *name, const gchar *label,
		const gchar *tip, const gchar *stock_id,
		GCallback activate, GError **error )
{
	GtkAction *action;
	gchar *accel_path;

	g_return_val_if_fail( SCREEM_IS_PLUGIN( plugin ), FALSE );

	action = gtk_action_new( name, label, tip, stock_id );
	g_signal_connect( G_OBJECT( action ), "activate", activate,
			plugin );


	accel_path = g_strconcat( "<Actions>", "/ScreemPluginActions/",
			name, NULL );
	gtk_action_set_accel_path( action, accel_path );
	g_free( accel_path );
	
	return screem_plugin_add_gtk_action( plugin, action, error );
}

gboolean screem_plugin_add_gtk_action( ScreemPlugin *plugin,
		GtkAction *action, GError **error  )
{
	ScreemPluginPrivate *priv;
	gchar *accel_path;

	g_return_val_if_fail( SCREEM_IS_PLUGIN( plugin ), FALSE );
	g_return_val_if_fail( GTK_IS_ACTION( action ), FALSE );
	
	priv = plugin->priv;

	g_return_val_if_fail( GTK_IS_ACTION_GROUP( priv->group ),
			FALSE );

	if( ! gtk_action_get_accel_path( action ) ) {
		accel_path = g_strconcat( "<Actions>", 
				"/ScreemPluginActions/",
				gtk_action_get_name( action ),
				NULL );
		gtk_action_set_accel_path( action, accel_path );
		g_free( accel_path );
	}
	
	gtk_action_group_add_action( priv->group, action );

	priv->has_actions = TRUE;
	
	return TRUE;
}

gboolean screem_plugin_add_interface( ScreemPlugin *plugin,
		const gchar *name, const gchar *label,
		const gchar *tip, const gchar *stock_id,
		GCallback activate, GError **error )
{
	gboolean ret;

	ret = TRUE;
	
	if( ! screem_plugin_add_action( plugin, name, label,
				tip, stock_id, activate, error ) ) {
		ret = FALSE;
	}
	
	if( ret && ! screem_plugin_add_menu( plugin,
				SCREEM_INSERT_WIZARD_MENU,
				name, error ) ) {
		ret = FALSE;
	}

	if( ret && ! screem_plugin_add_toolbar( plugin,
				SCREEM_WIZARD_TOOLBAR,
				name, error ) ) {
		ret = FALSE;
	}

	return ret;
}


void screem_plugin_restore_from_session( ScreemPlugin *plugin,
		GtkWidget *window )
{
	ScreemPluginPrivate *priv;
	ScreemApplication *app;
	ScreemSession *session;
	
	g_return_if_fail( SCREEM_IS_PLUGIN( plugin ) );
	g_return_if_fail( GTK_IS_WINDOW( window ) );
	
	priv = plugin->priv;

	g_return_if_fail( SCREEM_IS_WINDOW( priv->window ) );
	
	app = SCREEM_APPLICATION( priv->window->application );
	session = screem_application_get_session( app );

	screem_session_restore_dialog( session, window );
}

void screem_plugin_store_in_session( ScreemPlugin *plugin,
		GtkWidget *window )
{
	ScreemPluginPrivate *priv;
	ScreemApplication *app;
	ScreemSession *session;
	gint x;
	gint y;
	gint w;
	gint h;
	
	g_return_if_fail( SCREEM_IS_PLUGIN( plugin ) );
	g_return_if_fail( GTK_IS_WINDOW( window ) );
	
	priv = plugin->priv;

	g_return_if_fail( SCREEM_IS_WINDOW( priv->window ) );
	
	app = SCREEM_APPLICATION( priv->window->application );
	session = screem_application_get_session( app );

	gtk_window_get_position( GTK_WINDOW( window ), &x, &y );
	gtk_window_get_size( GTK_WINDOW( window ), &w, &h );
	
	screem_session_set_dialog( session,
			gtk_window_get_role( GTK_WINDOW( window ) ),
			x, y, w, h );
}

ScreemSite *screem_plugin_get_current_site( ScreemPlugin *plugin )
{
	ScreemPluginPrivate *priv;
	
	g_return_val_if_fail( SCREEM_IS_PLUGIN( plugin ), NULL );

	priv = plugin->priv;

	g_return_val_if_fail( SCREEM_IS_WINDOW( priv->window ), NULL );
	
	return screem_window_get_current( priv->window );
}

ScreemPage *screem_plugin_get_current_document( ScreemPlugin *plugin )
{
	ScreemPluginPrivate *priv;

	g_return_val_if_fail( SCREEM_IS_PLUGIN( plugin ), NULL );

	priv = plugin->priv;

	g_return_val_if_fail( SCREEM_IS_WINDOW( priv->window ), NULL );
	
	return screem_window_get_document( priv->window );
}

ScreemDTD *screem_plugin_get_current_dtd( ScreemPlugin *plugin )
{
	ScreemDTD *ret;
	ScreemPage *page;
	
	g_return_val_if_fail( SCREEM_IS_PLUGIN( plugin ), NULL );

	page = screem_plugin_get_current_document( plugin );
	ret = NULL;
	if( page ) {
		ret = screem_page_get_dtd( page );
	} else {
		/* FIXME: provide the default dtd if there is
		 * no page? */
	}
	
	return ret;
}

guint screem_plugin_get_cursor_position( ScreemPlugin *plugin )
{
	ScreemPluginPrivate *priv;
	ScreemWindowDetails *details;
	ScreemPage *page;
	
	g_return_val_if_fail( SCREEM_IS_PLUGIN( plugin ), 0 );

	priv = plugin->priv;

	g_return_val_if_fail( SCREEM_IS_WINDOW( priv->window ), 0 );
	
	page = screem_window_get_document( priv->window );

	g_return_val_if_fail( SCREEM_IS_PAGE( page ), 0 );

	details = priv->window->details;

	return screem_view_get_pos( SCREEM_VIEW( details->editor ) );
}

void screem_plugin_set_cursor_position( ScreemPlugin *plugin, guint pos )
{
	ScreemPluginPrivate *priv;
	ScreemWindowDetails *details;
	ScreemPage *page;
	
	g_return_if_fail( SCREEM_IS_PLUGIN( plugin ) );

	priv = plugin->priv;

	g_return_if_fail( SCREEM_IS_WINDOW( priv->window ) );
	
	page = screem_window_get_document( priv->window );

	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	details = priv->window->details;

	screem_view_set_pos( SCREEM_VIEW( details->editor ), pos );
}

void screem_plugin_insert( ScreemPlugin *plugin, gint pos,
		const gchar *text, guint length, gboolean indent )
{
	ScreemPluginPrivate *priv;
	ScreemWindowDetails *details;
	ScreemPage *page;
	GConfClient *client;
	
	g_return_if_fail( SCREEM_IS_PLUGIN( plugin ) );

	priv = plugin->priv;

	g_return_if_fail( SCREEM_IS_WINDOW( priv->window ) );
	
	page = screem_window_get_document( priv->window );

	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	details = priv->window->details;

	screem_view_insert( SCREEM_VIEW( details->editor ), pos, text );
		
	client = gconf_client_get_default();
	indent = ( indent && gconf_client_get_bool( client, 
				"/apps/screem/editor/auto_indent", 
				NULL ) );
	if( indent ) {
		/* select and structure based indent */
		screem_view_select_region( SCREEM_VIEW( details->editor ),
				pos, length );
		screem_editor_auto_indent( SCREEM_EDITOR( details->editor ),
				pos );
	}

	g_object_unref( client );
}

void screem_plugin_insert_markup( ScreemPlugin *plugin,
		const gchar *open, const gchar *close,
		gboolean indent )
{
	ScreemPluginPrivate *priv;
	ScreemWindowDetails *details;
	ScreemPage *page;
	
	g_return_if_fail( SCREEM_IS_PLUGIN( plugin ) );

	priv = plugin->priv;

	g_return_if_fail( SCREEM_IS_WINDOW( priv->window ) );
	
	page = screem_window_get_document( priv->window );

	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	details = priv->window->details;

	screem_view_insert_markup( SCREEM_VIEW( details->editor ), 
			open, close );

	if( indent ) {
		/* select and structure based indent */
	}
	
}

void screem_plugin_file_op( GnomeVFSMonitorEventType type,
		const gchar *uri, gpointer data )
{
	ScreemPlugin *plugin;
	ScreemPluginPrivate *priv;
	
	plugin = SCREEM_PLUGIN( data );
	priv = plugin->priv;

	screem_application_file_op( type, uri, priv->window->application );
}

void screem_plugin_show_message( ScreemPlugin *plugin,
		const gchar *message )
{
	ScreemPluginPrivate *priv;
	
	g_return_if_fail( SCREEM_IS_PLUGIN( plugin ) );

	priv = plugin->priv;

	g_return_if_fail( SCREEM_IS_WINDOW( priv->window ) );

	screem_window_show_message( priv->window, message, TRUE );
}

void screem_plugin_show_error( ScreemPlugin *plugin, const gchar *error )
{
	ScreemPluginPrivate *priv;
	
	g_return_if_fail( SCREEM_IS_PLUGIN( plugin ) );

	priv = plugin->priv;

	g_return_if_fail( SCREEM_IS_WINDOW( priv->window ) );

	screem_window_show_error( priv->window, error, TRUE );
}

