/*  Screem:  cssWizard.c,
 *  A css1 wizard
 *
 *  Copyright (C) 1999, 2000  David A Knight
 *
 *  TODO:
 *  when loading styles handle child styles properly, 
 *      ie "ul.mylist li"   should display the li as a child of a ul.mylist
 *                          style.
 *  (tree ascii art)
 *        |
 *       ul      mylist
 *        |
 *        -  li
 *        |   
 *
 *
 *
 *  support editing styles from a right click in the editor in a style context
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <gmodule.h>
#include <gnome.h>

#include <glade/glade.h>

#include "site.h"
#include "page.h"
#include "editor.h"
#include "support.h"
#include "fileops.h"

#include "link.xpm"

#include "css.h"

extern GtkWidget *app;
extern Site *current_site;

static GladeXML *xml;
static GtkWidget *window = NULL;
static GtkCTreeNode *node = NULL;
static GtkCTreeNode *pnode = NULL;
static gint current_page = CSS_FONT;

static gboolean closing = FALSE;

static void remove_styles( GtkCTreeNode *n );

static gchar* create_css( GtkCTree *tree, GtkCTreeNode *n );

static void store_current( void );
static void set_current( void );

G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module )
{
	g_print("cssWizard: check-init\n");
	return NULL;
}

G_MODULE_EXPORT void g_module_unload( GModule *module )
{
	g_print( "cssWizard: unloaded\n" );
}

G_MODULE_EXPORT void init() 
{
	GtkWidget *linkButton;
	GtkWidget *toolbar;
 
	GnomeUIInfo menuinfo[] = { 
                {
                        GNOME_APP_UI_ITEM, N_("CaSSIUS"),
                        N_("Open Stylesheet editor"),
                        cssWizard, NULL, NULL,
                        GNOME_APP_PIXMAP_STOCK,
                        GNOME_STOCK_MENU_BLANK,
                        0,
                        GDK_CONTROL_MASK, NULL
                },
                GNOMEUIINFO_END
        };

	toolbar = gtk_object_get_data( GTK_OBJECT( app ), "wizardbar" );

	/* place menu item after image under insert */
        gnome_app_insert_menus( GNOME_APP( app ),
                                _("_Tools/"), menuinfo);

	/* place a button on the wizards toolbar */
	linkButton = gnome_pixmap_new_from_xpm_d( LINK_XPM );
	gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
				 _("CSS Wizard"), "", linkButton ,
				 cssWizard, 0 );

	g_print( "cssWizard: initialised\n" );
}

void cssWizard()
{
        if( window ) {
                gdk_window_raise( window->window );
                gdk_window_show( window->window );
                return;
        }

	current_page = CSS_FONT;
	closing = FALSE;

	xml = glade_xml_new( GLADE_PATH"/css.glade", "CaSSIUS" );
	glade_xml_signal_autoconnect( xml );

	window = glade_xml_get_widget( xml, "CaSSIUS" );

	node = NULL;
	pnode = NULL;
	current_page = CSS_FONT;
}

void cassius_about()
{
	GladeXML *about_xml;
	GtkWidget *dialog;

	about_xml = glade_xml_new( GLADE_PATH"/css.glade", "about1" );
	glade_xml_signal_autoconnect( about_xml );

	dialog = glade_xml_get_widget( about_xml, "about1" );
}

void css_close()
{
	GtkWidget *tree;

	tree = glade_xml_get_widget( xml, "styles" );

	g_free( gtk_object_get_data( GTK_OBJECT( tree ), "filename" ) );
	gtk_object_set_data( GTK_OBJECT( tree ), "filename", NULL );

	pnode = gtk_ctree_node_nth( GTK_CTREE( tree ), 0 );
	remove_style();

	closing = TRUE;
	gtk_widget_destroy( window );
	window = NULL;
}

void css_dump_stylesheet()
{
	GtkWidget *tree;
	GtkCTreeNode *n;
	gchar *data;

	tree = glade_xml_get_widget( xml, "styles" );

	n = gtk_ctree_node_nth( GTK_CTREE( tree ), 0 );

	store_current();

	/* recursivley create css data */
	data = create_css( GTK_CTREE( tree ), n );

	screem_editor_insert( -1, data );

	g_free( data );
}

void css_dump_style()
{
	GtkWidget *tree;
	gchar *data;
	Style *style;

	if( ! node )
		return;

	tree = glade_xml_get_widget( xml, "styles" );

	store_current();

	style = gtk_ctree_node_get_row_data( GTK_CTREE( tree ), node );

	data = css_style_output( style );

	screem_editor_insert( -1, data );

	g_free( data );
}

void style_selected( GtkCTree *tree, GtkCTreeNode *row, gint column )
{
	GtkWidget *notebook;

	/* store details of currently selected node */
	store_current();
	
	/* enter details of clicked node */
	node = row;
	set_current();

	notebook = glade_xml_get_widget( xml, "notebook1" );
	gtk_widget_set_sensitive( notebook, TRUE );
}

void style_unselected( GtkCTree *tree, GtkCTreeNode *row, gint column )
{
	GtkWidget *notebook;
	
	store_current();

	node = NULL;
	pnode = NULL;
	
	notebook = glade_xml_get_widget( xml, "notebook1" );
	gtk_widget_set_sensitive( notebook, FALSE );
	
}

void style_clicked( GtkWidget *widget, GdkEventButton *event )
{
	GladeXML *menuxml;
	GtkWidget *menu;

	gint row;
	gint col;

	gtk_clist_get_selection_info( GTK_CLIST( widget ),
                                      event->x, event->y, &row, &col );
	
	if( row != - 1 )
		pnode = gtk_ctree_node_nth( GTK_CTREE( widget ), (guint)row );
	else
		pnode = NULL;

	if( event->button == 3 ) {
		/* popup menu */
		gtk_signal_emit_stop_by_name( GTK_OBJECT( widget ),
                                              "button_press_event" );

		menuxml = glade_xml_new( GLADE_PATH"/css.glade", "menu1" );
		glade_xml_signal_autoconnect( menuxml );
		menu = glade_xml_get_widget( menuxml, "menu1" );
		gnome_popup_menu_do_popup_modal( menu, 0, 0, event, 0 );
		gtk_widget_destroy( menu );
	}
}

Style* new_style( gchar* style_name, 
		  gchar* style_class, 
		  gchar* style_pseudo_class )
{
	Style *style;

	if( ! style_name )
		style_name = g_strdup( "unknown" );
	if( ! style_class )
		style_class = g_strdup( "" ); 
	if( ! style_pseudo_class )
		style_pseudo_class = g_strdup( "" );

	/* create a new style struct */
	style = css_style_new();
	style->name = g_strdup( style_name );
	style->class_name = g_strdup( style_class );
	style->pseudo = g_strdup( style_pseudo_class );

	return style;
}

void add_style()
{
	GladeXML *dialogxml;
	GtkWidget *dialog;
	GtkWidget *entry;
	gchar *name[ 4 ] = { NULL, NULL, NULL, NULL };
	gint button;
	Style *style;
	
	dialogxml = glade_xml_new( GLADE_PATH"/css.glade", "dialog1" );
	glade_xml_signal_autoconnect( dialogxml );

	dialog = glade_xml_get_widget( dialogxml, "dialog1" );

	button = gnome_dialog_run( GNOME_DIALOG( dialog ) );

	if( button == 0 ) {
		entry = glade_xml_get_widget( dialogxml, "style_name" );
		name[ 0 ] = gtk_entry_get_text( GTK_ENTRY( entry ) );
		entry = glade_xml_get_widget( dialogxml, "style_class" );
		name[ 1 ] = gtk_entry_get_text( GTK_ENTRY( entry ) );
		entry = glade_xml_get_widget( dialogxml,"style_pseudo_class" );
		name[ 2 ] = gtk_entry_get_text( GTK_ENTRY( entry ) );

		/* create a new style struct */
		style = new_style( name[ 0 ], name[ 1 ], name[ 2 ] );
		display_style( style );
     	}

	gtk_widget_destroy( dialog );
}

void display_style( Style *style )
{
	GtkCTreeNode *new_node;
	GtkWidget *tree;
	gchar *name[ 4 ] = { NULL, NULL, NULL, NULL };

	tree = glade_xml_get_widget( xml, "styles" );
	
	name[ 0 ] = style->name;
	name[ 1 ] = style->class_name;
	name[ 2 ] = style->pseudo;
	
	
	/* add new tree node */
	new_node = gtk_ctree_insert_node( GTK_CTREE( tree ), 
					  pnode, 
					  NULL, name,
					  3, NULL, NULL, NULL, NULL, 
					  (gboolean)node, FALSE );
	if( pnode )
		gtk_ctree_expand( GTK_CTREE( tree ), pnode );

	/* set it as this nodes row data */
	gtk_ctree_node_set_row_data( GTK_CTREE( tree ), new_node, style );

	/* store details of currently selected node */
	store_current();

	gtk_ctree_select( GTK_CTREE( tree ), new_node );
}

void remove_style()
{
	Style *style;
	GtkWidget *tree;
	GtkCTreeNode *n;

	if( ! pnode )
		return;

	tree = glade_xml_get_widget( xml, "styles" );

	n = pnode;

	if( pnode == node )
		style_unselected( GTK_CTREE( tree ), node, 0 );
       
	/* need to recursively destroy its child styles */
	remove_styles( GTK_CTREE_ROW( n )->children );
	/* destroy its own style */
	style = gtk_ctree_node_get_row_data( GTK_CTREE( tree ), n );
	css_style_destroy( style );

	/* remove the GtkCTreeNode */
	gtk_ctree_remove_node( GTK_CTREE( tree ), n );
}

static void remove_styles( GtkCTreeNode *n )
{
	Style *style;
	GtkWidget *tree;

	if( ! n )
		return;

	tree = glade_xml_get_widget( xml, "styles" );

	style = gtk_ctree_node_get_row_data( GTK_CTREE( tree ), n );
	css_style_destroy( style );	

	remove_styles( GTK_CTREE_ROW( n )->children );
	remove_styles( GTK_CTREE_ROW( n )->sibling );
}

gboolean page_switch( GtkNotebook *notebook, GtkNotebookPage *page, 
		      guint page_num )
{
	if( closing )
		return TRUE;

	store_current();

	current_page = page_num;
	set_current();

	return TRUE;
}

void color_set( GnomeColorPicker *cp, guint r, guint g, guint b, guint a,
		gchar *entry )
{
	gchar *colour;
	GtkWidget *e;

	colour = g_strdup_printf( "#%.2x%.2x%.2x", r >> 8, g >> 8, b >> 8 );

	e = glade_xml_get_widget( xml, entry );

	gtk_entry_set_text( GTK_ENTRY( e ), colour );

	g_free( colour );
}

void css_new( GtkWidget *item, const gchar *pathname )
{
	GtkWidget *tree;
	GladeXML *mxml;
	GtkWidget *widget;
	gint button;

	gchar *filename;

	if( item ) {
		/* we want a filename for the new stylesheet */
		filename = file_select( "Select filename for the new stylesheet" );
		
		if( ! filename )
			return;
		else if(  ! overwrite( filename ) ) {
			g_free( filename );
			return;
		}
	} else
		filename = g_strdup( pathname );

	tree = glade_xml_get_widget( xml, "styles" );
	
	if( gtk_ctree_node_nth( GTK_CTREE( tree ), 0 ) ) {
		mxml = glade_xml_new( GLADE_PATH"/css.glade", "saveMessage" );
		widget = glade_xml_get_widget( mxml, "saveMessage" );
		button = gnome_dialog_run_and_close( GNOME_DIALOG( widget ) );
		if( button == 2 )
			return;
		else if( button == 0 )
			css_save();
	}

	while( ( pnode = gtk_ctree_node_nth( GTK_CTREE( tree ), 0 ) ) )
		remove_style();

	/* store the filename of the new style */
	g_free( gtk_object_get_data( GTK_OBJECT( tree ), "filename" ) );
	gtk_object_set_data( GTK_OBJECT( tree ), "filename", filename );
}

void css_load()
{
	gchar *filename;
	gchar *buffer; 
	FILE  *in;
	GString *entireFile;
	
	/* Open File Selection Window */
	filename = file_select( _("Select CSS file to Open") );

        if( ! filename )
                return;

	in = fopen( filename, "r" );	

	if( ! in )
		return;

	css_new( NULL, filename );

	buffer = g_new0( gchar, BUFSIZ );	

	entireFile = g_string_new( "" );
	while( fread( buffer, 1, BUFSIZ, in ) )
     		g_string_append( entireFile, buffer );
	fclose( in );

	g_free( buffer );
	css_parse( entireFile );
}

void css_parse( GString *entireFile ) 
{
	gchar *tag;
	gchar *class;
	gchar *psudeoclass;
	gchar *currentChar;
	GString	*cssBuffer; 

	currentChar = entireFile->str;
	
	cssBuffer = g_string_new( "" );
	while( currentChar && *currentChar != '\0' ) {
		switch (*currentChar){
		case '/':
			/* Rule 1: comment */
			currentChar++;
			if( *currentChar == '*' ) {
				currentChar = strstr( currentChar, "*/" );
				if( ! currentChar ) {
					/* comment wasn't closed! */
					break;
				}
				currentChar += strlen( "*/" );
			}
			break;
		case ':':
		case '{':
			/* move the current position back one, so we
			   can use the code below */
			currentChar --;
		case '.':
			/* Rule 2 classes */
			
			/* Initalize Classes */	
			class = "";
			psudeoclass = "";
			tag = "";

			/* Get Tag name
			   FIXME: has a problem if the stylesheet is
			   contained within <!-- --> as it adds the <!--
			   to the start of the first name
			*/
			tag = g_strdup_printf( cssBuffer->str );
			g_string_assign( cssBuffer, "" );
			currentChar ++;

			/* the previous while loop here has been
			   placed in css_parse_class, which, if the
			   class has a pseudo name, calls itself,
			   as the code was the same for pseudo and non
			   pseudo */
			currentChar = css_parse_class( tag, cssBuffer,
						       currentChar );
			break;
		case ' ':
		case '\n':
		case '\t':
			currentChar++;
			break;
		default:
			g_string_append_c(cssBuffer,*currentChar);
			currentChar++;
			break;
		}
	}
	/* destroy GStrings */
	g_string_free( entireFile, TRUE );
	g_string_free( cssBuffer, TRUE );
}

gchar* parse_style_data( Style *style, GString *cssBuffer, gchar *currentChar )
{
	gboolean continue_parsing;
	gchar *styleName;
	gchar *styleData;

	continue_parsing = TRUE;
	styleName = NULL;
	while( *currentChar != '\0' && continue_parsing ) {
		switch( *currentChar ) {
		case '}':
			currentChar ++;
			g_string_assign( cssBuffer, "" );
			*currentChar = 0;
			continue_parsing = FALSE;
			break;
		case ':':
			styleName = g_strdup( cssBuffer->str );
			g_string_assign( cssBuffer, "" );
			currentChar ++;
			break;
		case ';':
			styleData = g_strdup( cssBuffer->str );
			g_string_assign( cssBuffer, "" );
			currentChar ++;
			input_style( styleName, styleData, style );
			break;
		case ' ':
			/* depends if we are handling style data or not,
			   if we are we want spaces */
			if( styleName && *cssBuffer->str != '\0' )
				g_string_append_c( cssBuffer, ' ' );
			
			currentChar ++;
			break;
		case '\t':
		case '\n':
			currentChar ++;
			break;
		default:
			g_string_append_c( cssBuffer,*currentChar );
			currentChar ++;
			break;
		}
	}
	return currentChar;
}

gchar* css_parse_class( gchar *tag, GString *cssBuffer, gchar *currentChar )
{
	gchar *class;
	gchar *psudeoclass;
	gboolean new_style_break;
	Style *style;

	static gboolean in_pseudo = FALSE;

	new_style_break = FALSE;
	class = NULL;

	while( *currentChar != '\0' && ! new_style_break ) {
		switch( *currentChar ) {
		case ':':
			class = g_strdup( cssBuffer->str );
			g_string_assign( cssBuffer, "" );
			currentChar ++;
			in_pseudo = TRUE;
			break;
		case '{':
			/* Enter Class Name */
			if( in_pseudo ){
				psudeoclass = g_strdup( cssBuffer->str );
				in_pseudo = FALSE;
			}else{
				class = g_strdup( cssBuffer->str );
				psudeoclass = g_strdup( "" );
			}
			g_string_assign( cssBuffer, "" );
			currentChar ++;
			
			style = new_style( tag, class, psudeoclass );
			
			currentChar = parse_style_data( style, cssBuffer, 
							currentChar );
			
			/*g_string_assign(cssBuffer,"");*/
			display_style( style );
			new_style_break = TRUE;
			currentChar ++;
			break;
		case ' ':
		case '\n':
		case '\t':
			currentChar ++;
			break;
		default:
			g_string_append_c( cssBuffer, *currentChar );
			currentChar ++;
			break;	
		}
	}

	in_pseudo = FALSE;
	return currentChar;
}

void css_save()
{
	const gchar *filename;
	gchar *data;
	GtkWidget *tree;
	GtkCTreeNode *n;
	FILE *file;

	tree = glade_xml_get_widget( xml, "styles" );
	filename = gtk_object_get_data( GTK_OBJECT( tree ), "filename" );

	/* if no filename is set then do a save as */
	if( ! filename )
		css_save_as();

	n = gtk_ctree_node_nth( GTK_CTREE( tree ), 0 );

	store_current();

	/* recursivley create css file */
	data = create_css( GTK_CTREE( tree ), n );

	/* we have the style sheet, save to the named file */
	if( ( file = fopen( filename, "w" ) ) ) {
		fwrite( data, 1, strlen( data ), file );
		fclose( file );
	}	
}

void css_save_as()
{
	GtkWidget *tree;
	gchar *filename;

	tree = glade_xml_get_widget( xml, "styles" );

	filename = file_select( _("Select filename for CSS file") );

        if( ! filename )
                return;

	g_free( gtk_object_get_data( GTK_OBJECT( tree ), "filename" ) );
	gtk_object_set_data( GTK_OBJECT( tree ), "filename", filename );

	css_save();
}

void input_style( const gchar* styleName, const gchar *styleData, Style *style )
{
	gchar *temp;

	if( ! styleName ) {
		g_print( "invalid style: name not given\n" );
		return;
	} else if( ! styleData ) {
		g_print( "We have no style data!" );
		return;
	} else if( ! style ) {
		g_print( "no style given" );
		return;
	}
	
	/* Font */
	if( ! strcmp( styleName, "font-family" ) ||
	    ! strcmp( styleName, "font-style" ) ||
	    ! strcmp( styleName, "font-variant" ) ||
	    ! strcmp( styleName, "font-weight" ) ||
	    ! strcmp( styleName, "font-size" ) ||
	    ! strcmp( styleName, "color" ) )
		css_font_style_set( style, styleName, styleData );
	
	/* Background */
	else if( ! strcmp( styleName, "background-color" ) ||
		 ! strcmp( styleName, "background-image" ) ||
		 ! strcmp( styleName, "background-repeat" ) ||
		 ! strcmp( styleName, "background-attachment" ) ||
		 ! strcmp( styleName, "background-position" ) )
		css_back_style_set( style, styleName, styleData );
       	/* Text */
	else if( ! strcmp( styleName, "word-spacing" ) ||
		 ! strcmp( styleName, "letter-spacing" ) ||
		 ! strcmp( styleName, "text-decoration" ) ||
		 ! strcmp( styleName, "vertical-align" ) ||
		 ! strcmp( styleName, "text-transform" ) ||
		 ! strcmp( styleName, "text-align" ) ||
		 ! strcmp( styleName, "text-indent" ) ||
		 ! strcmp( styleName, "line-height" ) )
		css_text_style_set( style, styleName, styleData );
	/* box, border, size place */
	else if( ! strcmp( styleName, "top-color" ) ||
		 ! strcmp( styleName, "right-color" ) ||
		 ! strcmp( styleName, "bottom-color" ) ||
		 ! strcmp( styleName, "left-color" ) ||
		 ! strcmp( styleName, "top-width" ) ||
		 ! strcmp( styleName, "right-width" ) ||
		 ! strcmp( styleName, "bottom-width" ) ||
		 ! strcmp( styleName, "left-width" ) ||
		 ! strcmp( styleName, "top-style" ) ||
		 ! strcmp( styleName, "right-style" ) ||
		 ! strcmp( styleName, "bottom-style" ) ||
		 ! strcmp( styleName, "left-style" ) ||
		 ! strcmp( styleName, "width" ) ||
		 ! strcmp( styleName, "height" ) ||
		 ! strcmp( styleName, "float" ) ||
		 ! strcmp( styleName, "clear" ) ||
		 ! strncmp( styleName, "margin", strlen( "margin" ) ) ||
		 ! strncmp( styleName, "padding", strlen( "padding" ) ) ||
		 ! strncmp( styleName, "border", strlen( "border" ) ) )
		 css_box_style_set( style, styleName, styleData );
	/* Classification */
	else if( ! strcmp( styleName, "display" ) ||
		 ! strcmp( styleName, "white-space" ) ||
		 ! strcmp( styleName, "list-style-type" ) ||
		 ! strcmp( styleName, "list-style-image" ) ||
		 ! strcmp( styleName, "list-style-position" ) )
		css_class_style_set( style, styleName, styleData );
	else {
		/* we don't know what it is! */
		temp = style->unknown;
		if( ! temp )
			temp = g_strdup( "" );
		
		style->unknown = g_strconcat( temp, "\n\t",
					      styleName, ": ",
					      styleData, ";", NULL );
		g_free( temp );
	}
}

static gchar* create_css( GtkCTree *tree, GtkCTreeNode *n )
{
	gchar *data;
	gchar *temp;
       	gchar *retval;
	Style *style;

	if( ! n )
		return NULL;

	/* create the style text for this node */
	style = gtk_ctree_node_get_row_data( GTK_CTREE( tree ), n );

	data = css_style_output( style );
		
	temp = create_css( tree, GTK_CTREE_ROW( n )->children );

	if( temp ) {
		retval = g_strconcat( data, "\n", style->name, " ", temp, 
				      NULL );
		g_free( temp );
	} else
		retval = g_strdup( data );

	g_free( data );

	temp = create_css( tree, GTK_CTREE_ROW( n )->sibling );

	if( temp ) {
		data = retval;
		retval = g_strconcat( data, "\n", temp, NULL );
		g_free( temp );
		g_free( data );
	}

	return retval;
}


static void store_current()
{
	Style *style;
	GtkWidget *tree;
	GtkWidget *widget;
	gint i;

	gchar *temp;

	const gchar *font[] = {
		"font-family",
		"font-style",
		"font-variant",
		"font-weight",
		"font-size",
		"color",
		NULL
	};
	const gchar *background[] = {
		"background-color",
		"background-image",
		"background-repeat",
		"background-attachment",
		"background-position",
		NULL
	};
	const gchar *text[] = {
		"word-spacing",
		"letter-spacing",
		"text-decoration",
		"vertical-align",
		"text-transform",
		"text-align",
		"text-indent",
		"line-height",
		NULL
	};
	const gchar *box[] = {
		"margin-top",
		"margin-right",
		"margin-bottom",
		"margin-left",
		"padding-top",
		"padding-right",
		"padding-bottom",
		"padding-left",
		NULL
	};
	const gchar *border[] = {
		"top-color",
		"right-color",
		"bottom-color",
		"left-color",
		"top-width",
		"right-width",
		"bottom-width",
		"left-width",
		"top-style",
		"right-style",
		"bottom-style",
		"left-style",
		NULL
	};
	const gchar *sizeplace[] = {
		"width",
		"height",
		"float",
		"clear",
		NULL
	};
	const gchar *classification[] = {
		"display",
		"white-space",
		"list-style-type",
		"list-style-image",
		"list-style-position",
		NULL
	};

	if( ! node )
		return;

	tree = glade_xml_get_widget( xml, "styles" );
	style = gtk_ctree_node_get_row_data( GTK_CTREE( tree ), node );

	/* store details from previous page */
     	switch( current_page ) {
	case CSS_FONT:
		for( i = 0; font[ i ]; i ++ ) {
			widget = glade_xml_get_widget( xml, font[ i ] );
			temp = gtk_entry_get_text( GTK_ENTRY(widget) );
			input_style( font[ i ], temp, style );
		}
		break;
	case CSS_BACKGROUND:
		for( i = 0; background[ i ]; i ++ ) {
			widget = glade_xml_get_widget( xml, background[ i ] );
			temp = gtk_entry_get_text( GTK_ENTRY(widget) );
			input_style( background[ i ], temp, style );
		}
		break;
	case CSS_TEXT:
		for( i = 0; text[ i ]; i ++ ) {
			widget = glade_xml_get_widget( xml, text[ i ] );
			temp = gtk_entry_get_text( GTK_ENTRY(widget) );
			input_style( text[ i ], temp, style );
		}
		break;
	case CSS_BOX: 
		for( i = 0; box[ i ]; i ++ ) {
			widget = glade_xml_get_widget( xml, box[ i ] );
			temp = gtk_entry_get_text( GTK_ENTRY(widget) );
			input_style( box[ i ], temp, style );
		}
		break;
	case CSS_BORDER: 
		for( i = 0; border[ i ]; i ++ ) {
			widget = glade_xml_get_widget( xml, border[ i ] );
			temp = gtk_entry_get_text( GTK_ENTRY(widget) );
			input_style( border[ i ], temp, style );
		}
		break;
	case CSS_SIZEPLACE:
		for( i = 0; sizeplace[ i ]; i ++ ) {
			widget = glade_xml_get_widget( xml, sizeplace[ i ] );
			temp = gtk_entry_get_text( GTK_ENTRY(widget) );
			input_style( sizeplace[ i ], temp, style );
		}
		break;
	case CSS_CLASSIFICATION:
		for( i = 0; classification[ i ]; i ++ ) {
			widget = glade_xml_get_widget( xml, classification[ i ] );
			temp = gtk_entry_get_text( GTK_ENTRY(widget) );
			input_style( classification[ i ], temp, style );
		}
		break;
	}
}

static void set_current()
{
	Style *style;
	GtkWidget *tree;
	GtkWidget *widget;

	if( ! node )
		return;

	tree = glade_xml_get_widget( xml, "styles" );
	g_return_if_fail( tree != NULL );
	style = gtk_ctree_node_get_row_data( GTK_CTREE( tree ), node );

	/* insert details for current_page */
	switch( current_page ) {
	case CSS_FONT:
		widget = glade_xml_get_widget( xml, "font-family" );
		if( style->font && style->font->family )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->font->family );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
 
		widget = glade_xml_get_widget( xml, "font-style" );
		if( style->font && style->font->style )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->font->style );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "font-variant" );
		if( style->font && style->font->variant )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->font->variant );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "font-weight" );
		if( style->font && style->font->weight )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->font->weight );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "font-size" );
		if( style->font && style->font->size )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->font->size );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "color" );
		if( style->font && style->font->color )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->font->color );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		break;
	case CSS_BACKGROUND:
		widget = glade_xml_get_widget( xml, "background-color" );
		if( style->back && style->back->color )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->back->color );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "background-image" );
		if( style->back && style->back->image )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->back->image );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "background-repeat" );
		if( style->back && style->back->repeat )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->back->repeat );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "background-attachment" );
		if( style->back && style->back->attachment )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->back->attachment );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "background-position" );
		if( style->back && style->back->position )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->back->position );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		break;
	case CSS_TEXT:
		widget = glade_xml_get_widget( xml, "word-spacing" );
		if( style->text && style->text->word_spacing )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->word_spacing );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "letter-spacing" );
		if( style->text && style->text->letter_spacing )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->letter_spacing );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "text-decoration" );
		if( style->text && style->text->decoration )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->decoration );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "vertical-align" );
		if( style->text && style->text->vertical_align )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->vertical_align );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "text-transform" );
		if( style->text && style->text->transform )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->transform );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "text-align" );
		if( style->text && style->text->align )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->align );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "text-indent" );
		if( style->text && style->text->indent )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->indent );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "line-height" );
		if( style->text && style->text->height )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->height );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		break;
	case CSS_BOX: 
		widget = glade_xml_get_widget( xml, "margin-top" );
		if( style->box && style->box->margin[ TOP ] )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->margin[ TOP ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "margin-right" );
		if( style->box && style->box->margin[ RIGHT ] )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->margin[ RIGHT ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "margin-bottom" );
		if( style->box && style->box->margin[ BOTTOM ] )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->margin[ BOTTOM ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "margin-left" );
		if( style->box && style->box->margin[ LEFT ] )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->margin[ LEFT ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "padding-top" );
		if( style->box && style->box->padding[ TOP ] )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->padding[ TOP ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "padding-right" );
		if( style->box && style->box->padding[ RIGHT ] )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->padding[ RIGHT ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "padding-bottom" );
		if( style->box && style->box->padding[ BOTTOM ] )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->padding[ BOTTOM ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "padding-left" );
		if( style->box && style->box->padding[ LEFT ] )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->padding[ LEFT ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		break;
	case CSS_BORDER: 
		widget = glade_xml_get_widget( xml, "top-color" );
		if( style->box && style->box->border_color[ TOP ] )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_color[ TOP ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "right-color" );
		if( style->box && style->box->border_color[ RIGHT ] )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_color[ RIGHT ]);
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "bottom-color" );
		if( style->box && style->box->border_color[ BOTTOM ] )
			gtk_entry_set_text(GTK_ENTRY( widget ),
					   style->box->border_color[ BOTTOM ]);
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "left-color" );
		if( style->box && style->box->border_color[ LEFT ] )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_color[ LEFT ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "top-width" );
		if( style->box && style->box->border_width[ TOP ] )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_width[ TOP ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "right-width" );
		if( style->box && style->box->border_width[ RIGHT ] )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_width[ RIGHT ]);
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "bottom-width" );
		if( style->box && style->box->border_width[ BOTTOM ] )
			gtk_entry_set_text(GTK_ENTRY( widget ),
					   style->box->border_width[ BOTTOM ]);
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "left-width" );
		if( style->box && style->box->border_width[ LEFT ] )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_width[ LEFT ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "top-style" );
		if( style->box && style->box->border_style[ TOP ] )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_style[ TOP ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "right-style" );
		if( style->box && style->box->border_style[ RIGHT ] )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_style[ RIGHT ]);
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "bottom-style" );
		if( style->box && style->box->border_style[ BOTTOM ] )
			gtk_entry_set_text(GTK_ENTRY( widget ),
					   style->box->border_style[ BOTTOM ]);
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "left-style" );
		if( style->box && style->box->border_style[ LEFT ] )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_style[ LEFT ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		break;
	case CSS_SIZEPLACE:
		widget = glade_xml_get_widget( xml, "width" );
		if( style->box && style->box->width )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->width );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );	
		
		widget = glade_xml_get_widget( xml, "height" );
		if( style->box && style->box->height )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->height );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "float" );
		if( style->box && style->box->float_ )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->float_ );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "clear" );
		if( style->box && style->box->clear )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->clear );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		break;
	case CSS_CLASSIFICATION:
		widget = glade_xml_get_widget( xml, "display" );
		if( style->class && style->class->display )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->class->display );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "white-space" );
		if( style->class && style->class->white_space )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->class->white_space );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "list-style-type" );
		if( style->class && style->class->list_style_type )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->class->list_style_type );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "list-style-image" );
		if( style->class && style->class->list_style_image )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->class->list_style_image );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "list-style-position" );
		if( style->class && style->class->list_style_position )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->class->list_style_position);
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		break;
	}
}




Style* css_style_new()
{
	Style *style;

	style = g_new0( Style, 1 );

	return style;
}

FontStyle* css_font_style_new()
{
	FontStyle *style;

	style = g_new0( FontStyle, 1 );

	return style;
}

BackStyle* css_back_style_new()
{
	BackStyle *style;

	style = g_new0( BackStyle, 1 );

	return style;
}

TextStyle* css_text_style_new()
{
	TextStyle *style;

	style = g_new0( TextStyle, 1 );

	return style;
}

BoxStyle* css_box_style_new()
{
	BoxStyle *style;

	style = g_new0( BoxStyle, 1 );

	return style;
}

ClassStyle* css_class_style_new()
{
	ClassStyle *style;

	style = g_new0( ClassStyle, 1 );

	return style;
}

void css_style_destroy( Style *style )
{
	if( ! style )
		return;

	css_font_style_destroy( style->font );
	css_back_style_destroy( style->back );
	css_text_style_destroy( style->text );
	css_box_style_destroy( style->box );
	css_class_style_destroy( style->class );

	g_free( style->name );
	g_free( style->class_name );
	g_free( style->pseudo );

	g_free( style->unknown );

	g_free( style );
}

void css_font_style_destroy( FontStyle *style )
{
	if( ! style )
		return;

	g_free( style->family );
	g_free( style->style );
	g_free( style->variant );
	g_free( style->weight );
	g_free( style->size );
	g_free( style->color );

	g_free( style );
}

void css_back_style_destroy( BackStyle *style )
{
	if( ! style )
		return;

	g_free( style->color );
	g_free( style->image );
	g_free( style->repeat );
	g_free( style->attachment );
	g_free( style->position );

	g_free( style );
}

void css_text_style_destroy( TextStyle *style )
{
	if( ! style )
		return;

	g_free( style->word_spacing );
	g_free( style->letter_spacing );
	g_free( style->decoration );
	g_free( style->vertical_align );
	g_free( style->transform );
	g_free( style->align );
	g_free( style->indent );
	g_free( style->height );

	g_free( style );
}

void css_box_style_destroy( BoxStyle *style )
{
	gint i;

	if( ! style )
		return;

	for( i = 0; i < 4; i ++ ) {
		g_free( style->margin[ i ] );
		g_free( style->padding[ i ] );
		g_free( style->border_color[ i ] );
		g_free( style->border_width[ i ] );
		g_free( style->border_style[ i ] );
	}
	g_free( style->width );
	g_free( style->height );
	g_free( style->float_ );
	g_free( style->clear );

	g_free( style );
}

void css_class_style_destroy( ClassStyle *style )
{
	if( ! style )
		return;

	g_free( style->display );
	g_free( style->white_space );
	g_free( style->list_style_type );
	g_free( style->list_style_image );
	g_free( style->list_style_position );

	g_free( style );
}

gchar* css_style_output( Style *style )
{
	gchar *data;
	gchar *temp;
	gchar *data2;
	gchar *name;

	if( ! style )
		return NULL;
	
	/* add the class on */
	if( strlen( style->class_name ) > 0 )
		name = g_strdup_printf("%s.%s", style->name,style->class_name);
	else
		name = g_strdup( style->name );

	/* add the pseudo class on */
	if( strlen( style->pseudo ) > 0 ) {
		data = name;
		name = g_strdup_printf( "%s:%s", name, style->pseudo );
		g_free( data );
	}
	if( ! ( temp = css_font_style_output( style->font ) ) )
		temp = g_strdup( "" );

	data = temp;

	if( ! ( temp = css_back_style_output( style->back ) ) )
		temp = g_strdup( "" );
	
	data2 = g_strconcat( data, temp, NULL );
	g_free( data );
	g_free( temp );

	if( ! ( temp = css_text_style_output( style->text ) ) )
		temp = g_strdup( "" );

	data = g_strconcat( data2, temp, NULL );
	g_free( data2 );
	g_free( temp );

	if( ! ( temp = css_box_style_output( style->box ) ) )
		temp = g_strdup( "" );

	data2 = g_strconcat( data, temp, NULL );
	g_free( data );
	g_free( temp );


	if( ! ( temp = css_class_style_output( style->class ) ) )
		temp = g_strdup( "" );

	data = g_strconcat( data2, temp, NULL );
	g_free( data2 );
	g_free( temp );

	/* now append any stuff the wizard didn't understand */
	temp = style->unknown;
	data2 = g_strconcat( data, temp, NULL );
	g_free( data );

	data = g_strdup_printf( "%s {%s\n}", name, data2 );

	g_free( data2 );
	g_free( name );

	return data;
}

gchar* css_font_style_output( FontStyle *style )
{
	gchar *data;
	gchar *temp;
	gchar *temp2;

	if( ! style )
		return NULL;

	if( ( style->family ) && ( strlen( style->family ) > 0 ) )
		data = g_strdup_printf( "\n\tfont-family: %s;", style->family );
	else
		data = g_strdup( "" );

	temp = data;

	if( ( style->style ) && ( strlen( style->style ) > 0 ) )
		data = g_strdup_printf( "\n\tfont-style: %s;", style->style );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->variant ) && ( strlen( style->variant ) > 0 ) )
		data = g_strdup_printf("\n\tfont-variant: %s;", style->variant);
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->weight ) && ( strlen( style->weight ) > 0 ) )
		data = g_strdup_printf( "\n\tfont-weight: %s;", style->weight );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->size ) && ( strlen( style->size ) > 0 ) )
		data = g_strdup_printf( "\n\tfont-size: %s;", style->size );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->color ) && ( strlen( style->color ) > 0 ) )
		data = g_strdup_printf( "\n\tcolor: %s;", style->color );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	return temp; 
}

gchar* css_back_style_output( BackStyle *style )
{
	gchar *data;
	gchar *temp;
	gchar *temp2;

	if( ! style )
		return NULL;

	if( ( style->color ) && ( strlen( style->color ) > 0 ) )
		data = g_strdup_printf( "\n\tbackground-color: %s;", 
					style->color );
	else
		data = g_strdup( "" );

	temp = data;

	if( ( style->image ) && ( strlen( style->image ) > 0 ) )
		data = g_strdup_printf( "\n\tbackground-image: url(%s);", 
					style->image );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->repeat ) && ( strlen( style->repeat ) > 0 ) )
		data = g_strdup_printf( "\n\tbackground-repeat: %s;", 
					style->repeat );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->attachment ) && ( strlen( style->attachment ) > 0 ) )
		data = g_strdup_printf( "\n\tbackground-attachment: %s;", 
					style->attachment );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->position ) && ( strlen( style->position ) > 0 ) )
		data = g_strdup_printf( "\n\tbackground-position: %s;", 
					style->position );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	return temp; 
}

gchar* css_text_style_output( TextStyle *style )
{
	gchar *data;
	gchar *temp;
	gchar *temp2;

	if( ! style )
		return NULL;

	if( ( style->word_spacing ) && ( strlen( style->word_spacing ) > 0 ) )
		data = g_strdup_printf( "\n\tword-spacing: %s;", 
					style->word_spacing );
	else
		data = g_strdup( "" );

	temp = data;

	if( ( style->letter_spacing ) && 
	    ( strlen( style->letter_spacing ) > 0 ) )
		data = g_strdup_printf( "\n\tletter-spacing: %s;", 
					style->letter_spacing );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->decoration ) && ( strlen( style->decoration ) > 0 ) )
		data = g_strdup_printf( "\n\ttext-decoration: %s;", 
					style->decoration );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->vertical_align ) && 
	    ( strlen( style->vertical_align ) > 0 ) )
		data = g_strdup_printf( "\n\tvertical-align: %s;", 
					style->vertical_align );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->transform ) && ( strlen( style->transform ) > 0 ) )
		data = g_strdup_printf( "\n\ttext-transform: %s;", 
					style->transform );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->align ) && ( strlen( style->align ) > 0 ) )
		data = g_strdup_printf( "\n\ttext-align: %s;", style->align );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->indent ) && ( strlen( style->indent ) > 0 ) )
		data = g_strdup_printf( "\n\ttext-indent: %s;", style->indent );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->height ) && ( strlen( style->height ) > 0 ) )
		data = g_strdup_printf( "\n\tline-height: %s;", style->height );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	return temp; 	
}

gchar* css_box_style_output( BoxStyle *style )
{
	gchar *data;
	gchar *temp;
	gchar *temp2;
	gint i;

	gboolean not_empty = TRUE;

	gchar *one;
	gchar *two;
	gchar *three;

	const gchar *side[] = { "top", "right", "bottom", "left" };

	if( ! style )
		return NULL;

	temp = g_strdup( "" );

	for( i = 0; i < 4; i ++ ) {
		if( ( style->margin[ i ] ) && 
		    ( strlen( style->margin[ i ] ) > 0 ) )
			data = g_strdup_printf( "\n\tmargin-%s: %s;",
						side[ i ],
						style->margin[ i ] );
		else
			data = g_strdup( "" );
		temp2 = temp;
		temp = g_strconcat( temp2, data, NULL );
		g_free( temp2 );
		g_free( data );
	}

	for( i = 0; i < 4; i ++ ) {
		if( ( style->padding[ i ] ) && 
		    ( strlen( style->padding[ i ] ) > 0 ) )
			data = g_strdup_printf( "\n\tpadding-%s: %s;",
						side[ i ],
						style->padding[ i ] );
		else
			data = g_strdup( "" );
		temp2 = temp;
		temp = g_strconcat( temp2, data, NULL );
		g_free( temp2 );
		g_free( data );
	}

	/* border-width border-style border-color */
	for( i = 0; i < 4; i ++ ) {
		not_empty = FALSE;
		if( style->border_width[ i ] && 
		    ( strlen( style->border_width[ i ] ) > 0 ) )
			not_empty = (gboolean)
				one = g_strconcat( " ", 
						   style->border_width[ i ],
						   NULL );
		else
			one = g_strdup( "" );
		if( style->border_style[ i ] && 
		    ( strlen( style->border_style[ i ] ) > 0 ) )
			not_empty = (gboolean)
				two = g_strconcat( " ",
						   style->border_style[ i ],
						   NULL );
		else
			two = g_strdup( "" );
		if( style->border_color[ i ] && 
		    ( strlen( style->border_color[ i ] ) > 0 ) ) 
			not_empty = (gboolean)
				three = g_strconcat( " ",
						     style->border_color[ i ],
						     NULL );
		else
			three = g_strdup( "" );

		if( not_empty ) {
			data = g_strdup_printf( "\n\tborder-%s: %s%s%s;",
						side[ i ],
						one, two, three );
			
			temp2 = temp;
			temp = g_strconcat( temp2, data, NULL );
			g_free( temp2 );
			g_free( data );
		}
	}

	if( ( style->width ) && ( strlen( style->width ) > 0 ) )
		data = g_strdup_printf( "\n\twidth: %s;", style->width );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->height ) && ( strlen( style->height ) > 0 ) )
		data = g_strdup_printf( "\n\theight: %s;", style->height );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->float_ ) && ( strlen( style->float_ ) > 0 ) )
		data = g_strdup_printf( "\n\tfloat: %s;", style->float_ );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->clear ) &&  ( strlen( style->clear ) > 0 ) )
		data = g_strdup_printf( "\n\tclear: %s;", style->clear );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	return temp;      
}

gchar* css_class_style_output( ClassStyle *style )
{
 	gchar *data;
	gchar *temp;
	gchar *temp2;

	if( ! style )
		return NULL;

	if( ( style->display ) && ( strlen( style->display ) > 0 ) )
		data = g_strdup_printf( "\n\tdisplay: %s;", style->display );
	else
		data = g_strdup( "" );

	temp = data;

	if( ( style->white_space ) && ( strlen( style->white_space ) > 0 ) )
		data = g_strdup_printf( "\n\twhite-space: %s;", 
					style->white_space );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->list_style_type ) && 
	    ( strlen( style->list_style_type ) > 0 ) )
		data = g_strdup_printf( "\n\tlist-style-type: %s;", 
					style->list_style_type );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->list_style_image ) && 
	    ( strlen( style->list_style_image ) > 0 ) )
		data = g_strdup_printf( "\n\tlist-style-image: url(%s);", 
					style->list_style_image );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->list_style_position ) && 
	    ( strlen( style->list_style_position ) > 0 ) )
		data = g_strdup_printf( "\n\tlist-style-position: %s;", 
					style->list_style_position );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	return temp;      
}

void css_font_style_set( Style *style, const gchar *param, const gchar *value )
{
	if( ! style->font )
		style->font = css_font_style_new();

	if( ! strcmp( param, "font-family" ) ) {
		g_free( style->font->family );
		style->font->family = g_strdup( value );
	} else if( ! strcmp( param, "font-style" ) ) {
		g_free( style->font->style );
		style->font->style = g_strdup( value );
	} else if( ! strcmp( param, "font-variant" ) ) {
		g_free( style->font->variant );
		style->font->variant = g_strdup( value );
	} else if( ! strcmp( param, "font-weight" ) ) {
		g_free( style->font->weight );
		style->font->weight = g_strdup( value );
	} else if( ! strcmp( param, "font-size" ) ) {
		g_free( style->font->size );
		style->font->size = g_strdup( value );
	} else if( ! strcmp( param, "color" ) ) {
		style->font->color = g_strdup( value );
	}
}

void css_back_style_set( Style *style, const gchar *param, const gchar *value )
{
	gchar *temp;

	if( ! style->back ) 
		style->back = css_back_style_new();

	if( ! strcmp( param, "background-color" ) ) {
		g_free( style->back->color );
		style->back->color = g_strdup( value );
	} else if( ! strcmp( param, "background-image" ) ) {
		if( ! strncmp( "url(", value, strlen( "url(" ) ) ) {
			temp = g_strdup( value + strlen( "url(" ) );
			temp[ strlen( temp ) - 1 ] = '\0';
			value = temp;
		} else
			temp = NULL;
		/* convert value to a path relative to the stylesheet */
		g_free( style->back->image );
		style->back->image = css_relative_path( value );
		g_free( temp );
	} else if( ! strcmp( param, "background-repeat" ) ) {
		g_free( style->back->repeat );
		style->back->repeat = g_strdup( value );
	} else if( strcmp( param, "background-attachment" ) ) {
		g_free( style->back->attachment );
		style->back->attachment = g_strdup( value );
	} else if( ! strcmp( param, "background-position" ) ) {
		g_free( style->back->position );
		style->back->position = g_strdup( value );
	}
}

void css_text_style_set( Style *style, const gchar *param, const gchar *value )
{
	if( ! style->text ) 
		style->text = css_text_style_new();

	if( ! strcmp( param, "word-spacing" ) ) {
		g_free( style->text->word_spacing );
		style->text->word_spacing = g_strdup( value );
	} else if( ! strcmp( param, "letter-spacing" ) ) {
		g_free( style->text->letter_spacing );
		style->text->letter_spacing = g_strdup( value );
	} else if( ! strcmp( param, "text-decoration" ) ) {
		g_free( style->text->decoration );
		style->text->decoration = g_strdup( value );
	} else if( ! strcmp( param, "vertical-align" ) ) {
		g_free( style->text->vertical_align );
		style->text->vertical_align = g_strdup( value );
	} else if( ! strcmp( param, "text-transform" ) ) {
		g_free( style->text->transform );
		style->text->transform = g_strdup( value );
	} else if( ! strcmp( param, "text-align" ) ) {
		g_free( style->text->align );
		style->text->align = g_strdup( value );
	} else if( ! strcmp( param, "text-indent" ) ) {
		g_free( style->text->indent );
		style->text->indent = g_strdup( value );
	} else if( ! strcmp( param, "line-height" ) ) {
		g_free( style->text->height );
		style->text->height = g_strdup( value );
	}
}

void css_box_style_set( Style *style, const gchar *param, const gchar *value )
{
	gchar **values;

	if( ! style->box ) 
		style->box = css_box_style_new();

	/* grrrr, this can get messy now,
	   value may be a space separated group of values */
	values = g_strsplit( value, " ", 16 );

	if( ! strcmp( param, "margin" ) ) {
		css_box_style_set( style, "margin-top", value );
		css_box_style_set( style, "margin-left", value );
		css_box_style_set( style, "margin-bottom", value );
		css_box_style_set( style, "margin-right", value );
	} else if( ! strcmp( param, "margin-top" ) ) { 
		g_free( style->box->margin[ TOP ] );
		style->box->margin[ TOP ] = g_strdup( value );
	 } else if( ! strcmp( param, "margin-right" ) ) { 
		g_free( style->box->margin[ RIGHT ] );
		style->box->margin[ RIGHT ] = g_strdup( value );
	 } else if( ! strcmp( param, "margin-bottom" ) ) { 
		g_free( style->box->margin[ BOTTOM ] );
		style->box->margin[ BOTTOM ] = g_strdup( value );
	 } else if( ! strcmp( param, "margin-left" ) ) { 
		g_free( style->box->margin[ LEFT ] );
		style->box->margin[ LEFT ] = g_strdup( value );
	 } else if( ! strcmp( param, "padding" ) ) {
		css_box_style_set( style, "padding-top", value );
		css_box_style_set( style, "padding-left", value );
		css_box_style_set( style, "padding-bottom", value );
		css_box_style_set( style, "padding-right", value );
	 } else if( ! strcmp( param, "padding-top" ) ) { 
		g_free( style->box->padding[ TOP ] );
		style->box->padding[ TOP ] = g_strdup( value );
	 } else if( ! strcmp( param, "padding-right" ) ) { 
		g_free( style->box->padding[ RIGHT ] );
		style->box->padding[ RIGHT ] = g_strdup( value );
	 } else if( ! strcmp( param, "padding-bottom" ) ) { 
		g_free( style->box->padding[ BOTTOM ] );
		style->box->padding[ BOTTOM ] = g_strdup( value );
	 } else if( ! strcmp( param, "padding-left" ) ) { 
		g_free( style->box->padding[ LEFT ] );
		style->box->padding[ LEFT ] = g_strdup( value );
	 } else if( ! strcmp( param, "top-color" ) ) { 
		g_free( style->box->border_color[ TOP ] );
		style->box->border_color[ TOP ] = g_strdup( value );
         } else if( ! strcmp( param, "right-color" ) ) { 
		g_free( style->box->border_color[ RIGHT ] );
		style->box->border_color[ RIGHT ] = g_strdup( value );
         } else if( ! strcmp( param, "bottom-color" ) ) { 
		g_free( style->box->border_color[ BOTTOM ] );
		style->box->border_color[ BOTTOM ] = g_strdup( value );
         } else if( ! strcmp( param, "left-color" ) ) { 
		g_free( style->box->border_color[ LEFT ] );
		style->box->border_color[ LEFT ] = g_strdup( value );
         } else if( ! strcmp( param, "top-width" ) ) {
 		g_free( style->box->border_width[ TOP ] );
		style->box->border_width[ TOP ] = g_strdup( value );
         } else if( ! strcmp( param, "right-width" ) ) {
  		g_free( style->box->border_width[ RIGHT ] );
		style->box->border_width[ RIGHT ] = g_strdup( value );
         } else if( ! strcmp( param, "bottom-width" ) ) { 
 		g_free( style->box->border_width[ BOTTOM ] );
		style->box->border_width[ BOTTOM ] = g_strdup( value );
         } else if( ! strcmp( param, "left-width" ) ) { 
 		g_free( style->box->border_width[ LEFT ] );
		style->box->border_width[ LEFT ] = g_strdup( value );
         } else if( ! strcmp( param, "top-style" ) ) { 
 		g_free( style->box->border_style[ TOP ] );
		style->box->border_style[ TOP ] = g_strdup( value );
	 } else if( ! strcmp( param, "right-style" ) ) { 
 		g_free( style->box->border_style[ RIGHT ] );
		style->box->border_style[ RIGHT ] = g_strdup( value );
	 } else if( ! strcmp( param, "bottom-style" ) ) { 
 		g_free( style->box->border_style[ BOTTOM ] );
		style->box->border_style[ BOTTOM ] = g_strdup( value );
	 } else if( ! strcmp( param, "left-style" ) ) { 
 		g_free( style->box->border_style[ LEFT ] );
		style->box->border_style[ LEFT ] = g_strdup( value );
	 } else if( ! strcmp( param, "width" ) ) {
		 g_free( style->box->width );
		 style->box->width = g_strdup( value );
	 } else if( ! strcmp( param, "height" ) ) {
		 g_free( style->box->height );
		 style->box->height = g_strdup( value );
	 } else if( ! strcmp( param, "float" ) ) {
		 g_free( style->box->float_ );
		 style->box->float_ = g_strdup( value );
	 } else if( ! strcmp( param, "clear" ) ) {
		 g_free( style->box->clear );
		 style->box->clear = g_strdup( value );
	 } 
	/* deal with border-* */
	else if( ! strcmp( param, "border" ) ) {
		/* sets all borders */
		css_box_style_set( style, "border-top", value );
		css_box_style_set( style, "border-left", value );
		css_box_style_set( style, "border-right", value );
		css_box_style_set( style, "border-bottom", value );
	}
	else if( ! strcmp( param, "border-top" ) ) {
		/* we don't handle incorrect styles properly,
		   we take the 1st value as the width, the 2nd
		   as the style, the 3rd as the colour */
		if( values[ 0 ] ) {
			g_free( style->box->border_width[ TOP ] ); 
			style->box->border_width[ TOP ] =
				g_strdup( values[ 0 ] );
		}
		if( values[ 1 ] ) {
			g_free( style->box->border_style[ TOP ] ); 
			style->box->border_style[ TOP ] =
				g_strdup( values[ 1 ] );
		}
		if( values[ 2 ] ) {
			g_free( style->box->border_color[ TOP ] ); 
			style->box->border_color[ TOP ] =
				g_strdup( values[ 2 ] );
		}
	} else if( ! strcmp( param, "border-left" ) ) {
		/* we don't handle incorrect styles properly,
		   we take the 1st value as the width, the 2nd
		   as the style, the 3rd as the colour */
		if( values[ 0 ] ) {
			g_free( style->box->border_width[ LEFT ] ); 
			style->box->border_width[ LEFT ] =
				g_strdup( values[ 0 ] );
		}
		if( values[ 1 ] ) {
			g_free( style->box->border_style[ LEFT ] ); 
			style->box->border_style[ LEFT ] =
				g_strdup( values[ 1 ] );
		}
		if( values[ 2 ] ) {
			g_free( style->box->border_color[ LEFT ] ); 
			style->box->border_color[ LEFT ] =
				g_strdup( values[ 2 ] );
		}
	} else if( ! strcmp( param, "border-bottom" ) ) {
		/* we don't handle incorrect styles properly,
		   we take the 1st value as the width, the 2nd
		   as the style, the 3rd as the colour */
		if( values[ 0 ] ) {
			g_free( style->box->border_width[ BOTTOM ] ); 
			style->box->border_width[ BOTTOM ] =
				g_strdup( values[ 0 ] );
		}
		if( values[ 1 ] ) {
			g_free( style->box->border_style[ BOTTOM ] ); 
			style->box->border_style[ BOTTOM ] =
				g_strdup( values[ 1 ] );
		}
		if( values[ 2 ] ) {
			g_free( style->box->border_color[ BOTTOM ] ); 
			style->box->border_color[ BOTTOM ] =
				g_strdup( values[ 2 ] );
		}
	} else if( ! strcmp( param, "border-right" ) ) {
		/* we don't handle incorrect styles properly,
		   we take the 1st value as the width, the 2nd
		   as the style, the 3rd as the colour */
		if( values[ 0 ] ) {
			g_free( style->box->border_width[ RIGHT ] ); 
			style->box->border_width[ RIGHT ] =
				g_strdup( values[ 0 ] );
		}
		if( values[ 1 ] ) {
			g_free( style->box->border_style[ RIGHT ] ); 
			style->box->border_style[ RIGHT ] =
				g_strdup( values[ 1 ] );
		}
		if( values[ 2 ] ) {
			g_free( style->box->border_color[ RIGHT ] ); 
			style->box->border_color[ RIGHT ] =
				g_strdup( values[ 2 ] );
		}
	}
	g_strfreev( values );
}

void css_class_style_set( Style *style, const gchar *param, const gchar *value )
{
	gchar *temp;

	if( ! style->class )
		style->class = css_class_style_new();
	
	if( ! strcmp( param, "display" ) ) {
		g_free( style->class->display );
		style->class->display = g_strdup( value );
	} else if( ! strcmp( param, "white-space" ) ) { 
		g_free( style->class->white_space );
		style->class->white_space = g_strdup( value );
	 } else if( ! strcmp( param, "list-style-type" ) ) { 
		g_free( style->class->list_style_type );
		style->class->list_style_type = g_strdup( value );
	} else if( ! strcmp( param, "list-style-image" ) ) {
		if( ! strncmp( "url(", value, strlen( "url(" ) ) ) {
			temp = g_strdup( value + strlen( "url(" ) );
			temp[ strlen( temp ) - 1 ] = '\0';
			value = temp;
		} else
			temp = NULL;
		/* convert value to a path relative to the stylesheet */
		g_free( style->class->list_style_image );
		style->class->list_style_image = css_relative_path( value );
		g_free( temp );
	} else if( ! strcmp( param, "list-style-position" ) ) { 
		g_free( style->class->list_style_position );
		style->class->list_style_position = g_strdup( value );
	}
}

gchar* css_relative_path( const gchar *path )
{
	GtkWidget *tree;
	gchar *sheetpath;
	gchar *ret;
	gchar *dir;

       	if( ! g_path_is_absolute( path ) )
		return g_strdup( path );

	tree = glade_xml_get_widget( xml, "styles" );

	sheetpath = gtk_object_get_data( GTK_OBJECT( tree ), "filename" );

	dir = g_dirname( sheetpath );
	ret = relative_path( path, dir );

	g_free( dir );

	return ret;
}
