/* 
   sitecopy local system protocol driver module
   Copyright (C) 2001, David A Knight <david@screem.org>
                                                   
   the ftp driver module was used as a skeleton for implementation
                  
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <config.h>

#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <ne_alloc.h>

#include "protocol.h"

typedef struct {
	struct site *site;
	char error[BUFSIZ];
} local_session;

extern void fe_transfer_progress(off_t progress, off_t total);


static int init(void **session, struct site *site)
{
	local_session *sess = ne_calloc(sizeof *sess);
	*session = sess;

	sess->site = site;
	return SITE_OK;
}

static void finish(void *session)
{
	local_session *sess = session;
	free( sess );
}

static int file_upload(void *session, const char *local, const char *remote,
		       int ascii)
{
       	FILE *l;
	FILE *r;
	char buffer[BUFSIZ];
	ssize_t total = 0;
	int ret = 0;
	local_session *sess = (local_session*)session;

	l = fopen( local, "r" );

	if( ! l ) {
		strcpy( sess->error, strerror( errno ) );
		return SITE_FAILED;
	}

	r = fopen( remote, "w" );

	if( ! r ) {
		strcpy( sess->error, strerror( errno ) );
		fclose( l );
		return SITE_FAILED;
	}

	while( 1 ) {
		if( ! fgets( buffer, BUFSIZ - 1, l) ) {
			if( ferror( l ) ) {
				strcpy( sess->error, strerror( errno ) );
				ret = SITE_FAILED;
				break;
			}
	       		ret = 0;
			break;
		}
		fputs( buffer, r );
		total += strlen( buffer );

		/* is this what we should be doing?
		   other drivers use the socket stuff in libneon
		   which ends up calling this */
		fe_transfer_progress(total, -1);
	}

	fclose( l );
	fclose( r );

	return ret;
}

static int file_get_modtime(void *session, const char *remote, time_t *modtime )
{
	struct stat s;
	local_session *sess = (local_session*)session;

        if( stat( remote, &s ) < 0 ) {
		strcpy( sess->error, strerror( errno ) );
                return SITE_FAILED;
	}

        *modtime = s.st_mtime;

        return SITE_OK;
}

static int file_upload_cond(void *session, const char *local, 
			    const char *remote, int ascii, time_t time)
{
	/* get modtime */
	time_t mtime;

	file_get_modtime(session, remote, &mtime);

	if( mtime != time )
		return SITE_FAILED;
       
	return file_upload(session, local, remote, ascii);
}

static int file_download(void *session, const char *local, const char *remote,
			 int ascii)
{
	/* reverse local/remote and call file_upload */
	return file_upload(session, remote, local, ascii);
}

static int file_read(void *session, const char *remote, 
		     sock_block_reader reader, void *userdata)
{
	int ret;
	FILE *r;
	char buffer[BUFSIZ];
	off_t done = 0;
	off_t length = -1;
	local_session *sess = (local_session*)session;

	r = fopen( remote, "r" );

	if( ! r ) {
		strcpy( sess->error, strerror( errno ) );
		return SITE_FAILED;
	}

	do {
		if( ! fgets( buffer, BUFSIZ - 1, r ) ) {
			if( ferror( r ) ) {
				strcpy( sess->error, strerror( errno ) );
				ret = SITE_FAILED;
				break;
			}
			ret = 0;
		} else
			ret = strlen( buffer );
		done += ret;
		fe_transfer_progress( done, length );
		(*reader)(userdata, buffer, ret);
	} while( ( done < length ) && ret );

	return ret;
}

static int file_delete(void *session, const char *filename)
{
	local_session *sess = (local_session*)session;

	if( unlink( filename ) < 0 ) {
		strcpy( sess->error, strerror( errno ) );
		return SITE_FAILED;
	}
	return SITE_OK;
}

static int file_move(void *session, const char *from, const char *to)
{
	/* move the file: copy then remove */
	if( file_upload(session, from, to, 0) == SITE_OK )
		return file_delete(session, from);
	return SITE_FAILED;
}

static int file_chmod(void *session, const char *filename, mode_t mode)
{
	local_session *sess = (local_session*)session;

	if( chmod( filename, mode ) < 0 ) {
		strcpy( sess->error, strerror( errno ) );
		return SITE_FAILED;
	}
	return SITE_OK;
}

static int dir_create(void *session, const char *dirname )
{
	local_session *sess = (local_session*)session;

	if( mkdir( dirname, 0x1e4 ) < 0 ) {
		if( errno == EEXIST ) {
			/* it exists, so everything is ok */
			return SITE_OK;
		}
		strcpy( sess->error, strerror( errno ) );
		return SITE_FAILED;
	}

	return SITE_OK;
}

static int dir_remove(void *session, const char *dirname)
{
	local_session *sess = (local_session*)session;

	if( rmdir( dirname ) < 0 ) {
		strcpy( sess->error, strerror( errno ) );
		return SITE_FAILED;
	}
	return SITE_OK;
}

static int fetch_list(void *session, const char *dirname, int need_modtimes,
		     struct proto_file **files)
{
	/* I'm to lazy to do this at the moment, well either that
	   or something to do with the following:
	   Sun Jan 28 03:07:39 GMT 2001

	   :-)
	*/

	return SITE_UNSUPPORTED;
}

static const char *error(void *session)
{
	return ((local_session*)session)->error;
}

static int get_dummy_port(struct site *site)
{
    return 0;
}


/* The protocol drivers */
const struct proto_driver local_driver = {
    init, 
    finish,
    file_move,
    file_upload,
    file_upload_cond,
    file_get_modtime,
    file_download,
    file_read,
    file_delete,
    file_chmod,
    dir_create,
    dir_remove,
    NULL, /* create link */
    NULL, /* change link target */
    NULL, /* delete link */
    fetch_list,
    error,
    get_dummy_port,
    get_dummy_port,
    "local"
};
