/*  Screem:  Helpers.c,
 *  Handles the helper application execution, loading, and creation of
 *  the toolbar.
 * 
 *  Copyright (C) 1999, 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gnome.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <glade/glade.h>

#include "editor.h"
#include "helpers.h"
#include "page.h"
#include "preferences.h"
#include "site.h"
#include "support.h"

#include "browser.xpm"

extern GtkWidget *app;
extern Site *current_site;
extern Preferences *cfg;

#ifndef STDIN
#define STDIN 0
#endif

#ifndef STDOUT
#define STDOUT 1
#endif

static void screem_helper_execute_all( void );

static gboolean helper_button_clicked( GtkWidget *widget,  GdkEventButton *event,
			    Helper *helper );
static void helper_configure( GtkWidget *widget, Helper *helper );

/*
 * screem_helper_new:
 *
 * Creates and initialises a new helper struct
 *
 * return values: a Helper
 */
Helper* screem_helper_new()
{
	Helper *helper;

	helper = (Helper*)g_malloc( sizeof( Helper ) );

	helper->path = NULL;
	helper->mime_type = NULL;
	helper->icon = NULL;

	return helper;
}

/*
 * screem_helper_destroy:
 * @helper: the helper to destroy
 *
 * Destroys the given helper freeing up any memory used by it.
 *
 * return values: none
 */
void screem_helper_destroy( Helper *helper )
{
	g_return_if_fail( helper != NULL );

	g_free( helper->path );
	g_free( helper->mime_type );

	g_free( helper );
}

/*
 * screem_helper_locate:
 * @path: the pathname of the helper to locate
 *
 * locates the helper with the given pathname
 *
 * return values: a Helper pointer
 */
Helper* screem_helper_locate( gchar *path )
{
	GList *list;
	Helper *helper = NULL;

	g_return_val_if_fail( path != NULL, NULL );

	for( list = cfg->hpage->helpers; list && ! helper; list = list->next ){
		helper = (Helper*)list->data;
		if( strcmp( path, helper->path ) )
			helper = NULL;
	}

	return helper;
}

/*
 * screem_helper_add:
 * @helper: the helper to add to the list
 *
 * adds the given helper to the list of helpers
 *
 * return values: none
 */
void screem_helper_add( Helper *helper )
{
	g_return_if_fail( helper != NULL );

	cfg->hpage->helpers = g_list_append( cfg->hpage->helpers, helper );
}

/*
 * screem_helper_remove:
 * @path: the pathname of the helper to remove
 *
 * removes the helper with the given pathname from the list of helpers
 *
 * return values: a Helper pointer
 */
Helper* screem_helper_remove( gchar *path )
{
	Helper *helper;

	helper = screem_helper_locate( path );

	if( helper )
		cfg->hpage->helpers = g_list_remove( cfg->hpage->helpers, 
						     helper );

	return helper;
}

/*
 * screem_helper_execute:
 * @path: the pathname of the helper to execute
 *
 * executes the helper
 *
 * return values: none
 */
void screem_helper_execute( gchar *path )
{
	Helper *helper;
	Page *page;
	int p[ 2 ];
	int q[ 2 ]; 
	gchar *backup;
	gint status;
	FILE *stream;

	pid_t cpid;
	gboolean complete = FALSE;

	g_return_if_fail( path != NULL );

	helper = screem_helper_locate( path );

	g_return_if_fail( helper != NULL );

	page = screem_site_get_current_page( current_site );

	g_return_if_fail( page != NULL );

	screem_editor_buffer_text();

	pipe( p );
	pipe( q );
	switch( ( cpid = fork() ) ) {
		/* error */
	case -1:
		break;
		/* child */
	case 0:
		close( STDIN );
                dup( p[ 0 ] );
                close( p[ 1 ] );
                close( STDOUT );
                dup( q[ 1 ] );
                close( q[ 0 ] );
                execl( "/bin/sh", "/bin/sh", "-c", path, NULL );
                _exit( 0 );
		break;
		/* parent */
	default:
		backup = g_strdup( screem_page_get_data( page ) );

		close( p[ 0 ] );
                close( q[ 1 ] );
                stream = fdopen( p[ 1 ], "w" );
                fprintf( stream, "%s", screem_page_get_data( page ) );
                fclose( stream );
                screem_page_read( page, q[ 0 ] );
                waitpid( cpid, &status, 0 );

		if( WIFEXITED( status ) != 0  ) {
			g_free( backup );
			backup = screem_page_get_data( page );
			if( backup ) {
				screem_editor_clear();
				screem_editor_insert( 0, screem_page_get_data(page) );
				backup = NULL;
				complete = TRUE;
			}
		} else {
			screem_page_set_data( page, backup );
		}
		g_free( backup );
		if( ! complete ) {
			/* the helper failed to complete properly */
			screem_do_show_error( _("The helper failed to complete successfully"), FALSE );
		}
		break;
	}
}

/*
 * screem_helper_menu:
 *
 * builds the menu heirachy for the helpers menu(s)
 *
 * return values: none
 */
GtkWidget *screem_helper_menu()
{
	GtkWidget *menu_item;
	GtkWidget *menu;

	GtkWidget *item;

	GList *list;
	Helper *helper;

	menu_item = gtk_menu_item_new_with_label( _("Exec Helpers") );
	gtk_widget_show( menu_item );

	menu = gtk_menu_new();
	gtk_menu_item_set_submenu( GTK_MENU_ITEM( menu_item ), menu );

	item = gtk_menu_item_new_with_label( _("All Helpers") );
	gtk_signal_connect( GTK_OBJECT( item ), "activate",
			    GTK_SIGNAL_FUNC( screem_helper_execute_all ), 0 );
	gtk_menu_append( GTK_MENU( menu ), item );
	gtk_widget_show( item );

	for( list = cfg->hpage->helpers; list; list = list->next ) {
		helper = list->data;
		item = gtk_menu_item_new_with_label(helper->path);
		gtk_signal_connect_object( GTK_OBJECT( item ), "activate", GTK_SIGNAL_FUNC( screem_helper_execute ), (gpointer)helper->path );
		gtk_menu_append( GTK_MENU( menu ), item );
		gtk_widget_show( item );
	}

	return menu_item;
}

static void screem_helper_execute_all()
{
	GList *list;
	Helper *helper;

	for( list = cfg->hpage->helpers; list; list = list->next ) {
		helper = (Helper*)list->data;
		screem_helper_execute( helper->path );
	}
}

void screem_helpers_build_toolbar()
{
	GtkToolbar *helpers_bar;
	GtkWidget *button;
	GtkWidget *widget;
	GList *list;
	Helper *helper;
	gchar *text;

	GtkToolbarChild *child;

	helpers_bar = GTK_TOOLBAR( gtk_object_get_data( GTK_OBJECT( app ), 
							"helperbar" ) );

	/* remove previous buttons, we ignore the first button as
	   its not a helper */
	list = helpers_bar->children->next;
	if( list ) {
		list->prev->next = NULL;
		list->prev = NULL;
	}
	while( list ) {
		child = list->data;
		
		gtk_widget_unparent( child->widget );
		g_free( child );
		helpers_bar->num_children --;

		list = list->next;
	}
	g_list_free( g_list_first( list ) );
	gtk_widget_queue_resize( GTK_WIDGET( helpers_bar ) );

	/* add new ones */
	for( list = cfg->hpage->helpers; list; list = list->next ) {
		helper = (Helper*)list->data;
		if( helper->icon )
			button =gnome_pixmap_new_from_file_at_size(helper->icon,
								   24, 24 );
		else
			button = gnome_pixmap_new_from_xpm_d(browser_xpm);

		gtk_widget_show( button );
		text = g_basename( helper->path );
		gtk_toolbar_append_item( GTK_TOOLBAR( helpers_bar ),
					 text, helper->path, "", button,
					 GTK_SIGNAL_FUNC( 0 ), NULL );
		widget = button->parent->parent;
		gtk_object_set_data( GTK_OBJECT( widget ), "pixmap",
				     button );
		gtk_signal_connect_after( GTK_OBJECT( widget ),
					  "button_press_event",
					  helper_button_clicked, helper );
	}
}

static gboolean helper_button_clicked(GtkWidget *widget,  GdkEventButton *event,
				      Helper *helper )
{
	if( event->button == 1 ) {
		/* execute this script */
		screem_helper_execute( helper->path );
	} else if( event->button == 3 ) {
		/* configure the button */
		helper_configure( widget, helper );
	}

	return TRUE;
}

static void helper_configure( GtkWidget *widget, Helper *helper )
{
	GtkWidget *dialog;
	GtkWidget *path;
	GtkWidget *icon;
	GtkTooltipsData *ttd;

	GtkWidget *pixmap;

	GladeXML *xml;
	gint button;

	gchar *temp;

	if( ! helper )
		return;

	xml = glade_xml_new( cfg->glade_path, "helpers_dialog" );

	dialog = glade_xml_get_widget( xml, "helpers_dialog" );
	path = glade_xml_get_widget( xml, "helper_path" );
	icon = glade_xml_get_widget( xml, "helper_icon" );

	/* set the path/icon */
	gtk_entry_set_text( GTK_ENTRY( path ), helper->path );

	gnome_icon_entry_set_icon(GNOME_ICON_ENTRY(icon), helper->icon );

   	gtk_widget_show_all( icon );

	button = gnome_dialog_run( GNOME_DIALOG( dialog ) );

	if( button == 0 ) {
		g_free( helper->path );
		temp = gtk_entry_get_text( GTK_ENTRY( path ) );
		helper->path = g_strdup( temp );

		temp = gtk_entry_get_text( GTK_ENTRY( gnome_icon_entry_gtk_entry( GNOME_ICON_ENTRY( icon ) ) ) );

		g_free( helper->icon );
		helper->icon = g_strdup( temp );

		pixmap = gtk_object_get_data( GTK_OBJECT( widget ),
					      "pixmap" );
		gnome_pixmap_load_file_at_size( GNOME_PIXMAP( pixmap ), 
						temp, 24, 24 );

		/* set the tooltip to the programs path */
		ttd = gtk_tooltips_data_get( widget );
		gtk_tooltips_set_tip( ttd->tooltips, ttd->widget, 
				      helper->path, "");
	}

	gtk_widget_destroy( dialog );
}

void edit_helper_scripts()
{
	edit_preferences();
}
