/*  Screem:  page.c,
 *  a tidied up page.c, separating ui from backend
 * 
 *  Copyright (C) 1999, 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <ctype.h>
#include <fcntl.h>
#include <gnome.h>
#include <stdio.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <libgnome/libgnome.h>

#ifdef HAVE_GNOME_VFS
#include <libgnomevfs/gnome-vfs-mime.h>
#endif

#include "page.h"
#include "pageUI.h"

typedef struct RealPage {
	gchar *pathname;     /* pathname for the page */
        gchar *data;         /* the actual page data */
	gint tab;            /* the tab number of the page if open */
	gchar *charset;      /* the charset name, or NULL */
	const gchar *mime_type;
	gboolean changed;
	time_t mod_time;     /* last modified time */
	GList *undo_list;
	GList *redo_list;
} RealPage;

/**
 * screem_page_new:
 *
 * Creates and initialises a new page struct
 *
 * return values: a Page
 */
Page *screem_page_new()
{
	RealPage *p;

	p = (RealPage*)g_new( RealPage, 1 );

	p->pathname = NULL;
	p->data = NULL;
	p->tab = -1;
	p->charset = NULL;
	p->changed = FALSE;
	p->mime_type = "text/plain";

	p->undo_list = NULL;
	p->redo_list = NULL;

	return (Page*)p;
}

/**
 * screem_page_destroy:
 * @page:  the page to destroy
 *
 * destroys the page struct and any memory it may be taking up
 *
 * return values: none
 */
void screem_page_destroy( Page *page )
{
	RealPage *p;

        g_return_if_fail( page != NULL );

	p = (RealPage*)page;

	g_free( p->pathname );
	g_free( p->data );
	g_free( p->charset );
	g_free( p );
}

/**
 * screem_page_set_pathname:
 * @page:  the page to set
 * @path:  the pathname to set to
 *
 * sets the given pages pathname to path
 *
 * return values: none
 */
void screem_page_set_pathname( Page *page, const gchar *path )
{
	RealPage *p;

	g_return_if_fail( page != NULL );

	p = (RealPage*)page;

	if( p->pathname ) {
		if( ! strcmp( p->pathname, path ) )
			return;
		else
			g_free( p->pathname );
	}

	p->pathname = g_strdup( path );

#ifndef HAVE_GNOME_VFS
	p->mime_type = gnome_mime_type( path );
#else
	p->mime_type = gnome_vfs_mime_type_from_name( path );
#endif
}

/**
 * screem_page_get_pathname:
 * @page:  the page
 *
 * obtain the pathname for a page
 *
 * return values: the pathname
 */
const gchar *screem_page_get_pathname( Page *page )
{
	g_return_val_if_fail( page != NULL, NULL );

	return ((RealPage*)page)->pathname;
}

const gchar *screem_page_get_mime_type( Page *page )
{
	g_return_val_if_fail( page != NULL, NULL );

	return ((RealPage*)page)->mime_type;
}

/**
 * screem_page_set_tab:
 * @page:  the page to set
 * @tab:   the tab number
 *
 * stores the notebook tab number of the page
 *
 * return values: none
 */
void screem_page_set_tab( Page *page, gint num )
{
	g_return_if_fail( page != NULL );

	((RealPage*)page)->tab = num;
}

/**
 * screem_page_get_tab:
 * @page:  the page to get from
 *
 * retrieves the notebook tab number of the page
 *
 * return values: the tab number of the page
 */
gint screem_page_get_tab( Page *page )
{
	g_return_val_if_fail( page != NULL, -1 );

	return ((RealPage*)page)->tab;
}

/**
 * screem_page_set_charset:
 * @page:  the page to set
 * @charset: the charset
 *
 * sets the charset name that the page uses
 *
 * return values: none
 */
void screem_page_set_charset( Page *page, const gchar *charset )
{
	RealPage *p;

	g_return_if_fail( page != NULL );

	p = (RealPage*)page;

	if( p->charset ) {
		if( ! strcmp( p->charset, charset ) )
			return;
		else
			g_free( p->charset );
	}

	p->charset = g_strdup( charset );
}

/**
 * screem_page_get_charset:
 * @page:  the page to get from
 *
 * retrieves the charset name for the page
 *
 * return values: the character set name the page is using
 */
const gchar *screem_page_get_charset( Page *page )
{
	g_return_val_if_fail( page != NULL, NULL );

	return ((RealPage*)page)->charset;
}


void screem_page_set_data( Page *page, const gchar *data )
{
	RealPage *p;

	g_return_if_fail( page != NULL );

	p = (RealPage*)page;

	if( p->data ) {
		if( data && ! strcmp( p->data, data ) ) {
			/* its already the same! */
			return;
		}
		g_free( p->data );
		screem_page_set_changed( page, TRUE );
	}

	if( data )
		p->data = g_strdup( data );
	else
		p->data = NULL;
}

const gchar *screem_page_get_data( Page *page )
{
	g_return_val_if_fail( page != NULL, NULL );

	return ((RealPage*)page)->data;
}

void screem_page_set_changed( Page *page, gboolean flag )
{
	g_return_if_fail( page != NULL );

	((RealPage*)page)->changed = flag;
}

gboolean screem_page_get_changed( Page *page )
{
	g_return_val_if_fail( page != NULL, FALSE );

	return ((RealPage*)page)->changed;
}

void screem_page_set_mod_time( Page *page, time_t t )
{
	g_return_if_fail( page != NULL );

	((RealPage*)page)->mod_time = t;
}

time_t screem_page_get_mod_time( Page *page )
{
	g_return_val_if_fail( page != NULL, 0 );

	return ((RealPage*)page)->mod_time;
}

/**
 * screem_page_read:
 * @page:  the page to read into
 * @fd:    the file descriptor to read from
 *
 * reads from file descriptor fd into page->data
 * returns TRUE if successful, FALSE on error
 * The fd is not closed, even if an error occurs, it is down to the calling
 * function to do this.
 *
 * return values: boolean
 */
gboolean screem_page_read( Page *p, int fd )
{
	int size = 1;
 	gchar buffer[ BUFSIZ + 1 ];
	gchar *temp;

	RealPage *page = (RealPage*)p;

        g_return_val_if_fail( page != NULL, FALSE );

	page->data = g_strdup( "" );

	while( size ) {
		size = read( fd, buffer, BUFSIZ );
		if( size == -1 ) {
			/* there was an error with the read */
			g_free( page->data );
			page->data = NULL;
			return FALSE;
		}
		buffer[ size ] = 0;
		temp = page->data;
		page->data = g_strconcat( temp, buffer, NULL );
		g_free( temp );
	}

	return TRUE;
}

/**
 * screem_page_load:
 * @page:  the page to load
 *
 * If the page isn't already in memory then this function will
 * load it.
 * returns FALSE on error, TRUE if successful.
 * the page already being loaded is not counted as an error
 *
 * return values: boolean
 */
gboolean screem_page_load( Page *page )
{
	int fd;
	gboolean ret;
	const gchar *path;
	struct stat s;

        g_return_val_if_fail( page, FALSE );

	if( screem_page_get_data( page ) )
		return TRUE;

	path = screem_page_get_pathname( page );

	/* page exist? */
	if( stat( path, &s ) < 0 )
		return FALSE;

	/* load the page */
	fd = open( path, O_RDONLY );

	if( fd == -1 ) {
		/* failed to open the page */
		return FALSE;
	}

	ret = screem_page_read( page, fd );

	/* set the page modtime */
    	screem_page_set_mod_time( page, s.st_mtime );

	close( fd );
	return ret;
}

/**
 * screem_page_revert:
 * @page:  the page to revert
 *
 * reverts the page to the version on disk
 *
 * return values: boolean
 */
gboolean screem_page_revert( Page *page )
{
	g_return_val_if_fail( page, FALSE );

	screem_page_set_data( page, NULL );

	return screem_page_load( page );
}

/**
 * screem_page_save:
 * @page:  the page to save
 *
 * saves the given page, if the page wasn't loaded then we just return
 *
 * return values: boolean
 */
gboolean screem_page_save( Page *page )
{
	int size;
	FILE *out;

	g_return_val_if_fail( page, FALSE );

	if( ! screem_page_get_data( page ) )
		return TRUE;

	if( ! screem_page_get_changed( page ) ) {
		g_print( "not saving %s as it hasn't changed\n",
			 screem_page_get_pathname( page ) );
		return TRUE;
	}

	if( ! screem_page_save_check( page ) )
		return TRUE;

	size = strlen( screem_page_get_data( page ) );

	out = fopen( screem_page_get_pathname( page ), "w" );
	if( ! out )
		return FALSE;

	fwrite( screem_page_get_data( page ), 1, size, out );
	fclose( out );

	screem_page_set_changed( page, FALSE );
	screem_page_set_mod_time( page, time( 0 ) );

	return TRUE;
}

Page *screem_page_from_tab_num( GList *list, gint num )
{
	Page *page = NULL;

	g_return_val_if_fail( num > -1, NULL );

	for( ; list && !page; list = list->next ) {
		if( num == screem_page_get_tab( (Page*)list->data ) )
			page = (Page*)list->data;
	}
	
	return page;
}

void screem_page_update_tab_nums( GList *list, gint num )
{
	Page *page = NULL;
	gint num2;

	g_return_if_fail( num > -1 );

	for( ; list; list = list->next ) {
		page = (Page*)list->data;
		num2 = screem_page_get_tab( page );
		if( num2 > num ) 
			screem_page_set_tab( page, --num2 );
		else if( num2 == num )
			screem_page_set_tab( page, -1 ); /* the closed one */
	}
}

gboolean screem_page_is_file_page( const gchar *path )
{
	const gchar *mime_type;

#ifndef HAVE_GNOME_VFS
	mime_type = gnome_mime_type( path );
#else
	mime_type = gnome_vfs_mime_type_from_name( path );
#endif

	return screem_page_is_mime_type_page( mime_type );
}

gboolean screem_page_is_mime_type_page( const gchar *mime_type )
{
	gboolean is_page;

	g_return_val_if_fail( mime_type != NULL, FALSE );

	is_page = ! strncmp( mime_type, "text/", strlen( "text/" ) );
	is_page |= ! strcmp( mime_type, "application/x-php" );
	is_page |= ! strcmp( mime_type, "application/x-asp" );
	is_page |= ! strcmp( mime_type, "application/x-cgi" );
	
	return is_page;
}

Undo* screem_page_get_undo_last_action( Page *page, gboolean remove )
{
	GList *list;
	Undo *undo;
	RealPage *rp;

	g_return_val_if_fail( page != NULL, NULL );

	rp = (RealPage*)page;

	list = rp->undo_list;

	if( ! list )
		return NULL;

	undo = (Undo*)list->data;
		
	/* remove the action from the list */
	if( remove )
		rp->undo_list = g_list_remove( rp->undo_list, undo );

	return undo;
}

void screem_page_set_undo_last_action( Page *page, gboolean insert, gint pos, 
				       const gchar *text )
{
	Undo *undo;
	RealPage *rp;

	g_return_if_fail( page != NULL );
	g_return_if_fail( text != NULL );

	undo = g_new( Undo, 1 );

	undo->insert = insert;
	undo->text = g_strdup( text );
	undo->pos = pos;

	rp = (RealPage*)page;

	rp->undo_list = g_list_prepend( rp->undo_list, undo );
}

void screem_page_add_undo_action( Page *page, Undo *undo )
{
	RealPage *rp;

	g_return_if_fail( page != NULL );
	g_return_if_fail( undo != NULL );

	rp = (RealPage*)page;

	rp->undo_list = g_list_prepend( rp->undo_list, undo );
}

void screem_page_clear_undo_list( Page *page )
{
	RealPage *rp;
	GList *list;
	Undo *undo;

	g_return_if_fail( page != NULL );

	rp = (RealPage*)page;
	for( list = rp->undo_list; list; list = list->next ) {
		undo = (Undo*)list->data;
		g_free( undo->text );
		g_free( undo );
	}
	g_list_free( rp->undo_list );
	rp->undo_list = NULL;
}

Undo* screem_page_get_redo_last_action( Page *page, gboolean remove )
{
	GList *list;
	Undo *undo;
	RealPage *rp;

	g_return_val_if_fail( page != NULL, NULL );

	rp = (RealPage*)page;

	list = rp->redo_list;

	if( ! list )
		return NULL;

	undo = (Undo*)list->data;
		
	/* remove the action from the list */
	if( remove )
		rp->redo_list = g_list_remove( rp->redo_list, undo );

	return undo;
}

void screem_page_set_redo_last_action( Page *page, Undo *undo )
{
	RealPage *rp;

	g_return_if_fail( page != NULL );
	g_return_if_fail( undo != NULL );

	rp = (RealPage*)page;

	rp->redo_list = g_list_prepend( rp->redo_list, undo );
}

void screem_page_clear_redo_list( Page *page )
{
	RealPage *rp;
	GList *list;
	Undo *undo;

	g_return_if_fail( page != NULL );

	rp = (RealPage*)page;
	for( list = rp->redo_list; list; list = list->next ) {
		undo = (Undo*)list->data;
		g_free( undo->text );
		g_free( undo );
	}
	g_list_free( rp->redo_list );
	rp->redo_list = NULL;
}

void screem_page_chdir( Page *page )
{
	const gchar *path;
	gchar *dir;

	g_return_if_fail( page != NULL );

	path = screem_page_get_pathname( page );
	dir = g_dirname( path );

	chdir( dir );

	g_free( dir );
}
