/*  Screem:  editMenu.c,
 *  This file handles find/replace
 *
 *  Copyright (C) 2000  David A Knight
 *
 *  TODO:
 *  
 *  ability to find backwards as well as forward?
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <sys/types.h>
#include <gnome.h>
#include <libgnome/gnome-regex.h>
#include <glade/glade.h>

#include "editor.h"
#include "editMenu.h"
#include "htmlfuncs.h"
#include "site.h"
#include "support.h"
#include "page.h"
#include "pageUI.h"

#include "preferences.h"

extern Site *current_site;
extern Preferences *cfg;

static void search_all_pages( GtkWidget *widget );
static gint search( const gchar *text, const gchar *find, gboolean use_regex,
		    gint start, gint *length );


void show_hide_matches( GtkWidget *widget );
void do_find( GtkWidget *widget );
void do_replace( GtkWidget *widget );
void do_replace_all( GtkWidget *widget );
void find_change_file( GtkCList *clist, gint row, gint column,
		       GdkEventButton *event, gpointer data );


/**
 * find_text:
 *
 * @string:  the string to search
 * @find:    the string to search for
 * @replace: the string to replace find with, NULL if not replacing
 * @length:  a pointer to the place to store the length of the match,
 *           NULL if not wanted.
 *
 * Find the string find in string, and replace with replace if given,
 * storing the length of the match in length, if given
 *
 * return: a pointer to the found position, or the new text if a replace was
 *         performed
 */
gchar *find_text( const gchar *string, const gchar *find, 
		  const gchar *replace, gint *length )
{
	gchar *chunk1;
	gchar *chunk2;
	gchar *newChunk;
	gchar *text = NULL;
	gint found_at;
	gint find_length;

       	found_at = search( string, find, TRUE, 0, &find_length );
    
	if( found_at == -1 )
		return NULL;

	/* replace the text if needed */
	if( replace ) {
		chunk1 = g_strndup( string, found_at );
		chunk2 = g_strdup( string + found_at + find_length );
		newChunk = g_strconcat( chunk1, replace, chunk2, NULL );
		g_free( chunk1 );
		g_free( chunk2 );
		text = newChunk;
	} else
		text = (gchar*)string + found_at;
      
	if( length )
		*length = find_length;

	return text;
}


/**
 * select_all:
 *
 * selects all the text in the current editor.
 *
 * return: none
 */
void select_all()
{
       	Page *page;

	page = screem_site_get_current_page( current_site );
     
	g_return_if_fail( page != NULL );

	screem_editor_select_region( 0, screem_editor_get_length() );
}

/**
 * screem_tags_to_upper:
 *
 * proxy to call to screem_tags_case_change with the correct flag
 *
 * returns: none
 */
void screem_tags_to_upper()
{
	screem_tags_case_change( TRUE );
}

/**
 * screem_tags_to_lower:
 *
 * proxy to call to screem_tags_case_change with the correct flag
 *
 * returns: none
 */
void screem_tags_to_lower()
{
	screem_tags_case_change( FALSE );
}

/**
 * screem_tags_case_change:
 *
 * @upper: TRUE if the change is to uppercase, FALSE if it is to lowercase.
 *
 * Changes the case of all tags and attributes in the current document
 *
 * returns: none
 */
void screem_tags_case_change( gboolean upper )
{
	Page *page;
	gchar *text;

	gint start = 0;
	gint len = -1;

	page = screem_site_get_current_page( current_site );

	if( ! page )
		return;

	if( screem_editor_has_selection( &start, &len ) )
		len = len - start;
    
	text = screem_editor_get_text( start, len );
	screem_tags_change_case( text, upper );
	screem_editor_delete_forward( start, len );
	screem_editor_insert( start, text );
	g_free( text );
}

/**
 * find:
 *
 * proxy to call find replace with the correct flag
 *
 * returns: none
 */
void find()
{
	find_replace( FALSE );
}

/**
 * replace:
 *
 * proxy to call find replace with the correct flag
 *
 * returns: none
 */
void replace()
{
	find_replace( TRUE );
}

/**
 * find_replace:
 *
 * @replace:   boolean flag to indicate if the replace features should
 *             be made sensitive.
 *
 * displays the find/replace dialog
 *
 * returns: none
 */
void find_replace( gboolean replace )
{
	GladeXML *xml;
	GtkWidget *widget;
	GtkTooltips *tooltips;

	xml = glade_xml_new( cfg->glade_path, "find_dialog" );

	/* setup the match list correctly */
	widget = glade_xml_get_widget( xml, "match_list" );
	gtk_clist_set_column_width( GTK_CLIST( widget ), 0, 16 );
       	gtk_clist_set_selection_mode( GTK_CLIST( widget ),
				      GTK_SELECTION_BROWSE );
	gtk_clist_column_titles_hide( GTK_CLIST( widget ) );
	gtk_object_set_data( GTK_OBJECT( widget ), "matches", NULL );

	/* add tooltips that glade won't let us do */
	widget = glade_xml_get_widget( xml, "check_all" );
	tooltips = gtk_tooltips_new ();
	gtk_tooltips_set_tip( tooltips, widget, _("When searching multiple files, any  which contain a match are shown below, selecting one will open the page in the editor." ), NULL );

	glade_xml_signal_autoconnect( xml );

	/* set last found position to -1 */
	widget = glade_xml_get_widget( xml, "replace_entry" );
	gtk_object_set_data( GTK_OBJECT( widget ), "found_at",
			     GINT_TO_POINTER( -1 ) );

	/* toggle the replace button if we are opened for replacing */
	widget = glade_xml_get_widget( xml, "replace_check" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), replace );
}

/**
 * show_hide_matches:
 *
 * @widget:  the scrolled window we want to show/hide
 *
 * hides/displays the window
 *
 * returns: none
 */
void show_hide_matches( GtkWidget *widget )
{
	if( ! GTK_WIDGET_VISIBLE( widget ) )
		gtk_widget_show( widget );
	else
		gtk_widget_hide( widget );
}

/**
 * search_all_pages:
 *
 * @widget:
 *
 * performs a single find on all pages in the site, and adds the
 * page to the found list if a match is found
 *
 * returns: none
 */
static void search_all_pages( GtkWidget *widget )
{
	GladeXML *xml;
	Page *page;
	GList *list;
	GList *matches;
	const gchar *text;

	gchar *find_string;
	const Icons *icon;
	const gchar *path;
	const gchar *mime_type;
	gint row;
	gchar *item[ 3 ] = { NULL, NULL, NULL };

	xml = glade_get_widget_tree( widget );
	
	widget = glade_xml_get_widget( xml, "find_entry" );
	find_string = gtk_entry_get_text( GTK_ENTRY( widget ) );

	/* find pages with matches */
	for( matches = NULL, list = screem_site_get_pages( current_site ); 
	     list; list = list->next ) {
		page = (Page*)list->data;

		path = screem_page_get_pathname( page );

		/* is path excluded? */
		if( screem_site_is_excluded( current_site, path ) )
			continue;

		screem_page_load( page );
		text = screem_page_get_data( page );
		if( find_text( text, find_string, NULL, NULL ) ) {
			/* add to list */
			matches = g_list_append( matches, page );
		}
	}

	/* display matches */
	widget = glade_xml_get_widget( xml, "match_list" );
	gtk_clist_clear( GTK_CLIST( widget ) );
	gtk_clist_freeze( GTK_CLIST( widget ) );
	gtk_object_set_data( GTK_OBJECT( widget ), "appending",
			     GINT_TO_POINTER( 1 ) );
	for( row = 0, list = matches; list; row ++, list = list->next ) {
		path = screem_page_get_pathname( (Page*)list->data );
		mime_type = screem_page_get_mime_type( (Page*)list->data );
		icon = icon_from_mime_type( path, mime_type );
		item[ 1 ] = (gchar*)path;
		gtk_clist_append( GTK_CLIST( widget ), item );
		gtk_clist_set_pixmap( GTK_CLIST( widget ), row, 0, icon->open,
				      icon->open_mask );
	}
	gtk_clist_columns_autosize( GTK_CLIST( widget ) );
	gtk_clist_thaw( GTK_CLIST( widget ) );

	gtk_object_set_data( GTK_OBJECT( widget ), "appending",
			     GINT_TO_POINTER( 0 ) );
	gtk_object_set_data( GTK_OBJECT( widget ), "matches", matches );
}

/**
 * do_find:
 *
 * @widget: the find next button
 *
 * performs the next find
 *
 * returns: none
 */
void do_find( GtkWidget *widget )
{
	GladeXML *xml;
	gchar *find_string;
	gboolean use_regex;

	gchar *text;
	gint found_at;
	gint length;
	gint pos;

	gboolean check_all;

	xml = glade_get_widget_tree( widget );

	/* if there is no match list, and the user wants to check all
	   pages then check them */
	widget = glade_xml_get_widget( xml, "match_list" );
	if( ! gtk_object_get_data( GTK_OBJECT( widget ), "matches" ) ) {
		widget = glade_xml_get_widget( xml, "check_all" );
		check_all = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(widget) );
		if( check_all )
			search_all_pages( widget );
	}

	widget = glade_xml_get_widget( xml, "find_entry" );
	find_string = gtk_entry_get_text( GTK_ENTRY( widget ) );

	widget = glade_xml_get_widget( xml, "regex" );
	use_regex = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(widget) );

	widget = glade_xml_get_widget( xml, "pos_start" );
	pos = 0;
	if( ! gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
		pos = screem_editor_get_pos();

	widget = glade_xml_get_widget( xml, "replace_entry" );
	found_at = (gint)gtk_object_get_data( GTK_OBJECT( widget ),
					      "found_at" );
	if( found_at != -1 )
		pos = found_at + 1;

	text = screem_editor_get_text( 0, -1 );

	found_at = search( text, find_string, use_regex, pos, &length );

	gtk_object_set_data( GTK_OBJECT( widget ), "found_at",
				     GINT_TO_POINTER( found_at ) );

	if( found_at != -1 ) {
		/* goto and highlight the string */
		screem_editor_set_pos( found_at );
		screem_editor_select_region( found_at, found_at + length );
	
		gtk_object_set_data( GTK_OBJECT( widget ), "length",
				     GINT_TO_POINTER( length ) );
	} else {
		/* inform user not found */

	}

	g_free( text );
}

/**
 * do_replace:
 *
 * @widget: the replace button
 *
 * replaces the last found word
 *
 * returns: none
 */
void do_replace( GtkWidget *widget )
{
	GladeXML *xml;
	gchar *replace_string;

	gint found_at;
	gint length;

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "replace_entry" );
	replace_string = gtk_entry_get_text( GTK_ENTRY( widget ) );

	found_at = (gint) gtk_object_get_data( GTK_OBJECT( widget ), 
					       "found_at" );
	length = (gint) gtk_object_get_data( GTK_OBJECT( widget ),
					     "length" );

	if( found_at == -1 ) {
		/* there was no string found */
		return;
	}

	screem_editor_delete_forward( found_at, length );
	screem_editor_insert( found_at, replace_string );
	found_at += strlen( replace_string );

	gtk_object_set_data( GTK_OBJECT( widget ), "found_at",
			     GINT_TO_POINTER( found_at ) );

	while( gtk_events_pending() )
		gtk_main_iteration();

	screem_editor_set_pos( found_at );

	/* find the next one automatically */
	do_find( widget );
}

/**
 * do_replace_all:
 *
 * @widget: the replace all button
 *
 * replaces all occurances of the find pattern
 *
 * returns: none
 */
void do_replace_all( GtkWidget *widget )
{
	GladeXML *xml;
	gint found_at;
	GList *list;
	GtkWidget *widget2;
	Page *page;
	xml = glade_get_widget_tree( widget );

	widget = glade_xml_get_widget( xml, "match_list" );
	list = (GList*)gtk_object_get_data( GTK_OBJECT( widget ), "matches" );

	widget = glade_xml_get_widget( xml, "replace_entry" );
	widget2 = glade_xml_get_widget( xml, "match_window" );

	while( list || ! GTK_WIDGET_VISIBLE( widget2 ) ) {
		found_at = (gint) gtk_object_get_data( GTK_OBJECT( widget ), 
						       "found_at" );
		
		if( list ) {
			/* insert the next page */
			page = (Page*)list->data;
			screem_page_insert( page );
			found_at = -1;
		}			

		if( found_at == -1 )
			do_find( widget );

		found_at = (gint) gtk_object_get_data( GTK_OBJECT( widget ), 
						       "found_at" );

		/* replace all occurances in this document */
		while( found_at != -1 ) {
			do_replace( widget );
			found_at = (gint) gtk_object_get_data( GTK_OBJECT( widget ), "found_at" );
		}

		while( gtk_events_pending() )
			gtk_main_iteration();

	

		if( list ) {
			list = list->next;
		}
		else
			break;

		while( gtk_events_pending() )
			gtk_main_iteration();
       	}
}

/**
 * find_change_file:
 * 
 * @list:  the clist displaying the matches
 * @row:   the row number that was selected
 * @col:   the column number, not really used, but here as the signal requires
 *         it
 * @event: the event that occured
 *
 * changes the page in the editor to the selected one, ready to be searched,
 * and replaced if wanted.
 *
 * returns: none
 */
void find_change_file( GtkCList *clist, gint row, gint column,
		       GdkEventButton *event, gpointer data )
{
	gchar *info[ 1 ];
	Page *page;

	GtkWidget *widget;
	GladeXML *xml;

	xml = glade_get_widget_tree( GTK_WIDGET( clist ) );
	widget = glade_xml_get_widget( xml, "replace_entry" );

	if( gtk_object_get_data( GTK_OBJECT( clist ), "appending" ) )
		return;

	if( event->type != GDK_2BUTTON_PRESS )
		return;

	gtk_clist_get_text( clist, row, 1, &info[ 0 ] );

	page = screem_site_locate_page( current_site, info[ 0 ] );

	if( ! page )
		return;
	
	screem_page_insert( page );

	/* open the page in the editor and perform a single page search */
	while( gtk_events_pending() )
		gtk_main_iteration();
	
	gtk_object_set_data( GTK_OBJECT( widget ), "found_at",
			     GINT_TO_POINTER( -1 ) );

	do_find( GTK_WIDGET( clist ) );
}

/**
 * search:
 *
 * @text:      the text to search
 * @find:      the string to find
 * @use_regex: search uses regular expressions
 * @start:     offset in string to start searching at
 * @length:    a pointer to an integer to store the length of the match in,
 *             can be NULL if the value isn't interested in
 *
 * finds a string inside a piece of text
 *
 * returns: the position in text that find was found
 */
static gint search( const gchar *text, const gchar *find, gboolean use_regex,
		    gint start, gint *length )
{
#define MAX_EXPRESSIONS 12

	gchar *pos;
	static GnomeRegexCache* regexCache = NULL;
	regex_t *expressions;
	static size_t nmatch = 12;
	static regmatch_t pmatch[ MAX_EXPRESSIONS ];

	gint retval = start;

	if( ! regexCache ) {
		regexCache = gnome_regex_cache_new();
		gnome_regex_cache_set_size( regexCache, MAX_EXPRESSIONS );
	}

	if( use_regex ) {
		expressions = gnome_regex_cache_compile( regexCache, find, 
							 REG_EXTENDED );
		if( regexec( expressions, text + start, nmatch, pmatch, 0 ) )
			retval = -1;
		 else
			 retval += pmatch[ 0 ].rm_so;

		if( length )
			*length = pmatch[0].rm_eo - pmatch[0].rm_so;
	} else {
		/* not using regular expressions */
		pos = strstr( text + start, find );
		if( ! pos )
			retval = -1;
		else
			retval = pos - text;

		if( length )
			*length = strlen( find );
	}

	return retval;
}
