/*  Screem:  linkView.c,
 *  Site wide link view, centered around the current page
 *
 *
 *  TODO:
 *  1) do the drag scrolling better
 *
 *  Copyright (C) 1999, 2000, 2001  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gnome.h>
#include <unistd.h>
#include <gmodule.h>

#include <gnome-xml/debugXML.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>

#include <gdk-pixbuf/gnome-canvas-pixbuf.h>

#include <glade/glade.h>

#include <sys/stat.h>
#include <math.h>

#ifdef HAVE_GNOME_VFS
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <libgnomevfs/gnome-vfs-context.h>
#include <libgnomevfs/gnome-vfs-cancellable-ops.h>
#endif

#include <setjmp.h>
#include <pthread.h>
#include <signal.h>

#include "editor.h"
#include "fileops.h"
#include "htmlfuncs.h"
#include "linkview.h"
#include "preferences.h"
#include "site.h"
#include "support.h"
#include "plugin.h"
#include "pageUI.h"

static int timeout_handle = -1;
static GHashTable *external_check_cache = NULL;

static gboolean screem_link_view_check_next( void );


#ifdef HAVE_GNOME_VFS

void screem_link_view_check_remote_fill( GNode *node, RemoteStatus status );

#endif

extern GtkWidget *app;
extern Site *current_site;
extern Preferences *cfg;

/* Yuk! look at all these horrible global variables */

static gdouble max_x, max_y, min_x, min_y;

static gint zoom = 100;
static gint depth = 3;

/* global for the sidebar_switch functions, needed so we can hide/show
   text nodes (I think - Dave) */
static GNode *link_view_tree = NULL;

pthread_mutex_t check_lock;
static GList *node_check = NULL;

gboolean abort_check = FALSE;

static void screem_link_view_drag_scroll( GnomeCanvasItem *item,
					  GdkEventButton *event );
void screem_link_view_zoom( GtkWidget *menuitem );

static GNode* build_normal_node(Page *page, const gchar *pathname, GNode *parent,
				GHashTable *upload_table ); 
static void build_image_node(Page *page, const gchar *pathname, GNode *parent,
			     GHashTable *upload_table ); 

static gint background_event( GnomeCanvasItem *item, GdkEvent *event );
GnomeCanvasItem *node_draw( GnomeCanvasGroup *root, gdouble x, gdouble y, 
			    gint node_type, gboolean legend );

gboolean link_view_labels_off( GNode* node, gpointer data );
gboolean link_view_labels_on( GNode* node, gpointer data );
static gint item_event( GnomeCanvasItem *group, GdkEvent *event, 
			GnomeCanvasItem *item );
gboolean traverse_for_remote(GNode *node, gpointer data );

/* the main canvas */
static 	GtkWidget *canvas;

/**
 * screem_link_view_create_icons:
 *
 * Creates a list of canvas items for each file in the site
 *
 * return values: none
 */
void screem_link_view_create()
{
	static GnomeCanvasGroup *root = NULL;
	static GNode *tree = NULL;
	GtkWidget *legend;
	GnomeCanvasGroup *mainroot;
	GnomeCanvasGroup *legendroot;
	static GnomeCanvasItem *background = NULL;
	GnomeCanvasItem *key;
	GtkType type;

	GladeXML *xml;
	GtkWidget *spin;
	GtkWidget *labels;

	Page *page;

	SiteSyncState screem_site_sync_status;
	GModule *uploadWiz;

	GHashTable *checked_files = NULL;
	GHashTable *upload_table = NULL;

	/* fake site does not support this */
	if( screem_site_get_fake_flag( current_site ) )
		return;

	uploadWiz = screem_plugin_get_sitecopy();
	
	screem_link_view_stop_check();

	page = screem_site_get_current_page( current_site );

	/* we must have a page open */
	if( ! page )
		return;

	screem_editor_buffer_text();

	if( uploadWiz ) {
		g_module_symbol( uploadWiz, "screem_site_get_sync_status", 
				 (gpointer*)&screem_site_sync_status );
		
		screem_site_sync_status( &upload_table );

		/* if failed table will be NULL, otherwise the status
		   of a file can be obtained via 
		   g_hash_table_lookup( table, filename );
		   the states are defined in linkView.h */
	}


	/* get the canvas */
	canvas = gtk_object_get_data( GTK_OBJECT( app ), "link_view" );
	
	legend = gtk_object_get_data( GTK_OBJECT( app ), "link_view_legend" );	
	legendroot = GNOME_CANVAS_GROUP(gnome_canvas_root(GNOME_CANVAS(legend)));
	
	/* add the key */
	key = (GnomeCanvasItem*)screem_link_view_add_key( legendroot,
							  -10,
							  0);
	gnome_canvas_item_raise_to_top( key );
	
	gtk_widget_hide( canvas );
	/* do we already have our root object ?
	   if yes then delete it, destroying all its children */
	if( root ) {
		gtk_object_destroy( GTK_OBJECT( background ) );
		gtk_object_destroy( GTK_OBJECT( root ) );
		remove_data ( tree );
		g_node_destroy( tree );
	}

	gnome_canvas_set_pixels_per_unit( GNOME_CANVAS( canvas ), zoom/100.0 );

	max_x = 0.0;
	max_y = 0.0;
	min_x = 0.0;
	min_y = 0.0;
	
	/* create our new root object */
	mainroot = GNOME_CANVAS_GROUP(gnome_canvas_root(GNOME_CANVAS(canvas)));
	type = gnome_canvas_group_get_type();
	root = GNOME_CANVAS_GROUP( gnome_canvas_item_new( mainroot, type,
							  "x", 0.0,
							  "y", 0.0,
							  NULL ) );

	/* get the depth from the spin button */
	xml = glade_get_widget_tree( canvas );
	spin = glade_xml_get_widget( xml, "link_view_depth" );
	depth = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( spin ) );
                                                       
	/* we now need to build up the tree of links */
	checked_files = g_hash_table_new( g_str_hash, g_str_equal );
	tree = build_link_tree( page, screem_page_get_pathname( page ),
				NULL, 0, checked_files, upload_table );
				
	/* we can destroy the checked_files hash table now
	   as we no longer need it */
	g_hash_table_destroy( checked_files );

	link_view_tree = tree;
	
	/* GNode tree built, now create the icons */
	build_icons( tree, root, 0.0, 0.0, 0,0, 1, 0 );
	
	/* the background is 8 times bigger than the true background
	   because I wanted to make sure the user only saw white*/
	type = gnome_canvas_rect_get_type();
	background = gnome_canvas_item_new( mainroot, type,
					    "x1", 
					    (gdouble)-canvas->allocation.width*4,
					    "y1", 
					    (gdouble)-canvas->allocation.height*4,
					    "x2", 
					    (gdouble)canvas->allocation.width*4,
					    "y2", 
					    (gdouble)canvas->allocation.height*4,
					    "fill_color", "white",
					    NULL );
	gnome_canvas_item_lower_to_bottom( background );

	gtk_signal_connect( GTK_OBJECT( background ), "event",
                            GTK_SIGNAL_FUNC( background_event ), NULL );
	
	gnome_canvas_set_scroll_region( GNOME_CANVAS( canvas ),
					min_x-80, min_y-80, max_x+80, max_y+80);
		
	/* if we aren't showing text labels hide them now */
	labels = glade_xml_get_widget( xml, "link_view_label_off" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( labels ) ) )
		screem_link_view_sidebar_switch_off( labels );
       
	/* show the canvas */
	gtk_widget_show( canvas );

	link_view_tree = g_node_get_root(link_view_tree);
	g_node_traverse(link_view_tree,G_IN_ORDER,G_TRAVERSE_ALL,-1,
			(GNodeTraverseFunc)traverse_for_remote,
			NULL);
	screem_show_message( _( "Checking for broken links (may take a while depending on your connection speed)" ) );


	/* timeout 2 seconds, */
	timeout_handle = gtk_timeout_add( 1000 * 2, (GtkFunction)
					  screem_link_view_check_next, NULL );
}
gboolean traverse_for_remote(GNode *node, gpointer data )
{
#ifdef HAVE_GNOME_VFS
	if( ( (HtmlFileData*)node->data )->remoteAddr ) {
		gnome_canvas_item_set( ((HtmlFileData*)node->data)->icon, 
				       "fill_color", "orange", NULL );
		pthread_mutex_lock( &check_lock );
		node_check = g_list_append( node_check, node );
		pthread_mutex_unlock( &check_lock );
	}
#endif
	return FALSE;
}

/* build_link_tree:
*
* this function creates the node tree that will be used
* in order to create the link view nodes. It is recursive.
*
* return values:
* the pointer to the root gnode
*/
GNode *build_link_tree( Page *page, const gchar *pathname, GNode *parent, 
			gint level, 
			GHashTable *checked_files,
			GHashTable *upload_table )
{
	GNode *node = NULL;
	GList *links;
	GList *list;
	Page *next;
	gchar *full;
	const gchar *text;
	const gchar *sitepath;
	const gchar *mime_type;
	struct stat s;

	gchar *path;
	gchar *temp;
	gchar *temp2;

	gchar cwd[ 16384 ];

	GList *this_level = NULL;
	gint check_level;

	getcwd( cwd, 16384 );

#ifndef HAVE_GNOME_VFS
	mime_type = gnome_mime_type_or_default( pathname, "no" );
#else
	mime_type = gnome_vfs_mime_type_from_name_or_default( pathname, "no" );
#endif	

	if(!strncmp( "image/", mime_type, strlen( "image/" ) )) {
		build_image_node(page, pathname, parent, upload_table);
	}else{
		node = build_normal_node(page, pathname, parent, upload_table);
		if (node == NULL)
			return NULL;		
	}	
	
	/* external links */
	if (page == NULL)
		return NULL;

	/* have we already checked this file? */
	if( (check_level = (gint)g_hash_table_lookup( checked_files, pathname )) ) {
		if( level > check_level ) {
			chdir( cwd );
			return node;
		}
	}

	screem_page_load( page );
	text = screem_page_get_data( page );
	links = screem_html_get_links( text );

	for( list = links; list; list = list->next ) {
		/* if link is local convert to absolute path
		   and locate a page for it */
		screem_page_chdir( page );

		if( ( temp = strchr( list->data, '#' ) ) ) {
			/* it links to an anchor in a page */
			path = g_strndup( list->data, 
					  temp - (gchar*)list->data );
		} else {
			path = g_strdup( list->data );
		}

		if( strlen( path ) == 0 ) {
			/* hmm, must be a link to this page */
			g_free( path );
			path = g_strconcat( screem_page_get_pathname( page ),
					    NULL );
		}

		/* sort out absolute links */
		if( path[ 0 ] == G_DIR_SEPARATOR ) {
			sitepath = screem_site_get_pathname( current_site );
			if( strncmp( path, sitepath, strlen( sitepath ) ) ) {
				temp2 = path;
				path = g_strconcat( sitepath, 
						    G_DIR_SEPARATOR_S,
						    temp2, NULL );
				g_free( temp2 );
			}
		}
		if( stat( path, &s ) < 0 ) {
			/* its not a local file, or a broken link */
			full = g_strdup( list->data );
		} else {
			/* local file */
			full =  relative_to_full( path ) ;
			if( temp ) {
				temp = g_strconcat( full, temp, NULL );
				g_free( full );
				full = temp;
			}
		}

		g_free( path );

		/* add full to a list of files at this level,
		   we don't want to build the next level until we
		   have a list of all files at this level */
		if( level < depth )
			this_level = g_list_append( this_level, full );

		/* add to hash table */
		if( ! g_hash_table_lookup( checked_files, full ) ) {
			g_hash_table_insert( checked_files, 
					     full, 
					     GINT_TO_POINTER( level + 1 ) );
		}
	}
	
	/* ok, now we can try and build the next level of links from the
	   this_level list */
	for( list = this_level; list; list = list->next ) {
		full = (gchar*)list->data;
		if( ! stat( full, &s ) )
			next = screem_site_locate_page( current_site, full );
		else
			next = NULL;

		if(!strstr(full,"mailto:") && !strstr(full,"javascript:"))
			build_link_tree( next, full, node, level + 1,
					 checked_files, upload_table );
	}
	g_list_foreach( this_level, (GFunc)g_free, NULL );
	g_list_free( this_level );

	chdir( cwd );
 
		/*
	  If we have a parent node then the node has already been
	  appended to it by build_normal_node, stopping this
	  removes the GLib Critical error
	
	  if( parent ) {
	  return g_node_append( parent, node );
	  } else
	*/
	return node;
}
/*
helper function of build_node_tree
*/
static GNode* build_normal_node(Page *page, const gchar *pathname, GNode *parent,
				GHashTable *upload_table)
{ 
	struct stat s;
	gchar *temp;
	HtmlFileData *ptrHFD;
	
	/* make normal node*/
	ptrHFD = g_new0( HtmlFileData, 1 );

	if ( page ){
		
		/* relative name*/
		temp = relative_path( pathname, screem_page_get_pathname( page ) );
		ptrHFD->name = g_strdup(temp+2);
		g_free( temp );
		
		ptrHFD->filename = g_strdup( pathname );
		ptrHFD->remoteAddr = NULL;
		
		/*exists?*/
		if (!stat( pathname, &s )) {
			ptrHFD->exists = TRUE;
			/*ptrHFD->filesize = s->st_blocks/2;*/
			if( upload_table )
				ptrHFD->uploadStatus = GPOINTER_TO_INT( g_hash_table_lookup( upload_table, pathname ) );
			else
				ptrHFD->uploadStatus = file_new;
		}else{
			ptrHFD->exists = FALSE;
		}
		
		/* ignored?*/
		ptrHFD->excluded = 
			(screem_site_is_excluded( current_site , ptrHFD->filename ) ||
			 screem_site_is_ignored( current_site , ptrHFD->filename ));
	} else {
		ptrHFD->name = NULL;
		ptrHFD->filename = NULL;	
		ptrHFD->remoteAddr = g_strdup( pathname );
	}
	ptrHFD->images = NULL;
	
	if( parent )
		return g_node_append_data( parent, ptrHFD );
	else
		return g_node_new(ptrHFD);
}

/*
	Helper function of build_node_tree
*/
static void build_image_node(Page *page, const gchar *pathname, GNode *parent,
			     GHashTable *upload_table)
{
	struct stat s;
	gchar *temp;
	ImageFileData *ptrIFD;

	/*image*/
	ptrIFD = g_new0( ImageFileData, 1 );
	
	if ( page ){
		/* relative name*/
		temp = relative_path( pathname, screem_page_get_pathname( page ) );
		ptrIFD->name = g_strdup(temp+2);
		g_free( temp );
		
		ptrIFD->filename = g_strdup( pathname );
		
		/*exists?*/
		if (!stat( pathname, &s )) {
			ptrIFD->exists = TRUE;
			/*ptrIFD->filesize = s->st_blocks/2;*/
			if( upload_table )
				ptrIFD->uploadStatus = GPOINTER_TO_INT( g_hash_table_lookup( upload_table, pathname ) );
			else
				ptrIFD->uploadStatus = file_new;
		}else{
			ptrIFD->exists = FALSE;
		}
		
		/* ignored?*/
		ptrIFD->excluded = 
			(screem_site_is_excluded( current_site , ptrIFD->filename ) ||
			 screem_site_is_ignored( current_site , ptrIFD->filename ));

		ptrIFD->external = FALSE;	
	} else {
		ptrIFD->name = g_strdup( pathname );
		ptrIFD->filename = g_strdup( pathname );
		ptrIFD->external = TRUE;	
	}
	ptrIFD->nextImage = NULL;
	
	append_image_node( parent, ptrIFD );
}

/* build_icons:
*
* this function creates the link map for all the nodes and draws the links
* underneath it calls build_icon to do the actual building of the nodes
*
* return values:
* none
*/
void build_icons( GNode *node, GnomeCanvasGroup *root, gdouble root_x, 
		  gdouble root_y, gdouble root_angle, gint level, 
		  gint numOfNodesAtLevel, gint nthNode )
{
	gint siblings;
	gint currentNode = 0;
	GNode *temp;

	GnomeCanvasPoints *points;
	GnomeCanvasItem *line;

	gdouble y;
	gdouble x = 10.0;
	static gdouble width = 0.0;

	GtkType type;
	
	gdouble temp_x;
	gdouble temp_y;
	gdouble temp_angle;
	
	gdouble angle = 0.0;


	if( ! node )
		return;
       

	if( level == 0 ) {
		/* Base Node Case*/
		
		/* Count siblings*/
		siblings = 0;
		for( temp = g_node_first_child( node ); temp;
		     temp = g_node_next_sibling( temp ) ) {
			siblings++;     	
		}
		
		/*Build siblings */
		for( temp = g_node_first_child( node ); temp;
		     temp = g_node_next_sibling( temp ) ) {
		     							       
			build_icons( temp, root, root_x, root_y, level, 1, 
				     siblings ,currentNode);
			currentNode++;			
		}
		
		/* build the root icon */
		build_icon( node, root, root_x, root_y, angle, &width );
     	} else if( level <= depth ) {
		/* this is the links from the root icon, create an icon
		   then call build_icons again */
							   
		/* formulas to create a circle */
		if (level > 1){
			angle = ((M_PI/2)/numOfNodesAtLevel)*(nthNode+1)+root_angle-.785;
			if (numOfNodesAtLevel>4){
				x = cos(angle)*30*numOfNodesAtLevel;
				y = sin(angle)*30*numOfNodesAtLevel;
			}else {
				x = cos(angle)*160;
				y = sin(angle)*160;				
			}
		}else{
			angle = (2*M_PI/numOfNodesAtLevel)*(nthNode+1);
			x = cos(angle)*180;
			y = sin(angle)*180;
			type = gnome_canvas_line_get_type();
			points = gnome_canvas_points_new( 2 );
			points->coords[ 0 ] = (gdouble)0;
			points->coords[ 1 ] = (gdouble)0;
			points->coords[ 2 ] = (gdouble)x;
			points->coords[ 3 ] = (gdouble)y;		
			line = gnome_canvas_item_new( root, type,
						      "points", points,
						      "fill_color", "green",
						      NULL );
			gnome_canvas_points_unref(points);
			gnome_canvas_item_lower_to_bottom(line);
		}
										   
		/* build this icon */					   
		build_icon( node, root, root_x+x, root_y+y, angle, &width );
		
		/* build children */
		 if( ( temp = g_node_first_child( node ) ) ) {		
			/**/
			siblings = 0;
			while( temp ) {
				siblings ++;
				temp = g_node_next_sibling( temp );
			}
			
			temp = g_node_first_child( node );

			currentNode = 0;
			while( temp ) {
				build_icons( temp, root, root_x+x, root_y+y, angle, level +1,
					     siblings ,currentNode);

				/* draw a line*/
				type = gnome_canvas_line_get_type();
				points = gnome_canvas_points_new( 2 );
				points->coords[ 0 ] = (gdouble)0 +root_x+ x;
				points->coords[ 1 ] = (gdouble)0 +root_y+ y;
				
				/* compute new x,y*/
				temp_angle = ((M_PI/2)/siblings)*(currentNode+1)+angle-.785;
				if (siblings>5){
					temp_x = cos(temp_angle)*30*siblings;
					temp_y = sin(temp_angle)*30*siblings;
				}else {
					temp_x = cos(temp_angle)*160;
					temp_y = sin(temp_angle)*160;				
				}
				
				points->coords[ 2 ] = (gdouble)temp_x + root_x+x;
				points->coords[ 3 ] = (gdouble)temp_y + root_y+y;		
				line = gnome_canvas_item_new( root, type,
							      "points", points,
							      "fill_color", "green",
							      NULL );
				gnome_canvas_points_unref(points);
				gnome_canvas_item_lower_to_bottom(line);
				temp = g_node_next_sibling( temp );
				currentNode++;
			}
		}
	}
}

/* build_icon:
*
* this function creates the node icon and the
* text label by parsing the node data struct
*
* return values:
* none
*/
void build_icon( GNode *node, GnomeCanvasGroup *root, 
		 gdouble x, gdouble y, gdouble angle,
		 gdouble *width )
{
	GnomeCanvasGroup *group;
	GnomeCanvasItem *item1;
	GnomeCanvasItem *item2;
	GtkType type;
	gint type_of_icon;
	gchar *path;
	
	gdouble true_x;
	gdouble true_y;

	if( ! node )
		return;

		
	/* Set the label and choose the correct icon*/
	if ( (gchar*) ((HtmlFileData*)node->data)->name ) {

		/*Local File */
		path = (gchar*) ((HtmlFileData*)node->data)->name;
			
		/*Is it a bad link?*/
		if ( ((HtmlFileData*)node->data)->exists ){
			if ( ((HtmlFileData*)node->data)->excluded){
				type_of_icon = NONODE;
			}else{
				if ( ((HtmlFileData*)node->data)->uploadStatus == file_changed ||
					((HtmlFileData*)node->data)->uploadStatus == file_moved ||
					((HtmlFileData*)node->data)->uploadStatus == file_deleted
					){
					type_of_icon = UPDATENODE;
				}else{
					type_of_icon = GOODNODE;
				}
			}
		}else{
			type_of_icon = BADNODE;
		}
		if ( ((HtmlFileData*)node->data)->images ){
			/*for (temp = ((HtmlFileData*)node->data)->images; 
			     temp; 
			     temp =  ((ImageFileData*)temp)->nextImage){
			  	g_print("IMAGES: %s\n", ((ImageFileData*)temp)->filename);
			}
			g_print("\n");*/
		}
	}else{
		/*Remote File*/
		path = (gchar*) ((HtmlFileData*)node->data)->remoteAddr;
		type_of_icon = EXTERNALNODE;
	}
	
	/* create the group */
	type = gnome_canvas_group_get_type();
	group = GNOME_CANVAS_GROUP( gnome_canvas_item_new( root, type,
							   "x", x,
							   "y", y,
							   NULL ) );

	/*update the node data*/
	((HtmlFileData*)node->data)->drawnX = x;
	((HtmlFileData*)node->data)->drawnY = y;							   

	/* create the image */
	type = gnome_canvas_pixbuf_get_type();
	item1 = node_draw( group, -5.0, -5.0, type_of_icon, FALSE);
	 
	((HtmlFileData*)node->data)->icon = item1;		      
	gnome_canvas_item_raise_to_top( item1 );
	
	/*Create the text*/
	type = gnome_canvas_text_get_type();
	item2 = gnome_canvas_item_new( group, type, 
				       "x", 0.0,
				       "y", 15.0,
				       "text", path,
				       "font", cfg->mpage->font_name,
				       "fill_color", "black",
				       "anchor", GTK_ANCHOR_CENTER,
				       NULL );
	/* set the label as a data item of item1 */
	gtk_object_set_data( GTK_OBJECT( item1 ), "label", item2 );

	/*add a pointer to the label on the node*/			       
	((HtmlFileData*)node->data)->label = item2;		       

	/*Expand the background if necessary*/
	/*	gnome_canvas_item_i2w(item1, &true_x, &true_y);*/

	/* Use the group x,y as they are correct, using the world
	   coords of the item doesn't work, try switching depth and
	   the view shifts, and may hide some of it it so that it
	   can't be scrolled to */
	true_x = x; true_y = y;
	if (min_x > true_x) min_x = true_x;
	if (max_x < true_x) max_x = true_x;
	if (min_y > true_y) min_y = true_y;
	if (max_y < true_y) max_y = true_y; 
	
	
	/*gtk_object_get( GTK_OBJECT( item2 ), "text_width", width, NULL );*/
	if (((HtmlFileData*)node->data)->remoteAddr){
			
		gtk_signal_connect( GTK_OBJECT( group ), "event",
				    GTK_SIGNAL_FUNC( group_event ), 
				    ((HtmlFileData*)node->data)->remoteAddr );
	}else if (((HtmlFileData*)node->data)->name){
		gtk_signal_connect( GTK_OBJECT( group ), "event",
				    GTK_SIGNAL_FUNC( group_event ), 
				    ((HtmlFileData*)node->data)->filename );		
	}
		
	gtk_signal_connect( GTK_OBJECT( group ), "event", 
			    GTK_SIGNAL_FUNC( item_event ), item1 );
}

/* item_event:
 */
static gint item_event( GnomeCanvasItem *group, GdkEvent *event, 
			GnomeCanvasItem *item )
{
	GnomeCanvasItem *label;
	gint handled = FALSE;

	switch( event->type ) {
	case GDK_ENTER_NOTIFY:
		gnome_canvas_item_set(item, "outline_color", "gray",NULL);
		gnome_canvas_item_request_update(item);
		label = gtk_object_get_data( GTK_OBJECT( item ), "label" );
		if( ! ( label->object.flags & GNOME_CANVAS_ITEM_VISIBLE ) ) {
			gtk_object_set_data( GTK_OBJECT( label ), "hidden",
					     GINT_TO_POINTER( TRUE ) );
			gnome_canvas_item_show( label );
		}
		handled = TRUE;
		break;
	case GDK_LEAVE_NOTIFY:
		gnome_canvas_item_set(item, "outline_color", "black",NULL);
		gnome_canvas_item_request_update(item);
		label = gtk_object_get_data( GTK_OBJECT( item ), "label" );
		if( gtk_object_get_data( GTK_OBJECT( label ), "hidden" ) ) {
			gnome_canvas_item_hide( label );
			gtk_object_set_data( GTK_OBJECT( label ), "hidden",
					     NULL );
		}
		handled = TRUE;
		break;
	default:
		break;
	}

	return handled;
}

/* group_event:
*
* connected to all the node groups to handle events
*
* the item in this case is the group which holds the icon
*
* return values:
* handled or not
*/
gint group_event( GnomeCanvasItem *item, GdkEvent *event, gchar *path )
{
	Page *page;
	GladeXML *xml;
	GtkWidget *menu;
	gboolean handled = FALSE;

	GtkWidget *widget;
	GtkWidget *tree;
	GtkCTreeNode *node;

	switch( event->type ) {
	case GDK_BUTTON_PRESS:
		if( event->button.button != 3 )
			break;

		gtk_object_set_data( GTK_OBJECT( app ), "EEK_ANOTHER_HACK",
				     GINT_TO_POINTER( 1 ) );

		/* popup menu for the icons, path holds the 
		   pathname of the file clicked on */
		   
		xml =  glade_xml_new( cfg->glade_path, "file_tree_menu" );
		menu = glade_xml_get_widget( xml, "file_tree_menu" );

		gtk_object_set_data( GTK_OBJECT( menu ), "path",path );
		
		/* find upload flags for the file/directory */
		screem_set_upload_flags( current_site, xml, path );
		glade_xml_signal_autoconnect( xml );
		
		widget = gtk_object_get_data( GTK_OBJECT( app ), "sitebook" );
		xml = glade_get_widget_tree( widget );
		tree = glade_xml_get_widget( xml, "tree" );
		gtk_object_set_data( GTK_OBJECT( menu ), "tree",tree );

		node = gtk_ctree_node_nth( GTK_CTREE( tree ), 0 );
		node = gtk_ctree_find_by_row_data_custom(GTK_CTREE( tree ),
							 node, path,
							 (GCompareFunc)strcmp);
		gtk_object_set_data( GTK_OBJECT( menu ), "node", node );
		gtk_object_set_data( GTK_OBJECT( tree ), "node", node );
	
		gnome_popup_menu_do_popup_modal( menu, 0, 0, 
						 (GdkEventButton*)event, 0);
		gtk_widget_destroy( menu );
		
		break;
	case GDK_2BUTTON_PRESS:
		if( event->button.button == 1 ) {
			/* the item was double clicked on, switch to the 
			   page */
			page = screem_site_locate_page( current_site, path );
			if( page ) {
				/* switch to it! */
				screem_page_insert( page );
			}
			handled = TRUE;
		}
		break;
	default:
		break;
	}
	
	return handled;
}

/* background_event:
*
* connected to the background canvas item to handle events
*
* the item in this case is the group which holds the icon
*
* return values:
* true ?
*/
static gint background_event( GnomeCanvasItem *item, GdkEvent *event )
{
	GladeXML *xml;
	GtkWidget *menu;
	guint32 t;

	gint x;
	gint y;

	gint cx;
	gint cy;
	GtkWidget *canvas;
	gint dx;
	gint dy;

       	switch( event->type ) {
	case GDK_BUTTON_PRESS:
		if( event->button.button == 1 ) {
			screem_link_view_drag_scroll( item, (GdkEventButton*)event );
			gtk_object_set_data( GTK_OBJECT( item ), "x",
					     GINT_TO_POINTER((gint)event->button.x));
			gtk_object_set_data( GTK_OBJECT( item ), "y",
					     GINT_TO_POINTER((gint)event->button.y));
		} else if( event->button.button == 3 ) {
			/* popup menu */
			gtk_signal_emit_stop_by_name( GTK_OBJECT( item ),
						      "event" );
			xml = glade_xml_new( cfg->glade_path,
					     "linkview_popup" );
			menu = glade_xml_get_widget( xml, "linkview_popup" );
			glade_xml_signal_autoconnect( xml );
			gnome_popup_menu_do_popup_modal(menu, 0, 0, 
							(GdkEventButton*)event,
							0);
			gtk_widget_destroy( menu );
		}
		break;
	case GDK_BUTTON_RELEASE:
		if( gtk_object_get_data( GTK_OBJECT( item ), "grabbed" ) ) {
			t = gdk_event_get_time( event );
			gtk_object_set_data(GTK_OBJECT( item ), "grabbed", 0 );
			gdk_pointer_ungrab( t );
		}
		break;
	case GDK_MOTION_NOTIFY:
		if( gtk_object_get_data( GTK_OBJECT( item ), "grabbed" ) ) {
			x = (gint)
				gtk_object_get_data( GTK_OBJECT( item ), "x" );
			y = (gint)
				gtk_object_get_data( GTK_OBJECT( item ), "y" );

			gtk_object_set_data( GTK_OBJECT( item ), "x",
					     GINT_TO_POINTER((gint)event->motion.x));
			gtk_object_set_data( GTK_OBJECT( item ), "y",
					     GINT_TO_POINTER((gint)event->motion.y));

			canvas = gtk_object_get_data( GTK_OBJECT( app ),
						      "link_view" );

			gnome_canvas_get_scroll_offsets( GNOME_CANVAS(canvas),
							 &cx, &cy );

			dx = event->motion.x - x;
			dy = event->motion.y - y;

			if( dx > 0 || dx < 0 )
				cx += dx * zoom / 100;
		
			if( dy > 0 || dy < 0 )
				cy += dy * zoom / 100;
			
			gnome_canvas_scroll_to( GNOME_CANVAS( canvas ), cx, cy );
		}
		break;
	default:
		break;
	}

	return TRUE;
}

static void screem_link_view_drag_scroll( GnomeCanvasItem *item,
					  GdkEventButton *event )
{
	static GdkCursor *cursor = NULL;
	guint32 t = gdk_event_get_time( (GdkEvent*)event );
#if 0
	if( ! cursor ) 
		cursor = gdk_cursor_new( GDK_???? );
#endif
	gtk_object_set_data( GTK_OBJECT( item ), "grabbed", GINT_TO_POINTER( 1 ) );
	gdk_pointer_grab( event->window, FALSE, 
			  GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
			  FALSE, cursor, t );
}

/* free_mem_from_html_node:
*
* used to remove the memory from the nodes 
*
* before they are removed by glib
*
* return values:
* none
*/
void free_mem_from_html_node (GNode* node, gpointer data)
{
	HtmlFileData *fdata = (HtmlFileData*)node->data;

	g_free( fdata->name);
	
	g_free( fdata->filename);
	
	g_free( fdata->remoteAddr);

	if( fdata->images )
		free_mem_from_image_node( (ImageFileData*) fdata->images);
	
	g_free( fdata );	
}

/* free_mem_from_image_node:
*
* used to remove the memory from the nodes 
*
* before they are removed by free_mem_from_html_node
*
* return values:
* none
*/
void free_mem_from_image_node (ImageFileData* node)
{
	g_return_if_fail( node != NULL );

	g_free( node->name);
	
	g_free( node->filename );

	if( node->nextImage )
		free_mem_from_image_node( (ImageFileData*)node->nextImage );
       
	g_free(node);
}

/* remove_data:
*
* function to invoke the above two functions 
*
* on every node in the tree before glib deletes the tree
*
* return values:
* none
*/
void remove_data (GNode* parent)
{
	g_node_traverse(parent,G_IN_ORDER,G_TRAVERSE_ALL,-1,
			(GNodeTraverseFunc)free_mem_from_html_node,
			NULL);
}

/* append_image_node:
*
* adds the image node to the ll of the htmlfiledata struct
*
* return values:
* pointer to parent
*/
GNode* append_image_node( GNode *parent, ImageFileData* node)
{
	struct ImageFileData *temp;
	
	if (((HtmlFileData*)parent->data)->images == NULL)
		((HtmlFileData*)parent->data)->images = 
			(struct ImageFileData*)node;
	else {
		for (temp = ((HtmlFileData*)parent->data)->images; 
			 ((ImageFileData*)temp)->nextImage; 
			 temp =  ((ImageFileData*)temp)->nextImage);
		((ImageFileData*)temp)->nextImage = (struct ImageFileData*)node;
	}
	return parent;
}

/**
 * node_draw:
 * @root:   the canvas group to be added to
 * @x:      the x coord to place the key at
 * @y:      the y coord to place the key at
 * @type:   the type of node
 * @legend: is the node being drawn for the legend
 * draws node using canvas functions
 *
 * returns: the canvas group of the key
 */
GnomeCanvasItem* node_draw( GnomeCanvasGroup *root, gdouble x,
			    gdouble y, gint node_type, gboolean legend )
{
	GnomeCanvasItem *node;
	GtkType type;
	
	type = gnome_canvas_ellipse_get_type();
	node = gnome_canvas_item_new( root, type,
				      "x1", x,
				      "y1", y,
				      "x2", x + 10.0,
				      "y2", y +10.0,
				      "fill_color", "yellow",
				      "outline_color", "black",
				      "width_units", 2.0,
				      NULL);
	switch( node_type ) {
	case GOODNODE:
		gnome_canvas_item_set( node, "fill_color", "green", NULL );
		break;
	case EXTERNALNODE:
		gnome_canvas_item_set( node, "fill_color", "blue", NULL );
		break;
	case BADNODE:
		gnome_canvas_item_set( node, "fill_color", "red", NULL );
		break;
	case NONODE:
		gnome_canvas_item_set( node, "fill_color", "gray", NULL );
		break;
	case UPDATENODE:
		gnome_canvas_item_set( node, "fill_color", "yellow", NULL );
		break;
	case CHECKNODE:
		gnome_canvas_item_set( node, "fill_color", "orange", NULL );
		break;
	}

	return node;
}
 
/**
 * screem_link_view_add_key:
 * @root:   the main canvas group
 * @x:      the x coord to place the key at
 * @y:      the y coord to place the key at
 *
 * creates a canvas item showing a key for the link view
 *
 * returns: the canvas group of the key
 */
GnomeCanvasGroup* screem_link_view_add_key( GnomeCanvasGroup *root, gdouble x,
					    gdouble y)
{
	GnomeCanvasGroup *group;
	GnomeCanvasItem *image;
	GnomeCanvasItem *text;
	GtkWidget *legend;
	GtkType type;

	gdouble x2 = 20.0;
	gdouble y2;
	gdouble w;
	gdouble h;

	gint icons[] = {
		GOODNODE,
		EXTERNALNODE,
		BADNODE,
		NONODE,
		UPDATENODE,
		CHECKNODE,
		-1
	};
	gchar *names[] = {
		_( "Link/File exists" ),
		_( "External Link/File" ),
		_( "Broken Link" ),
		_( "Ignored" ),
		_( "Requires Upload" ),
		_( "Checking Link" ),
		NULL
	};
	gint i;


	/* create the group */
	type = gnome_canvas_group_get_type();
	group = GNOME_CANVAS_GROUP( gnome_canvas_item_new( root, type,
							   "x", x,
							   "y", y,
							   NULL ) );
   
	/* create the image */
	y2 = 4.0;
	for( i = 0; icons[ i ] != -1; i ++ ) {
		
		image = node_draw( group,4.0, y2, icons[ i ], TRUE );
					       
		type = gnome_canvas_text_get_type();
		text = gnome_canvas_item_new( group, type, 
					      "x", 20.0,
					      "y", y2+5,
					      "anchor", GTK_ANCHOR_WEST,
					      "text", names[ i ],
					      "font", cfg->mpage->font_name,
					       "fill_color", "black",
					      NULL );
		
		gtk_object_get( GTK_OBJECT( text ), "text_width", &w, NULL );
		gtk_object_get( GTK_OBJECT( text ), "text_height", &h, NULL );

		w += 20.0;
		if( w > x2 )
			x2 = w;

		if( h < 20.0 )
			h = 20.0;

		/* next */
		y2 += h;
	}

	/* y2 will be the height we want to set the legend canvas to */
	legend = gtk_object_get_data( GTK_OBJECT( app ), "link_view_legend" );

	gtk_widget_set_usize( legend, -1, y2 + h );

	return group;
}

void screem_link_view_zoom( GtkWidget *menuitem )
{
	GladeXML *xml;
	GtkWidget *spin;
	gchar *temp;
	
	temp = gtk_widget_get_name( menuitem );

	zoom = atoi( temp );

	if( zoom == 0 )
		zoom = 100;

	/* set the spinbutton */
	xml = glade_get_widget_tree( app );
	spin = glade_xml_get_widget( xml, "sidebar_zoom" );
	gtk_spin_button_set_value( GTK_SPIN_BUTTON( spin ), zoom );
}

void screem_link_view_sidebar_zoom( GtkWidget *widget )
{
	zoom = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widget));
	gnome_canvas_set_pixels_per_unit( GNOME_CANVAS( canvas ), zoom/100.0 );
}

gboolean link_view_labels_off(GNode *node, gpointer data)
{
	gnome_canvas_item_hide(((HtmlFileData*)node->data)->label);
	return FALSE;
}

gboolean link_view_labels_on(GNode *node, gpointer data)
{
	gnome_canvas_item_show(((HtmlFileData*)node->data)->label);
	return FALSE;
}

void screem_link_view_sidebar_switch_on( GtkWidget *widget )
{
	g_node_traverse(link_view_tree,G_IN_ORDER,G_TRAVERSE_ALL,-1,
		(GNodeTraverseFunc)link_view_labels_on,
		NULL);
}

void screem_link_view_sidebar_switch_off( GtkWidget *widget )
{
	g_node_traverse(link_view_tree,G_IN_ORDER,G_TRAVERSE_ALL,-1,
			(GNodeTraverseFunc)link_view_labels_off,
			NULL);
}

void screem_link_view_check_remote_fill( GNode *node, RemoteStatus status )
{
	switch( status ) {
	case REMOTE_EXISTS:
		/* HACK, also not right as any external links with #
		   will look like local ones, infact the local ones
		   shouldn't get through to here, but they get a remoteAddr
		   set for some reason */
		if( strchr( ((HtmlFileData*)node->data)->remoteAddr, '#' ) )
			gnome_canvas_item_set(((HtmlFileData*)node->data)->icon,
					      "fill_color", "green", NULL );
		else
			gnome_canvas_item_set(((HtmlFileData*)node->data)->icon,
					      "fill_color", "blue", NULL );
		break;
	case REMOTE_BROKEN:
		gnome_canvas_item_set( ((HtmlFileData*)node->data)->icon, 
				       "fill_color", "red", NULL );
		break;
	case REMOTE_UNREACHABLE:
		/* not used yet */
		break;
	default:
		g_assert( FALSE );  /* should never happen */
		break;
	}
}

void screem_link_view_stop_check()
{
	pthread_mutex_lock( &check_lock );
	if( node_check ) {
		g_list_free( node_check );
		node_check = NULL;
	}
	pthread_mutex_unlock( &check_lock );
}

#ifdef HAVE_GNOME_VFS
static void* screem_link_view_check( GNode *node )
{
	GnomeVFSURI *uri;
	GnomeVFSHandle *handle;
	GnomeVFSContext **context;
	HtmlFileData *data;
	GnomeVFSResult res;
	RemoteStatus status;

	pthread_detach( pthread_self() );

	data = (HtmlFileData*)node->data;

	context = (GnomeVFSContext**)data->handle;
	*context = gnome_vfs_context_new();

	/* BUG: could get stuck creating the uri */
	uri = gnome_vfs_uri_new( data->remoteAddr );
	
	res = gnome_vfs_open_uri_cancellable( &handle, 
					      uri, 
					      GNOME_VFS_OPEN_READ,
					      *context );
	switch( res ) {
	case GNOME_VFS_OK:
		/* exists */
		data->status = REMOTE_EXISTS;
		gnome_vfs_close( handle );
		break;
	default:
		data->status = REMOTE_BROKEN;
		break;
	}

	return NULL;
}
#endif

static gboolean screem_link_view_check_next()
{
	static GList *current;
	static GNode *node = NULL;
	const gchar *remote_addr;
	RemoteStatus status;
#ifdef HAVE_GNOME_VFS
	static GnomeVFSContext *context = NULL;
	HtmlFileData *data;

	pthread_mutex_lock( &check_lock );

	/* */
	if( context ) {
		gchar *temp;

		data = (HtmlFileData*)node->data;

		if( data->status == REMOTE_NOT_CHECKED ) {
			gnome_vfs_cancellation_cancel(gnome_vfs_context_get_cancellation(context));
			temp = g_strdup_printf( _( "Timeout checking: %s" ),
						data->remoteAddr );
			screem_show_message( temp );
			g_free( temp );
			data->status = REMOTE_BROKEN;
		}
		screem_link_view_check_remote_fill( node, data->status );
		
		g_hash_table_insert( external_check_cache, 
				     ((HtmlFileData*)node->data)->remoteAddr,
				     GINT_TO_POINTER( data->status ) );
		gnome_vfs_context_unref( context );
		context = NULL;
	}

	if( ! external_check_cache )
		external_check_cache = g_hash_table_new( g_str_hash, 
							 g_str_equal );

	if( node_check ) {
		/* pop next node to check from the list */
		current = node_check;
		node = (GNode*)current->data;
	
		data = (HtmlFileData*)node->data;

		remote_addr =  data->remoteAddr;
		data->handle = &context;
		
		if( remote_addr ) {
			status = (RemoteStatus)
				g_hash_table_lookup( external_check_cache, 
						     remote_addr );
			if( status == REMOTE_NOT_CHECKED ) {
				pthread_t thread;
				data->status = REMOTE_NOT_CHECKED;

				/*pthread_create( &thread, NULL, 
				  screem_link_view_check, node );*/
				/* OK, too much trouble getting this working,
				   lets just forget about checking broken links
				   for now */
				status = REMOTE_EXISTS;
			} /*else*/
				screem_link_view_check_remote_fill( node,status );

		}
		node_check = g_list_remove( node_check, current->data );
	} else {
		g_hash_table_destroy( external_check_cache );
		external_check_cache = NULL;
	}

	pthread_mutex_unlock( &check_lock );

	return (node_check != NULL || context != NULL );
#else
	return FALSE;
#endif
}
