/*  Screem:  main.c,
 *  This contains the main() function for Screem, along with handling the
 *  Gnome session management features and other initialisation issues
 * 
 *  Copyright (C) 1999, 2000, 2001  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gmodule.h>
#include <gnome.h>
#include <unistd.h>
#include <errno.h>

#include <glade/glade.h>
#include <libgnorba/gnorba.h>
#include <sys/stat.h>

#ifdef HAVE_GUILE
#include <libguile.h>
#include <guile/gh.h>
#endif

#ifdef USE_GCONF
#include <gconf/gconf.h>
#endif

#ifdef HAVE_GNOME_VFS
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#endif

#include "editor.h"
#include "interface.h"
#include "site.h"
#include "siteUI.h"
#include "page.h"
#include "pageUI.h"
#include "plugin.h"
#include "php3.h"
#include "preferences.h"
#include "preview.h"
#include "linkview.h"
#include "siteTreeUI.h"
#include "fileops.h"
#include "helpers.h"
#include "guile.h"
#include "dtd.h"

GtkWidget *app;
GnomeClient *client = NULL;
Preferences *cfg;

Site *current_site = NULL;
GList *loaded_sites = NULL;

/* CORBA stuff */
CORBA_ORB my_orb;
CORBA_Environment ev;
CORBA_Object ns;

static gboolean clean_exit = FALSE;

static gint no_splash;
static gchar *glade_path;

/* passed parameters stuff */
static poptContext ctx;
static const struct poptOption options[] = {
	{ "no-splash", '\0', POPT_ARG_NONE, &no_splash, 1,
	  N_("Disables splash screen on start up"), NULL },
	{ "glade-file", '\0', POPT_ARG_STRING, &glade_path, 1,
	  N_("Use the given glade file instead of the default"), N_( "PATH" ) },
        { NULL, '\0', 0, NULL, 0 }
};

static const char** start_files = NULL;

static GtkWidget *text;
static GtkWidget *splash;

#define update_splash(message) 	gtk_text_insert(GTK_TEXT( text ), 0, 0, 0, message, -1);\
while( gtk_events_pending() )\
gtk_main_iteration();

static void main_real( int argc, char *argv[] );
static gboolean save_sites( void );
static gint
save_yourself( GnomeClient *client, gint phase, GnomeSaveStyle save_style,
               gint is_shutdown, GnomeInteractStyle interact_style,
               gint is_fast, gpointer client_data );
static void parse_args( int argc, char *argv[] );

void exit_confirm( void );


/* hint stuff */
void screem_startup_hint( void );
void hint_on_startup( GtkWidget *widget );
void hide_hint_window( GtkWidget *close_button );
void next_hint( GtkWidget *widget );
void prev_hint( GtkWidget *widget );
static void read_hints_file( gchar *filename );
static gint hint_count = 0;
static gchar **hint_text = NULL;

int main( int argc, char *argv[] )
{
#ifdef ENABLE_NLS
	/* initialise i18n stuff */
        bindtextdomain( PACKAGE, GNOMELOCALEDIR );
        textdomain( PACKAGE );
#endif	
	
	/* initialise gnome, gtk corba etc */
	g_thread_init( NULL );
        CORBA_exception_init( &ev );
	my_orb = gnome_CORBA_init_with_popt_table( PACKAGE, VERSION, &argc,
						   argv, options, 0, &ctx,
						   0, &ev );
        if( ev._major != CORBA_NO_EXCEPTION )
                exit( 1 ); /* error initializing ORB */
	
	/* init gconf if needed */
#ifdef USE_GCONF
	{
		GConfError *gconferror;
		
		gconf_init( argc, argv, &gconferror );
	}
#endif
	/* init libglade */
	glade_gnome_init();
	
        ns = gnome_name_service_get();
	
        /* parse arguments */
        parse_args( argc, argv );
	
	/* show splash */
	splash = create_splash();
	text = (GtkWidget*)gtk_object_get_data( GTK_OBJECT( splash ),"text1" );
	
	if( ! no_splash )
		gtk_widget_show_all( splash );
	else
		gtk_widget_realize( text );
	
	gdk_threads_enter();
#ifdef HAVE_GUILE
	/* init guile */
	update_splash( _("Initialising Guile\n") );
	gh_enter(argc, argv, main_real);
#else
	main_real( argc, argv );
#endif
	gdk_threads_leave();
	
	return 0;
}

static void main_real( int argc, char *argv[] )
{
	struct stat s;
	const gchar *mime_type;
	gchar cwd[ 16384 ];
	Site *site;
	
	const gchar *file;
	gint i;
	guint j;
	GnomeDock *dock;
	GtkWidget *widget;
	gchar **temp;

	screem_init_guile();

	update_splash( _("Loading DTDs\n") );
	screem_dtd_load_catalog();

#ifdef HAVE_GNOME_VFS
	update_splash( _("Initialising gnome-vfs\n") );
	gnome_vfs_init();
#endif
	
	update_splash( _("Creating Interface\n") );
	
	/* create interface */
	app = create_app( FALSE, PACKAGE, "Screem:" );

	/* create the fake site */
	site = screem_site_new();
	/* make it a fake site, handles the setting of site pathname
	   etc */
	screem_site_set_fake_flag( site, TRUE );
     	current_site = site;
	
	/* add to list of loaded sites */
	loaded_sites = g_list_append( loaded_sites, site );
	screem_build_sites_combo_list();
	
	/* load modules */
	if( g_module_supported() ) {
		update_splash( _("Loading Modules...\n") );
		scan_plugins( text );
		update_splash( _("Modules loaded\n") );
	} else {
		update_splash( _("Modules not supported\n") );
	}
	
	screem_helpers_build_toolbar();
	
	screem_php_build_function_hash_table();
	
	/* show the main interface */
	gtk_widget_show_all( app );
	gtk_widget_set_usize( app, app->allocation.width,
			      app->allocation.height );
	
	/* hide the dock items that the user has hidden */
	dock = gnome_app_get_dock( GNOME_APP( app ) );
	temp = g_strsplit( cfg->mpage->hide_string, "\\", 256 );
	for( i = 0; temp && temp[ i ]; i += 2 ) {
		widget = GTK_WIDGET( gnome_dock_get_item_by_name( dock, temp[ i ], &j,
								  &j, &j, &j ) );
		/* bit of a hack always hiding it and then
		   showing again if needed, but it was the simplest
		   way to get the view menu item entries working
		   correctly */
		if( ! widget )
			continue;
		gtk_widget_hide( widget );
		if( *temp[ i + 1 ] == '1' )
			continue;
		show_hide( widget, temp[ i ] );
		
	}
	g_strfreev( temp );
    	
     	getcwd( cwd, 16385 );
	screem_browse_list_show_dir( cwd );
	
	build_recent_sites_menu();
	build_recent_pages_menu();
	
	/* destroy splash */
	gtk_widget_destroy( splash );
	
	screem_preview_display( NULL );

	/* load any sites that were open last time */
	load_open();
	
	for( i = 0; start_files && start_files[ i ]; i ++ ) {
		file = start_files[ i ];
		/* is it a project file or an html file? */
#ifndef HAVE_GNOME_VFS
		mime_type = gnome_mime_type( file );
#else
		mime_type = gnome_vfs_mime_type_from_name( file );
#endif
		if( ! strcmp( mime_type, "application/x-screem" ) ) {
			/* does it even exist? */
			if( ! stat( file, &s ) )
				screem_site_open_with_filename( file );
		} else {
			/* FIXME: doesn't work if we have loaded a site */
			screem_page_open_with_filename( file );
		}
	}
	poptFreeContext( ctx );
	
	screem_startup_hint();

	gtk_main();

	if( clean_exit ) {
		save_open( loaded_sites );
		save_preferences();
	}
#ifdef HAVE_GNOME_VFS
	g_print( "CALLING GNOME VFS SHUTDOWN\n" );
	gnome_vfs_shutdown();
#endif
}

void exit_confirm()
{
	gboolean quit;

	quit = save_sites();

	if( quit ) {
		clean_exit = TRUE;
		gtk_widget_hide( app );
		screem_link_view_stop_check();

		gtk_main_quit();
	} else {
		/* Why the hell do I need this?  If it isn't here then the main 
		   window disappears (Sawfish 0.34) even though we never destroy
		   it */
		gtk_widget_show( app );
	}
}

static gboolean save_sites()
{
	GList *list;
	GList *list2;
	Site *site;
	const gchar *site_name;
	const gchar *site_path;

	gboolean confirm = TRUE;

	GtkWidget *prompt;
        gint button;
        gchar *message;

	/* request confirmation to save any opened sites / pages */
	for( list = loaded_sites; list; list = list->next ) {
		site = (Site*)list->data;
		if( ! screem_site_get_fake_flag( site ) ) {
			site_name = screem_site_get_name( site );
			site_path = screem_site_get_pathname( site );
			message = g_strdup_printf( _("Save site: %s\n(%s)"),
						   site_name, site_path );
			prompt = gnome_message_box_new(message,
						       GNOME_MESSAGE_BOX_WARNING,
						       GNOME_STOCK_BUTTON_YES,
						       GNOME_STOCK_BUTTON_NO,
						       GNOME_STOCK_BUTTON_CANCEL,
						       NULL );
			g_free( message );
			button = gnome_dialog_run_and_close( GNOME_DIALOG( prompt ) );
			if( button == 0 )
				screem_site_save( site );
			else if( button == 2 )
				return FALSE;
		} else {
			/* this is the fake site, so ask about saving the
			   individual pages in it */
			button = 0;
			for( list2 = screem_site_get_pages( site ); 
			     list2 && button != 2 ; list2 = list2->next )
				button = screem_page_save_confirmation( (Page*)list2->data );
			if( button == 2 )
				return FALSE;
		}
	
	}

	return confirm;
}

static void parse_args( int argc, char *argv[] )
{
        /* passed arguements */
        start_files = poptGetArgs( ctx );

        /* Session Managment stuff */
        client = gnome_master_client ();
        gtk_signal_connect (GTK_OBJECT (client), "save_yourself",
                            GTK_SIGNAL_FUNC (save_yourself), NULL);
        gtk_signal_connect (GTK_OBJECT (client), "die",
                            GTK_SIGNAL_FUNC (gtk_main_quit), NULL);

        /* load the preferences */
        load_preferences();

	if( ! glade_path )
		glade_path = GLADE_PATH;
	cfg->glade_path = g_strdup( glade_path );

        return;
}

static gint
save_yourself( GnomeClient *client, gint phase, GnomeSaveStyle save_style,
               gint is_shutdown, GnomeInteractStyle interact_style,
               gint is_fast, gpointer client_data )
{
        gchar* args[4] = { "rm", "-r", NULL, NULL };

	save_preferences();

	args[2] = gnome_config_get_real_path
                    (gnome_client_get_config_prefix (client));
        gnome_client_set_discard_command (client, 3, args);

        return TRUE;
}


/* this is here as I can't think of a better place yet, it is 
   essentially the GIMP tip of the day code, modified slightly for
   SCREEM, and my coding style */
void screem_startup_hint()
{
	static GladeXML *xml = NULL;
	GtkWidget *hint_window;
	GtkWidget *check;
	GtkWidget *hint_label;

	gchar *hint_file;

	gint w;
	gint h;

	if( ! xml ) {
		xml = glade_xml_new( cfg->glade_path, "hint_window" );
		hint_window = glade_xml_get_widget( xml, "hint_window" );

		check = glade_xml_get_widget( xml, "hint_on_startup" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( check ),
					      cfg->hints );

		/* load the hints */
		hint_file = g_strdup_printf( "%s%c%s", HINTS_PATH,
					     G_DIR_SEPARATOR,
					     _("screem_hints.txt") );
		read_hints_file( hint_file );
		g_free( hint_file );

		if( cfg->last_hint >= hint_count || cfg->last_hint < 0 )
			cfg->last_hint = 0;
		
		hint_label = glade_xml_get_widget( xml, "hint_label" );
		gtk_label_set( GTK_LABEL( hint_label ), 
			       hint_text[ cfg->last_hint ] );

		w = hint_window->allocation.width;
		h = hint_window->allocation.height;
		if( w < 450 )
			gtk_widget_set_usize( hint_window, 450, -1 );
		if( h < 150 )
			gtk_widget_set_usize( hint_window, -1, 150 );

		glade_xml_signal_autoconnect( xml );
		if( cfg->hints )
			gtk_widget_show( hint_window );
	} else {
		hint_window = glade_xml_get_widget( xml, "hint_window" );
		gtk_widget_show( hint_window );
	}
}

void hint_on_startup( GtkWidget *widget )
{
	cfg->hints = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(widget) );
}

void hide_hint_window( GtkWidget *close_button )
{
	GladeXML *xml;
	GtkWidget *hint_window;

	xml = glade_get_widget_tree( close_button );
	hint_window = glade_xml_get_widget( xml, "hint_window" );

	gtk_widget_hide( hint_window );
}

void next_hint( GtkWidget *widget )
{
	GladeXML *xml;
	GtkWidget *hint_label;

	xml = glade_get_widget_tree( widget );
	hint_label = glade_xml_get_widget( xml, "hint_label" );

	cfg->last_hint ++;
	
	if( cfg->last_hint >= hint_count )
		cfg->last_hint = 0;
	
	gtk_label_set( GTK_LABEL( hint_label ), hint_text[ cfg->last_hint ] );
}

void prev_hint( GtkWidget *widget )
{
	GladeXML *xml;
	GtkWidget *hint_label;

	xml = glade_get_widget_tree( widget );
	hint_label = glade_xml_get_widget( xml, "hint_label" );

	cfg->last_hint --;
	
	if( cfg->last_hint < 0 )
		cfg->last_hint = hint_count - 1;
	
	gtk_label_set( GTK_LABEL( hint_label ), hint_text[ cfg->last_hint ] );
}

static void store_tip( gchar *str )
{
	hint_count++;
	hint_text = (gchar**)
		g_realloc (hint_text, sizeof (gchar *) * hint_count);
	hint_text[hint_count - 1] = str;
}

static void read_hints_file( gchar *filename )
{
	FILE *fp;
	gchar *tip = NULL;
	gchar *str = NULL;
	gchar *temp = NULL;

	fp = fopen( filename, "rt" );
	if( !fp ) {
		store_tip( _("Your SCREEM tips file appears to be missing!\n"
			     "There should be a file called screem_hints.txt in the\n"
			     "SCREEM data directory.  Please check your installation."));
		return;
	}

	str = g_new( char, 1024 );
	while( ! feof( fp ) ) {
		if( ! fgets( str, 1024, fp ) )
			continue;
   
		if( str[ 0 ] == '#' || str[ 0 ] == '\n' ) {
			if( tip ) {
				tip[ strlen( tip ) - 1 ] = '\000';
				store_tip( tip );
				tip = NULL;
			}
		} else 	if( ! tip ) {
			tip = g_strdup( str );
		} else {
			temp = tip;
			tip = g_strconcat( tip, str, NULL );
			g_free( temp );
		}
	}
	if( tip )
		store_tip( tip );
	g_free( str );
	fclose( fp );
}
