/*  Screem:  page_guile.c,
 *  Guile glue functions for accessing a screem page
 * 
 *  Copyright (C) 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */
#include <config.h>

#ifdef HAVE_GUILE
#include <guile/gh.h>
#endif

#include "page.h"
#include "page_guile.h"

/* "scheme name", C function name, required args, optional args,
   takes rest list */
#ifdef HAVE_GUILE
static gpointer page_procedures[] = {
	"screem-page-new", (gpointer)scm_screem_page_new, 
	GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-destroy", (gpointer)scm_screem_page_destroy, 
	GINT_TO_POINTER( 1 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-set-pathname", (gpointer)scm_screem_page_set_pathname, 
	GINT_TO_POINTER( 2 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-get-pathname", (gpointer)scm_screem_page_get_pathname,
	GINT_TO_POINTER( 1 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-get-mime-type", (gpointer)scm_screem_page_get_mime_type,
	GINT_TO_POINTER( 1 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-set-tab", (gpointer)scm_screem_page_set_tab, 
	GINT_TO_POINTER( 2 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-get-tab", (gpointer)scm_screem_page_get_tab, 
	GINT_TO_POINTER( 1 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-set-charset", (gpointer)scm_screem_page_set_charset, 
	GINT_TO_POINTER( 2 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-get-charset", (gpointer)scm_screem_page_get_charset, 
	GINT_TO_POINTER( 1 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-set-data", (gpointer)scm_screem_page_set_data, 
	GINT_TO_POINTER( 2 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-get-data", (gpointer)scm_screem_page_get_data,
	GINT_TO_POINTER( 1 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-set-changed", (gpointer)scm_screem_page_set_changed, 
	GINT_TO_POINTER( 2 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-get-changed", (gpointer)scm_screem_page_get_changed, 
	GINT_TO_POINTER( 1 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-set-mod_time", (gpointer)scm_screem_page_set_mod_time,
	GINT_TO_POINTER( 2 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-get-mod_time", (gpointer)scm_screem_page_get_mod_time, 
	GINT_TO_POINTER( 1 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-read", (gpointer)scm_screem_page_read, 
	GINT_TO_POINTER( 2 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-load", (gpointer)scm_screem_page_load, 
	GINT_TO_POINTER( 1 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-revert", (gpointer)scm_screem_page_revert, 
	GINT_TO_POINTER( 1 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-save", (gpointer)scm_screem_page_load, 
	GINT_TO_POINTER( 1 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-from-tab-num", (gpointer)scm_screem_page_from_tab_num, 
	GINT_TO_POINTER( 2 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-update-tab-nums", 
	(gpointer)scm_screem_page_update_tab_nums, 
	GINT_TO_POINTER( 2 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-is-file-page", (gpointer)scm_screem_page_is_file_page, 
	GINT_TO_POINTER( 1 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-is-mime-type-page", 
	(gpointer)scm_screem_page_is_mime_type_page, 
	GINT_TO_POINTER( 1 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-get-undo-last-action", 
	(gpointer)scm_screem_page_get_undo_last_action, 
	GINT_TO_POINTER( 1 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-set-undo-last-action", 
	(gpointer)scm_screem_page_set_undo_last_action, 
	GINT_TO_POINTER( 4 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-add-undo-action", 
	(gpointer)scm_screem_page_add_undo_action, 
	GINT_TO_POINTER( 2 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-clear-undo-list", 
	(gpointer)scm_screem_page_clear_undo_list, 
	GINT_TO_POINTER( 1 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-set-redo-last-action", 
	(gpointer)scm_screem_page_set_redo_last_action, 
	GINT_TO_POINTER( 2 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-get-redo-last-action", 
	(gpointer)scm_screem_page_get_redo_last_action, 
	GINT_TO_POINTER( 1 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-page-clear-redo-list", 
	(gpointer)scm_screem_page_clear_redo_list, 
	GINT_TO_POINTER( 1 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	NULL
};
#endif

void screem_page_init_guile()
{
	gint i;
#ifdef HAVE_GUILE
      	for( i = 0; page_procedures[ i ]; i += 5 ) {
		gh_new_procedure( page_procedures[ i ],
				  page_procedures[ i + 1 ],
				  (gint)page_procedures[ i + 2 ],
				  (gint)page_procedures[ i + 3 ],
				  (gint)page_procedures[ i + 4 ] );
	}
#endif
}

#ifdef HAVE_GUILE

SCM scm_screem_page_new()
{
	Page *page;

	page = screem_page_new();

	return PTR2SCM( page );
}

SCM scm_screem_page_destroy( SCM p )
{
	Page *page = (Page*)SCM2PTR( p );

	screem_page_destroy( page );

	return SCM_EOL;
}

SCM scm_screem_page_set_pathname( SCM p, SCM pth )
{
	Page *page = (Page*)SCM2PTR( p );
	gchar *path;
	gint len;

	path = gh_scm2newstr( pth, &len );

	screem_page_set_pathname( page, path );

	return SCM_EOL;
}

SCM scm_screem_page_get_pathname( SCM p )
{
	Page *page = (Page*)SCM2PTR( p );
	const gchar *path;

	path = screem_page_get_pathname( page );

	return gh_str2scm( path, strlen( path ) );
}

SCM scm_screem_page_get_mime_type( SCM p )
{
	Page *page = (Page*)SCM2PTR( p );
	const gchar *type;

	type = screem_page_get_mime_type( page );

	return gh_str2scm( type, strlen( type ) );
}

SCM scm_screem_page_set_tab( SCM p, SCM t )
{
	Page *page = (Page*)SCM2PTR( p );
	gint num;

	num = gh_scm2int( t );

	screem_page_set_tab( page, num );

	return SCM_EOL;
}

SCM scm_screem_page_get_tab( SCM p )
{
	Page *page = (Page*)SCM2PTR( p );
	gint num;

	num = screem_page_get_tab( page );

	return gh_int2scm( num );
}

SCM scm_screem_page_set_charset( SCM p, SCM pth )
{
	Page *page = (Page*)SCM2PTR( p );
	gchar *path;
	gint len;

	path = gh_scm2newstr( pth, &len );

	screem_page_set_charset( page, path );

	return SCM_EOL;
}

SCM scm_screem_page_get_charset( SCM p )
{
	Page *page = (Page*)SCM2PTR( p );
	const gchar *path;

	path = screem_page_get_charset( page );

	return gh_str2scm( path, strlen( path ) );
}

SCM scm_screem_page_set_data( SCM p, SCM pth )
{
	Page *page = (Page*)SCM2PTR( p );
	const gchar *path;
	gint len;

	path = gh_scm2newstr( pth, &len );

	screem_page_set_data( page, path );

	return SCM_EOL;
}

SCM scm_screem_page_get_data( SCM p )
{
	Page *page = (Page*)SCM2PTR( p );
	const gchar *path;

	path = screem_page_get_data( page );

	return gh_str2scm( path, strlen( path ) );
}

SCM scm_screem_page_set_changed( SCM p, SCM f )
{
	Page *page = (Page*)SCM2PTR( p );
	gboolean flag;

	flag = gh_scm2bool( f );

	screem_page_set_changed( page, flag );

	return SCM_EOL;
}

SCM scm_screem_page_get_changed( SCM p )
{
	Page *page = (Page*)SCM2PTR( p );
	gboolean flag;

	flag = screem_page_get_changed( page );

	return gh_bool2scm( flag );
}

SCM scm_screem_page_set_mod_time( SCM p, SCM t )
{
	Page *page = (Page*)SCM2PTR( p );
	time_t tm;

	tm = gh_scm2int( t );

	screem_page_set_mod_time( page, tm );

	return SCM_EOL;
}

SCM scm_screem_page_get_mod_time( SCM p )
{
	Page *page = (Page*)SCM2PTR( p );
	time_t tm;

	tm = screem_page_get_mod_time( page );

	return gh_int2scm( tm );
}

SCM scm_screem_page_read( SCM p, SCM f )
{
	Page *page = (Page*)SCM2PTR( p );
	gint fd;
	gboolean ret;

	fd = gh_scm2int( f );

	ret = screem_page_read( page, fd );

	return gh_bool2scm( ret );
}

SCM scm_screem_page_load( SCM p )
{
	Page *page = (Page*)SCM2PTR( p );
	gboolean ret;

	ret = screem_page_load( page );

	return gh_bool2scm( ret );
}

SCM scm_screem_page_revert( SCM p )
{
	Page *page = (Page*)SCM2PTR( p );
	gboolean ret;

	ret = screem_page_revert( page );

	return gh_bool2scm( ret );
}

SCM scm_screem_page_save( SCM p )
{
	Page *page = (Page*)SCM2PTR( p );
	gboolean ret;

	ret = screem_page_save( page );

	return gh_bool2scm( ret );
}

SCM scm_screem_page_from_tab_num( SCM l, SCM n )
{
	GList *list = (GList*)SCM2PTR( l );
	gint num;
	Page *page;

	num = gh_scm2int( n );

	page = screem_page_from_tab_num( list, n );

	return PTR2SCM( page );
}

SCM scm_screem_page_update_tab_nums( SCM l, SCM n )
{
	GList *list = (GList*)SCM2PTR( l );
	gint num;

	num = gh_scm2int( n );

	screem_page_update_tab_nums( list, n );

	return SCM_EOL;
}

SCM scm_screem_page_is_file_page( SCM p )
{
	gchar *path;
	gint len;
	gboolean ret;

	path = gh_scm2newstr( p, &len );

	ret = screem_page_is_file_page( path );

	return gh_bool2scm( ret );
}

SCM scm_screem_page_is_mime_type_page( SCM p )
{
	gchar *path;
	gint len;
	gboolean ret;

	path = gh_scm2newstr( p, &len );

	ret = screem_page_is_file_page( path );

	return gh_bool2scm( ret );
}

SCM scm_screem_page_get_undo_last_action( SCM p )
{
	Page *page = (Page*)SCM2PTR( p );
	Undo *undo;
	
	undo = screem_page_get_undo_last_action( page, TRUE );

	return PTR2SCM( undo );
}

SCM scm_screem_page_set_undo_last_action( SCM p, SCM i, SCM pos, SCM t )
{
	Page *page = (Page*)SCM2PTR( p );
	gboolean insert;
	gint position;
	gchar *text;
	gint len;

	insert = gh_scm2bool( i );
	position = gh_scm2int( pos );
	text = gh_scm2newstr( t, &len );

	screem_page_set_undo_last_action( page, insert, position, text );

	return SCM_EOL;
}

SCM scm_screem_page_add_undo_action( SCM p, SCM u )
{
	Page *page = (Page*)SCM2PTR( p );
	Undo *undo = (Undo*)SCM2PTR( u );

	screem_page_add_undo_action( page, undo );

	return SCM_EOL;
}

SCM scm_screem_page_clear_undo_list( SCM p )
{
	Page *page = (Page*)SCM2PTR( p );

	screem_page_clear_undo_list( page );

	return SCM_EOL;
}

SCM scm_screem_page_get_redo_last_action( SCM p )
{
	Page *page = (Page*)SCM2PTR( p );
	Undo *redo;
	
	redo = screem_page_get_redo_last_action( page, TRUE );

	return PTR2SCM( redo );
}

SCM scm_screem_page_set_redo_last_action( SCM p, SCM u )
{
	Page *page = (Page*)SCM2PTR( p );
	Undo *redo = (Undo*)SCM2PTR( u );

	screem_page_set_redo_last_action( page, redo );

	return SCM_EOL;
}

SCM scm_screem_page_clear_redo_list( SCM p )
{
	Page *page = (Page*)SCM2PTR( p );

	screem_page_clear_redo_list( page );

	return SCM_EOL;
}


#endif


