/*  Screem:  siteTemplate.c,
 *  Handles the site template creation gui
 *
 *  Copyright (C) 1999, 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gnome.h>

#include <glade/glade.h>
#include <gnome-xml/debugXML.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>

#ifdef HAVE_GNOME_VFS
#include <libgnomevfs/gnome-vfs-mime.h>
#endif

#include "fileops.h"
#include "site.h"
#include "siteTemplate.h"
#include "support.h"
#include "preferences.h"

extern Preferences *cfg;

static GladeXML *xml;
static GtkCTreeNode *current_row;

typedef enum _NodeType {
	DIRECTORY = (1 << 0),
	FILE_T = (1 << 1)
} NodeType;

static void build_site( const gchar *stemplate, xmlNodePtr node );
static void build_template_tree( GtkCTree *tree, GtkCTreeNode *tnode, 
				 xmlNodePtr node );
static void build_template( GtkCTree *tree, GtkCTreeNode *tnode,
			    xmlNodePtr node );

void site_template_window()
{
	xml = glade_xml_new( cfg->glade_path, "site_template_window" );
	glade_xml_signal_autoconnect( xml );
}

void site_template_close()
{
	GtkWidget *widget;

	widget = glade_xml_get_widget( xml, "site_template_window" );

	gtk_widget_destroy( widget );
}

void site_template_new()
{
	GtkWidget *tree;
	GtkWidget *frame;

	tree = glade_xml_get_widget( xml, "template_tree" );
	frame = glade_xml_get_widget( xml, "details_frame" );

	if( ! GTK_WIDGET_IS_SENSITIVE( tree ) )
		change_state( tree, NULL );
	if( ! GTK_WIDGET_IS_SENSITIVE( frame ) )
		change_state( frame, NULL );

	gtk_clist_freeze( GTK_CLIST( tree ) );
	gtk_clist_clear( GTK_CLIST( tree ) );
	gtk_clist_thaw( GTK_CLIST( tree ) );

	current_row = NULL;
}

/* add a new node to the template */
void site_template_add_new()
{
	gchar *text[ 2 ] = { NULL, NULL };
	GtkWidget *widget;
	GtkCTreeNode *node;
	GtkWidget *tpath;
	NodeType type;

	gchar *name;
	gchar *tpathname;
	const gchar *mime_type;

	widget = glade_xml_get_widget( xml, "dir_type" );

	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
		type = DIRECTORY;
	else
		type = FILE_T;

	widget = glade_xml_get_widget( xml, "entry_name" );
	name = gtk_entry_get_text( GTK_ENTRY( widget ) );

	if( type == FILE_T ) {
#ifndef HAVE_GNOME_VFS 
		mime_type = gnome_mime_type( name );
#else
		mime_type = gnome_vfs_mime_type_from_name( name );
#endif
	} else {
		mime_type = NULL;
	}

	widget = glade_xml_get_widget( xml, "template_tree" );
	tpath = glade_xml_get_widget( xml, "template_path" );
	
	text[ 0 ] = name;
	node = gtk_ctree_insert_node( GTK_CTREE( widget ), current_row, 
				      NULL, text,
				      3, NULL, NULL, NULL, NULL, 
				      (gboolean)current_row, FALSE );

	if( GTK_WIDGET_IS_SENSITIVE( tpath ) ) {
		tpath = glade_xml_get_widget( xml, "template_path_entry" );

		tpathname = gtk_entry_get_text( GTK_ENTRY( tpath ) );
	} else {
		tpathname = NULL;
	}

	gtk_ctree_node_set_text( GTK_CTREE( widget ), node, 1, tpathname );

	gtk_ctree_node_set_row_data( GTK_CTREE( widget ), node, 
				     (gpointer)type );

	if( current_row )
		gtk_ctree_expand( GTK_CTREE( widget ), current_row );
}

void site_template_remove_item()
{
	GtkWidget *widget;
	GtkCTreeNode *node = current_row;

	g_return_if_fail( current_row != NULL );

	current_row = GTK_CTREE_ROW( node )->parent;

	widget = glade_xml_get_widget( xml, "template_tree" );
	gtk_ctree_remove_node( GTK_CTREE( widget ), node );
}

void site_template_update_details()
{
	GtkWidget *widget;
	GtkCTreeNode *node = current_row;

	g_return_if_fail( current_row != NULL );

	current_row = GTK_CTREE_ROW( node )->parent;

	widget = glade_xml_get_widget( xml, "template_tree" );
	gtk_ctree_remove_node( GTK_CTREE( widget ), node );

	site_template_add_new();
}

void site_template_row_selected( GtkCTree *tree, GtkCTreeNode *row, 
				 gint column )
{
	current_row = row;
}

void site_template_row_unselected( GtkCTree *tree, GtkCTreeNode *row, 
				   gint column )
{
	current_row = NULL;
}

void site_template_save()
{
	GtkWidget *widget;
	xmlDocPtr doc;
	gchar *filename;
	GtkCTreeNode *node;

	widget = glade_xml_get_widget( xml, "template_tree" );

	filename = file_select( _("Select filename for site template") );

	if( ! filename )
		return;

	doc = xmlNewDoc( XML_DEFAULT_VERSION );
	doc->root = xmlNewDocNode( doc, NULL, "SCREEM_SITE_TEMPLATE", NULL );

	node = gtk_ctree_node_nth( GTK_CTREE( widget ), 0 );

	build_template( GTK_CTREE( widget ), node, doc->root );

	xmlSaveFile( filename, doc );
}

void site_template_load()
{
	GtkWidget *widget;
	GtkWidget *frame;
	xmlDocPtr doc;
	gchar *filename;

	widget = glade_xml_get_widget( xml, "template_tree" );
	frame = glade_xml_get_widget( xml, "details_frame" );

	filename = file_select( _("Select filename for site template") );

	if( ! filename )
		return;

	doc = xmlParseFile( filename );

	build_template_tree( GTK_CTREE( widget ), NULL, doc->root->childs );

	if( ! GTK_WIDGET_IS_SENSITIVE( widget ) )
		change_state( widget, NULL );
	if( ! GTK_WIDGET_IS_SENSITIVE( frame ) )
		change_state( frame, NULL );
}

void build_site_with_template( Site *site, gchar *filename )
{
	xmlDocPtr doc;

	const gchar *template;

	g_return_if_fail( filename != NULL || site != NULL );

	doc = xmlParseFile( filename );

	template = screem_site_get_template_path( site );

	build_site( template, doc->root->childs );
}

static void build_site( const gchar *stemplate, xmlNodePtr node )
{
	gchar *name;
	gchar *type;
	gchar *template;

	gboolean status = FALSE;
	FILE *f;

	const gchar *mime_type;

	g_return_if_fail( node != NULL );

	while( xmlNodeIsText( node ) ) {
		node = node->next;
		g_return_if_fail( node != NULL );
	}

	name = xmlNodeGetContent( node );
	type = xmlGetProp( node, "type" );
	template = xmlGetProp( node, "template" );

   	if( ! strcmp( "directory", type ) ) {
		/* create a directory */
		if( template ) {
			/* we don't support site templates for directories
			   yet, there is a huge danger if we do as
			   it could create a NASTY recursive loop */
		} else {
			status =  mkdir_recursive( name );
		}
		
		/* if status is NULL we failed to create the dir so
		   we don't try to create its children */
		if( status ) {
			build_site( stemplate, node->childs );
			/* move up a directory */
			if( node )
				chdir( ".." );
		}
	} else {
		/* create a file */
#ifndef HAVE_GNOME_VFS
		mime_type = gnome_mime_type_or_default( name, "" );
#else
		mime_type = gnome_vfs_mime_type_from_name_or_default( name, "" );
#endif

		if( template ) {
			/* copy template */
			copy_file( template, name );
		} else if( stemplate && ! strcmp( "text/html", mime_type ) ) {
			/* default page template for the site */
			copy_file( stemplate, name );
		} else {
			/* new file  */
			f = fopen( name, "w" );
			fclose( f );
		}
	}

	build_site( stemplate, node->next );
}


static void build_template_tree( GtkCTree *tree, GtkCTreeNode *tnode, 
				 xmlNodePtr node )
{
	GtkCTreeNode *new_node;
	gchar *text[ 2 ] = { NULL, NULL };
	gchar *type;
	gchar *template;
	NodeType t;

	g_return_if_fail( node != NULL );

	while( xmlNodeIsText( node ) ) {
		node = node->next;
	       return;
	}

	text[ 0 ] = xmlNodeGetContent( node );
	type = xmlGetProp( node, "type" );
	template = xmlGetProp( node, "template" );

    	new_node = gtk_ctree_insert_node( tree, tnode, 
					  NULL, text,
					  3, NULL, NULL, NULL, NULL, 
					  (gboolean)current_row, FALSE );

	gtk_ctree_node_set_text( tree, new_node, 1, template );

     	if( ! strcmp( "directory", type ) )
		t = DIRECTORY;
	else
		t = FILE_T;

	gtk_ctree_node_set_row_data( tree, new_node, (gpointer)t );

	build_template_tree( tree, new_node, node->childs );
	build_template_tree( tree, tnode, node->next );

	if( tnode )
		gtk_ctree_expand( tree, tnode );
}

static void build_template( GtkCTree *tree, GtkCTreeNode *tnode, 
			    xmlNodePtr node )
{
	xmlNodePtr new_node;
	NodeType type;
	gchar *pathname = NULL;
	gchar *tpathname = NULL;


	if( ! node )
		return;

	type = (NodeType)gtk_ctree_node_get_row_data( tree ,tnode );

	gtk_ctree_get_node_info( tree, tnode, &pathname, 0, 0, 0, 0, 0, 0, 0);

	gtk_ctree_node_get_text( tree, tnode, 1, &tpathname );

	new_node = xmlNewChild( node, NULL, "node", pathname );

	if( type == DIRECTORY )
		xmlSetProp( new_node, "type", "directory" );
	else
		xmlSetProp( new_node, "type", "file" );

	if( tpathname )
		xmlSetProp( new_node, "template", tpathname );
	
	build_template( tree, GTK_CTREE_ROW( tnode )->children, new_node );

	build_template( tree, GTK_CTREE_ROW( tnode )->sibling, node );
}
