/*  Screem:  tagtree.c,
 *  Handles creation of / interaction with the tag tree
 *
 *  Copyright (C) 1999, 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <dirent.h>
#include <gnome.h>
#include <gnome-xml/debugXML.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>
#include <sys/stat.h>
#include <unistd.h>

#ifdef HAVE_GNOME_VFS
#include <libgnomevfs/gnome-vfs-mime.h>
#endif

#include "editor.h"
#include "tag_tree.h"
#include "xml.h"

static GList *g_name = NULL;  /* group name */
static GList *g_node = NULL;  /* group node */

static GtkCTreeNode* get_node( GtkCTree *tree, GtkCTreeNode *parent, 
			       gchar *line );

static void tag_tree_clicked( GtkWidget *widget, GdkEventButton *event,
			      gpointer data );



/*
 * screem_tag_tree_build:
 * @tree: the tree
 *
 * build a tag tree from the given xml tree
 *
 * return values: none
 */
GtkWidget* screem_tag_tree_build( GtkWidget *t, xmlDocPtr tree )
{
	xmlNodePtr node;

	g_return_val_if_fail( tree != NULL, t );

	node = tree->root;

	/* not interested in the first element */
	node = node->childs;

	screem_tag_tree_add_node( GTK_CTREE( t ), node, NULL );

	add_bluefish_fn( GTK_CTREE( t ) );

    	gtk_signal_connect( GTK_OBJECT( t ), "button_press_event",
			    GTK_SIGNAL_FUNC( tag_tree_clicked ), 0 );

	GTK_WIDGET_UNSET_FLAGS( t, GTK_CAN_FOCUS );

	return t;
}

/*
 * screem_tag_tree_load:
 *
 * load the global and user tag trees
 *
 * return values: an xml tree
 */
xmlDocPtr screem_tag_tree_load()
{
	xmlDocPtr doc;
	xmlDocPtr user_doc;
	xmlNodePtr node;

	const gchar *home;
	gchar *path;
	struct dirent **namelist;
	gint num;
	gint pos;
	gchar *file;
	const gchar *mime_type;

	/* load the global file */
	doc = xmlParseFile( TAG_TREE_FILE );

	/* load user tag trees */
	home = g_get_home_dir();
	path = g_strconcat( home, G_DIR_SEPARATOR_S, ".screem", NULL );
	num = scandir( path, &namelist, 0, alphasort );
	g_free( path );

	for( pos = 0; pos < num; pos ++ ) {
		file = namelist[ pos ]->d_name;
#ifndef HAVE_GNOME_VFS
		mime_type = gnome_mime_type( file );
#else
		mime_type = gnome_vfs_mime_type_from_name( file );
#endif
		if( ! strcmp( "application/x-screem-tag-tree",
			      mime_type ) )	{
			path = g_strconcat( home, G_DIR_SEPARATOR_S, 
					    ".screem", G_DIR_SEPARATOR_S,
					    file, NULL );
			user_doc = xmlParseFile( path );
			g_free( path );
			/* copy the root node */
			node = user_doc->root;
			node = xmlCopyNode( node->childs, TRUE );
			/* add the node to the main tree */
			xmlAddChild( doc->root, node );
			/* delete user_doc */
			xmlFreeDoc( user_doc );
		}
		free( namelist[ pos ] );
	}

	return doc;
}

void screem_tag_tree_add_node( GtkCTree *t, xmlNodePtr node, GtkCTreeNode *parent )
{
	gchar *text[ 2 ] = { NULL, NULL };
	GtkCTreeNode *this;
	gchar *sname;
	gchar *name;
	gchar *open;
	gchar *close;
    	TagData *td;

	name = xml_get_value( node, "name" );
	if( ! name )
		name = "Invalid tree";

	/* does name already match the name of a sibling node of parent ? */
	if( ! parent )
		this = gtk_ctree_node_nth( t, 0 );
	else {
		this = parent;
		this = GTK_CTREE_ROW( this )->children;
	}

	while( this ) {
	       gtk_ctree_node_get_pixtext( t, this, 0, &sname, NULL, NULL,
					   NULL );
	       if( sname && ( ( ! strcmp( name, sname ) ) || 
			      ( sname[ 0 ] == G_DIR_SEPARATOR &&
				! strcmp( name, sname + 1) ) ) )
		       break;
	       this = GTK_CTREE_ROW( this )->sibling;
	}

	/* add the node */
	if( ! this ) {
		text[ 0 ] = name;
		this = gtk_ctree_insert_node( t, parent, NULL, text, 3, 
					      NULL, NULL, NULL, NULL, 
					      (gboolean)parent, FALSE );
	}
	g_free( name );

	/* has children? */
	if( node->childs )
		screem_tag_tree_add_node( t, node->childs, this );

	/* move on to the next sibling */
	if( node->next )
		screem_tag_tree_add_node( t, node->next, parent );
	
	/* set up the row data for the node we added if needed */
	open = xml_get_value( node, "open" );
	close = xml_get_value( node, "close" );
	
	td = g_new( TagData, 1 );
	td->hack = '\0';
	td->open = open;
	td->close = close;
	gtk_ctree_node_set_row_data( GTK_CTREE( t ), this, td );

	return;
}

static void tag_tree_clicked( GtkWidget *widget, GdkEventButton *event,
			      gpointer data )
{
	GtkCTreeNode *node;
        gint row = -1;
        gint col;
	TagData *td;

  	gtk_clist_get_selection_info( GTK_CLIST( widget ),
                                      event->x, event->y, &row, &col );

	if( row == -1 )
		return;

	node = gtk_ctree_node_nth( GTK_CTREE( widget ), ( guint )row );
       
	if( node && event->button == 1 ) {
		td = gtk_ctree_node_get_row_data( GTK_CTREE( widget ), node );
		if( td && ! td->hack && ( td->open || td->close ) ) 
			screem_editor_insert_markup( td->open, td->close );
	}
}

void add_bluefish_fn( GtkCTree *tree )
{
	const gchar *home;
	
	/* 
	 * the first element in a pair is the file name, the second is
	 * the group name for the tag tree
	 */
	gchar *fn_files[] = {
		"php3_functions", "PHP3",
		"rxml_functions", "RXML",
		"ssi_functions", "SSI",
		NULL
	};

	gchar *fn_path;
	gchar *fn_file;
	gint i;

	FILE *file;
	struct stat s;
	gchar *data;
	gint size;

	gchar *search;
	gchar *line;

	gchar *text[ 2 ] = { NULL, NULL };
	GtkCTreeNode *node;
	GtkCTreeNode *nnode;

	GList *list;

	home = g_get_home_dir();

	fn_path = g_strjoin( G_DIR_SEPARATOR_S, home, ".bluefish", NULL );

	for( i = 0; fn_files[ i ]; i += 2 ) {
		fn_file = g_strjoin( G_DIR_SEPARATOR_S, fn_path, fn_files[ i ],
				     NULL );
		if( stat( fn_file, &s ) < 0 ) {
			g_free( fn_file );
			continue;
		}
		if( ! ( file = fopen( fn_file, "r" ) ) ) {
			g_free( fn_file );
			continue;
		}
		g_free( fn_file );
		data = (gchar*)g_malloc( s.st_size );
		size = fread( data, 1, s.st_size, file );
		fclose( file );
		data[ size ] = '\0';

		/* file now read, start processing */
		text[ 0 ] = fn_files[ i + 1 ];
		node = gtk_ctree_insert_node( tree, NULL, NULL, text, 3, 
					      NULL, NULL, NULL, NULL, 
					      FALSE, FALSE );
		for( search = data; ( line = strtok( search, "\n" ) ); ) {
			nnode = get_node( tree, node, line );
			search = NULL;
		}
		g_free( data );
	}

	for( list = g_name; list; list = list->next )
		g_free( list->data );
	g_list_free( g_name );
	g_list_free( g_node );
       
	g_free( fn_path );
}

static GtkCTreeNode* get_node( GtkCTree *tree, GtkCTreeNode *parent, 
			       gchar *line )
{
	gchar *group;
	gchar *function;
	
	GList *list;
	gint num = 0;
	gboolean found = FALSE;

	GtkCTreeNode *pnode;
	GtkCTreeNode *node;
	gchar *text[ 2 ] = { NULL, NULL };

	TagData *td;

	group = line;
	function = strchr( group, ':' );
	*function = 0;
	function ++;
	function = g_strdup( function );

	/* find group in g_name */
	for( list = g_name; list && ! found ; list = list->next ) {
		num ++;
		found = ! strcmp( group, (gchar*)list->data );
	}

	if( found ) {
		list = g_list_nth( g_node, --num );
		pnode = list->data;
	} else {
		text[ 0 ] = group;
		pnode = gtk_ctree_insert_node( tree, parent, NULL, 
					       text, 3, NULL, NULL, NULL, 
					       NULL, FALSE, FALSE );
		g_name = g_list_append( g_name, g_strdup( group ) );
		g_node = g_list_append( g_node, pnode );
	}

	/* add the function as a child of pnode */
	text [ 0 ] = function;
	node = gtk_ctree_insert_node( tree, pnode, NULL, 
				      text, 3, NULL, NULL, NULL, 
				      NULL, TRUE, FALSE );

	td = g_new( TagData, 1 );
	td->hack = '\0';
	td->open = function;
	td->close = NULL;
	gtk_ctree_node_set_row_data( GTK_CTREE( tree ), node, td );


	return node;
}
