/*  Screem:  toolbar.c,
 *  Handles creation of / interaction with the toolbars
 *
 *  Copyright (C) 1999, 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gnome.h>

#include <glade/glade.h>

#include "editMenu.h"
#include "editor.h"
#include "site.h"
#include "siteUI.h"
#include "toolbar.h"
#include "todo.h"
#include "pageUI.h"
#include "preferences.h"
#include "preview.h"

#include "browser.xpm"

static Browser b1;
static Browser b2;
static Browser b3;

extern Site *current_site;
extern Page *current_page;
extern GtkWidget *app;
extern Preferences *cfg;
extern GnomeClient *client;

extern void spell_check( void );

typedef enum _Align {
	LEFT,
	CENTRE,
	JUSTIFY,
	RIGHT
} Align;

/* html elements for toolbar tag help */
static gchar* html_tags[] = {
	"!--", "!doctype", "a", "abbr", "acronym", "address", "applet",
	"area", "b", "base", "basefont", "bdo", "big", "blockquote", "body",
	"br", "button",	"caption", "center", "cite", "code", "col", 
	"colgroup", "dd", "del", "dfn", "dir", "div", "dl", "dt", "em",
	"fieldset", "font", "form", "frame", "frameset", "h1", "h2", "h3",
	"h4", "h5", "h6", "head", "hr", "html", "i", "iframe", "img", "input", 
	"ins", "isindex", "kdb", "label", "legend", "li", "link", "map", 
	"menu", "meta",	"noframes", "noscript", "object", "ol", "optgroup",
	"option", "p", "param", "pre", "q", "s", "samp", "script", "select",
	"small", "span", "strike", "strong", "style", "sub", "table", "tbody",
	"td", "textarea", "tfoot", "th", "thead", "title", "tr", "tt",
	"u", "ul", "var", NULL
};

void single_markup( GtkWidget *widget, gchar *tag );
void toolbar_markup( GtkWidget *widget, gchar *open );
void indent( GtkWidget *widget, gchar *alt_data );
void tag_help( GtkWidget *widget );
void external_view( GtkWidget *widget, gpointer data );
void align( GtkWidget *widget, gchar *data );


static gboolean browser_clicked( GtkWidget *widget, GdkEventButton *event, 
			     Browser *b );
static void browser_launch_url( gchar *path );
static void browser_launch( gchar *path );
static void browser_settings( Browser *b );
void browsers_deleted( GtkWidget *widget, GdkEvent *event, Browser *b );
void browsers_clicked( GtkWidget *widget, gint button, Browser *b );

GtkWidget *create_sidebar()
{
	GtkWidget *widget;
	GladeXML *xml;

	gint i;
	const gchar *buttons[] = {
		"editor_button", "preview_button",
		"link_button", "help_button", NULL
	};

	GSList *group = NULL;

	xml = glade_xml_new( cfg->glade_path, "sidebar" );

	/* sort out the groups for the radio buttons */
	for( i = 0; buttons[ i ]; i ++ ) {
		widget = glade_xml_get_widget( xml,  buttons[ i ] );
		gtk_radio_button_set_group( GTK_RADIO_BUTTON( widget ),
					    group );
		group = gtk_radio_button_group( GTK_RADIO_BUTTON( widget ) );
	}
	widget = glade_xml_get_widget( xml, buttons[ 1 ] );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), TRUE );

	glade_xml_signal_autoconnect( xml );
	widget = glade_xml_get_widget( xml, "sidebar" );
	return widget;
}

GtkWidget *create_toolbar1()
{
	GladeXML *xml;
	GtkWidget *widget;

	xml = glade_xml_new( cfg->glade_path, "toolbar1" );
	glade_xml_signal_autoconnect( xml );
	widget = glade_xml_get_widget( xml, "toolbar1" );
	gtk_object_unref( GTK_OBJECT( xml ) );
	return widget;
}

GtkWidget *create_toolbar2()
{
	GladeXML *xml;
	GtkWidget *widget;

	xml = glade_xml_new( cfg->glade_path, "toolbar2" );
	glade_xml_signal_autoconnect( xml );
	widget = glade_xml_get_widget( xml, "toolbar2" );
	gtk_object_unref( GTK_OBJECT( xml ) );
	return widget;
}

GtkWidget *create_toolbar3()
{
	GladeXML *xml;
	GtkWidget *widget;

	xml = glade_xml_new( cfg->glade_path, "toolbar3" );
	glade_xml_signal_autoconnect( xml );
	widget = glade_xml_get_widget( xml, "toolbar3" );
	gtk_object_unref( GTK_OBJECT( xml ) );
	return widget;
}

GtkWidget *create_toolbar4()
{
        GtkWidget *tag_combo;
        GList *tag_strings = NULL;
        gint num;
	GladeXML *xml;
	GtkWidget *widget;

	xml = glade_xml_new( cfg->glade_path, "toolbar4" );
	glade_xml_signal_autoconnect( xml );

     	for( num = 1; html_tags[ num ]; num ++ )
		tag_strings = g_list_append( tag_strings, html_tags[ num ] );
         
	tag_combo = glade_xml_get_widget( xml, "tag_combo" );
	gtk_combo_set_popdown_strings( GTK_COMBO( tag_combo ), tag_strings );
        g_list_free( tag_strings );
	gtk_widget_set_usize( tag_combo, 128, - 1 );
      	widget = glade_xml_get_widget( xml, "toolbar4" );
	gtk_object_unref( GTK_OBJECT( xml ) );
	return widget;
}

GtkWidget *create_browsers_bar()
{
	GtkWidget *toolbar;
	GtkWidget *widget;
	gchar *tip;
	gchar *icon;

	/* load the browser settings */
	gnome_config_push_prefix(gnome_client_get_config_prefix(client));
	b1.path = gnome_config_get_string( "External Browsers/path1=not set" );
	b1.use_url = gnome_config_get_int( "External Browsers/url1=0" );
	b2.path = gnome_config_get_string( "External Browsers/path2=not set" );
	b2.use_url = gnome_config_get_int( "External Browsers/url2=0" );
	b3.path = gnome_config_get_string( "External Browsers/path3=not set" );
	b3.use_url = gnome_config_get_int( "External Browsers/url3=0" );

	if( ! strcmp( b1.path, "not set" ) ) {
		g_free( b1.path );
		b1.path = NULL;
	}
	if( ! strcmp( b2.path, "not set" ) ) {
		g_free( b2.path );
		b2.path = NULL;
	}
	if( ! strcmp( b3.path, "not set" ) ) {
		g_free( b3.path );
		b3.path = NULL;
	}

	icon = gnome_config_get_string( "External Browsers/icon1=no icon" );
	if( ! strcmp( "no icon", icon ) ) {
		g_free( icon );
		icon = NULL;
	}
	b1.iconFile = icon;
  
	icon = gnome_config_get_string( "External Browsers/icon2=no icon" );
	if( ! strcmp( "no icon", icon ) ) {
		g_free( icon );
		icon = NULL;
	}
	b2.iconFile = icon;

	icon = gnome_config_get_string( "External Browsers/icon3=no icon" );
	if( ! strcmp( "no icon", icon ) ) {
		g_free( icon );
		icon = NULL;
	}
	b3.iconFile = icon;
	gnome_config_pop_prefix();

	b1.dialog = NULL;
	b2.dialog = NULL;
	b3.dialog = NULL;

	toolbar = gtk_toolbar_new( GTK_ORIENTATION_HORIZONTAL, 
				   GTK_TOOLBAR_ICONS);

	/* create 3 buttons for external browsers */
	if( ! b1.iconFile )
		b1.widget = gnome_pixmap_new_from_xpm_d( browser_xpm );
	else
		b1.widget = gnome_pixmap_new_from_file_at_size( b1.iconFile, 
								24, 24 );

	if( ! b2.iconFile )
		b2.widget =  gnome_pixmap_new_from_xpm_d( browser_xpm );
	else
		b2.widget = gnome_pixmap_new_from_file_at_size( b2.iconFile,
								24, 24 );

	if( ! b3.iconFile )
		b3.widget =  gnome_pixmap_new_from_xpm_d( browser_xpm );
	else
		b3.widget = gnome_pixmap_new_from_file_at_size( b3.iconFile,
								24, 24 );

	tip = b1.path ? b1.path : _("Not set");
	gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "", tip, "",
				 b1.widget, 0, 0 );

	tip = b2.path ? b2.path : _("Not set");
	gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "", tip, "", 
				 b2.widget, 0, 0 );

;	tip = b3.path ? b3.path : _("Not set");
	gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "", tip, "",
				 b3.widget, 0, 0 );

	widget = b1.widget->parent->parent;
	gtk_signal_connect_after( GTK_OBJECT( widget ),"button_press_event",
				  browser_clicked, &b1 );
	widget = b2.widget->parent->parent;
	gtk_signal_connect_after( GTK_OBJECT( widget ),"button_press_event",
				  browser_clicked, &b2 );
	widget = b3.widget->parent->parent;
	gtk_signal_connect_after( GTK_OBJECT( widget ),"button_press_event",
				  browser_clicked, &b3 );

	return toolbar;
}

/* deals with tags which have no close tag */
void single_markup( GtkWidget *widget, gchar *tag )
{
	screem_editor_insert_markup( tag, NULL );
}

/* deals with B I U UL OL etc */
void toolbar_markup( GtkWidget *widget, gchar *open )
{
	gchar *close;
	gboolean has_selection;
	gboolean pressed;
	static gboolean toggle = FALSE;

	if( toggle ) {
		toggle = FALSE;
		return;
	}

	pressed = !gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	close = g_strdup_printf( "%c/%s", open[ 0 ], &open[ 1 ] );
	has_selection = screem_editor_has_selection( NULL, NULL );

	if( has_selection ) {
		toggle = TRUE;
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
					      pressed );
	} else if( pressed ) {
		open = NULL;
	} else {
		g_free( close );
		close = NULL;
	}

	screem_editor_insert_markup( open, close );

	g_free( close );
}

void indent( GtkWidget *widget, gchar *alt_data )
{
    	static gint indent_level = 0;

	gint alt = atoi( alt_data );

	indent_level += alt;

	if( indent_level < 0 ) {
		indent_level = 0;
		return;
	}

	if( alt < 0 )
		screem_editor_insert_markup( NULL, "</blockquote>" );
	else
		screem_editor_insert_markup( "<blockquote>", NULL );
}

void external_view( GtkWidget *widget, gpointer data )
{
	Page *page;
	const gchar *pathname;

	page = screem_site_get_current_page( current_site );

	g_return_if_fail( page != NULL );

	pathname = screem_page_get_pathname( page );

	g_return_if_fail( pathname != NULL );

	if( screem_page_save_confirmation( page ) == 2 )
		return;

	gnome_url_show( pathname );
}

void tag_help( GtkWidget *widget )
{
	gchar *tag;

	tag = gtk_entry_get_text( GTK_ENTRY( widget ) );

	screem_help_show( tag );
}

void align( GtkWidget *widget, gchar *data )
{
	static gboolean toggle = FALSE;
	static GtkWidget *current = NULL;
	static Align pa = LEFT;
	gchar *tag = NULL;

	Align a;

	if( ! strcmp( "LEFT", data ) )
		a = LEFT;
	else if( ! strcmp( "RIGHT", data ) )
		a = RIGHT;
	else if( ! strcmp( "CENTRE", data ) )
		a = CENTRE;
	else
		a = JUSTIFY;

	if( toggle ) {
		toggle = FALSE;
		return;
	}

	/* is the click on the same alignement we are already set on? */
	if( current == widget )
		return;
  
	if( current ) {
		toggle = TRUE;
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( current ),
					      FALSE );
		if( pa == CENTRE ) {
			tag = "</center>";
		} else if( pa != LEFT ) {
			tag = "</div>";
		}
		screem_editor_insert_markup( NULL, tag );
	}

	/* now we set the new alignment */
	current = widget;
	pa = a;
	switch( a ) {
	case CENTRE:
		tag = "<center>";
		break;
	case RIGHT:
		tag = "<div align=\"right\">";
		break;
	case JUSTIFY:
		tag = "<div align=\"justify\">";
		break;
	case LEFT:
		return;
		break;
	}

	screem_editor_insert_markup( tag, NULL );
}

static gboolean browser_clicked( GtkWidget *widget, GdkEventButton *event, 
			     Browser *b )
{
	if( event->button == 1 ) {
		/* launch the browser */
		if( b->path && ! b->use_url ) 
			browser_launch( b->path );
		else if( b->path )
			browser_launch_url( b->path );
		else
			/* if the browser isn't configured then lets open
			   the dialog to configure it */
			event->button = 3;
	}
	
	if( event->button == 3 ) {
		/* configure the button */
		browser_settings( b );
	}

	return TRUE;
}

static void browser_launch_url( gchar *path )
{
	Page *page;
	const gchar *site_path;
	const gchar *http_url;
	const gchar *page_path;

	gchar *url;

	page = screem_site_get_current_page( current_site );

	g_return_if_fail( page != NULL );

	site_path = screem_site_get_pathname( current_site );
	http_url = screem_site_get_http_url( current_site );
	page_path = screem_page_get_pathname( page );

	url = g_strconcat( path, " http://", http_url, 
			   page_path + strlen( site_path ), NULL );

	if( screem_page_save_confirmation( page ) != 2 ) {
		gnome_execute_shell( NULL, url );
	}

	g_free( url );
}

static void browser_launch( gchar *path )
{
	Page *page;
	gchar *file;
	gchar *rest;
	gint pos;

	const gchar *pathname;

	page = screem_site_get_current_page( current_site );
	
        g_return_if_fail( page != NULL );

	pathname = screem_page_get_pathname( page );

	file = strstr( path, "%f" );

	if( file ) {
		pos = file - path;
		rest = file + 2;
		if( rest > ( path + strlen( path ) ) )
			rest = "";
		file = g_strndup( path, pos );
	} else {
		/* if there was no %f then we will assume that we should just
		   pass the path on the end */
		file = g_strdup( path );
		rest = "";
	}

	path = g_strdup_printf( "%s %s %s", file, pathname, rest );
	g_free( file );

	if( screem_page_save_confirmation( page ) != 2 ) {
		/* path is now the program + args we should execute */
		gnome_execute_shell( NULL, path );
	}
	g_free( path );
}

static void browser_settings( Browser *b )
{
	GtkWidget *dialog;
	GtkWidget *path;
	GtkWidget *icon;
	GtkWidget *check;

	GladeXML *xml;

	if( ! b )
		return;

	if( b->dialog ) {
		gdk_window_raise( b->dialog->window );
                gdk_window_show( b->dialog->window );
                return;
	}

	xml = glade_xml_new( cfg->glade_path, "browsers_dialog" );

	dialog = glade_xml_get_widget( xml, "browsers_dialog" );
	path = glade_xml_get_widget( xml, "browser_entry" );
	icon = glade_xml_get_widget( xml, "icon" );
	check = glade_xml_get_widget( xml, "url_check" );

	gtk_object_unref( GTK_OBJECT( xml ) );

	/* set the path/icon */
	if( b->path )
		gtk_entry_set_text( GTK_ENTRY( path ), b->path );

	if( b->iconFile )
		gnome_icon_entry_set_icon(GNOME_ICON_ENTRY(icon), b->iconFile);

	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( check ), b->use_url );

   	gtk_widget_show_all( icon );

	gtk_object_set_data( GTK_OBJECT( dialog ), "path", path );
	gtk_object_set_data( GTK_OBJECT( dialog ), "icon", icon );
	gtk_object_set_data( GTK_OBJECT( dialog ), "check", check );

	/* 
	 * have to do the signals here as glade can't handle the passing of
	 * a data pointer 
	 */
	gtk_signal_connect( GTK_OBJECT( dialog ), "clicked",
			    GTK_SIGNAL_FUNC( browsers_clicked ), b );
	gtk_signal_connect( GTK_OBJECT( dialog ), "delete_event",
			    GTK_SIGNAL_FUNC( browsers_deleted ), b );

	b->dialog = dialog;
}

void browsers_deleted( GtkWidget *widget, GdkEvent *event, Browser *b )
{
	b->dialog = NULL;
}

void browsers_clicked( GtkWidget *widget, gint button, Browser *b )
{
	GtkWidget *path;
	GtkWidget *icon;
	GtkTooltipsData *ttd;
	GtkWidget *check;

	gchar *bpath;
	gchar *bicon;

	if( button < 2 ) {
		path = gtk_object_get_data( GTK_OBJECT( widget ), "path" );
		icon = gtk_object_get_data( GTK_OBJECT( widget ), "icon" );
		check = gtk_object_get_data( GTK_OBJECT( widget ), "check" );

		bpath = gtk_entry_get_text( GTK_ENTRY( path ) );
		
		if( b->path )
			g_free( b->path );
		
		b->path = g_strdup( bpath );
		
		bicon = gtk_entry_get_text( GTK_ENTRY( gnome_icon_entry_gtk_entry( GNOME_ICON_ENTRY( icon ) ) ) );

		if( b->iconFile )
			g_free( b->iconFile );

		b->iconFile = g_strdup( bicon );

		b->use_url = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( check ) );

		gnome_pixmap_load_file_at_size( GNOME_PIXMAP( b->widget ), 
						bicon, 24, 24 );

		/* set the tooltip to the programs path */
		ttd = gtk_tooltips_data_get( b->widget->parent->parent );
		gtk_tooltips_set_tip( ttd->tooltips, ttd->widget, b->path, "");

		/* save the settings */
		gnome_config_push_prefix(gnome_client_get_config_prefix(client));
		if( b == &b1 ) {
			gnome_config_set_string( "External Browsers/path1",
						 b->path );
			gnome_config_set_string( "External Browsers/icon1",
						 b->iconFile);
			gnome_config_set_int( "External Browsers/url1",
					      b->use_url );
		} else if( b == &b2 ) {
			gnome_config_set_string( "External Browsers/path2",
						 b->path );
			gnome_config_set_string( "External Browsers/icon2",
						 b->iconFile);
			gnome_config_set_int( "External Browsers/url2",
					      b->use_url );
		} else {
			gnome_config_set_string( "External Browsers/path3",
						 b->path );
			gnome_config_set_string( "External Browsers/icon3",
						 b->iconFile);
			gnome_config_set_int( "External Browsers/url3",
					      b->use_url );
		}
		gnome_config_pop_prefix();
	}

	if( button != 1 ) {
		gtk_widget_destroy( widget );
		b->dialog = NULL;
	}
}
