/*-------------------------------------------------------------------------

  device.c - Accomodates subtle variations in PIC16 devices

   Written By -  Scott Dattalo scott@dattalo.com

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any
   later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
-------------------------------------------------------------------------*/

/*
  PIC device abstraction

  There are dozens of variations of PIC microcontrollers. This include
  file attempts to abstract those differences so that SDCC can easily
  deal with them.
*/

#ifndef  __DEVICE_H__
#define  __DEVICE_H__

#if defined(__BORLANDC__) || defined(_MSC_VER)
#define STRCASECMP stricmp
#else
#define STRCASECMP strcasecmp
#endif

#define CONFIGURATION_WORDS	20
#define IDLOCATION_BYTES	20

typedef struct {
	int sfrLoAddr;
	int sfrHiAddr;
} sfrRangeInfo_t;
	

typedef struct {
	unsigned int mask;
	int emit;
	unsigned int value;
} configRegInfo_t;

typedef struct {
	int confAddrStart;	/* starting address */
	int confAddrEnd;	/* ending address */
	configRegInfo_t crInfo[ CONFIGURATION_WORDS ];
} configWordsInfo_t;

typedef struct {
	unsigned char emit;
	unsigned char value;
} idRegInfo_t;

typedef struct {
	int idAddrStart;	/* starting ID address */
	int idAddrEnd;		/* ending ID address */
	idRegInfo_t irInfo[ IDLOCATION_BYTES ];
} idBytesInfo_t;


#define PROCESSOR_NAMES    4
/* Processor unique attributes */
typedef struct PIC16_device {
  char *name[PROCESSOR_NAMES];/* aliases for the processor name */

  int maxRAMaddress;		/* maximum value for a data address */
  int RAMsize;			/* size of Data RAM - VR 031120 */
  int acsSplitOfs;		/* access bank split offset */
  int extMIface;		/* device has external memory interface */
  sfrRangeInfo_t sfrRange;	/* SFR range */
  configWordsInfo_t cwInfo;	/* configuration words info */
  idBytesInfo_t idInfo;		/* ID Locations info */
} PIC16_device;

/* Given a pointer to a register, this macro returns the bank that it is in */
#define REG_ADDR(r)        ((r)->isBitField ? (((r)->address)>>3) : (r)->address)

#define OF_LR_SUPPORT		0x00000001
#define OF_OPTIMIZE_GOTO	0x00000002
#define OF_OPTIMIZE_CMP		0x00000004
#define OF_OPTIMIZE_DF		0x00000008

typedef struct {
  int no_banksel;
  int opt_banksel;
  int omit_configw;
  int omit_ivt;
  int leave_reset;
  int stack_model;
  int ivt_loc;
  int nodefaultlibs;
  int dumpcalltree;
  char *crt_name;
  int no_crt;
  int ip_stack;
  unsigned long opt_flags;
  int gstack;
  unsigned int debgen;
  int CATregs;
} pic16_options_t;

extern int xinst;

#define STACK_MODEL_SMALL	(pic16_options.stack_model == 0)
#define STACK_MODEL_LARGE	(pic16_options.stack_model == 1)

extern set *fix_idataSymSet;
extern set *rel_idataSymSet;

#if 0
/* This is an experimental code for #pragma inline
   and is temporarily disabled for 2.5.0 release */
extern set *asmInlineMap;
#endif  /* 0 */

typedef struct {
  unsigned long isize;
  unsigned long adsize;
  unsigned long udsize;
  unsigned long idsize;
  unsigned long intsize;
} stats_t;

extern stats_t statistics;

extern pic16_options_t pic16_options;
extern PIC16_device *pic16;

/****************************************/
void pic16_assignConfigWordValue(int address, unsigned int value);
void pic16_assignIdByteValue(int address, char value);
int pic16_isREGinBank(regs *reg, int bank);
int pic16_REGallBanks(regs *reg);
void pic16_setMaxRAM(int size);
int PIC16_IS_CONFIG_ADDRESS(int address);
int PIC16_IS_IDLOC_ADDRESS(int address);
int PIC16_IS_HWREG_ADDRESS(int address);

int checkAddReg(set **set, regs *reg);
int checkAddSym(set **set, symbol *reg);
int checkSym(set *set, symbol *reg);

#endif  /* __DEVICE_H__ */
