#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>

#include "../include/string.h"
#include "../include/strexp.h"
#include "../include/os.h"

#include "gw.h"

#include "obj.h"
#include "messages.h"
#include "simutils.h"
#include "cmd.h"
#include "sar.h"
#include "config.h"


void SARCommandSet(void *client_data, const char *arg);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *	General purpose object parameter value manipulation.
 */
void SARCommandSet(void *client_data, const char *arg)
{
	sar_core_struct *core_ptr = client_data;
	sar_scene_struct *scene;
	int obj_num = -1;
	sar_object_struct *obj_ptr = NULL;
	char **argv, **strv, *strptr;
	int argc, strc;
	char *parm, *val;


	if((core_ptr == NULL) ||
	   (arg == NULL)
	)
	    return;

	scene = core_ptr->scene;

	/* No argument given? */
	if((*arg) == '\0')
	{
            SARMessageAdd(
                scene,
                "Usage: set <object>=<parameter>:<value>"
            );
            return;
        }

	/* Parse arguments, format is:
	 *	<object_name>=<parameter>:<value>
	 */
	argv = strchrexp(arg, '=', &argc);
	if(argv == NULL)
	    return;

	/* First argument is always object name or number, so
	 * match it.
	 */
	strptr = ((argc > 0) ? argv[0] : NULL);
	if(strptr != NULL)
	{
	    while(ISBLANK(*strptr))
		strptr++;

	    /* First character the number sign? */
	    if((*strptr) == '#')
	    {
		/* Match obj_ptr explicitly. */
		strptr++;
		obj_num = atoi(strptr);
		if(SARObjIsAllocated(
		    core_ptr->object, core_ptr->total_objects,
		    obj_num
		))
		{
		    obj_ptr = core_ptr->object[obj_num];
		}
		else
		{
		    obj_num = -1;
		    obj_ptr = NULL;
		}
	    }
	    else
	    {
		/* Match obj_ptr by name. */
		StringStripSpaces(strptr);
		obj_ptr = SARObjMatchPointerByName(
		    core_ptr->scene,
		    core_ptr->object, core_ptr->total_objects,
                    strptr, &obj_num
		);
	    }
	}

	/* No such object? */
	if((obj_ptr == NULL) && (argc > 0))
	{
	    char *text = (char *)malloc(
		256 + strlen(argv[0])
	    );
	    if(text != NULL)
	    {
		sprintf(text, "%s: %s.",
		    argv[0], SAR_MESG_NO_SUCH_OBJECT
		);
                SARMessageAdd(scene, text);
		free(text);
	    }
            StringFreeArray(argv, argc);
	    return;
	}


	/* Second argument is the "<parameter>:<value>". */
	if(argc > 1)
	    strv = strchrexp(argv[1], ':', &strc);
	else
	    strv = NULL;
	if((strv == NULL) || (strc <= 1))
	{
	    SARMessageAdd(
		scene,
		"Usage: set <object>=<parameter>:<value>"
	    );
	    StringFreeArray(argv, argc);
	    StringFreeArray(strv, strc);
	    return;
	}

	/* Get links to parm and val in the strv array. */
	if(strc > 0)
	    parm = strv[0];
	else
	    parm = NULL;

	if(strc > 1)
	    val = strv[1];
        else
            val = NULL;

	if((parm == NULL) || (val == NULL))
	{   
            SARMessageAdd(
                scene,
                "Usage: set <object>=<parameter>:<value>"
            );
            StringFreeArray(argv, argc);
            StringFreeArray(strv, strc);
            return;
        } 

	StringStripSpaces(parm);
	StringStripSpaces(val);


	/* ********************************************************* */
	/* Begin handling parm. */

	/* Set new x coordinate (in meters)? */
	if(!strcasecmp(parm, "x"))
	{
	    if(strpfx(val, "++"))
	    {
		/* Relative move. */
		double dx = atof(val + 2);
                obj_ptr->pos.x += dx;
	    }
	    else if(strpfx(val, "--"))
	    {
                /* Relative move. */
                double dx = atof(val + 1);
                obj_ptr->pos.x += dx;
	    }
	    else
	    {
		/* Explicit move. */
		double x = atof(val);
		obj_ptr->pos.x = x;
	    }
	    /* Realize new position. */
	    SARSimWarpObject(
		scene, obj_ptr,
		&obj_ptr->pos, NULL
	    );
	}
	/* Set new y coordinate (in meters)? */
        else if(!strcasecmp(parm, "y"))
        {
            if(strpfx(val, "++"))
            {
                /* Relative move. */
                double dy = atof(val + 2);
                obj_ptr->pos.y += dy;
            }
            else if(strpfx(val, "--"))
            {
                /* Relative move. */
                double dy = atof(val + 1);
                obj_ptr->pos.y += dy;
            }
            else
            {
                /* Explicit move. */
                double y = atof(val);
                obj_ptr->pos.y = y;
            }
            /* Realize new position. */
            SARSimWarpObject(
                scene, obj_ptr,
                &obj_ptr->pos, NULL
            );
        }
        /* Set new z coordinate (in meters)? */
        else if(!strcasecmp(parm, "z"))
        {
            if(strpfx(val, "++"))
            {
                /* Relative move. */
                double dz = atof(val + 2);
                obj_ptr->pos.z += dz;
            }
            else if(strpfx(val, "--"))
            {
                /* Relative move. */
                double dz = atof(val + 1);
                obj_ptr->pos.z += dz;
            }
            else
            {
                /* Explicit move. */
                double z = atof(val);
                obj_ptr->pos.z = z;
            }
            /* Realize new position. */
            SARSimWarpObject(
                scene, obj_ptr,
                &obj_ptr->pos, NULL
            );
        }

        /* Set new fuel amount (in kg)? */
        else if(!strcasecmp(parm, "fuel"))
        {
	    sar_object_aircraft_struct *obj_aircraft_ptr;

	    switch(obj_ptr->type)
	    {
	      case SAR_OBJ_TYPE_AIRCRAFT:
		obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
		if(obj_aircraft_ptr != NULL)
		{
		    obj_aircraft_ptr->fuel = MAX(atof(val), 0.0);
		}
		break;
	    }
        }
        /* Service ceiling (in feet)? */
        else if(!strcasecmp(parm, "service_ceiling") ||
                !strcasecmp(parm, "serviceceiling")
	)
        {
            sar_object_aircraft_struct *obj_aircraft_ptr;

            switch(obj_ptr->type)
            {
              case SAR_OBJ_TYPE_AIRCRAFT:
                obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
                if(obj_aircraft_ptr != NULL)
                {
                    obj_aircraft_ptr->service_ceiling = MAX(atof(val), 0.0);
                }
                break;
            }
        }

	/* Switch to slew mode? */
        else if(!strcasecmp(parm, "slew"))
        {
            sar_object_aircraft_struct *obj_aircraft_ptr;

            switch(obj_ptr->type)
            {
              case SAR_OBJ_TYPE_AIRCRAFT:
                obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
                if(obj_aircraft_ptr != NULL)
                {
		    /* Turn on slew mode? */
		    if(StringIsYes(val))
		    {
			/* Flight model currently not of type slew? */
			if(obj_aircraft_ptr->flight_model_type !=
			    SAR_FLIGHT_MODEL_SLEW
			)
			{
			    /* Record current flight model type. */
			    obj_aircraft_ptr->last_flight_model_type =
				obj_aircraft_ptr->flight_model_type;

			    /* Set flight model to slew mode. */
			    obj_aircraft_ptr->flight_model_type =
				SAR_FLIGHT_MODEL_SLEW;
			}
		    }
		    /* Turn off slew. */
		    else
		    {
                        /* Flight model currently not of type slew? */
                        if(obj_aircraft_ptr->flight_model_type ==
                            SAR_FLIGHT_MODEL_SLEW
                        )
                        {
                            /* Restore previous flight model type. */
                            obj_aircraft_ptr->flight_model_type =
                                obj_aircraft_ptr->last_flight_model_type;
                        }
		    }
                }
                break;
            }
        }



	/* Deallocate exploded argument strings. */
	StringFreeArray(argv, argc);

	/* Exploded parameter and value strings. */
	StringFreeArray(strv, strc);

	/* Links to parm and val strings are now invalid. */
	parm = NULL;
	val = NULL;

	return;
}
