#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>

#ifdef JS_SUPPORT
# include <jsw.h>
#endif

#include "gw.h"		/* Need to know about GW key codes. */

#include "gctl.h"
#include "sar.h"


int GCtlInit(gctl_struct *gc, int type, gctl_flags_t options);
void GCtlUpdate(gctl_struct *gc);
void GCtlHandlePointer(
        gw_display_struct *display, gctl_struct *gc,
	int type,			/* One of GWEventType*. */
	int button,			/* Button number. */
	int x, int y,
        long t				/* Time stamp. */
);
void GCtlHandleKey(
	gw_display_struct *display, gctl_struct *gc,
	int k, int kstate,		/* Key code and state. */
	int alt_state, int ctrl_state,	/* Modifiers. */
	int shift_state,
	long t				/* Time stamp. */
);
void GCtlShutdown(gctl_struct *gc);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *	Initializes the game controller, closing it if its
 *	already opened before initializing it again.
 */
int GCtlInit(gctl_struct *gc, int type, gctl_flags_t options)
{
	int status = 0;
#ifdef JSW_H
	int js_num, jsstatus;
	js_data_struct *jsd;
	char js_dev_name[PATH_MAX + NAME_MAX];
#endif	/* JSW_H */


	if(gc == NULL)
	    return(-1);


	/* Close current game controller as needed and deallocate
	 * resources.
	 */
	GCtlShutdown(gc);


	/* Set new game controller type. */
	gc->type = type;
	gc->options = options;

	/* Open game controller device. */
	switch(gc->type)
	{
	  case GCTL_CONTROLLER_JOYSTICK:
#ifdef JSW_H
	    /* Calculate total number of joysticks. */
	    if(options & GCTL_OPT_JS0_INIT)
		gc->total_joysticks = 1;
            if(options & GCTL_OPT_JS1_INIT)
                gc->total_joysticks = 2;

	    /* Allocate an array of joystick data structures. */
	    jsd = (js_data_struct *)calloc(
		gc->total_joysticks, sizeof(js_data_struct)
	    );
	    if(jsd == NULL)
	    {
		gc->total_joysticks = 0;
		return(-1);
	    }
	    else
	    {
		gc->joystick = jsd;
	    }

	    /* Initialize each joystick. */
	    for(js_num = 0; js_num < gc->total_joysticks; js_num++)
	    {
		sprintf(js_dev_name, "/dev/js%i", js_num);
	        jsstatus = JSInit(
		    &jsd[js_num],
		    js_dev_name,
		    NULL,
		    JSFlagNonBlocking
		);
		if(jsstatus != JSSuccess)
		    status = -1;
	    }

	    /* Set up js0 axis mappings. */
	    if(1)
	    {
		int	*axis_heading = &gc->js0_axis_heading,
			*axis_bank = &gc->js0_axis_bank,
			*axis_pitch = &gc->js0_axis_pitch,
			*axis_throttle = &gc->js0_axis_throttle,
			*axis_hat_x = &gc->js0_axis_hat_x,
			*axis_hat_y = &gc->js0_axis_hat_y;

		/* Reset axis mappings. */
		(*axis_heading) = -1;
		(*axis_bank) = -1;
		(*axis_pitch) = -1;
		(*axis_throttle) = -1;
		(*axis_hat_x) = -1;
		(*axis_hat_y) = -1;

		/* Throttle and rudder unit? */
		if(options & GCTL_OPT_JS0_AS_THROTTLE_AND_RUDDER)
		{
		    (*axis_heading) = 0;
		    (*axis_throttle) = 1;
		}
		/* Joystick as a standard composite unit. */
		else
		{
		    /* Get bank axis. */
		    if(options & GCTL_OPT_JS0_BANK)
			(*axis_bank) = 0;
		    /* Get heading axis. */
		    if(options & GCTL_OPT_JS0_HEADING)
			(*axis_heading) = 2;
		    /* Get pitch axis. */
		    if(options & GCTL_OPT_JS0_PITCH)
			(*axis_pitch) = 1;

		    /* Get throttle axis. */
		    if(options & GCTL_OPT_JS0_THROTTLE)
		    {
			/* Check if heading axis is valid, if not then
			 * throttle is axis 2.
			 */
			if((*axis_heading) < 0)
			    (*axis_throttle) = 2;
			else
			    (*axis_throttle) = 3;
                    }

		    /* Get hat x and y axis. */
		    if(options & GCTL_OPT_JS0_HAT)
		    {
			/* Hat x. */
			if((*axis_heading) < 0)
			{
			    if((*axis_throttle) < 0)
                                (*axis_hat_x) = 2;
                            else
                                (*axis_hat_x) = 3;
                        }
                        else if((*axis_throttle) < 0)
                        {
                            (*axis_hat_x) = 3;
                        }
                        else
                        {
                            (*axis_hat_x) = 4;
                        }
                        /* Hat y (the axis after hat x). */
                        (*axis_hat_y) = (*axis_hat_x) + 1;
                    }
                }	/* Joystick as a standard composite unit. */
	    }
            /* Set up js1 axis mappings. */
	    if(1)
	    {
                int     *axis_heading = &gc->js1_axis_heading,
                        *axis_bank = &gc->js1_axis_bank,
                        *axis_pitch = &gc->js1_axis_pitch,
                        *axis_throttle = &gc->js1_axis_throttle,
                        *axis_hat_x = &gc->js1_axis_hat_x,
                        *axis_hat_y = &gc->js1_axis_hat_y;

                /* Reset axis mappings. */
                (*axis_heading) = -1;
                (*axis_bank) = -1;
                (*axis_pitch) = -1;
                (*axis_throttle) = -1;
                (*axis_hat_x) = -1;
                (*axis_hat_y) = -1;

                /* Throttle and rudder unit? */
                if(options & GCTL_OPT_JS1_AS_THROTTLE_AND_RUDDER)
                {
                    (*axis_heading) = 0;
                    (*axis_throttle) = 1;
                }
                /* Joystick as a standard composite unit. */
                else
                {
                    /* Get bank axis. */
                    if(options & GCTL_OPT_JS1_BANK)
                        (*axis_bank) = 0;
                    /* Get heading axis. */
                    if(options & GCTL_OPT_JS1_HEADING)
                        (*axis_heading) = 2;
                    /* Get pitch axis. */
                    if(options & GCTL_OPT_JS1_PITCH)
                        (*axis_pitch) = 1;
                         
                    /* Get throttle axis. */
                    if(options & GCTL_OPT_JS1_THROTTLE)
                    {
                        /* Check if heading axis is valid, if not then
                         * throttle is axis 2.
                         */
                        if((*axis_heading) < 0)
                            (*axis_throttle) = 2;
                        else
                            (*axis_throttle) = 3;
                    }

                    /* Get hat x and y axis. */
                    if(options & GCTL_OPT_JS1_HAT)
                    {
                        /* Hat x. */
                        if((*axis_heading) < 0)
                        {
                            if((*axis_throttle) < 0)
                                (*axis_hat_x) = 2;
                            else
                                (*axis_hat_x) = 3;
                        }
                        else if((*axis_throttle) < 0)
                        {
                            (*axis_hat_x) = 3;
                        }
                        else
                        {
                            (*axis_hat_x) = 4;
                        }
                        /* Hat y (the axis after hat x). */
                        (*axis_hat_y) = (*axis_hat_x) + 1;
                    }
                }       /* Joystick as a standard composite unit. */
            }




#else
	    fprintf(
		stderr,
 "GCtlInit: Joystick support not compiled.\n"
	    );
#endif	/* JSW_H */
	    break;

	  /* Keyboard and pointer. */
          case GCTL_CONTROLLER_KEYBOARD:

	    /* Begin resetting keyboard values. */
	    /* Keyboard states. */
	    gc->heading_kb_state = 0;
	    gc->pitch_kb_state = 0;
	    gc->bank_kb_state = 0;
	    gc->throttle_kb_state = 0;
	    gc->hat_x_kb_state = 0;
	    gc->hat_y_kb_state = 0;
	    /* Last keypress time stamps. */
	    gc->heading_kb_last = 0;
	    gc->pitch_kb_last = 0;
	    gc->bank_kb_last = 0;
	    gc->throttle_kb_last = 0;
	    gc->hat_x_kb_last = 0;
	    gc->hat_y_kb_last = 0;

	    /* Begin resetting pointer values. */
	    /* Pressed buttons mask. */
	    gc->pointer_buttons = 0;
	    /* Set default pointer box size. */
	    gc->pointer_box_x = 0;
	    gc->pointer_box_y = 0;
	    gc->pointer_box_width = 100;
	    gc->pointer_box_height = 70;
	    /* Reset pointer coordinates. */
            gc->pointer_x = (gc->pointer_box_width / 2);
            gc->pointer_y = (gc->pointer_box_height / 2);

	    break;
	}

	return(status);
}

/*
 *	Gets new game controller values.
 *
 *	For handling keyboard keys, see GCtlHandleKey(). Keyboard
 *	keys should be sent to GCtlHandleKey() even if the controller
 *	type is not set to GCTL_CONTROLLER_KEYBOARD.
 */
void GCtlUpdate(gctl_struct *gc)
{
	gctl_flags_t options;
	time_t gc_last_updated_ms, cur_ms, dms;
#ifdef JSW_H
	int axis_num, button_num, js_num, status;
	js_data_struct *jsd;
	js_axis_struct *axis_ptr;
	js_button_struct *button_ptr;
#endif  /* JSW_H */
	Boolean	heading_has_nz = True,
		pitch_has_nz = True,
		bank_has_nz = True;
        int *btn_kb_state;
        double *btn_kb_coeff;
        time_t *btn_kb_last;
	double btn_kb_decell_coeff;


        if(gc == NULL)
            return;

	/* Set null zone states based on flight difficulty. */
	switch(option.flight_physics)
	{
	  case FLIGHT_PHYSICS_REALISTIC:
	    heading_has_nz = False;
	    pitch_has_nz = False;
	    bank_has_nz = False;
	    break;

	  case FLIGHT_PHYSICS_MODERATE:
	    pitch_has_nz = False;
            bank_has_nz = False;
            break;
	}

	/* Get game controller options. */
	options = gc->options;

	/* Get timmings. */
	gc_last_updated_ms = gc->last_updated;
	cur_ms = cur_millitime;
	dms = MAX(cur_ms - gc_last_updated_ms, 0);

	switch(gc->type)
	{
	  case GCTL_CONTROLLER_JOYSTICK:
#ifdef JSW_H
	    jsd = (js_data_struct *)gc->joystick;
	    if(jsd == NULL)
		break;

/* Is hat value x centered at center y (both in raw units)? */
#define HAT_IS_CEN_RAW(v,c)	((v) == (c))
/* Is hat center at coefficient 0.0? */
#define HAT_IS_CEN_COEFF(v)	((v) == 0.0)

/* Calculates the coefficient of the current time (c) minus the previous (l) time
 * divided by delta time (d). If (d) is not positive then 0.0 is returned.
 */
#define DELTA_TIME_TO_COEFF(c,l,d)	(((d) > 0) ? \
	(double)CLIP((double)((c) - (l)) / (double)(d), 0.0, 1.0) : 0.0 \
)

	    /* Handle first joystick, js0.
	     *
	     * Note that the procedure for handling each joystick here
	     * is the same, only differences are the js_num and the
	     * GCTL_OPT_JS* flags that are checked.
	     */
	    js_num = 0;
	    if(gc->total_joysticks > js_num)
	    {
		int	axis_heading = gc->js0_axis_heading,
			axis_bank = gc->js0_axis_bank,
			axis_pitch = gc->js0_axis_pitch,
			axis_throttle = gc->js0_axis_throttle,
			axis_hat_x = gc->js0_axis_hat_x,
			axis_hat_y = gc->js0_axis_hat_y;

		/* If ALT state is on, then bank axis becomes
		 * heading axis.
		 */
		if(gc->alt_state)
		{
		    if(options & GCTL_OPT_JS0_BANK)
		    {
			axis_heading = axis_bank;
		        axis_bank = -1;
		    }
		}

		/* Get next event from joystick if any. */
		status = JSUpdate(&jsd[js_num]);
		if(status == JSGotEvent)
		{
		    /* Got event, begin fetching values for each axis,
		     * any axis number that is -1 will be ignored.
		     */
		    /* Heading. */
		    if(axis_heading >= 0)
			gc->heading = ((heading_has_nz) ?
			    JSGetAxisCoeffNZ(jsd, axis_heading) :
			    JSGetAxisCoeff(jsd, axis_heading)
                        );
		    /* Bank. */
		    if(axis_bank >= 0)
			gc->bank = ((bank_has_nz) ?
			    JSGetAxisCoeffNZ(jsd, axis_bank) :
			    JSGetAxisCoeff(jsd, axis_bank)
			);
		    /* Pitch. */
		    if(axis_pitch >= 0)
			gc->pitch = ((pitch_has_nz) ?
			    JSGetAxisCoeffNZ(jsd, axis_pitch) :
			    JSGetAxisCoeff(jsd, axis_pitch)
			);
		    /* Throttle. */
		    if(axis_throttle >= 0)
		    {
			gc->throttle = (JSGetAxisCoeff(
			    jsd, axis_throttle
			) + 1) / 2;
			if(gc->throttle > 1.0)
                            gc->throttle = 1.0;
			else if(gc->throttle < 0.0)
			    gc->throttle = 0;
		    }
		    /* Hat x. */
		    if(axis_hat_x >= 0)
		    {
			axis_num = axis_hat_x;
			if(JSIsAxisAllocated(jsd, axis_num))
			{
                            axis_ptr = jsd->axis[axis_num];

                            if(HAT_IS_CEN_RAW(axis_ptr->cur, axis_ptr->cen) &&
                               !HAT_IS_CEN_RAW(axis_ptr->prev, axis_ptr->cen) &&
                               HAT_IS_CEN_COEFF(gc->hat_x)
                            )
                            {
                                /* Pressed and released during busy loop. */
                                gc->hat_x = DELTA_TIME_TO_COEFF(
                                    axis_ptr->time, axis_ptr->last_time, dms) *
                                    ((axis_ptr->prev > axis_ptr->cen) ? 1.0 : -1.0);
                                if(axis_ptr->flags & JSAxisFlagFlipped)
                                    gc->hat_x *= -1;
                            }
                            else
                            {
                                gc->hat_x = JSGetAxisCoeffNZ(jsd, axis_num);
                            }
                        }
		    }
                    /* Hat y. */
                    if(axis_hat_y >= 0)
                    {
                        axis_num = axis_hat_y;
                        if(JSIsAxisAllocated(jsd, axis_num))
                        {
                            axis_ptr = jsd->axis[axis_num];

                            if(HAT_IS_CEN_RAW(axis_ptr->cur, axis_ptr->cen) &&
                               !HAT_IS_CEN_RAW(axis_ptr->prev, axis_ptr->cen) &&
                               HAT_IS_CEN_COEFF(gc->hat_y)
                            )
                            {
                                /* Pressed and released during busy loop. */
                                gc->hat_y = DELTA_TIME_TO_COEFF(
                                    axis_ptr->time, axis_ptr->last_time, dms) *
                                    ((axis_ptr->prev > axis_ptr->cen) ? 1.0 : -1.0);
                                if(axis_ptr->flags & JSAxisFlagFlipped)
                                    gc->hat_y *= -1;
                            }
                            else
                            {
                                gc->hat_y = JSGetAxisCoeffNZ(jsd, axis_num);
                            }
                        }
                    }

		    /* Begin handling buttons. */

		    /* Zoom in button state and coefficient. */
		    button_num = gc->js0_btn_zoom_in;
		    if(JSIsButtonAllocated(jsd, button_num))
		    {
			button_ptr = jsd->button[button_num];

			if((button_ptr->changed_state == JSButtonChangedStateOnToOff) &&
                           !gc->zoom_in_state
			)
			{
			    /* Pressed and released during busy loop. */
			    gc->zoom_in_coeff = DELTA_TIME_TO_COEFF(
				button_ptr->time, button_ptr->last_time, dms
			    );
			    gc->zoom_in_state = ((button_ptr->state) ? 1 : 0);
			}
			else
			{
			    gc->zoom_in_state = ((button_ptr->state) ? 1 : 0);
			    gc->zoom_in_coeff = ((button_ptr->state) ? 1.0 : 0.0);
			}
		    }
                    /* Zoom out button state and coefficient. */
                    button_num = gc->js0_btn_zoom_out; 
                    if(JSIsButtonAllocated(jsd, button_num))
                    {
                        button_ptr = jsd->button[button_num];

                        if((button_ptr->changed_state == JSButtonChangedStateOnToOff) &&
                           !gc->zoom_out_state
                        )
                        {        
                            /* Pressed and released during busy loop. */
                            gc->zoom_out_coeff = DELTA_TIME_TO_COEFF(
				button_ptr->time, button_ptr->last_time, dms
			    );
                            gc->zoom_out_state = ((button_ptr->state) ? 1 : 0);
                        }
                        else
                        {
                            gc->zoom_out_state = ((button_ptr->state) ? 1 : 0);
                            gc->zoom_out_coeff = ((button_ptr->state) ? 1.0 : 0.0);
                        }
		    }

                    /* Hoist up button state and coefficient. */
                    button_num = gc->js0_btn_hoist_up;
                    if(JSIsButtonAllocated(jsd, button_num))
                    {
                        button_ptr = jsd->button[button_num];

                        if((button_ptr->changed_state == JSButtonChangedStateOnToOff) &&
                           !gc->hoist_up_state
                        )
                        {
                            /* Pressed and released during busy loop. */
			    gc->hoist_up_coeff = DELTA_TIME_TO_COEFF(
				button_ptr->time, button_ptr->last_time, dms
			    );
                            gc->hoist_up_state = ((button_ptr->state) ? 1 : 0);
                        }
                        else
                        {
                            gc->hoist_up_state = ((button_ptr->state) ? 1 : 0);
                            gc->hoist_up_coeff = ((button_ptr->state) ? 1.0 : 0.0);
                        }
                    }
                    /* Hoist down button state and coefficient. */
                    button_num = gc->js0_btn_hoist_down;
                    if(JSIsButtonAllocated(jsd, button_num))
                    {
                        button_ptr = jsd->button[button_num];
                                
                        if((button_ptr->changed_state == JSButtonChangedStateOnToOff) &&
                           !gc->hoist_down_state
                        )
                        {
                            /* Pressed and released during busy loop. */
			    gc->hoist_down_coeff = DELTA_TIME_TO_COEFF(
				button_ptr->time, button_ptr->last_time, dms
			    );
                            gc->hoist_down_state = ((button_ptr->state) ? 1 : 0);
                        }
                        else
                        {
                            gc->hoist_down_state = ((button_ptr->state) ? 1 : 0);
                            gc->hoist_down_coeff = ((button_ptr->state) ? 1.0 : 0.0);
                        }
                    }

		    /* Alt button state (for switching bank axis to
		     * act as heading axis).
		     */
		    gc->alt_state = ((JSGetButtonState(jsd,
		        gc->js0_btn_rotate)) ? 1 : 0
		    );

		    /* Air brakes. */
                    button_num = gc->js0_btn_air_brakes;
                    if(JSIsButtonAllocated(jsd, button_num))
                    {
                        button_ptr = jsd->button[button_num];

                        if((button_ptr->changed_state == JSButtonChangedStateOnToOff) &&
                           !gc->air_brakes_state
                        )
                        {
                            /* Pressed and released during busy loop. */
			    gc->air_brakes_coeff = DELTA_TIME_TO_COEFF(
				button_ptr->time, button_ptr->last_time, dms
			    );
                            gc->air_brakes_state = ((button_ptr->state) ? 1 : 0);
                        }
                        else
                        {
                            gc->air_brakes_state = ((button_ptr->state) ? 1 : 0);
                            gc->air_brakes_coeff = ((button_ptr->state) ? 1.0 : 0);
                        }
                    }

		    /* Wheel brakes. */
                    button_num = gc->js0_btn_wheel_brakes;
                    if(JSIsButtonAllocated(jsd, button_num))
                    {
                        button_ptr = jsd->button[button_num];
                         
                        if((button_ptr->changed_state == JSButtonChangedStateOnToOff) &&
                           !gc->wheel_brakes_state
                        )
                        {
                            /* Pressed and released during busy loop. */
			    gc->wheel_brakes_coeff = DELTA_TIME_TO_COEFF(
				button_ptr->time, button_ptr->last_time, dms
			    );
                        }
                        else
                        {
                            gc->wheel_brakes_coeff = ((button_ptr->state) ? 1.0 : 0.0);
                        }
                    }
		    if(gc->wheel_brakes_coeff > 0.0)
		    {
		        if(gc->shift_state)
			    gc->wheel_brakes_state = 2;
		        else
			    gc->wheel_brakes_state = 1;
		    }
		    else
		    {
		        if(gc->wheel_brakes_state != 2)
			    gc->wheel_brakes_state = 0;
		    }
		}
		else
		{
		    /* No events to report, reset some place holding variables. */

		    /* Reset hat values (this is needed). */
                    axis_num = axis_hat_x;
                    if((options & GCTL_OPT_JS0_HAT) &&
                       JSIsAxisAllocated(jsd, axis_num)
                    )
                    {
                        axis_ptr = jsd->axis[axis_num];

		        if(HAT_IS_CEN_RAW(axis_ptr->cur, axis_ptr->cen) &&
                           (gc->hat_x != 0.0)
			)
                            gc->hat_x = 0.0;
		    }
                    axis_num = axis_hat_y;
                    if((options & GCTL_OPT_JS0_HAT) &&
                       JSIsAxisAllocated(jsd, axis_num)
                    )
                    {
                        axis_ptr = jsd->axis[axis_num];

                        if(HAT_IS_CEN_RAW(axis_ptr->cur, axis_ptr->cen) &&
                           (gc->hat_y != 0.0)
                        )
                            gc->hat_y = 0.0;
                    }

		}
	    }	/* Handle first joystick, js0. */
	    /* ****************************************************** */
	    /* Handle second joystick, js1. */
            js_num = 1;
            if(gc->total_joysticks > js_num)
            {
                int     axis_heading = gc->js1_axis_heading,
                        axis_bank = gc->js1_axis_bank,
                        axis_pitch = gc->js1_axis_pitch,
                        axis_throttle = gc->js1_axis_throttle,
                        axis_hat_x = gc->js1_axis_hat_x,
                        axis_hat_y = gc->js1_axis_hat_y;

                /* If ALT state is on, then bank axis becomes
                 * heading axis.
                 */
                if(gc->alt_state)
                {
                    if(options & GCTL_OPT_JS0_BANK)
                    {   
                        axis_heading = axis_bank;
                        axis_bank = -1;
                    }
		}

		/* Get next event from joystick if any. */
                status = JSUpdate(&jsd[js_num]);
                if(status == JSGotEvent)
                {
                    /* Got event, begin fetching values for each axis,
                     * any axis number that is -1 will be ignored.
                     */   
                    /* Heading. */
                    if(axis_heading >= 0) 
                        gc->heading = ((heading_has_nz) ?
                            JSGetAxisCoeffNZ(jsd, axis_heading) :
                            JSGetAxisCoeff(jsd, axis_heading)
                        );
                    /* Bank. */
                    if(axis_bank >= 0)
                        gc->bank = ((bank_has_nz) ?
                            JSGetAxisCoeffNZ(jsd, axis_bank) :
                            JSGetAxisCoeff(jsd, axis_bank)
                        );
                    /* Pitch. */
                    if(axis_pitch >= 0)
                        gc->pitch = ((pitch_has_nz) ?
                            JSGetAxisCoeffNZ(jsd, axis_pitch) :
                            JSGetAxisCoeff(jsd, axis_pitch)
                        );
                    /* Throttle. */
                    if(axis_throttle >= 0)
                    {
                        gc->throttle = (JSGetAxisCoeff(
                            jsd, axis_throttle
                        ) + 1) / 2;
                        if(gc->throttle > 1.0)
                            gc->throttle = 1.0;
                        else if(gc->throttle < 0.0)
                            gc->throttle = 0;
                    }
                    /* Hat x. */
                    if(axis_hat_x >= 0)
                    {
                        axis_num = axis_hat_x;
                        if(JSIsAxisAllocated(jsd, axis_num))
                        {
                            axis_ptr = jsd->axis[axis_num];

                            if(HAT_IS_CEN_RAW(axis_ptr->cur, axis_ptr->cen) &&    
                               !HAT_IS_CEN_RAW(axis_ptr->prev, axis_ptr->cen) &&
                               HAT_IS_CEN_COEFF(gc->hat_x)
                            )
                            {
                                /* Pressed and released during busy loop. */
                                gc->hat_x = DELTA_TIME_TO_COEFF(
                                    axis_ptr->time, axis_ptr->last_time, dms) *
                                    ((axis_ptr->prev > axis_ptr->cen) ? 1.0 : -1.0);
                                if(axis_ptr->flags & JSAxisFlagFlipped)
                                    gc->hat_x *= -1;
                            }
                            else
                            {
                                gc->hat_x = JSGetAxisCoeffNZ(jsd, axis_num);
                            }
                        }
                    }
                    /* Hat y. */
                    if(axis_hat_y >= 0)
                    {
                        axis_num = axis_hat_y;
                        if(JSIsAxisAllocated(jsd, axis_num))
                        {
                            axis_ptr = jsd->axis[axis_num];

                            if(HAT_IS_CEN_RAW(axis_ptr->cur, axis_ptr->cen) &&
                               !HAT_IS_CEN_RAW(axis_ptr->prev, axis_ptr->cen) &&
                               HAT_IS_CEN_COEFF(gc->hat_y)
                            )
                            {
                                /* Pressed and released during busy loop. */
                                gc->hat_y = DELTA_TIME_TO_COEFF(
                                    axis_ptr->time, axis_ptr->last_time, dms) *
                                    ((axis_ptr->prev > axis_ptr->cen) ? 1.0 : -1.0);
                                if(axis_ptr->flags & JSAxisFlagFlipped)
                                    gc->hat_y *= -1;
                            }
                            else
                            {
                                gc->hat_y = JSGetAxisCoeffNZ(jsd, axis_num);
                            }
                        }
                    }

                    /* Begin handling buttons. */

                    /* Zoom in button state and coefficient. */
                    button_num = gc->js1_btn_zoom_in;
                    if(JSIsButtonAllocated(jsd, button_num))
                    {
                        button_ptr = jsd->button[button_num];

                        if((button_ptr->changed_state == JSButtonChangedStateOnToOff) &&
                           !gc->zoom_in_state
                        )
                        {
                            /* Pressed and released during busy loop. */
			    gc->zoom_in_coeff = DELTA_TIME_TO_COEFF(
				button_ptr->time, button_ptr->last_time, dms
			    );
                            gc->zoom_in_state = ((button_ptr->state) ? 1 : 0);
                        }
                        else
                        {
                            gc->zoom_in_state = ((button_ptr->state) ? 1 : 0);
                            gc->zoom_in_coeff = ((button_ptr->state) ? 1.0 : 0.0);
                        }
                    }
                    /* Zoom out button state and coefficient. */
                    button_num = gc->js1_btn_zoom_out;
                    if(JSIsButtonAllocated(jsd, button_num))
                    {
                        button_ptr = jsd->button[button_num];

                        if((button_ptr->changed_state == JSButtonChangedStateOnToOff) &&
                           !gc->zoom_out_state
                        )
                        {
                            /* Pressed and released during busy loop. */
			    gc->zoom_out_coeff = DELTA_TIME_TO_COEFF(
				button_ptr->time, button_ptr->last_time, dms
			    );
                            gc->zoom_out_state = ((button_ptr->state) ? 1 : 0);
                        }
                        else
                        {
                            gc->zoom_out_state = ((button_ptr->state) ? 1 : 0);
                            gc->zoom_out_coeff = ((button_ptr->state) ? 1.0 : 0.0);
                        }
                    }

                    /* Hoist up button state and coefficient. */
                    button_num = gc->js1_btn_hoist_up;
                    if(JSIsButtonAllocated(jsd, button_num))
                    {
                        button_ptr = jsd->button[button_num];

                        if((button_ptr->changed_state == JSButtonChangedStateOnToOff) &&
                           !gc->hoist_up_state
                        )
                        {
                            /* Pressed and released during busy loop. */
			    gc->hoist_up_coeff = DELTA_TIME_TO_COEFF(
				button_ptr->time, button_ptr->last_time, dms
			    );
                            gc->hoist_up_state = ((button_ptr->state) ? 1 : 0);
                        }
                        else
                        {
                            gc->hoist_up_state = ((button_ptr->state) ? 1 : 0);
                            gc->hoist_up_coeff = ((button_ptr->state) ? 1.0 : 0.0);
                        }
                    }
                    /* Hoist down button state and coefficient. */
                    button_num = gc->js1_btn_hoist_down;
                    if(JSIsButtonAllocated(jsd, button_num))
                    {
                        button_ptr = jsd->button[button_num];

                        if((button_ptr->changed_state == JSButtonChangedStateOnToOff) &&
                           !gc->hoist_down_state
                        )
                        {
                            /* Pressed and released during busy loop. */
			    gc->hoist_down_coeff = DELTA_TIME_TO_COEFF(
				button_ptr->time, button_ptr->last_time, dms
			    );
                            gc->hoist_down_state = ((button_ptr->state) ? 1 : 0);
                        }
                        else
                        {       
                            gc->hoist_down_state = ((button_ptr->state) ? 1 : 0);
                            gc->hoist_down_coeff = ((button_ptr->state) ? 1.0 : 0.0);
                        }
                    }

                    /* Alt button state (for switching bank axis to
                     * act as heading axis).
                     */
                    gc->alt_state = ((JSGetButtonState(jsd,
                        gc->js1_btn_rotate)) ? 1 : 0
                    );

 		    /* Air brakes. */
                    button_num = gc->js1_btn_air_brakes;
                    if(JSIsButtonAllocated(jsd, button_num))
                    {
                        button_ptr = jsd->button[button_num];

                        if((button_ptr->changed_state == JSButtonChangedStateOnToOff) &&
                           !gc->air_brakes_state
                        )
                        {
                            /* Pressed and released during busy loop. */
			    gc->air_brakes_coeff = DELTA_TIME_TO_COEFF(
				button_ptr->time, button_ptr->last_time, dms
			    );
                            gc->air_brakes_state = ((button_ptr->state) ? 1 : 0);
                        }
                        else
                        {
                            gc->air_brakes_state = ((button_ptr->state) ? 1 : 0);
                            gc->air_brakes_coeff = ((button_ptr->state) ? 1.0 : 0);
                        }
                    }

		    /* Wheel brakes. */
                    button_num = gc->js1_btn_wheel_brakes;
                    if(JSIsButtonAllocated(jsd, button_num))
                    {
                        button_ptr = jsd->button[button_num];

                        if((button_ptr->changed_state == JSButtonChangedStateOnToOff) &&
                           !gc->wheel_brakes_state
                        )
                        {
                            /* Pressed and released during busy loop. */
			    gc->wheel_brakes_coeff = DELTA_TIME_TO_COEFF(
				button_ptr->time, button_ptr->last_time, dms
			    );
                        }
                        else
                        {
                            gc->wheel_brakes_coeff = ((button_ptr->state) ? 1.0 : 0.0);
                        }
                    }
		    if(gc->wheel_brakes_coeff > 0.0)
		    {
		        if(gc->shift_state)
			    gc->wheel_brakes_state = 2;
		        else
			    gc->wheel_brakes_state = 1;
		    }
		    else
		    {
		        if(gc->wheel_brakes_state != 2)
			    gc->wheel_brakes_state = 0;
		    }
                }
                else
                {
                    /* No events to report, reset some place holding
		     * variables.
		     */

                    /* Reset hat values (this is needed). */
                    axis_num = axis_hat_x;
                    if((options & GCTL_OPT_JS1_HAT) &&
                       JSIsAxisAllocated(jsd, axis_num)
                    )
                    {
                        axis_ptr = jsd->axis[axis_num];

                        if(HAT_IS_CEN_RAW(axis_ptr->cur, axis_ptr->cen) &&
                           (gc->hat_x != 0.0)
                        )
                            gc->hat_x = 0.0;
                    }
                    axis_num = axis_hat_y;
                    if((options & GCTL_OPT_JS1_HAT) &&
                       JSIsAxisAllocated(jsd, axis_num)
                    )
                    {
                        axis_ptr = jsd->axis[axis_num];
                        
                        if(HAT_IS_CEN_RAW(axis_ptr->cur, axis_ptr->cen) &&
                           (gc->hat_y != 0.0)
                        )   
                            gc->hat_y = 0.0;
                    }

                }
            }	/* Handle second joystick, js1. */

#undef HAT_IS_CEN_RAW
#undef HAT_IS_CEN_COEFF
#undef DELTA_TIME_TO_COEFF

#endif	/* JSW_H */
	    break;

          case GCTL_CONTROLLER_KEYBOARD:
/* Reduces the btn_kb_coeff towards 0.0 if btn_kb_state is false.
 * The bigger btn_kb_decell_coeff is, the faster btn_kb_coeff reaches
 * 0.0.
 */
#define DO_KB_BTN_UPDATE	\
{ \
 if(!(*btn_kb_state)) \
 { \
  if((*btn_kb_coeff) > 0.0) \
   (*btn_kb_coeff) = MAX( \
    (*btn_kb_coeff) - (btn_kb_decell_coeff * time_compensation), \
    0.0 \
   ); \
  else \
   (*btn_kb_coeff) = MIN( \
    (*btn_kb_coeff) + (btn_kb_decell_coeff * time_compensation), \
    0.0 \
   ); \
 } \
}

	    /* Begin reducing coefficients for keyboard key states. */

	    /* Heading. */
	    btn_kb_decell_coeff = 3.0;
	    btn_kb_state = &gc->heading_kb_state;
            btn_kb_last = &gc->heading_kb_last;
            btn_kb_coeff = &gc->heading;
            DO_KB_BTN_UPDATE

	    /* Pitch. */
	    btn_kb_decell_coeff = 3.0;
            btn_kb_state = &gc->pitch_kb_state;
            btn_kb_last = &gc->pitch_kb_last;
            btn_kb_coeff = &gc->pitch;
            DO_KB_BTN_UPDATE

	    /* Bank. */
	    btn_kb_decell_coeff = 3.0;
            btn_kb_state = &gc->bank_kb_state;
            btn_kb_last = &gc->bank_kb_last;
            btn_kb_coeff = &gc->bank;
            DO_KB_BTN_UPDATE

	    /* Leave throttle alone. */

	    /* Hat X. */
	    btn_kb_decell_coeff = 2.0;
            btn_kb_state = &gc->hat_x_kb_state;
            btn_kb_last = &gc->hat_x_kb_last;
            btn_kb_coeff = &gc->hat_x;
            DO_KB_BTN_UPDATE

	    /* Hat Y. */
	    btn_kb_decell_coeff = 2.0;
            btn_kb_state = &gc->hat_y_kb_state;
            btn_kb_last = &gc->hat_y_kb_last;
            btn_kb_coeff = &gc->hat_y;
            DO_KB_BTN_UPDATE


	    /* Wheel brakes. */
	    btn_kb_decell_coeff = 0.5;
	    btn_kb_state = &gc->wheel_brakes_state; 
            btn_kb_last = &gc->wheel_brakes_kb_last;
            btn_kb_coeff = &gc->wheel_brakes_coeff;
            DO_KB_BTN_UPDATE

	    /* Leave air brakes alone. */

	    /* Zoom in. */
	    btn_kb_decell_coeff = 1.6;
	    btn_kb_state = &gc->zoom_in_state;
	    btn_kb_last = &gc->zoom_in_kb_last;
	    btn_kb_coeff = &gc->zoom_in_coeff;
	    DO_KB_BTN_UPDATE

	    /* Zoom out. */
	    btn_kb_decell_coeff = 1.6;
            btn_kb_state = &gc->zoom_out_state;
            btn_kb_last = &gc->zoom_out_kb_last;
            btn_kb_coeff = &gc->zoom_out_coeff;
	    DO_KB_BTN_UPDATE

#undef DO_KB_BTN_UPDATE
	    break;
	}

	/* Update last gc time updated in ms to the current time. */
	gc->last_updated = cur_ms;

	return;
}

/*
 *	Handles a pointer event with respect to the given game 
 *	controller.
 */
void GCtlHandlePointer(
        gw_display_struct *display, gctl_struct *gc,
        int type,                       /* One of GWEventType*. */
        int button,                     /* Button number. */
        int x, int y,
        long t                          /* Time stamp. */ 
)
{
        if(gc == NULL)
            return;  






	return;
}


/*
 *	Handles the key, where state indicates if it is pressed or
 *	released.
 */
void GCtlHandleKey(
	gw_display_struct *display, gctl_struct *gc,
	int k, int kstate,
	int alt_state, int ctrl_state, int shift_state,
	long t
)
{
	int *btn_kb_state;
	double *btn_kb_coeff;
	time_t *btn_kb_last;
	double btn_bk_to_coeff;


	if(gc == NULL)
	    return;

	/* Alt key state. */
	gc->alt_state = ((alt_state) ? 1 : 0);
        gc->ctrl_state = ((ctrl_state) ? 1 : 0);
	gc->shift_state = ((shift_state) ? 1 : 0);


#define DO_HAS_NO_AUTOREPEAT    \
GWKeyboardAutoRepeat(display, (Boolean)!kstate);

#define DO_BTN_KB_UPDATE	\
if((kstate ? 1 : 0) != ((*btn_kb_state) ? 1 : 0)) \
{ \
 time_t dt = MAX(t - (*btn_kb_last), 0); \
 if(kstate) \
 { \
  /* Pressed down. */ \
  (*btn_kb_state) = 1; \
  (*btn_kb_coeff) = 1.0; \
  (*btn_kb_last) = t; \
 } \
 else if(dt > lapsed_millitime) \
 { \
  /* Released up. */ \
  (*btn_kb_state) = 0; \
  (*btn_kb_last) = t; \
 } \
 else \
 { \
  /* Released up quickly from a prior press which \
   * occured during this cycle. \
   */ \
  (*btn_kb_state) = 0; \
  (*btn_kb_coeff) = (double)dt / (double)lapsed_millitime; \
  (*btn_kb_last) = t; \
 } \
}

#define DO_BTN_KB_AXIS_UPDATE	\
if((kstate ? 1 : 0) != ((*btn_kb_state) ? 1 : 0)) \
{ \
 time_t dt = MAX(t - (*btn_kb_last), 0); \
 if(kstate) \
 { \
  /* Pressed down. */ \
  (*btn_kb_state) = 1; \
  (*btn_kb_coeff) = btn_bk_to_coeff; \
  (*btn_kb_last) = t; \
 } \
 else if(dt > lapsed_millitime) \
 { \
  /* Released up. */ \
  (*btn_kb_state) = 0; \
  (*btn_kb_last) = t; \
 } \
 else \
 { \
  /* Released up quickly from a prior press which \
   * occured during this cycle. \
   */ \
  (*btn_kb_state) = 0; \
  (*btn_kb_coeff) = btn_bk_to_coeff * (double)dt / (double)lapsed_millitime; \
  (*btn_kb_last) = t; \
 } \
}


	switch(k)
	{
	  case GWKeyLeft:
	    DO_HAS_NO_AUTOREPEAT
	    btn_bk_to_coeff = -1.0;
	    if(gc->ctrl_state)
	    {
		btn_kb_state = &gc->heading_kb_state;
		btn_kb_coeff = &gc->heading;
		btn_kb_last = &gc->heading_kb_last;
	    }
	    else if(gc->shift_state)
	    {
		btn_kb_state = &gc->hat_x_kb_state;
                btn_kb_coeff = &gc->hat_x;
                btn_kb_last = &gc->hat_x_kb_last;
	    }
	    else
	    {
		btn_kb_state = &gc->bank_kb_state;
                btn_kb_coeff = &gc->bank;  
                btn_kb_last = &gc->bank_kb_last;
	    }
	    DO_BTN_KB_AXIS_UPDATE
	    if(!kstate)
	    {
                gc->heading_kb_state = 0;
                gc->hat_x_kb_state = 0;
		gc->bank_kb_state = 0;
	    }
	    break;

          case GWKeyRight:
	    DO_HAS_NO_AUTOREPEAT
            btn_bk_to_coeff = 1.0;
            if(gc->ctrl_state)
            {
                btn_kb_state = &gc->heading_kb_state;
                btn_kb_coeff = &gc->heading;
                btn_kb_last = &gc->heading_kb_last;
            }
            else if(gc->shift_state)
            {
                btn_kb_state = &gc->hat_x_kb_state;
                btn_kb_coeff = &gc->hat_x;
                btn_kb_last = &gc->hat_x_kb_last;
            }
            else
            {
                btn_kb_state = &gc->bank_kb_state;
                btn_kb_coeff = &gc->bank;
                btn_kb_last = &gc->bank_kb_last;
            }
            DO_BTN_KB_AXIS_UPDATE
            if(!kstate)
            {
                gc->heading_kb_state = 0;
                gc->hat_x_kb_state = 0;
                gc->bank_kb_state = 0;
            }
            break;

          case GWKeyUp:
            DO_HAS_NO_AUTOREPEAT
            btn_bk_to_coeff = 1.0;
            if(gc->shift_state)
            {
                btn_kb_state = &gc->hat_y_kb_state;
                btn_kb_coeff = &gc->hat_y;
                btn_kb_last = &gc->hat_y_kb_last;
            }
            else
            {
                btn_kb_state = &gc->pitch_kb_state;
                btn_kb_coeff = &gc->pitch;
                btn_kb_last = &gc->pitch_kb_last;
            }
            DO_BTN_KB_AXIS_UPDATE
            if(!kstate)
            {
                gc->pitch_kb_state = 0;
                gc->hat_y_kb_state = 0;
            }
            break;

          case GWKeyDown:
	    DO_HAS_NO_AUTOREPEAT
            btn_bk_to_coeff = -1.0;
            if(gc->shift_state)
            {
                btn_kb_state = &gc->hat_y_kb_state;
                btn_kb_coeff = &gc->hat_y;
                btn_kb_last = &gc->hat_y_kb_last;
            }
            else
            {
                btn_kb_state = &gc->pitch_kb_state;
                btn_kb_coeff = &gc->pitch;
                btn_kb_last = &gc->pitch_kb_last;
            }
            DO_BTN_KB_AXIS_UPDATE
            if(!kstate)
            {
                gc->pitch_kb_state = 0;
                gc->hat_y_kb_state = 0;
            }
            break;

          case GWKeyPageUp:
	    if(kstate)
	    {
		gc->throttle += ((gc->shift_state) ? 0.1 : 0.01);
		if(gc->throttle > 1)
		    gc->throttle = 1;
	    }
	    break;

          case GWKeyPageDown:
            if(kstate)
	    {
		gc->throttle -= ((gc->shift_state) ? 0.1 : 0.01);
		if(gc->throttle < 0)
		    gc->throttle = 0;
	    }
            break;

	  case GWKeyDelete: case '.': case '>':
	    DO_HAS_NO_AUTOREPEAT
            btn_kb_state = &gc->wheel_brakes_state;
            btn_kb_coeff = &gc->wheel_brakes_coeff;
            btn_kb_last = &gc->wheel_brakes_kb_last;
            DO_BTN_KB_UPDATE
	    break;

	  case '=': case '+':
	    DO_HAS_NO_AUTOREPEAT
	    btn_kb_state = &gc->zoom_in_state;
	    btn_kb_coeff = &gc->zoom_in_coeff;
	    btn_kb_last = &gc->zoom_in_kb_last;
	    DO_BTN_KB_UPDATE
	    break;

	  case '-': case '_':
	    DO_HAS_NO_AUTOREPEAT
            btn_kb_state = &gc->zoom_out_state;
            btn_kb_coeff = &gc->zoom_out_coeff;
            btn_kb_last = &gc->zoom_out_kb_last;
	    DO_BTN_KB_UPDATE
	    break;

	  default:
	    break;
	}

#undef DO_BTN_KB_UPDATE
#undef DO_BTN_KB_AXIS_UPDATE
#undef DO_HAS_NO_AUTOREPEAT

	return;
}

/*
 *	Shuts down the game controller.
 */
void GCtlShutdown(gctl_struct *gc)
{
#ifdef JSW_H
	int js_num;
	js_data_struct *jsd;
#endif  /* JSW_H */


	if(gc == NULL)
	    return;

	/* Close device. */
	switch(gc->type)
	{
          case GCTL_CONTROLLER_JOYSTICK:
#ifdef JSW_H
	    jsd = (js_data_struct *)gc->joystick;
	    if(jsd != NULL)
	    {
	        for(js_num = 0; js_num < gc->total_joysticks; js_num++)
                    JSClose(&jsd[js_num]);
	    }
#endif	/* JSW_H */
	    if(gc->joystick != NULL)
	    {
                free(gc->joystick);
                gc->joystick = NULL;
	    }
	    gc->total_joysticks = 0;
            break;

	  default:
            if(gc->joystick != NULL)
            {
                free(gc->joystick);
                gc->joystick = NULL;
            }
            gc->total_joysticks = 0;
	    break;
	}

        /* Reset entire gctl structure. */
        memset(gc, 0x00, sizeof(gctl_struct));

	return;
}


