#include <stdio.h>
#include <string.h>
#include <stdlib.h>   

#include "../include/os.h"

#ifndef __MSW__
#include <unistd.h>

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/Xproto.h>
#include <X11/Xatom.h>
#include <X11/cursorfont.h>
#include <X11/extensions/shape.h>

#ifdef HAVE_LIBXPM
# include <X11/xpm.h>
# ifndef XPM_H
#  define XPM_H
# endif
# ifndef XpmDefaultColorCloseness
#  define XpmDefaultColorCloseness	40000
# endif
# include "icon_info.xpm"
# include "icon_warning.xpm"
# include "icon_error.xpm"
# include "icon_question.xpm"
#endif

#ifdef HAVE_MWMUTIL_H
# include "../include/MwmUtil.h"
# ifndef MWMUTIL_H
#  define MWMUTIL_H
# endif
#endif

#include "../include/string.h"
#include "gw.h"


static Pixel GWXCreatePixel(
	gw_display_struct *display,
        u_int8_t r, u_int8_t g, u_int8_t b
);
static void GWXDestroyPixel(gw_display_struct *display, Pixel pixel);

static XFontStruct *GWXCreateFont(
	gw_display_struct *display, const char *name
);
static void GWXDestroyFont(gw_display_struct *display, XFontStruct *font);
static int GWXGetFontStringLength(XFontStruct *font, const char *s);

int GWXButtonCreate(
        gw_display_struct *display, gwx_button_struct *btn,
        Window parent,
        int x, int y,
        unsigned int width, unsigned int height,
        const char *label, 
        void *client_data,
        void (*func_cb)(void *, void *)
);
void GWXButtonResize(
	gw_display_struct *display, gwx_button_struct *btn
);
void GWXButtonDraw(gw_display_struct *display, gwx_button_struct *btn);
int GWXButtonManage(
	gw_display_struct *display, gwx_button_struct *btn,
        XEvent *event
);
void GWXButtonDestroy(
	gw_display_struct *display, gwx_button_struct *btn
);

int GWXDialogCreate(
        gw_display_struct *display, gwx_dialog_struct *md,
        int type	/* Dialog type, one of GWX_DIALOG_TYPE_*. */
);
void GWXDialogLoadIcon(
	gw_display_struct *display, gwx_dialog_struct *md,
        int icon_code
);
void GWXDialogResize(
	gw_display_struct *display, gwx_dialog_struct *md
);
void GWXDialogDraw(
	gw_display_struct *display, gwx_dialog_struct *md
);
int GWXDialogManage(
	gw_display_struct *display, gwx_dialog_struct *md,
        XEvent *event
);
void GWXDialogDestroy(
	gw_display_struct *display, gwx_dialog_struct *md
);

void GWXDialogNoBtnCB(void *object, void *client_data);
void GWXDialogYesBtnCB(void *object, void *client_data);
void GWXDialogCancelBtnCB(void *object, void *client_data);
void GWXDialogOKBtnCB(void *object, void *client_data);
void GWXDialogSetMesg(
	gw_display_struct *display, gwx_dialog_struct *md,
	const char *title,
	const char *mesg,
	const char *details
);
void GWXDialogMap(
	gw_display_struct *display, gwx_dialog_struct *md
);
int GWXDoBlockUntilConf(gw_display_struct *display);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))


/* Default button sizes and margins. */
#define GWX_BTN_DEF_WIDTH	80
#define GWX_BTN_DEF_HEIGHT	25
#define GWX_BTN_XMARGIN		5
#define GWX_BTN_YMARGIN		5

/* Widget style colors (a, r, g, b). */
static u_int8_t style_fg_color[]	= {0xff, 0x00, 0x00, 0x00};
static u_int8_t style_bg_color[]	= {0xff, 0xd7, 0xd7, 0xd7};
static u_int8_t style_fg_insensitive[]	= {0xff, 0x80, 0x80, 0x80};
static u_int8_t style_bg_highlight_color[]	= {0xff, 0xe8, 0xe8, 0xe8};
static u_int8_t style_highlight_color[]	= {0xff, 0xf0, 0xf0, 0xf0};
static u_int8_t style_shade_color[]	= {0xff, 0x80, 0x80, 0x80};



/* Local global for recording the last confermation dialog return
 * code, one of GWConfirmation*.
 */
static int last_conf_dialog_code;


/*
 *	Allocates a new pixel color with the given rgb values.
 *
 *	The returned Pixel value needs to be deallocated by calling
 *	GWXDestroyPixel().
 */
static Pixel GWXCreatePixel(
	gw_display_struct *display,
	u_int8_t r, u_int8_t g, u_int8_t b
)
{
        Display *dpy;
	Colormap cmap;
	XColor c;


        if(display == NULL)
	    return(0);

        dpy = display->display;
        if(dpy == NULL)
	    return(0);

	cmap = display->colormap;
	if(cmap == None)
	    return(0);


        c.flags = DoRed | DoGreen | DoBlue;
        c.red = (u_int16_t)((r == 0xff) ? 0xffff : ((u_int16_t)r << 8));
        c.green = (u_int16_t)((g == 0xff) ? 0xffff : ((u_int16_t)g << 8));
        c.blue = (u_int16_t)((b == 0xff) ? 0xffff : ((u_int16_t)b << 8));

        if(XAllocColor(dpy, cmap, &c))
	    return(c.pixel);
	else
	    return(0);
}
/*
 *	Destroys a pixel returned by GWXAllocColor().
 */
static void GWXDestroyPixel(gw_display_struct *display, Pixel pixel)
{
        Display *dpy;
        Colormap cmap;

        if(display == NULL)
            return;

        dpy = display->display;
        if(dpy == NULL)
            return;

        cmap = display->colormap;
        if(cmap == None)
            return;

	XFreeColors(dpy, cmap, &pixel, 1, 0);

	return;
}

/*
 *	Creates (loads) a font from the given font name.
 *
 *	Can return None on error.
 */
static XFontStruct *GWXCreateFont(
	gw_display_struct *display, const char *name
)
{
	Display *dpy;


	if((display == NULL) ||
	   (name == NULL)
	)
	    return(NULL);

	dpy = display->display;
	if(dpy == NULL)
	    return(NULL);

	return(XLoadQueryFont(dpy, name));
}
/*
 *	Destroys a font returned by GWXCreateFont().
 */
static void GWXDestroyFont(gw_display_struct *display, XFontStruct *font)
{
	Display *dpy;


        if((display == NULL) ||
           (font == NULL)
        )
            return;

        dpy = display->display;
        if(dpy == NULL)
            return;

	XFreeFont(dpy, font);

	return;
}

/*
 *	Returns the length of the string is in pixels with respect to
 *	the size of the font.
 *
 *	The string is terminated by either a '\0' or '\n' character
 *	(exclusive), so it will never be counted beyond its end or
 *	end of line.
 *
 *	Can return 0 on error.
 */
static int GWXGetFontStringLength(XFontStruct *font, const char *s)
{
	int len = 0;		/* In pixels. */
	XCharStruct *csp;


	if((font == NULL) || (s == NULL))
	    return(len);

	/* Do we have per character geometry information? */
	if(font->per_char != NULL)
	{
	    /* 8 bits per character type? */
	    if(!font->min_byte1 && !font->max_byte1)
	    {
		int c;
		int i = 0;
		int m = (int)font->max_char_or_byte2 -
			(int)font->min_char_or_byte2;

		/* Itterate through given string. */
		while(((*s) != '\0') &&
                      ((*s) != '\n') && ((*s) != '\r')
		)
		{
		    c = *s++;	/* Get current string char and increment. */
		    c -= (int)font->min_char_or_byte2;	/* Offset for per_char index. */

		    /* Clip, check if string char in bounds? */
		    if((c >= i) && (c <= m))
		    {
			csp = &font->per_char[c];	/* Ptr to geometry. */
			len += csp->width;		/* Inc str len. */
		    }
		}
	    }
	    else
	    {
		/* 16 bits per character, ouch, code this in later. */


	    }
	}
	else
	{
	    /* No per character info, use max bounds. */
	    csp = &font->max_bounds;

	    /* Itterate through given string. */
	    while(((*s) != '\0') &&
	          ((*s) != '\n') && ((*s) != '\r')
	    )
		len += csp->width;
	}

	return(len);
}


/*
 *	Creates a button by setting up the given button structure's
 *	values. Returns non-zero on error.
 */
int GWXButtonCreate(
	gw_display_struct *display, gwx_button_struct *btn,
	Window parent,
	int x, int y,
	unsigned int width, unsigned int height,
	const char *label,
	void *client_data,
	void (*func_cb)(void *, void *)
)
{
        Display *dpy;
	Window w;
	XFontStruct *font;
	u_int8_t *sc;		/* Style color. */

        if((display == NULL) ||
           (btn == NULL) ||
           (parent == None)
        )
            return(-1);

        dpy = display->display;
        if(dpy == NULL)
            return(-1);

	memset(btn, 0x00, sizeof(gwx_button_struct));

	btn->x = x;
	btn->y = y;
	btn->width = width;
	btn->height = height;
	btn->parent = parent;

	btn->flags = (	GWX_BUTTON_FLAG_CAN_DEFAULT |
			GWX_BUTTON_FLAG_SENSITIVE
	);

	/* Create button's toplevel window. */
	w = GWCreateWindow(
	    display, parent,
	    x, y,
	    width, height,
	    label
        );
	if(w == None)
	    return(-1);
	else
	    btn->toplevel = w;
	XSelectInput(
	    dpy, w, 
	    ButtonPressMask | ButtonReleaseMask |
            ExposureMask |
            EnterWindowMask | LeaveWindowMask
	);

	sc = style_fg_color;
	btn->color_fg = GWXCreatePixel(display, sc[1], sc[2], sc[3]);
	sc = style_bg_color;
	btn->color_bg = GWXCreatePixel(display, sc[1], sc[2], sc[3]);
	sc = style_fg_insensitive;
	btn->color_fg_insensitive = GWXCreatePixel(display, sc[1], sc[2], sc[3]);
	sc = style_bg_highlight_color;
	btn->color_bg_highlighted = GWXCreatePixel(display, sc[1], sc[2], sc[3]);
	sc = style_highlight_color;
	btn->color_highlight = GWXCreatePixel(display, sc[1], sc[2], sc[3]);
	sc = style_shade_color;
	btn->color_shade = GWXCreatePixel(display, sc[1], sc[2], sc[3]);

	btn->colors_initialized = True;

	btn->font = font = GWXCreateFont(
	    display, display->def_xfont_name
	);

	btn->label = StringCopyAlloc(label);
	btn->label_len_pixels = GWXGetFontStringLength(
	    font, label
	);
	if(font == NULL)
	    btn->label_height_pixels = 0;
	else
	    btn->label_height_pixels = font->max_bounds.ascent +
		font->max_bounds.descent;

	btn->client_data = client_data;
	btn->func_cb = func_cb;

	GWXButtonResize(display, btn);

	return(0);
}

/*
 *	Resizes all button resources to its newly resized toplevel window.
 */
void GWXButtonResize(gw_display_struct *display, gwx_button_struct *btn)
{
        Display *dpy;
	GC gc;
	Window w;
	XWindowAttributes wattr;
	Boolean size_changed = False;


        if((display == NULL) ||
           (btn == NULL)
	)
	    return;

	dpy = display->display;
	gc = display->gc;
	w = btn->toplevel;
	if((dpy == NULL) ||
	   (gc == None) ||
           (w == None)
	)
	    return;

	XSync(dpy, False);
	XGetWindowAttributes(dpy, w, &wattr);
	if((wattr.width != btn->width) ||
           (wattr.height != btn->height)
	)
	    size_changed = True;

	btn->x = wattr.x;
	btn->y = wattr.y;
	btn->width = wattr.width;
	btn->height = wattr.height;

        if(size_changed || (btn->toplevel_buf == None))
        {
	    if(btn->toplevel_buf != None)
	        XFreePixmap(dpy, btn->toplevel_buf);

	    btn->toplevel_buf = XCreatePixmap(
	        dpy, btn->toplevel, btn->width, btn->height, display->depth
	    );
	}

	return;
}

/*
 *	Redraws the button.
 */
void GWXButtonDraw(gw_display_struct *display, gwx_button_struct *btn)
{
        Display *dpy;
        GC gc;
        Window w;
	Pixmap pm;
	Pixel c_fg, c_bg, c_highlight, c_shade;
	unsigned int width, height;
	XPoint p[3];
	char *label;
	XFontStruct *font;
	int label_len_pixels, label_height_pixels;
	XGCValues gcv;


        if((display == NULL) ||
           (btn == NULL)
        )
            return;

        dpy = display->display;
	gc = display->gc;
	w = btn->toplevel;
	pm = btn->toplevel_buf;
	if((dpy == NULL) ||
           (gc == None) ||
           (w == None) ||
           (pm == None)
	)
	    return;

	width = btn->width;
	height = btn->height;

	if(!btn->map_state)
	{
	    XMapRaised(dpy, w);
	    btn->map_state = True;
	}

	switch(btn->state)
	{
	  case GWX_BUTTON_STATE_ARMED:
	    c_fg = btn->color_fg;
            c_bg = btn->color_bg_highlighted;
            c_highlight = btn->color_shade;
            c_shade = btn->color_highlight;
            break;

	  case GWX_BUTTON_STATE_HIGHLIGHTED:
	    c_fg = btn->color_fg;
	    c_bg = btn->color_bg_highlighted;
	    c_highlight = btn->color_highlight;
	    c_shade = btn->color_shade;
	    break;

          default:
            c_fg = btn->color_fg;
            c_bg = btn->color_bg;
            c_highlight = btn->color_highlight;
            c_shade = btn->color_shade;
            break;
	}
	font = btn->font;


	/* Draw (clear) background. */
	XSetForeground(dpy, gc, c_bg);
	XFillRectangle(dpy, pm, gc, 0, 0, width, height);

	/* Draw highlight. */
	XSetForeground(dpy, gc, c_highlight);
	XSetLineAttributes(
	    dpy, gc,
	    2, LineSolid, CapRound, JoinMiter
	);
	p[0].x = 1;
	p[0].y = (int)((int)height - 1);
	p[1].x = 1;
	p[1].y = 1;
	p[2].x = (int)((int)width - 1);
	p[2].y = 1;
	XDrawLines(
	    dpy, pm, gc,
	    &(p[0]), 3,		/* Point array and number of points. */
	    CoordModeOrigin
	);

	/* Draw shadow. */
        XSetForeground(dpy, gc, c_shade);
        p[0].x = (int)((int)width - 1);
        p[0].y = 0;
        p[1].x = (int)((int)width - 1);
        p[1].y = (int)((int)height - 1);
        p[2].x = 0;
        p[2].y = (int)((int)height - 1);
        XDrawLines(
            dpy, pm, gc,
            &(p[0]), 3,         /* Point array and number of points. */
            CoordModeOrigin
        );


	/* Get pointer to button label. */
	label = btn->label;
	label_len_pixels = btn->label_len_pixels;
	label_height_pixels = btn->label_height_pixels;
	/* Got enough info to draw button's label? */
	if((font != NULL) && (label != NULL) &&
           (label_len_pixels > 0) && (label_height_pixels > 0)
	)
	{
	    int x, y, len;

	    /* Calculate length and position to draw label. */
	    len = strlen(label);
	    x = (int)(((int)width / 2) - (label_len_pixels / 2));
	    y = (int)(((int)height / 2) - (label_height_pixels / 2));

	    /* Set up GC for button label drawing. */
	    if(font->fid != None)
		XSetFont(dpy, gc, font->fid);
	    XSetForeground(
		dpy, gc,
		(btn->flags & GWX_BUTTON_FLAG_SENSITIVE) ?
		    c_fg : btn->color_fg
	    );

	    /* Draw the button's label. */
	    XDrawString(
		dpy, pm, gc,
		x, y + font->max_bounds.ascent,
		label, len
	    );

	    /* Button has default? */
	    if((btn->flags & GWX_BUTTON_FLAG_CAN_DEFAULT) &&
	       (btn->flags & GWX_BUTTON_FLAG_HAS_DEFAULT)
	    )
	    {
		int focus_margin = 3;
		unsigned long gcv_mask = (	GCFunction | GCCapStyle |
						GCJoinStyle | GCLineWidth |
						GCLineStyle | GCDashOffset |
						GCDashList
		);

		/* Set up GC to draw `focus rectangle'. */
		gcv.function = GXinvert;
		gcv.cap_style = CapNotLast;
		gcv.join_style = JoinBevel;
		gcv.line_width = 1;
		gcv.line_style = LineOnOffDash;
		gcv.dash_offset = 0;
		gcv.dashes = 1;
		XChangeGC(dpy, gc, gcv_mask, &gcv);

		/* Draw `focus rectangle'. */
		XDrawRectangle(
		    dpy, pm, gc,
		    0 + focus_margin, 0 + focus_margin,
		    width - (focus_margin * 2) - 1,
		    height - (focus_margin * 2) - 1
		);

		/* Restore GC. */
		gcv.function = GXcopy;
		gcv.cap_style = CapNotLast;
		gcv.join_style = JoinMiter;
		gcv.line_width = 1;
		gcv.line_style = LineSolid;
                gcv.dash_offset = 0;
		gcv.dashes = 2;
		XChangeGC(dpy, gc, gcv_mask, &gcv);
	    }
	}

	/* Put Pixmap buffer to Window. */
	XCopyArea(dpy, pm, w, gc, 0, 0, width, height, 0, 0);

	return;
}

/*
 *	Manages the given event if it is for the button.
 */
int GWXButtonManage(
	gw_display_struct *display,
	gwx_button_struct *btn,
	XEvent *event
)
{
	Display *dpy;
	Window w, ew;
	int events_handled = 0;


        if((display == NULL) ||
           (btn == NULL) ||
           (event == NULL)
        )
            return(events_handled);

	dpy = display->display;
	if(dpy == NULL)
	    return(events_handled);

	w = btn->toplevel;
	if(w == None)
	    return(events_handled);

	ew = event->xany.window;

	switch(event->type)
	{
	  case ButtonPress:
	    if(w == ew)
	    {
		btn->state = GWX_BUTTON_STATE_ARMED;
		GWXButtonDraw(display, btn);
		events_handled++;
	    }
	    break;

	  case ButtonRelease:
            if(w == ew)
            {
		if(btn->state == GWX_BUTTON_STATE_ARMED)
		{
                    btn->state = GWX_BUTTON_STATE_HIGHLIGHTED;
                    GWXButtonDraw(display, btn);
		    events_handled++;

		    if(btn->func_cb != NULL)
		    {
			btn->func_cb(
			    (void *)btn,	/* This object. */
			    btn->client_data	/* Client data. */
			);
		    }
		}
            }
	    break;

	  case Expose:
	    if(w == ew)
            {
                GWXButtonDraw(display, btn);
                events_handled++;
            }
	    break;

	  case EnterNotify:
	    if(w == ew)
            {
                btn->state = GWX_BUTTON_STATE_HIGHLIGHTED;
                GWXButtonDraw(display, btn);
                events_handled++;
            }
	    break;

	  case LeaveNotify:
            if(w == ew)
            {
                btn->state = GWX_BUTTON_STATE_UNARMED;
                GWXButtonDraw(display, btn);
                events_handled++;
            }
	    break;
	}

	return(events_handled);
}

/*
 *	Deallocates all resources on the button structure but does not
 *	deallocate the structure itself.
 */
void GWXButtonDestroy(gw_display_struct *display, gwx_button_struct *btn)
{
	Display *dpy;
	Pixmap *pm;
	Window *w;
	XFontStruct **font;


	if((display == NULL) ||
           (btn == NULL)
	)
	    return;

	dpy = display->display;
	if(dpy == NULL)
	    return;


	w = &btn->toplevel;
	if((*w) != None)
        {
            XDestroyWindow(dpy, *w);
            (*w) = None;
        }

	pm = &btn->toplevel_buf;
	if((*pm) != None)
	{
	    XFreePixmap(dpy, *pm);
	    (*pm) = None;
	}

	if(btn->colors_initialized)
	{
	    GWXDestroyPixel(display, btn->color_fg);
	    GWXDestroyPixel(display, btn->color_bg);
	    GWXDestroyPixel(display, btn->color_fg_insensitive);
	    GWXDestroyPixel(display, btn->color_bg_highlighted);
	    GWXDestroyPixel(display, btn->color_highlight);
	    GWXDestroyPixel(display, btn->color_shade);
	    btn->colors_initialized = False;
	}

	font = &btn->font;
	if((*font) != NULL)
	{
	    GWXDestroyFont(display, *font);
	    (*font) = NULL;
	}

	free(btn->label);
	btn->label = NULL;

	GWAcceleratorListDelete(
	    &btn->accelerator, &btn->total_accelerators
	);

	memset(btn, 0x00, sizeof(gwx_button_struct));

	return;
}


/*
 *	Create new message dialog by setting up the values in the given
 *	message dialog structure.
 */
int GWXDialogCreate(
        gw_display_struct *display, gwx_dialog_struct *md,
        int type        /* Dialog type, one of GWX_DIALOG_TYPE_*. */
)
{
        Display *dpy;
        Window w, parent;
	const char *cstrptr;
	int len;
	u_int8_t *sc;
#ifdef MWMUTIL_H
	PropMwmHints mwm_prop;
#endif	/* MWMUTIL_H */


        if((display == NULL) ||
           (md == NULL)
        )
            return(-1);

        dpy = display->display;
	parent = display->root;
        if((dpy == NULL) ||
           (parent == None)
	)
            return(-1);

        memset(md, 0x00, sizeof(gwx_dialog_struct));

        md->type = type;
	md->x = 0;
	md->y = 0;
	md->width = 200;
	md->height = 100;
	md->margin = 10;

        w = GWCreateWindow(
            display, parent,
            md->x, md->y,
            md->width, md->height,
            "Message Dialog"
        );
        if(w == None)
            return(-1);
        else
	    md->toplevel = w;

        XSelectInput(
            dpy, w,
            KeyPressMask | KeyReleaseMask |
	    ExposureMask | FocusChangeMask |
	    VisibilityChangeMask | StructureNotifyMask
        );

#ifdef MWMUTIL_H
	/* Set up decorations. */
	mwm_prop.flags =   MWM_HINTS_FUNCTIONS | MWM_HINTS_DECORATIONS
/*
			 | MWM_HINTS_INPUT_MODE | MWM_HINTS_STATUS
 */
	;
	mwm_prop.functions = (/* MWM_FUNC_RESIZE | */
                              MWM_FUNC_MOVE     |
                              /* MWM_FUNC_MINIMIZE | */
                              /* MWM_FUNC_MAXIMIZE | */
                              MWM_FUNC_CLOSE
        );
        mwm_prop.decorations = (MWM_DECOR_BORDER   |
                                /* MWM_DECOR_RESIZEH  | */
                                MWM_DECOR_TITLE |
                                MWM_DECOR_MENU
                                /* MWM_DECOR_MINIMIZE | */
                                /* MWM_DECOR_MAXIMIZE */
        );
        mwm_prop.inputMode = 0;
        mwm_prop.status = 0;
	XChangeProperty(
	    dpy, w,
	    display->atom_wm_motif_hints,	/* Property atom. */
	    display->atom_wm_motif_hints,	/* Type atom. */
	    32,					/* Bits. */
	    PropModeReplace,			/* Format. */
	    (unsigned char *)&mwm_prop,		/* Data. */
	    PROP_MWM_HINTS_ELEMENTS		/* Number of elements. */
	);
#endif	/* MWMUTIL_H */

	/* Set transient for the toplevel window on the display structure. */
	if(display->toplevel != None)
	    XSetTransientForHint(dpy, w, display->toplevel);

	sc = style_fg_color;
        md->color_fg = GWXCreatePixel(display, sc[1], sc[2], sc[3]);
	sc = style_bg_color;
        md->color_bg = GWXCreatePixel(display, sc[1], sc[2], sc[3]);
	sc = style_highlight_color;
	md->color_highlight = GWXCreatePixel(display, sc[1], sc[2], sc[3]);
	sc = style_shade_color;
        md->color_shade = GWXCreatePixel(display, sc[1], sc[2], sc[3]);

	md->colors_initialized = True;

        md->font = GWXCreateFont(
            display, display->def_xfont_name
        );


	switch(type)
	{
	  case GWX_DIALOG_TYPE_CONF_CANCEL:
            cstrptr = "Cancel";
            len = strlen(cstrptr);
            GWXButtonCreate(
                display, &md->cancel_btn, w,
                0, 0,
                GWX_BTN_DEF_WIDTH,
                GWX_BTN_DEF_HEIGHT,
                cstrptr,
                (void *)display, GWXDialogCancelBtnCB
            );
            md->cancel_btn.flags |= GWX_BUTTON_FLAG_HAS_DEFAULT;

	    /* Fall through. */

	  case GWX_DIALOG_TYPE_CONF:
            cstrptr = "No";
            len = strlen(cstrptr);
            GWXButtonCreate(
                display, &md->no_btn, w,
                0, 0,
                GWX_BTN_DEF_WIDTH,
                GWX_BTN_DEF_HEIGHT,
                cstrptr,
                (void *)display, GWXDialogNoBtnCB
            );
            cstrptr = "Yes";
            len = strlen(cstrptr);
            GWXButtonCreate(
                display, &md->yes_btn, w,
                0, 0,
                GWX_BTN_DEF_WIDTH,
                GWX_BTN_DEF_HEIGHT,
                cstrptr,
                (void *)display, GWXDialogYesBtnCB
            );
	    break;

	  case GWX_DIALOG_TYPE_MESG:
	    cstrptr = "OK";
	    len = strlen(cstrptr);
	    GWXButtonCreate(
	        display, &md->ok_btn, w,
	        0, 0,
		GWX_BTN_DEF_WIDTH,
		GWX_BTN_DEF_HEIGHT,
	        cstrptr,
	        (void *)display, GWXDialogOKBtnCB
	    );
	    md->ok_btn.flags |= GWX_BUTTON_FLAG_HAS_DEFAULT;
	    break;
	}


        GWXDialogResize(display, md);

	return(0);
}

/*
 *	Loads the message dialog icon (the icon displayed next to
 *	dialog's text, not the WM icon).
 */
void GWXDialogLoadIcon(
	gw_display_struct *display,
        gwx_dialog_struct *md,
	int icon_code		/* One of GWX_ICON_*. */
)
{
        Display *dpy;
        GC gc;
        Window w, *icon_w;
	Pixmap *icon, *icon_mask;
	unsigned int btn_height = 0;
#ifdef XPM_H
	int xpm_status;
	XpmAttributes xpm_attr;
	char **xpm_data;
#endif


        if((display == NULL) ||
           (md == NULL)
        )
            return;

        dpy = display->display;
        gc = display->gc;
	w = md->toplevel;
        if((dpy == NULL) ||
           (gc == None) ||
           (w == None)
        )
            return;

	/* Calculate height based on type of dialog. */
	switch(md->type)
	{
	  case GWX_DIALOG_TYPE_CONF_CANCEL:
	  case GWX_DIALOG_TYPE_CONF:
	    btn_height = md->yes_btn.height;
	    break;

	  case GWX_DIALOG_TYPE_MESG:
	    btn_height = md->ok_btn.height;
            break;
	}

	/* Get pointers to icon window, pixmap, and pixmap mask on dialog
	 * structure.
	 */
	icon_w = &md->icon_w;
	icon = &md->icon;
	icon_mask = &md->icon_mask;


	/* Destroy old icons and window if any. */
	if(*icon_w != None)
	    XDestroyWindow(dpy, *icon_w);
	if(*icon != None)
	    XFreePixmap(dpy, *icon);
        if(*icon_mask != None)
            XFreePixmap(dpy, *icon_mask);


#ifdef XPM_H
	/* Load new icon xpm. */  
	switch(icon_code)
	{
	  case GWX_ICON_WARNING:
	    xpm_data = icon_warning_xpm;
	    break;

	  case GWX_ICON_ERROR:
            xpm_data = icon_error_xpm;
	    break;

          case GWX_ICON_QUESTION:
            xpm_data = icon_question_xpm;
            break;

	  default:
	    xpm_data = icon_info_xpm;
	    break;
	}

        memset(&xpm_attr, 0x00, sizeof(XpmAttributes));
        xpm_attr.closeness = XpmDefaultColorCloseness;
        xpm_attr.valuemask = XpmSize | XpmCloseness | XpmDepth;
        xpm_attr.depth = display->depth;
 
	xpm_status = XpmCreatePixmapFromData(
            dpy, w, xpm_data,
	    icon, icon_mask,
	    &xpm_attr
        );
        if(xpm_status == XpmSuccess)
        {
	    md->icon_width = xpm_attr.width,
	    md->icon_height = xpm_attr.height;

	    (*icon_w) = GWCreateWindow(
		display, w,
		0, 0,
		md->icon_width, md->icon_height,
		NULL
	    );
	    if((*icon_w) != None)
	    {
		XSelectInput(dpy, *icon_w, ExposureMask);
		XShapeCombineMask(
		    dpy, *icon_w,
		    ShapeBounding,
		    0, 0,
		    *icon_mask,
		    ShapeSet
		);
		XMapRaised(dpy, *icon_w);
	    }
	}

#endif	/* XPM_H */

	XResizeWindow(
	    dpy, w,
	    (unsigned int)(((md->icon_w == None) ?
		(md->margin * 2) : (md->margin * 3))
		+ md->icon_width + md->longest_line_pixels
	    ),
	    (unsigned int)MAX(
		(md->margin * 4) + btn_height + md->icon_height,
		(md->margin * 4) + btn_height + (md->lines_height_pixels)
	    )
	);
	GWXDialogResize(display, md);

	return;
}

/*
 *	Resizes message dialog resources with respect to the dialog's
 *	toplevel window.
 */
void GWXDialogResize(gw_display_struct *display, gwx_dialog_struct *md)
{
        Display *dpy;
        GC gc; 
        Window w;
	Boolean size_changed = False;
        XWindowAttributes wattr;
	gwx_button_struct *btn;
	int x_inc;


        if((display == NULL) ||
           (md == NULL)
        )
            return;

        dpy = display->display;
        gc = display->gc;
        w = md->toplevel;
        if((dpy == NULL) || 
           (gc == None) ||
           (w == None)
        )
            return;

	XSync(dpy, False);
        XGetWindowAttributes(dpy, w, &wattr);
	if((wattr.width != md->width) ||
           (wattr.height != md->height)
	)
	    size_changed = True;

	md->x = wattr.x;
	md->y = wattr.y;
        md->width = wattr.width;
        md->height = wattr.height;

	if(size_changed || (md->toplevel_buf == None))
	{
	    if(md->toplevel_buf != None)
		XFreePixmap(dpy, md->toplevel_buf);

	    md->toplevel_buf = XCreatePixmap(
		dpy, md->toplevel, md->width, md->height, display->depth
	    );
	}

	/* Move icon window? */
	if(md->icon_w != None)
	    XMoveWindow(
		dpy, md->icon_w,
		(int)md->margin,
		(int)md->margin + (int)MAX(
		    (md->lines_height_pixels / 2) - ((int)md->icon_height / 2),
		    0
		)
	    );

	/* Reposition buttons. */
	switch(md->type)
	{
	  case GWX_DIALOG_TYPE_CONF_CANCEL:
	    x_inc = (int)((int)md->width / 4);

	    btn = &md->cancel_btn;
            XMoveWindow(
                dpy, btn->toplevel,
                (int)((x_inc * 3) - ((int)btn->width / 2)),
                (int)((int)md->height - (int)md->margin - (int)btn->height)
            );
	    GWXButtonResize(display, btn);

            btn = &md->no_btn;
            XMoveWindow(
                dpy, btn->toplevel,
                (int)((x_inc * 2) - ((int)btn->width / 2)),
                (int)((int)md->height - (int)md->margin - (int)btn->height)
            );
            GWXButtonResize(display, btn);

            btn = &md->yes_btn;
            XMoveWindow(
                dpy, btn->toplevel,
                (int)((x_inc * 1) - ((int)btn->width / 2)),
                (int)((int)md->height - (int)md->margin - (int)btn->height)
            );
            GWXButtonResize(display, btn);

	    break;

          case GWX_DIALOG_TYPE_CONF:
            x_inc = (int)((int)md->width / 3);

            btn = &md->no_btn;
            XMoveWindow(
                dpy, btn->toplevel,
                (int)((x_inc * 2) - ((int)btn->width / 2)),
                (int)((int)md->height - (int)md->margin - (int)btn->height)
            );
            GWXButtonResize(display, btn);

            btn = &md->yes_btn;
            XMoveWindow(
                dpy, btn->toplevel,
                (int)((x_inc * 1) - ((int)btn->width / 2)),
                (int)((int)md->height - (int)md->margin - (int)btn->height)
            );
            GWXButtonResize(display, btn);

            break;

	  case GWX_DIALOG_TYPE_MESG:
            x_inc = (int)((int)md->width / 2);

	    btn = &md->ok_btn;
            XMoveWindow(
                dpy, btn->toplevel,
                (int)((x_inc * 1) - ((int)btn->width / 2)),
                (int)((int)md->height - (int)md->margin - (int)btn->height)
            ); 
            GWXButtonResize(display, btn);
	    break;
	}

	return;
}

/*
 *	Redraws message dialog.
 */
void GWXDialogDraw(gw_display_struct *display, gwx_dialog_struct *md)
{
        Display *dpy;
        GC gc;
        Window w;
        Pixmap pm;
        Pixel c_fg, c_bg, c_highlight, c_shade;
        unsigned int width, height, margin;
	unsigned int btn_height = 0;
        XFontStruct *font;
	char **strv;
	int strc;
	int line_height_pixels;
	XPoint p[2];


        if((display == NULL) ||
           (md == NULL)   
        )
            return;

        dpy = display->display;
        gc = display->gc;
        w = md->toplevel;
        pm = md->toplevel_buf;
        if((dpy == NULL) ||
           (gc == None) ||
           (w == None) ||
           (pm == None)
        )
            return;
        
        width = md->width;
        height = md->height;
	margin = md->margin;

        if(!md->map_state)
        {
            XMapRaised(dpy, w);
            md->map_state = True;
        }

        switch(md->type)
        {
          case GWX_DIALOG_TYPE_CONF_CANCEL:
          case GWX_DIALOG_TYPE_CONF:
            btn_height = md->yes_btn.height;
            break;
            
          case GWX_DIALOG_TYPE_MESG:
            btn_height = md->ok_btn.height;
            break;
        }
            
        c_fg = md->color_fg;
        c_bg = md->color_bg;
	c_highlight = md->color_highlight;
	c_shade = md->color_shade;

	font = md->font;


	/* Draw background. */
        XSetForeground(dpy, gc, c_bg);
        XFillRectangle(dpy, pm, gc, 0, 0, width, height);


	/* Draw HR. */
	XSetForeground(dpy, gc, c_shade);
        XSetLineAttributes(
            dpy, gc,
            2, LineSolid, CapRound, JoinMiter
        );
        p[0].x = 0;
        p[0].y = (int)((int)height - ((int)md->margin * 2) - (int)btn_height);
        p[1].x = (int)width;
        p[1].y = p[0].y;
        XDrawLines(
            dpy, pm, gc,
            &(p[0]), 2,         /* Point array and number of points. */
            CoordModeOrigin
        );

        XSetForeground(dpy, gc, c_highlight);
        p[0].y += 1;
        p[1].y = p[0].y;
        XDrawLines(
            dpy, pm, gc,
            &(p[0]), 2,         /* Point array and number of points. */
            CoordModeOrigin
        );


	/* Get values for drawing message lines. */
	strv = md->strv;
	strc = md->strc;
	line_height_pixels = md->line_height_pixels;

	/* Got enough information to draw message lines? */
        if((font != NULL) && (strv != NULL) &&
           (line_height_pixels > 0)
        )
        {
            int	x = (int)((md->margin * 2) + md->icon_width),
		y = (int)md->margin,
		i, len;
	    const char *strptr;

	    /* Set up GC for message lines drawing. */
            if(font->fid != None)
                XSetFont(dpy, gc, font->fid);
            XSetForeground(dpy, gc, c_fg);

	    /* Draw each message line. */
	    for(i = 0; i < strc; i++)
	    {
		strptr = (const char *)strv[i];
		if(strptr == NULL)
		    continue;

		len = strlen(strptr);
		if(len > 0)
		    XDrawString(
                        dpy, pm, gc,
                        x, y + font->max_bounds.ascent,
                        strptr, len
                    );

		y += line_height_pixels;
	    }
        }

	/* Copy Pixmap buffer to Window. */
        XCopyArea(dpy, pm, w, gc, 0, 0, width, height, 0, 0);


	/* Draw icon. */
	w = md->icon_w;
	pm = md->icon;
	width = md->icon_width;
	height = md->icon_height;
	if((w != None) &&
           (pm != None) &&
           (width > 0) &&
           (height > 0)
	)
	{
	    XCopyArea(dpy, pm, w, gc, 0, 0, width, height, 0, 0);
	}

	return;
}

/*
 *	Manages the event if it is for the given message dialog.
 */
int GWXDialogManage(
        gw_display_struct *display,
        gwx_dialog_struct *md,
        XEvent *event
)
{
        Display *dpy;
        Window w, ew;
        int events_handled = 0;


        if((display == NULL) ||
           (md == NULL) ||
           (event == NULL)
        )
            return(events_handled);

        dpy = display->display;
        if(dpy == NULL)
            return(events_handled);

        w = md->toplevel;
        if(w == None)
            return(events_handled);

        ew = event->xany.window; 

        switch(event->type)
        {
          case Expose:
	    if(w == ew)
	    {
		GWXDialogDraw(display, md);
		events_handled++;
	    }
	    break;

	  case ConfigureNotify:
	    if(w == ew)
	    {
		GWXDialogResize(display, md);
                events_handled++;
	    }
	    break;

	  case VisibilityNotify:
            if(w == ew)
            {


                events_handled++;
            }
	    break;

	  case KeyPress:
	    if(md->in_focus)
	    {
		

                events_handled++;
            }
            break;

          case KeyRelease:
            if(md->in_focus)
            {


                events_handled++;
            }
            break;

          case FocusOut:
            if(w == ew)
            {
                md->in_focus = False;
                events_handled++;
            }
            break;

	  case FocusIn:
	    if(w == ew)
            {
                md->in_focus = True;
                events_handled++;
            }
            break;

	  case ClientMessage:
	    if((event->xclient.format == 32) &&
               (event->xclient.data.l[0] == display->atom_wm_delete_window) &&
	       (w == ew)
	    )
	    {
		switch(md->type)
		{
		  case GWX_DIALOG_TYPE_CONF_CANCEL:
		    GWXDialogCancelBtnCB(
                        &md->cancel_btn,
                        md->cancel_btn.client_data
                    );
                    break;

                  case GWX_DIALOG_TYPE_CONF:
                    GWXDialogCancelBtnCB(
                        &md->no_btn,
                        md->no_btn.client_data
                    );
                    break;

		  default:
		    GWXDialogOKBtnCB(
		        &md->ok_btn,
		        md->ok_btn.client_data
		    );
		    break;
		}
		events_handled++;
	    }
	    break;
	}

	switch(md->type)
	{
	  case GWX_DIALOG_TYPE_CONF_CANCEL:
	    if(!events_handled)
		events_handled += GWXButtonManage(display, &md->cancel_btn, event);
	  case GWX_DIALOG_TYPE_CONF:
            if(!events_handled)
                events_handled += GWXButtonManage(display, &md->no_btn, event);
            if(!events_handled)
                events_handled += GWXButtonManage(display, &md->yes_btn, event);
	    break;

	  case GWX_DIALOG_TYPE_MESG:
	    if(!events_handled)
                events_handled += GWXButtonManage(display, &md->ok_btn, event);
            break;
	}

	return(events_handled);
}

/*
 *	Destroys and deallocates all resources on the given dialog.
 */
void GWXDialogDestroy(
        gw_display_struct *display,
        gwx_dialog_struct *md
)
{
        Display *dpy;
        Pixmap *pm;
        Window *w;
        XFontStruct **font;


        if((display == NULL) ||
           (md == NULL)
        )
            return;

        dpy = display->display;
        if(dpy == NULL)
            return;

	switch(md->type)
	{
	  case GWX_DIALOG_TYPE_CONF_CANCEL:
	    GWXButtonDestroy(display, &md->cancel_btn);
	  case GWX_DIALOG_TYPE_CONF:
            GWXButtonDestroy(display, &md->yes_btn);
            GWXButtonDestroy(display, &md->no_btn);
	    break;

	  case GWX_DIALOG_TYPE_MESG:
	    GWXButtonDestroy(display, &md->ok_btn);
	    break;
	}

#define DO_DESTROY_WINDOW	\
if((*w) != None) \
{ \
 XWMHints *wm_hints = XGetWMHints(dpy, *w); \
 if(wm_hints != NULL) \
 { \
  if(wm_hints->flags & IconWindowHint) \
   if(wm_hints->icon_window != None) \
    XDestroyWindow(dpy, wm_hints->icon_window); \
  if(wm_hints->flags & IconPixmapHint) \
   if(wm_hints->icon_pixmap != None) \
    XFreePixmap(dpy, wm_hints->icon_pixmap); \
  if(wm_hints->flags & IconMaskHint) \
   if(wm_hints->icon_mask != None) \
    XFreePixmap(dpy, wm_hints->icon_mask); \
  XFree(wm_hints); \
 } \
 \
 XDestroyWindow(dpy, *w); \
 (*w) = None; \
}
	w = &md->icon_w;
	DO_DESTROY_WINDOW

        w = &md->toplevel;
	DO_DESTROY_WINDOW

#define DO_DESTROY_PIXMAP	\
if(*pm != None) \
{ \
 XFreePixmap(dpy, *pm); \
 *pm = None; \
}
        pm = &md->icon;  
        DO_DESTROY_PIXMAP

        pm = &md->icon_mask;
	DO_DESTROY_PIXMAP

        pm = &md->toplevel_buf;
	DO_DESTROY_PIXMAP

	if(md->colors_initialized)
	{
            GWXDestroyPixel(display, md->color_fg);
            GWXDestroyPixel(display, md->color_bg);
            GWXDestroyPixel(display, md->color_highlight);
            GWXDestroyPixel(display, md->color_shade);
	    md->colors_initialized = False;
	}

        font = &md->font;
        if((*font) != NULL)
        {
            GWXDestroyFont(display, *font);
            (*font) = NULL;
        }

	StringFreeArray(md->strv, md->strc);
	md->strv = NULL;
	md->strc = 0;

	memset(md, 0x00, sizeof(gwx_dialog_struct));

#undef DO_DESTROY_WINDOW
#undef DO_DESTROY_PIXMAP

	return;
}


/*
 *      Confirmation dialog no button callback.
 */
void GWXDialogNoBtnCB(
        void *object, void *client_data
)
{
        gw_display_struct *display = client_data;
        gwx_button_struct *btn = object;
        gwx_dialog_struct *md;


        if((display == NULL) ||
           (btn == NULL)
        )
            return;

        md = &display->conf_dialog;
        if(&md->no_btn == btn)
            last_conf_dialog_code = GWConfirmationNo;

        return;
}

/*
 *	Confirmation dialog yes button callback.
 */
void GWXDialogYesBtnCB(
        void *object, void *client_data
)
{
        gw_display_struct *display = client_data;
        gwx_button_struct *btn = object;
        gwx_dialog_struct *md;


        if((display == NULL) ||
           (btn == NULL)
        )
            return;

        md = &display->conf_dialog;
        if(&md->yes_btn == btn)
	    last_conf_dialog_code = GWConfirmationYes;

	return;
}

/*
 *      Confirmation dialog cancel button callback.
 */
void GWXDialogCancelBtnCB(
        void *object, void *client_data
)
{
        gw_display_struct *display = client_data;
        gwx_button_struct *btn = object;
        gwx_dialog_struct *md;


        if((display == NULL) ||
           (btn == NULL)
        )
            return;

        md = &display->conf_dialog;
        if(&md->cancel_btn == btn)
            last_conf_dialog_code = GWConfirmationCancel;

        return;
}

/*
 *	Dialog OK button callback.
 */
void GWXDialogOKBtnCB(
	void *object, void *client_data
)
{
	gw_display_struct *display = client_data;
	gwx_button_struct *btn = object;
	gwx_dialog_struct *md;


	if((display == NULL) ||
           (btn == NULL)
	)
	    return;

	/* Is this our message dialog's ok button? */
	md = &display->mesg_dialog;
	if(&md->ok_btn == btn)
	    GWXDialogUnmap(display, md);

	return;
}

/*
 *	Sets the dialog messages, parsinging it into seperate lines
 *	and updates the longest line value and resizes the dialog.
 */
void GWXDialogSetMesg(
	gw_display_struct *display,
	gwx_dialog_struct *md,
	const char *title,
	const char *mesg,
	const char *details
)
{
        Display *dpy;
        Window w;
	XFontStruct *font;
	unsigned int btn_height = 0;


        if((display == NULL) ||
           (md == NULL)
        )
            return;

	/* Remove old message lines (if any) from message dialog. */
	StringFreeArray(md->strv, md->strc);
	md->strv = NULL;
	md->strc = 0;
	md->longest_line = 0;
	md->longest_line_pixels = 0;
	md->line_height_pixels = 0;
	md->lines_height_pixels = 0;


        dpy = display->display;
	w = md->toplevel;
        if((dpy == NULL) ||
           (w == None)
	)
            return;

	/* Get button height depending on dialog type. */
        switch(md->type)  
        {
          case GWX_DIALOG_TYPE_CONF_CANCEL:
          case GWX_DIALOG_TYPE_CONF:
            btn_height = md->yes_btn.height;
            break;
            
          case GWX_DIALOG_TYPE_MESG:
            btn_height = md->ok_btn.height;
            break;
        }

	/* Update dialog title. */
	if(title != NULL)
            XStoreName(dpy, w, title);

	/* Get pointer to message dialog's font structure. */
	font = md->font;

	/* Parse message and store each line, updating line bound 
	 * values.
	 */
	if(mesg != NULL)
	{
	    int i, len;
	    int len_pixels;
	    const char *strptr, *strend;


	    strptr = mesg;
	    /* strc and strv should already be reset. */

	    while(strptr != NULL)
	    {
		/* Seek new line deliminator. */
		strend = strchr(strptr, '\n');
		if(strend == NULL)
		    strend = strchr(strptr, '\r');

		/* Allocate more lines. */
		i = md->strc;
		md->strc++;
		md->strv = (char **)realloc(
		    md->strv,
		    md->strc * sizeof(char *)
		);
		if(md->strv == NULL)
		{
		    md->strc = 0;
		    break;
		}

		/* No new line deliminter at end? */
		if(strend == NULL)
		{
		    /* This is the last line. */
		    len = strlen(strptr);
		    md->strv[i] = StringCopyAlloc(strptr);

                    len_pixels = GWXGetFontStringLength(
                        font, strptr
                    );
		}
		else
		{
		    /* Got new line deliminator, so copy this line segment. */
		    char *strptr2;

		    len = MAX(strend - strptr, 0);
		    md->strv[i] = (char *)malloc((len + 1) * sizeof(char));
		    strptr2 = md->strv[i];
		    if(strptr2 != NULL)
		    {
			if(len > 0)
			    strncpy(strptr2, strptr, len);
			strptr2[len] = '\0';
		    }

		    len_pixels = GWXGetFontStringLength(
			font, strptr2
		    );
		}
		/* Now len and len_pixels should be set properly. */

		/* If this line is longer than the longest. */
		if(len > md->longest_line)
		    md->longest_line = len;
		if(len_pixels > md->longest_line_pixels)
		    md->longest_line_pixels = len_pixels;

		/* Seek to next line if strend is not NULL. */
		strptr = ((strend == NULL) ? NULL : strend + 1);
	    }
	}

	/* Update message lines heights. */
	if(font != NULL)
	{
	    md->line_height_pixels = font->max_bounds.ascent +
		font->max_bounds.descent;
	    md->lines_height_pixels = md->line_height_pixels * md->strc;
	}

	/* Resize message dialog's troplevel window. */
        XResizeWindow(
            dpy, w,
            (unsigned int)(((md->icon_w == None) ?
		(md->margin * 2) : (md->margin * 3))
                + md->icon_width + md->longest_line_pixels
	    ),
            (unsigned int)MAX(
                (md->margin * 4) + btn_height + md->icon_height,
                (md->margin * 4) + btn_height + md->lines_height_pixels
            )   
        );
	GWXDialogResize(display, md);

	return;
}

/*
 *	Maps the dialog, the dialog will be resized and drawn.
 */
void GWXDialogMap(
        gw_display_struct *display,
        gwx_dialog_struct *md
)
{
	Display *dpy;
	Window w;


	if((display == NULL) ||
           (md == NULL)
	)
	    return;

	/* Move dialog to center over toplevel window. */
	dpy = display->display;
	w = md->toplevel;
	if((dpy != NULL) &&
           (w != None)
	)
	{
	    int x = (int)(display->x + ((int)display->width / 2) -
		((int)md->width / 2));
	    int y = (int)(display->y + ((int)display->height / 2) -
		((int)md->height / 2));

	    if((x + (int)md->width) > (int)display->root_width)
		x = (int)display->root_width - (int)md->width;
            if((y + (int)md->height) > (int)display->root_height)
                y = (int)display->root_height - (int)md->height;

	    if(x < 0)
		x = 0;
	    if(y < 0)
		y = 0;

	    XMoveWindow(dpy, w, x, y);
	}

	/* Map dialog by calling the drawing function, it will map
	 * it and redraw it.
	 */
	GWXDialogDraw(display, md);

	/* Map buttons and subwindows. */
	switch(md->type)
	{
	  case GWX_DIALOG_TYPE_CONF_CANCEL:
	    GWXButtonDraw(display, &md->cancel_btn);
	  case GWX_DIALOG_TYPE_CONF:
	    GWXButtonDraw(display, &md->no_btn);
            GWXButtonDraw(display, &md->yes_btn);
	    break;

	  case GWX_DIALOG_TYPE_MESG:
            GWXButtonDraw(display, &md->ok_btn);
            break;
	}

	return;
}

/*
 *	Unmaps the dialog.
 */
void GWXDialogUnmap(
        gw_display_struct *display, 
        gwx_dialog_struct *md
)
{
	if((display != NULL) &&
           (md != NULL)
	)
	{
	    Display *dpy = display->display;
            Window w = md->toplevel;

            if((dpy != NULL) &&
               (w != None)
            )
                XUnmapWindow(dpy, w);

            md->map_state = False;
	}
	return;
}

/*
 *	Blocks client program execution untill confirmation is
 *	recieved.
 *
 *	Calling function is responsible for checking reentry to this
 *	function.
 *
 *	Confirmation dialog is already assumed to have been set
 *	up and mapped prior to this call.
 *
 *	The confermation dialog will not be unmapped by this call,
 *	it is up the calling function to do that.
 */
int GWXDoBlockUntilConf(gw_display_struct *display)
{
	if(display == NULL)
	    return(GWConfirmationNo);


	/* Reset local global confirmation code. */
	last_conf_dialog_code = GWConfirmationNotAvailable;

	/* Keep managing here untill confirmation dialog code
	 * has been set or display has been closed.
	 */
	while((last_conf_dialog_code == GWConfirmationNotAvailable) &&
              (display->display != NULL)
	)
	{
	    GWManage(display);
	    usleep(1000);
	}

	return(last_conf_dialog_code);
}


#endif	/* Not __MSW__ */
