#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include <ctype.h>
#include <sys/stat.h>
#include <math.h>

#include "../include/fio.h"
#include "../include/string.h"
#include "../include/strexp.h"
#include "../include/disk.h"

#include "sfm.h"

#include "obj.h"
#include "simutils.h"
#include "menu.h"
#include "sarreality.h"
#include "weather.h"
#include "mission.h"
#include "sar.h"
#include "sarfio.h"

#include "config.h"

char *SARMissionFSeekGetOpString(FILE *fp);

void SARMissionDestroy(sar_mission_struct *mission);

int SARMissionGetInterceptNameFromScene(
	const char *filename,
	const char *obj_name,
        sar_obj_intercept_struct *intercept
);

static int SARMissionLoadMenuMapSceneInfo(
	sar_core_struct *core_ptr,
	sar_menu_struct *m,
	sar_menu_map_struct *map_ptr,
	const char *filename
);

char *SARMissionLoadDescription(const char *filename);
void SARMissionLoadMenuMap(
	sar_core_struct *core_ptr,
	sar_menu_struct *m,
	const char *filename
);

sar_mission_struct *SARMissionLoadFromFile(
	sar_core_struct *core_ptr,
	const char *filename,
	void *client_data,
	int (*progress_func)(void *, long, long)
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))

#define ISCOMMENT(c)    ((c) == SAR_COMMENT_CHAR)
#define ISCR(c)         (((c) == '\n') || ((c) == '\r'))


/*
 *	Return a statically allocated string indicating the
 *      operation fetched from the pointed to fp.  Return can be
 *      an empty string if there was no op string to be found or
 *      error.
 *
 *      fp is positioned at the end of the op string which the next
 *      fetch can be used to get its argument. If a new line character
 *      is detected at the end of string on file, then the fp will be
 *      repositioned at that new line character. The next reading of fp
 *      will read that new line character.
 */
char *SARMissionFSeekGetOpString(FILE *fp)
{
        int c, i;
#define len     80
        static char rtn_str[len];


        (*rtn_str) = '\0';

        if(fp == NULL)
            return(rtn_str);

        /* Seek past spaces. */
        FSeekPastSpaces(fp);

	/* Itterate to get parameter. */
        for(i = 0; i < len; i++)
        {
            c = fgetc(fp);
            if((c == EOF) ||
               ISBLANK(c)
            )
            {
                rtn_str[i] = '\0';
                break;
            }
            /* Escape sequence? */
            else if(c == '\\')
            {
                c = fgetc(fp);
                if(c == EOF)
                {
                    rtn_str[i] = '\0';
                    break;
                }

                if(c != '\\')
                    c = fgetc(fp);

		if(c == EOF)
                {
                    rtn_str[i] = '\0';
                    break;
                }
            }
	    /* New line? */
            else if(ISCR(c))
            {
                /* New line right at the end of the op string, seek   
                 * back one character.
                 */
                fseek(fp, -1, SEEK_CUR);  
   
                rtn_str[i] = '\0';
                break;
            }

            rtn_str[i] = (char)c;
        }
        
#undef len
        return(rtn_str);
}


/*
 *	Deallocates mission structure and all its substructures,
 *	including the mission structure itself.
 */
void SARMissionDestroy(sar_mission_struct *mission)
{
	sar_mobj_arrive_struct *arrive_objective;
	sar_mobj_rescue_struct *rescue_objective;

	if(mission == NULL)
	    return;

	switch(mission->type)
	{
	  case MISSION_TYPE_ARRIVE:
	    arrive_objective = mission->objective;
	    if(arrive_objective == NULL)
		break;

	    free(arrive_objective->arrive_at_name);
            free(arrive_objective->message_success);
            free(arrive_objective->message_fail);
	    break;

          case MISSION_TYPE_RESCUE:
            rescue_objective = mission->objective;
            if(rescue_objective == NULL)
                break;

            free(rescue_objective->arrive_at_name);
            free(rescue_objective->message_success);
            free(rescue_objective->message_fail);
            break;
	}


	free(mission->title);
	free(mission->description);

	free(mission->scene_file);
        free(mission->player_model_file);
        free(mission->player_stats_file);

        free(mission->objective);

	free(mission);

	return;
}

/*
 *	Load intercept by name from scene file, returns 0 on success
 *	or -1 on no match.
 */
int SARMissionGetInterceptNameFromScene(
	const char *filename, const char *obj_name,
	sar_obj_intercept_struct *intercept
)
{
	int i, status, ptype, total_parms;
	char *cur_obj_name = NULL;
	void *p, **parm;
	sar_parm_object_name_struct *p_object_name;
	sar_parm_translate_struct *p_translate;
	sar_parm_rotate_struct *p_rotate;
	Boolean matched_translate = False, matched_rotate = False;



	if((filename == NULL) || (obj_name == NULL) || (intercept == NULL))
	    return(-1);

	/* Load parms from scene file. */
	status = SARParmLoadFromFile(
	    filename, SAR_FILE_FORMAT_SCENE,
	    &parm, &total_parms,
	    -1,
	    NULL, NULL
	);
	if(status)
	    return(-1);

	/* Itterate through loaded parms. */
	for(i = 0; i < total_parms; i++)
	{
	    p = parm[i];
	    if(parm == NULL)
		continue;
	    else
		ptype = (*(int *)p);

	    /* Handle by parm type. */
	    switch(ptype)
	    {
	      case SAR_PARM_OBJECT_NAME:
		p_object_name = (sar_parm_object_name_struct *)p;
	        /* Update current object name. */
		free(cur_obj_name);
                cur_obj_name = StringCopyAlloc(p_object_name->name);
		/* Reset all other context values. */
		matched_translate = False;
		matched_rotate = False;
		break;

	      case SAR_PARM_TRANSLATE:
		p_translate = (sar_parm_translate_struct *)p;
		if((cur_obj_name != NULL) && !matched_translate)
		{
		    if(!strcasecmp(cur_obj_name, obj_name))
		    {
			intercept->x = p_translate->translate.x;
			intercept->y = p_translate->translate.y;
			intercept->z = p_translate->translate.z;

			intercept->radius = 40;	/* Explicitly set. */

			matched_translate = True;
		    }
		}
		break;

	      case SAR_PARM_ROTATE:
		p_rotate = (sar_parm_rotate_struct *)p;
                if((cur_obj_name != NULL) && !matched_rotate)
                {
                    if(!strcasecmp(cur_obj_name, obj_name))
                    {
/* Work on this later, it's not really needed right now. */

			matched_rotate = True;
		    }
		}
		break;
            }
	}

	/* Deallocate all loaded parms. */
	SARParmDeleteAll(&parm, &total_parms);

	/* Free local names. */
	free(cur_obj_name);

	return(0);
}

/*
 *	Loads information from a scene file relivent to a mission map,
 *	such as map background file and landmarks.
 *
 *	Intercept points not loaded. Returns -1 on error or 0 on success.
 *
 *	This function is intended to be called by SARMissionLoadMenuMap().
 */
static int SARMissionLoadMenuMapSceneInfo(
	sar_core_struct *core_ptr, sar_menu_struct *m,
	sar_menu_map_struct *map_ptr,
	const char *filename			/* Scene file. */
)
{
	int i, n, status, ptype, total_parms;
	const char *img_path;
	void *p, **parm;
	sar_menu_map_marking_struct *marking_ptr;
        Boolean	got_translate,
		got_rotate,
		got_object_name,
		got_object_map_description;

	sar_parm_scene_map_struct *p_scene_map;
	sar_parm_new_object_struct *p_new_object;
	sar_parm_new_helipad_struct *p_new_helipad;
	sar_parm_new_runway_struct *p_new_runway;
	sar_parm_new_human_struct *p_new_human;
	sar_parm_new_premodeled_struct *p_new_premodeled;
	sar_parm_translate_struct *p_translate;
	sar_parm_rotate_struct *p_rotate;
	sar_parm_object_name_struct *p_object_name;
	sar_parm_object_map_description_struct *p_object_map_description;

        if((filename == NULL) || (map_ptr == NULL))
            return(-1);

        /* Load parms from scene file. */
        status = SARParmLoadFromFile(
            filename, SAR_FILE_FORMAT_SCENE,
            &parm, &total_parms,
            -1,
            NULL, NULL
        );
        if(status)
            return(-1);

        /* Itterate through loaded parms. */
        for(i = 0; i < total_parms; i++)
        {
            p = parm[i];
            if(parm == NULL)
                continue;
            else 
                ptype = (*(int *)p);
                     
            /* Handle by parm type. */
            switch(ptype)
            {
              case SAR_PARM_SCENE_MAP:
		p_scene_map = (sar_parm_scene_map_struct *)p;
		/* Width and height in meters. */
                map_ptr->bg_tex_width = MAX(p_scene_map->width, 10.0);
		map_ptr->bg_tex_height = MAX(p_scene_map->height, 10.0);

		/* Texture image path. */
                img_path = p_scene_map->file;
                if(img_path != NULL)   
                {
		    const char *new_path;
		    char tmp_path[PATH_MAX + NAME_MAX];

                    if(ISPATHABSOLUTE(img_path))
                    {           
                        strncpy(tmp_path, img_path, PATH_MAX + NAME_MAX);
                    }
                    else
                    {
                        struct stat stat_buf;

                        new_path = PrefixPaths(dname.local_data, img_path);
                        if(new_path != NULL)
                        {
                            if(stat(new_path, &stat_buf))
                                new_path = PrefixPaths(
                                    dname.global_data,
                                    img_path
                                );
                        }
                        strncpy(
                            tmp_path,
                            (new_path == NULL) ? img_path : new_path,
                            PATH_MAX + NAME_MAX
                        );
		    }

                    /* Unload menu map's background texture as needed. */
                    V3DTextureDestroy(map_ptr->bg_tex);

		    /* Load new texture for menu map's background. */
                    map_ptr->bg_tex = V3DTextureLoadFromFile2D(
                        tmp_path,
                        "mission_map_tex",	/* Not used. */
                        V3D_TEX_FORMAT_RGB,	/* Destination format. */
                        NULL,			/* Client data. */
                        NULL			/* Progress notify cb. */
                    );
		    V3DTexturePriority(map_ptr->bg_tex, 0.95);
                }
		break;

            }	/* Handle by parm type. */
        }	/* Itterate through loaded parms. */


/* Resets all local contexts. */
#define RESET_CONTEXTS	\
{ \
 marking_ptr = NULL; \
 got_translate = False; \
 got_rotate = False; \
 got_object_name = False; \
 got_object_map_description = False; \
}

/* Adds marking_ptr to the map_ptr. */
#define ADD_MARKING	\
{ \
 /* Allocate more pointers. */ \
 n = map_ptr->total_markings; \
 map_ptr->total_markings++; \
 map_ptr->marking = (sar_menu_map_marking_struct **)realloc( \
  map_ptr->marking, \
  map_ptr->total_markings * sizeof(sar_menu_map_marking_struct *) \
 ); \
 if(map_ptr->marking == NULL) \
 { \
  map_ptr->total_markings = 0; \
  free(marking_ptr); \
  RESET_CONTEXTS \
 } \
 else \
 { \
  map_ptr->marking[n] = marking_ptr; \
 } \
}

	/* Reset local contexts. */
	RESET_CONTEXTS

        /* Itterate through loaded parms (second time around). */ 
        for(i = 0; i < total_parms; i++)
        {
            p = parm[i];
            if(parm == NULL)
                continue;
            else
                ptype = (*(int *)p);

	    /* Handle by parm type. */
	    switch(ptype)
	    {
	      case SAR_PARM_NEW_OBJECT:
		p_new_object = (sar_parm_new_object_struct *)p;
                /* Reset local contexts. */
                RESET_CONTEXTS
/* Ignore for now. */
		break;

	      case SAR_PARM_NEW_HELIPAD:
		p_new_helipad = (sar_parm_new_helipad_struct *)p;
                /* Reset local contexts. */
                RESET_CONTEXTS
                /* Allocate new marking on map for intercept. */
                marking_ptr = (sar_menu_map_marking_struct *)calloc(
                    1,
                    sizeof(sar_menu_map_marking_struct)
                );
                if(marking_ptr != NULL)
                {
                    marking_ptr->type = SAR_MENU_MAP_MARKING_TYPE_ICON;
                    marking_ptr->fg_color.a = 1.0;
                    marking_ptr->fg_color.r = 0.0;
                    marking_ptr->fg_color.g = 1.0;
                    marking_ptr->fg_color.b = 0.0;
                    marking_ptr->x = 0.0;
                    marking_ptr->y = 0.0;
                    marking_ptr->icon = core_ptr->menumap_helipad_img;

		    ADD_MARKING
		}
		break;

	      case SAR_PARM_NEW_RUNWAY:
		p_new_runway = (sar_parm_new_runway_struct *)p;
                /* Reset local contexts. */
                RESET_CONTEXTS
/* Ignore for now. */ 
                break;

	      case SAR_PARM_NEW_HUMAN:
		p_new_human = (sar_parm_new_human_struct *)p;
		/* Reset local contexts. */
                RESET_CONTEXTS
/* Ignore for now. */
                break;

	      case SAR_PARM_NEW_PREMODELED:
		p_new_premodeled = (sar_parm_new_premodeled_struct *)p;
		/* Reset local contexts. */
                RESET_CONTEXTS
/* Ignore for now. */
                break;

	      case SAR_PARM_TRANSLATE:
		p_translate = (sar_parm_translate_struct *)p;
                if((marking_ptr != NULL) && !got_translate)
                {
                    marking_ptr->x = p_translate->translate.x;
                    marking_ptr->y = p_translate->translate.y;
		    got_translate = True;
                }
		break;

	      case SAR_PARM_ROTATE:
		p_rotate = (sar_parm_rotate_struct *)p;
		if((marking_ptr != NULL) && !got_rotate)
                {
/* Ignore for now. */
                    got_rotate = True;
                }
		break;

	      case SAR_PARM_OBJECT_NAME:
		p_object_name = (sar_parm_object_name_struct *)p;
		if((marking_ptr != NULL) && !got_object_name)
                {
/* Ignore for now. */


		    got_object_name = True;
                }
		break;

              case SAR_PARM_OBJECT_MAP_DESCRIPTION:
                p_object_map_description = (sar_parm_object_map_description_struct *)p;
		if((marking_ptr != NULL) && !got_object_map_description)
		{
		    free(marking_ptr->desc);
		    marking_ptr->desc = StringCopyAlloc(p_object_map_description->description);

		    got_object_map_description = True;
		}
		break;

	    }	/* Handle by parm type. */
	}	/* Itterate through loaded parms (second time around). */


        /* Deallocate all loaded parms. */
        SARParmDeleteAll(&parm, &total_parms);

#undef ADD_MARKING
#undef RESET_CONTEXTS

        return(0);
}


/*
 *	Returns a pointer to a dynamically allocated description fetched
 *	from the specified mission file.
 */
char *SARMissionLoadDescription(const char *filename)
{
	int i, status, ptype, total_parms;
	void *p, **parm;
	sar_parm_description_struct *p_description;
	char *desc_ptr = NULL;


	if(filename == NULL)
            return(desc_ptr);

        /* Load parms from mission file. */
        status = SARParmLoadFromFile(
            filename, SAR_FILE_FORMAT_MISSION,
            &parm, &total_parms,
            SAR_PARM_DESCRIPTION,
            NULL, NULL  
        );
        if(status)
            return(desc_ptr);

        /* Itterate through loaded parms. */
        for(i = 0; i < total_parms; i++)
        {
            p = parm[i];
            if(parm == NULL)
                continue;
            else
                ptype = (*(int *)p);

	    switch(ptype)
	    {
	      case SAR_PARM_DESCRIPTION:
		p_description = (sar_parm_description_struct *)p;
		free(desc_ptr);
		desc_ptr = StringCopyAlloc(p_description->description);
		break;
            }
	}

        /* Deallocate all loaded parms. */
        SARParmDeleteAll(&parm, &total_parms);

	return(desc_ptr);
}

/*
 *	Procedure to load `mission map' data from mission file and put 
 *	into the menu's map object.
 */
void SARMissionLoadMenuMap(
	sar_core_struct *core_ptr,
	sar_menu_struct *m,
	const char *filename		/* Mission file name. */
)
{
	int i, n, status, ptype, total_parms;
	void *p, **parm;
	sar_parm_mission_scene_file_struct *p_mission_scene_file;
	sar_parm_mission_begin_at_struct *p_mission_begin_at;
	sar_parm_mission_arrive_at_struct *p_mission_arrive_at;
	sar_parm_mission_add_intercept_struct *p_mission_add_intercept;
	int map_num = -1;
	sar_menu_map_struct *map_ptr = NULL;
	const char *cstrptr;
	char	*scene_filename = NULL,
		*arrive_at_name = NULL,
		*begin_at_name = NULL;
	sar_menu_map_marking_struct	*marking_ptr = NULL,
					*intercept_marking_ptr = NULL;
	int intercepts_set = 0;


	if((core_ptr == NULL) || (m == NULL))
	    return;

	/* Get first map object on the menu. */
	for(i = 0; i < m->total_objects; i++)
	{
	    p = m->object[i];
	    if(p == NULL)
		continue;
	    else
		ptype = (*(int *)p);

	    if(ptype == SAR_MENU_OBJECT_TYPE_MAP)
	    {
		map_num = i;
		map_ptr = (sar_menu_map_struct *)p;
		break;
	    }
	}
	if(map_ptr == NULL)
	    return;

	/* Delete all markings on menu's map object. */
	SARMenuMapDeleteAllMarkings(map_ptr);

	/* If no file name given then deleting all markings is all we needed
	 * to do.
	 */
	if(filename == NULL)
	    return;

	/* Note, order of which parameters appear in the mission file
	 * is important! We expect to parse certain parameters before
	 * others in this function.
	 */

        /* Load parms from mission file. */
        status = SARParmLoadFromFile(
            filename, SAR_FILE_FORMAT_MISSION,
            &parm, &total_parms,   
            -1,
            NULL, NULL
        );
        if(status)
            return;

/* Adds marking_ptr to the map_ptr. */
#define ADD_MARKING	\
{ \
 /* Allocate more pointers. */ \
 n = map_ptr->total_markings; \
 map_ptr->total_markings++; \
 map_ptr->marking = (sar_menu_map_marking_struct **)realloc( \
  map_ptr->marking, \
  map_ptr->total_markings * sizeof(sar_menu_map_marking_struct *) \
 ); \
 if(map_ptr->marking == NULL) \
 { \
  map_ptr->total_markings = 0; \
  free(marking_ptr); \
  intercept_marking_ptr = NULL; \
 } \
 else \
 { \
  map_ptr->marking[n] = marking_ptr; \
 } \
}

        /* Itterate through loaded parms. */
        for(i = 0; i < total_parms; i++)
        {
            p = parm[i];
            if(parm == NULL)
                continue;
            else
                ptype = (*(int *)p);

            switch(ptype)
            {
	      case SAR_PARM_MISSION_SCENE_FILE:
		p_mission_scene_file = (sar_parm_mission_scene_file_struct *)p;
		cstrptr = (const char *)p_mission_scene_file->file;
		if(cstrptr != NULL)
                {
		    /* Complete scene file name. */
                    if(ISPATHABSOLUTE(cstrptr))
                    {
			free(scene_filename);
			scene_filename = strdup(cstrptr);
                    }
                    else
                    {
			const char *cstrptr2;
			struct stat stat_buf;

			cstrptr2 = PrefixPaths(dname.local_data, cstrptr);
                        if(cstrptr2 != NULL)
                        {  
                            if(stat(cstrptr2, &stat_buf))
                                cstrptr2 = PrefixPaths(
                                    dname.global_data,
                                    cstrptr
                                );
                        }
			if(cstrptr2 != NULL)
			{
                            free(scene_filename);
                            scene_filename = strdup(cstrptr2);
			}
                    }

		    /* Load information from scene file which are
		     * relivent to the mission map, such as map background
		     * texture file and landmarks.
		     */                 
		    SARMissionLoadMenuMapSceneInfo(
			core_ptr, m, map_ptr, scene_filename
		    );
                }
		break;

	      case SAR_PARM_MISSION_BEGIN_AT:
		p_mission_begin_at = (sar_parm_mission_begin_at_struct *)p;
		if(p_mission_begin_at->name != NULL)
		{
		    sar_obj_intercept_struct intercept;
		    memset(&intercept, 0x00, sizeof(sar_obj_intercept_struct));

		    /* Update begin at name. */
		    free(begin_at_name);
		    begin_at_name = strdup(p_mission_begin_at->name);

		    /* Match intercept by name. */
		    if(!SARMissionGetInterceptNameFromScene(
			scene_filename, begin_at_name, &intercept
		    ))
		    {
			/* Allocate new marking on map for intercept. */
			marking_ptr = (sar_menu_map_marking_struct *)calloc(
			    1, sizeof(sar_menu_map_marking_struct)
			);
			intercept_marking_ptr = marking_ptr;
			if(marking_ptr != NULL)
			{
			    marking_ptr->type = SAR_MENU_MAP_MARKING_TYPE_INTERCEPT_LINE;
			    marking_ptr->fg_color.a = 1.0;
			    marking_ptr->fg_color.r = 0.0;
                            marking_ptr->fg_color.g = 1.0;
                            marking_ptr->fg_color.b = 0.0;
                            marking_ptr->x = intercept.x;
                            marking_ptr->y = intercept.y;

			    intercepts_set++;

			    ADD_MARKING

                            /* Reset meters to pixels coeff zoom. */
                            map_ptr->m_to_pixels_coeff = SAR_MAP_DEF_MTOP_COEFF;

                            /* Set initial scroll position on map with
			     * respect to the begin at location.
			     */
			    map_ptr->scroll_x = (int)(-intercept.x *
				map_ptr->m_to_pixels_coeff);
			    map_ptr->scroll_y = (int)(-intercept.y *
				map_ptr->m_to_pixels_coeff);

			    /* Reset selected_marking on menu map. */
			    map_ptr->selected_marking = -1;
			}
		    }
		}
		break;

	      case SAR_PARM_MISSION_ARRIVE_AT:
		p_mission_arrive_at = (sar_parm_mission_arrive_at_struct *)p;
		if(p_mission_arrive_at->name)
		{
		    free(arrive_at_name);
		    arrive_at_name = strdup(p_mission_arrive_at->name);
		}
		break;

	      case SAR_PARM_MISSION_ADD_INTERCEPT:
		p_mission_add_intercept = (sar_parm_mission_add_intercept_struct *)p;
		if(1)
		{
		    sar_obj_intercept_struct intercept;
		    memset(&intercept, 0x00, sizeof(sar_obj_intercept_struct));

		    /* Set intercept by referance code. */
		    switch(p_mission_add_intercept->ref_code)
		    {
		      case 3:		/* Arrive at location. */
			SARMissionGetInterceptNameFromScene(
			    scene_filename, arrive_at_name,
			    &intercept
			);
                        break;

                      case 2:		/* Begin at location. */
			SARMissionGetInterceptNameFromScene(
			    scene_filename, begin_at_name,
			    &intercept
			);
			break;

		      default:		/* Standard intercept. */
			intercept.x = p_mission_add_intercept->pos.x;
			intercept.y = p_mission_add_intercept->pos.y;
			intercept.z = p_mission_add_intercept->pos.z;
			intercept.radius = p_mission_add_intercept->radius;
			intercept.urgency = p_mission_add_intercept->urgency;
			break;
		    }

		    /* Allocate new map marking structure, note previous
		     * marking structure should already be allocated from
		     * another add intercept parameter or begin at name
		     * parameter.
		     */
		    if((marking_ptr != NULL) &&
		       (intercept_marking_ptr != NULL)
		    )
		    {
			/* Allocate new intercept marking structure and
			 * copy over end position values from previous
			 * marking structure.
			 */
			sar_menu_map_marking_struct *new_ptr =
			    (sar_menu_map_marking_struct *)calloc(
                                1, sizeof(sar_menu_map_marking_struct)
			    );

			/* Set end position on last marking structure. */
			marking_ptr->x_end = intercept.x;
			marking_ptr->y_end = intercept.y;

			intercepts_set++;

			/* New marking structure allocated successfully? */
			if(new_ptr != NULL)
			{
			    new_ptr->type = SAR_MENU_MAP_MARKING_TYPE_INTERCEPT_LINE;
			    /* Copy over fg color from previous marking. */
			    memcpy(
			        &new_ptr->fg_color,
			        &marking_ptr->fg_color,
			        sizeof(sar_menu_color_struct)
			    );
			    new_ptr->x = intercept.x;
			    new_ptr->y = intercept.y;

			    intercepts_set++;

                            intercept_marking_ptr = new_ptr;
                            marking_ptr = new_ptr;

			    ADD_MARKING
		        }
			else
			{
			    intercept_marking_ptr = NULL;
			    marking_ptr = NULL;
			}

			/* Allocate another marking structure to represent
			 * way point icon.
			 */
			new_ptr = (sar_menu_map_marking_struct *)calloc(
			    1, sizeof(sar_menu_map_marking_struct)
			);
			if(new_ptr != NULL)
			{
			    new_ptr->type = SAR_MENU_MAP_MARKING_TYPE_ICON;
			    new_ptr->x = intercept.x;
			    new_ptr->y = intercept.y;
			    new_ptr->icon = core_ptr->menumap_intercept_img;

                            n = map_ptr->total_markings;
                            map_ptr->total_markings++;
                            map_ptr->marking = (sar_menu_map_marking_struct **)realloc(
                                map_ptr->marking,
				map_ptr->total_markings * sizeof(sar_menu_map_marking_struct *)
			    );
			    if(map_ptr->marking == NULL)
			    {
				map_ptr->total_markings = 0;
				free(new_ptr);
                            }
                            else
                            {
                                map_ptr->marking[n] = new_ptr;
                            }
		        }
                    }
		}
		break;
	    }	/* Handle parm by type. */
	}	/* Itterate through loaded parms. */


        /* Deallocate all loaded parms. */
        SARParmDeleteAll(&parm, &total_parms);


	/* Deallocate local context names. */
	free(scene_filename);
	scene_filename = NULL;

	free(arrive_at_name);
	arrive_at_name = NULL;

	free(begin_at_name);
	begin_at_name = NULL;


	/* If the number of intercepts set on the map is an odd number,
	 * then that implies that the last intercept line marking is
	 * extraneous.
	 */
	if(intercepts_set & 1)
	{
	    /* Delete the last marking pointer of type
	     * SAR_MENU_MAP_MARKING_TYPE_INTERCEPT_LINE. Itterate from
	     * last to first of all markings on the map.
	     */
	    for(i = map_ptr->total_markings - 1; i >= 0; i--)
	    {
		marking_ptr = map_ptr->marking[i];
		if(marking_ptr == NULL)
		    continue;

		/* Check marking type. */
		if(marking_ptr->type == SAR_MENU_MAP_MARKING_TYPE_INTERCEPT_LINE)
		{
		    /* Need to delete this marking on the map, set the
		     * pointer to NULL after deleting it.
		     */
		    free(marking_ptr);
		    map_ptr->marking[i] = NULL;

		    /* That's it, don't delete any more markings. */
		    break;
		}
	    }
	}

#undef ADD_MARKING

	return;
}


/*
 *	Loads a new mission
 *
 *	Note that the scene structure in the core structure must be
 *	allocated and with all values reset.
 */
sar_mission_struct *SARMissionLoadFromFile(
	sar_core_struct *core_ptr,
	const char *filename,
	void *client_data,
	int (*progress_func)(void *, long, long)
)
{
	int i, status, ptype, total_parms;
	void *p, **parm;
	sar_parm_name_struct *p_name;
	sar_parm_description_struct *p_description;
	sar_parm_player_model_file_struct *p_player_model_file;
	sar_parm_weather_struct *p_weather;
        sar_parm_texture_base_directory_struct *p_texture_base_directory;
        sar_parm_texture_load_struct *p_texture_load;
	sar_parm_mission_type_struct *p_mission_type;
	sar_parm_mission_scene_file_struct *p_mission_scene_file;
	sar_parm_mission_time_left_struct *p_mission_time_left;
	sar_parm_mission_begin_at_struct *p_mission_begin_at;
	sar_parm_mission_arrive_at_struct *p_mission_arrive_at;
	sar_parm_mission_message_success_struct *p_mission_message_success;
	sar_parm_mission_message_fail_struct *p_mission_message_fail;
	sar_parm_mission_humans_tally_struct *p_mission_humans_tally;
	sar_parm_mission_add_intercept_struct *p_mission_add_intercept;
	sar_parm_new_object_struct *p_new_object;
	sar_parm_new_helipad_struct *p_new_helipad;
	sar_parm_new_runway_struct *p_new_runway;
	sar_parm_new_human_struct *p_new_human;
        sar_parm_new_fire_struct *p_new_fire;
	sar_parm_new_smoke_struct *p_new_smoke;
	sar_parm_new_premodeled_struct *p_new_premodeled;
	sar_parm_translate_struct *p_translate;
	sar_parm_translate_random_struct *p_translate_random;
	sar_parm_rotate_struct *p_rotate;
        sar_parm_no_depth_test_struct *p_no_depth_test;
        sar_parm_polygon_offset_struct *p_polygon_offset;
	sar_parm_human_message_enter_struct *p_human_message_enter;
	sar_parm_human_referance_struct *p_human_referance;

	Boolean loaded_scene = False;
	sar_scene_struct *scene;
	sar_mission_struct *mission;
	sar_mobj_arrive_struct *arrive_objective = NULL;
	sar_mobj_rescue_struct *rescue_objective = NULL;

	int humans_need_rescue = 0;
	char *begin_at_name = NULL, *arrive_at_name = NULL;

	char *weather_preset_name = NULL;

	int obj_num = -1;
	sar_object_struct *obj_ptr = NULL, ***ptr;
	int *total;
        sar_object_aircraft_struct *obj_aircraft_ptr = NULL;   
        sar_object_ground_struct *obj_ground_ptr = NULL;
	sar_object_human_struct *obj_human_ptr = NULL;
        sar_object_smoke_struct *obj_smoke_ptr = NULL;
	sar_object_fire_struct *obj_fire_ptr = NULL;
	sar_position_struct begin_pos;
	sar_direction_struct begin_dir;


#define DO_RESET_SUBSTRUCTURE_PTRS	\
{ \
 obj_aircraft_ptr = NULL; \
 obj_ground_ptr = NULL; \
 obj_human_ptr = NULL; \
 obj_smoke_ptr = NULL; \
 obj_fire_ptr = NULL; \
}


        if((core_ptr == NULL) || (filename == NULL))
            return(NULL);

	/* Get pointer to scene structure. */
	scene = core_ptr->scene;
	if(scene == NULL)
	    return(NULL);

	ptr = &core_ptr->object;
	total = &core_ptr->total_objects;

        /* Note, order of which parameters appear in the mission file
         * is important! We expect to parse certain parameters before
         * others in this function.
         */

	/* Load parms from mission file. */
        status = SARParmLoadFromFile(
            filename, SAR_FILE_FORMAT_MISSION,
            &parm, &total_parms,
            -1,
            NULL, NULL
        );
        if(status)
            return(NULL);

	/* Allocate a new mission structure. */
	mission = (sar_mission_struct *)calloc(
	    1, sizeof(sar_mission_struct)
	);
	if(mission == NULL)
	{
	    SARParmDeleteAll(&parm, &total_parms);
	    return(NULL);
	}

	/* Set up default mission values. */
	mission->type = MISSION_TYPE_NONE;
	mission->state = MISSION_STATE_IN_PROGRESS;
	mission->title = NULL;
	mission->time_spent = 0.0;
	mission->check_int = 1000;		/* In milliseconds. */
	mission->next_check = cur_millitime + mission->check_int;

	/* Itterate through loaded parms. */
	for(i = 0; i < total_parms; i++)
	{
	    p = parm[i];
	    if(p == NULL)
		continue;
	    else
		ptype = (*(int *)p);

	    /* Handle by parm type. */
	    switch(ptype)
	    {
	      case SAR_PARM_NAME:
		p_name = (sar_parm_name_struct *)p;
		free(mission->title);
		mission->title = StringCopyAlloc(p_name->name);
		break;

	      case SAR_PARM_DESCRIPTION:
		p_description = (sar_parm_description_struct *)p;
                free(mission->description);
                mission->description = StringCopyAlloc(p_description->description);
		break;

	      case SAR_PARM_PLAYER_MODEL_FILE:
		p_player_model_file = (sar_parm_player_model_file_struct *)p;
                if((p_player_model_file->file != NULL) && loaded_scene)
                {
		    const char *player_file = (const char *)p_player_model_file->file;
		    char tmp_path[PATH_MAX + NAME_MAX];

                    /* Complete path. */
                    if(ISPATHABSOLUTE(player_file))
                    {
                        strncpy(tmp_path, player_file, PATH_MAX + NAME_MAX);
                    }
                    else
                    {
			struct stat stat_buf;
                        const char *cstrptr2 = (const char *)PrefixPaths(
			    dname.local_data, player_file
			);
                        if(cstrptr2 != NULL)
                        {
                            if(stat(cstrptr2, &stat_buf))  
                                cstrptr2 = (const char *)PrefixPaths(
				    dname.global_data, player_file
				);
                        }
                        strncpy(
                            tmp_path,
                            ((cstrptr2 == NULL) ? player_file : cstrptr2),
                            PATH_MAX + NAME_MAX
                        );
                    }
                    tmp_path[PATH_MAX + NAME_MAX - 1] = '\0';

                    /* Update player model file name on mission and
                     * core structure.
                     */
                    free(mission->player_model_file);
                    mission->player_model_file = strdup(tmp_path);

                    free(core_ptr->cur_player_model_file);
                    core_ptr->cur_player_model_file = strdup(tmp_path);

                    /* Create player object. */
                    SARSceneAddPlayerObject(
                        core_ptr, scene,
                        tmp_path,
                        &begin_pos, NULL
                    );

                    /* Update current object context. */
                    obj_num = scene->player_obj_num;
                    obj_ptr = scene->player_obj_ptr;
		}
		break;

	      case SAR_PARM_WEATHER:
		p_weather = (sar_parm_weather_struct *)p;
                if(loaded_scene)
                {
                    fprintf(stderr,
 "%s: Cannot specify `weather' after loading scene.\n",
                        filename
                    );
                }
                else if(p_weather->weather_preset_name != NULL)
                {
                    free(weather_preset_name);
                    weather_preset_name = strdup(p_weather->weather_preset_name);
                }
		break;

/*
	      case SAR_PARM_TIME_OF_DAY:
		p_time_of_day = (sar_parm_time_of_day_struct *)p;
                if(loaded_scene)
                {
		    scene->tod = ???
		    scene->tod_code = ???
                }
                else
                {
                    fprintf(stderr,
 "%s: Cannot specify `time_of_day' before loading scene.\n",
                        filename
                    );
                }
		break;
*/

              case SAR_PARM_TEXTURE_BASE_DIRECTORY:
                p_texture_base_directory = (sar_parm_texture_base_directory_struct *)p;
                break;

              case SAR_PARM_TEXTURE_LOAD:
                p_texture_load = (sar_parm_texture_load_struct *)p;
                SARObjLoadTexture(
                    core_ptr, scene, p_texture_load
                );
                break;

	      case SAR_PARM_MISSION_TYPE:
		p_mission_type = (sar_parm_mission_type_struct *)p;
		switch(p_mission_type->mission_type)
                {
                  case MISSION_TYPE_NONE:
                    mission->type = p_mission_type->mission_type;
                    mission->objective = NULL;
                    break;

		  case MISSION_TYPE_ARRIVE:
		    arrive_objective = (sar_mobj_arrive_struct *)calloc(
			1, sizeof(sar_mobj_arrive_struct)
		    );
		    if(arrive_objective != NULL)
		    {
		        mission->type = p_mission_type->mission_type;
			mission->objective = arrive_objective;

			arrive_objective->type = p_mission_type->mission_type;
			arrive_objective->time_left = 0.0;
			arrive_objective->arrive_at_name = NULL;
			arrive_objective->message_success = NULL;
			arrive_objective->message_fail = NULL;
		    }
		    break;

                  case MISSION_TYPE_RESCUE:
                    rescue_objective = (sar_mobj_rescue_struct *)calloc(
                        1, sizeof(sar_mobj_rescue_struct)
                    );
                    if(rescue_objective != NULL)
		    {
                        mission->type = p_mission_type->mission_type;
                        mission->objective = rescue_objective;

		        rescue_objective->type = p_mission_type->mission_type;
			rescue_objective->time_left_type = MISSION_TIME_LEFT_TYPE_RESCUE_ALL;
			rescue_objective->time_left = 0.0;
			rescue_objective->humans_rescued = 0;
                        rescue_objective->humans_need_rescue = 0;
                        rescue_objective->humans_died = 0;
                        rescue_objective->message_success = NULL;
                        rescue_objective->message_fail = NULL;
                        break;
		    }
		    break;

		  default:
		    fprintf(
			stderr,
			"%s: Unsupported mission type `%i'\n",
			filename, p_mission_type->mission_type
		    );
		    break;
		}
		break;

	      case SAR_PARM_MISSION_SCENE_FILE:
		p_mission_scene_file = (sar_parm_mission_scene_file_struct *)p;
		if(loaded_scene)
		{
		    fprintf(
			stderr,
			"%s: Redefination of scene_file ignored.\n",
                        filename
                    );
		}
		else if(p_mission_scene_file->file != NULL)
		{
		    const char *scene_file = (const char *)p_mission_scene_file->file;
		    char tmp_path[PATH_MAX + NAME_MAX];

		    /* Complete path. */
		    if(ISPATHABSOLUTE(scene_file))
                    {
                        strncpy(tmp_path, scene_file, PATH_MAX + NAME_MAX);
                    }
		    else
		    {
			struct stat stat_buf;
			const char *cstrptr2 = (const char *)PrefixPaths(
			    dname.local_data, scene_file
			);
			if(cstrptr2 != NULL)
                        {
                            if(stat(cstrptr2, &stat_buf))
                                cstrptr2 = (const char *)PrefixPaths(
				    dname.global_data, scene_file
				);
			}
			strncpy(
			    tmp_path,
			    ((cstrptr2 == NULL) ? scene_file : cstrptr2),
			    PATH_MAX + NAME_MAX
			);
		    }

		    /* Set new scene file on mission structure. */
		    free(mission->scene_file);
		    mission->scene_file = strdup(tmp_path);

		    /* Load scene. */
		    status = SARSceneLoadFromFile(
			core_ptr, scene,
			mission->scene_file,
			weather_preset_name,
			client_data, progress_func
		    );
		    if(status)
		    {
			/* Failed to load scene. */
			fprintf(
			    stderr,
 "%s: Error occured while loading scene `%s'.\n",
			    filename, mission->scene_file
			);
			SARSceneDestroy(
			    core_ptr, scene,
			    ptr, total
			);
			SARMissionDestroy(mission);
			mission = NULL;

			break;
		    }
		    else
		    {
			loaded_scene = True;
		    }
		}
		break;

	      case SAR_PARM_MISSION_TIME_LEFT:
		p_mission_time_left = (sar_parm_mission_time_left_struct *)p;
                if(arrive_objective != NULL)
                {
                    /* Ignore time_left_type for arrive objectives. */
                    arrive_objective->time_left =
			p_mission_time_left->time_left;
                }
                else if(rescue_objective != NULL)
                {
                    rescue_objective->time_left_type =
			p_mission_time_left->time_left_type;
                    rescue_objective->time_left =
			p_mission_time_left->time_left;
                }
		break;

	      case SAR_PARM_MISSION_BEGIN_AT:
		p_mission_begin_at = (sar_parm_mission_begin_at_struct *)p;
		if(p_mission_begin_at->name != NULL)
		{
		    sar_object_struct *begin_obj_ptr;

		    /* Update mission begin at name. */
                    free(begin_at_name);
                    begin_at_name = strdup(p_mission_begin_at->name);

                    /* Match begin at object. */
		    begin_obj_ptr = SARObjMatchPointerByName(
                        scene, *ptr, *total,
                        begin_at_name, NULL
                    );
                    if(begin_obj_ptr == NULL)
		    {
			fprintf(
			    stderr,
"%s: Begin at object name `%s' not found.\n",
			    filename, begin_at_name
			);
		    }
		    else
                    {
			/* Record begin at position. */
                        memcpy(
			    &begin_pos, &begin_obj_ptr->pos,
			    sizeof(sar_position_struct)
			);
/* Need to work on getting begin_dir based on location of matched
 * begin_at_name object.
 */
		    }
		}
		break;

	      case SAR_PARM_MISSION_ARRIVE_AT:
		p_mission_arrive_at = (sar_parm_mission_arrive_at_struct *)p;
		if(p_mission_arrive_at->name != NULL)
		{
		    sar_object_struct *arrive_obj_ptr;
		    char **storage = NULL;

		    free(arrive_at_name);
                    arrive_at_name = strdup(p_mission_arrive_at->name);

		    if(arrive_objective != NULL)
                        storage = &arrive_objective->arrive_at_name;
                    else if(rescue_objective != NULL)
                        storage = &rescue_objective->arrive_at_name;

                    if(storage != NULL)
                    {
                        free(*storage);
                        (*storage) = strdup(p_mission_arrive_at->name);
                    }

                    /* Match arrive at object (just to verify). */
                    arrive_obj_ptr = SARObjMatchPointerByName(
                        scene, *ptr, *total,
                        arrive_at_name, NULL
                    );
                    if(arrive_obj_ptr == NULL)
                    {
                        fprintf(
                            stderr,
"%s: Arrive at object name `%s' not found.\n",
                            filename, arrive_at_name
                        );
                    }
		}
		break;

	      case SAR_PARM_MISSION_MESSAGE_SUCCESS:
		p_mission_message_success = (sar_parm_mission_message_success_struct *)p;
		if(p_mission_message_success->message != NULL)
		{
		    char **storage = NULL;

		    if(arrive_objective != NULL)
                        storage = &arrive_objective->message_success;
                    else if(rescue_objective != NULL)
                        storage = &rescue_objective->message_success;

                    if(storage != NULL)
                    {
                        free(*storage);
                        (*storage) = strdup(p_mission_message_success->message);
		    }
		}
		break;

              case SAR_PARM_MISSION_MESSAGE_FAIL:
                p_mission_message_fail = (sar_parm_mission_message_fail_struct *)p;
                if(p_mission_message_fail->message != NULL)
                {
                    char **storage = NULL;

                    if(arrive_objective != NULL)
                        storage = &arrive_objective->message_fail;
                    else if(rescue_objective != NULL)
                        storage = &rescue_objective->message_fail;
            
                    if(storage != NULL)
                    {
                        free(*storage);   
                        (*storage) = strdup(p_mission_message_fail->message);
                    }
                }
                break;

	      case SAR_PARM_MISSION_HUMANS_TALLY:
		p_mission_humans_tally = (sar_parm_mission_humans_tally_struct *)p;
                if(rescue_objective != NULL)
                {
                    rescue_objective->humans_need_rescue =
			p_mission_humans_tally->need_rescue;

                    /* Check if the number of humans that need rescue
                     * that were actually added are fewer than
                     * this amount (this parameter should be specified
                     * after all the humans that need rescue have been
                     * added).
                     */
                    if(humans_need_rescue <
                        rescue_objective->humans_need_rescue
                    )
                        fprintf(
			    stderr,
 "%s: Warning: Specified humans that need rescue %i less than actual %i.\n",
                            filename,
                            rescue_objective->humans_need_rescue,
                            humans_need_rescue
                        );
                }
		break;

	      case SAR_PARM_MISSION_ADD_INTERCEPT:
		p_mission_add_intercept = (sar_parm_mission_add_intercept_struct *)p;
		if(obj_ptr != NULL)
                {
                    sar_obj_intercept_struct intercept;
                    sar_object_struct *tar_obj_ptr;

		    memset(&intercept, 0x00, sizeof(sar_obj_intercept_struct));

		    DO_RESET_SUBSTRUCTURE_PTRS
		    if(obj_ptr->type == SAR_OBJ_TYPE_AIRCRAFT)
			obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;

                    /* Handle by intercept referance code. */
                    switch(p_mission_add_intercept->ref_code)
                    {
                      case 3:   /* Intercept arrive at location. */   
                        tar_obj_ptr = SARObjMatchPointerByName(
                            scene,
                            *ptr, *total,
                            arrive_at_name, NULL
                        );
                        if(tar_obj_ptr != NULL)
                        {
                            intercept.x = tar_obj_ptr->pos.x;
                            intercept.y = tar_obj_ptr->pos.y;
                            intercept.z = tar_obj_ptr->pos.z;
                            intercept.radius = p_mission_add_intercept->radius;
                            intercept.urgency = p_mission_add_intercept->urgency;
                        }
                        break;

                      case 2:   /* Intercept begin at location. */
                        tar_obj_ptr = SARObjMatchPointerByName(
                            scene,
                            *ptr, *total,
                            begin_at_name, NULL
                        );
                        if(tar_obj_ptr != NULL)
                        {
                            intercept.x = tar_obj_ptr->pos.x;
                            intercept.y = tar_obj_ptr->pos.y;
                            intercept.z = tar_obj_ptr->pos.z;
                            intercept.radius = p_mission_add_intercept->radius;
                            intercept.urgency = p_mission_add_intercept->urgency;
                        }
                        break;
                    
                      default:  /* Standard intercept. */
                        intercept.x = p_mission_add_intercept->pos.x;
                        intercept.y = p_mission_add_intercept->pos.y;
                        intercept.z = p_mission_add_intercept->pos.z;
                        intercept.radius = p_mission_add_intercept->radius;
                        intercept.urgency = p_mission_add_intercept->urgency;
                        break;
                    }

                    /* Allocate new intercept structure on object. */
		    if(obj_aircraft_ptr != NULL)
		    {
			SARObjCreateIntercept(
			    scene,
                            &obj_aircraft_ptr->intercept,
                            &obj_aircraft_ptr->total_intercepts,
                            intercept.flags,
                            intercept.x, intercept.y, intercept.z,
                            intercept.radius, intercept.urgency,
                            intercept.name
                        );
                    }
		}
		break;

	      case SAR_PARM_NEW_OBJECT:
		p_new_object = (sar_parm_new_object_struct *)p;
                obj_num = SARObjCreate(
                    scene, ptr, total,
                    p_new_object->object_type
                );
                obj_ptr = ((obj_num < 0) ? NULL : (*ptr)[obj_num]);

                /* Reset all substructure type pointers. */
                DO_RESET_SUBSTRUCTURE_PTRS

                if(obj_ptr != NULL)
                {
                    /* Get pointer to substructure. */
                    switch(obj_ptr->type)
                    {
                      case SAR_OBJ_TYPE_AIRCRAFT:
                        obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
                        break;

                      case SAR_OBJ_TYPE_GROUND:
                        obj_ground_ptr = (sar_object_ground_struct *)obj_ptr->data;
                        break;

		      case SAR_OBJ_TYPE_HUMAN:
			obj_human_ptr = (sar_object_human_struct *)obj_ptr->data;
			break;
                    }
                }
                break;

              case SAR_PARM_NEW_HELIPAD:
                p_new_helipad = (sar_parm_new_helipad_struct *)p;
                DO_RESET_SUBSTRUCTURE_PTRS
                obj_num = SARObjLoadHelipad(
                    core_ptr, scene, p_new_helipad
                );
                obj_ptr = ((obj_num < 0) ? NULL : (*ptr)[obj_num]);
                break;

              case SAR_PARM_NEW_RUNWAY:
                p_new_runway = (sar_parm_new_runway_struct *)p;
                DO_RESET_SUBSTRUCTURE_PTRS
                obj_num = SARObjCreate(
                    scene, ptr, total,
                    SAR_OBJ_TYPE_RUNWAY
                );
                obj_ptr = ((obj_num < 0) ? NULL : (*ptr)[obj_num]);
                if(obj_ptr != NULL)
                {
                    sar_object_runway_struct *obj_runway_ptr =
                        (sar_object_runway_struct *)obj_ptr->data;

                    obj_ptr->range = p_new_runway->range;

                    if(obj_runway_ptr != NULL)
                    {
                        obj_runway_ptr->flags |= SAR_RUNWAY_FLAG_NORTH_LABEL;
                        obj_runway_ptr->flags |= SAR_RUNWAY_FLAG_SOUTH_LABEL;
                        obj_runway_ptr->flags |= SAR_RUNWAY_FLAG_EDGE_LIGHTING;
                        obj_runway_ptr->flags |= SAR_RUNWAY_FLAG_NORTH_LIGHTING;
                        obj_runway_ptr->flags |= SAR_RUNWAY_FLAG_SOUTH_LIGHTING;
                
                        obj_runway_ptr->length = p_new_runway->length;
                        obj_runway_ptr->width = p_new_runway->width;
                
                        SARObjAddContactBoundsRectangular(
                            obj_ptr,
                            SAR_CRASH_FLAG_SUPPORT_SURFACE, 0,
                            -(obj_runway_ptr->width / 2),
                            (obj_runway_ptr->width / 2),
                            -(obj_runway_ptr->length / 2),
                            (obj_runway_ptr->length / 2),
                            0.0, 0.0   
                        );

/*
                        obj_runway_ptr->north_label =
                        obj_runway_ptr->south_label =
 */

                        obj_runway_ptr->light_spacing =
                            obj_runway_ptr->length / 50;

                        /* Load texture. */
                        obj_runway_ptr->tex_num = SARGetTextureRefNumberByName(
                            scene, SAR_STD_TEXNAME_RUNWAY
                        );
                    }
                }
                break;

              case SAR_PARM_NEW_HUMAN:
                p_new_human = (sar_parm_new_human_struct *)p;
                DO_RESET_SUBSTRUCTURE_PTRS
                obj_num = SARObjLoadHuman(
                    core_ptr, scene, p_new_human
                );
                obj_ptr = ((obj_num < 0) ? NULL : (*ptr)[obj_num]);

                obj_human_ptr = (sar_object_human_struct *)obj_ptr->data;
		if(obj_human_ptr != NULL)
		{
                    /* Increment number of humans that need rescue that
                     * this mission file has specified.
                     */
                    if(obj_human_ptr->flags & SAR_HUMAN_FLAG_NEED_RESCUE)
                        humans_need_rescue++;
		}

                /* Update total on rescue objective structure if
                 * this is a rescue mission.
                 */
                if(rescue_objective != NULL)
                {
                    rescue_objective->humans_need_rescue =
                        humans_need_rescue;
                }
                break;

              case SAR_PARM_NEW_FIRE:     
                p_new_fire = (sar_parm_new_fire_struct *)p;
                DO_RESET_SUBSTRUCTURE_PTRS
                obj_num = SARObjLoadFire(
                    core_ptr, scene, p_new_fire
                );
                obj_ptr = ((obj_num < 0) ? NULL : (*ptr)[obj_num]);
                obj_fire_ptr = (sar_object_fire_struct *)obj_ptr->data;
                break;

              case SAR_PARM_NEW_SMOKE:
                p_new_smoke = (sar_parm_new_smoke_struct *)p;
                DO_RESET_SUBSTRUCTURE_PTRS
                obj_num = SARObjLoadSmoke(
                    core_ptr, scene, p_new_smoke
                );
                obj_ptr = ((obj_num < 0) ? NULL : (*ptr)[obj_num]);
                obj_smoke_ptr = (sar_object_smoke_struct *)obj_ptr->data;
                break;

              case SAR_PARM_NEW_PREMODELED:
                p_new_premodeled = (sar_parm_new_premodeled_struct *)p;
		DO_RESET_SUBSTRUCTURE_PTRS
                obj_num = SARObjCreatePremodeled(
                    core_ptr, scene,
                    p_new_premodeled->model_type,
                    p_new_premodeled->argc,
                    p_new_premodeled->argv
                );
                obj_ptr = ((obj_num < 0) ? NULL : (*ptr)[obj_num]);
                break;

/*
	      case SAR_PARM_SELECT_OBJECT_BY_NAME:
		p_select_object_by_name = (sar_parm_select_object_by_name *)p;
		if(p_select_object_by_name->name != NULL)
		{
                    obj_ptr = SARObjMatchPointerByName(
                        scene,
                        core_ptr->object, core_ptr->total_objects,
                        p_select_object_by_name->name, &obj_num
                    );
                }
		break;
 */
	      case SAR_PARM_TRANSLATE:
		p_translate = (sar_parm_translate_struct *)p;
	        SARObjLoadTranslate(
		    core_ptr, scene,
		    obj_ptr, p_translate
		);
		break;

	      case SAR_PARM_TRANSLATE_RANDOM:
		p_translate_random = (sar_parm_translate_random_struct *)p;
		SARObjLoadTranslateRandom(
		    core_ptr, scene,
		    obj_ptr, p_translate_random
		);
		break;

	      case SAR_PARM_ROTATE:
		p_rotate = (sar_parm_rotate_struct *)p;
                if(obj_ptr != NULL)
                    SARSimWarpObject(
                        scene, obj_ptr,
                        NULL, &p_rotate->rotate
                    );
		break;

              case SAR_PARM_NO_DEPTH_TEST:
                p_no_depth_test = (sar_parm_no_depth_test_struct *)p;
                if(obj_ptr != NULL)
                {
                    obj_ptr->flags |= SAR_OBJ_FLAG_NO_DEPTH_TEST;
                }
                break;

              case SAR_PARM_POLYGON_OFFSET:
                p_polygon_offset = (sar_parm_polygon_offset_struct *)p;
                if(obj_ptr != NULL)
                {
                    obj_ptr->flags |= p_polygon_offset->flags;
                }
                break;

              case SAR_PARM_HUMAN_MESSAGE_ENTER:
                p_human_message_enter = (sar_parm_human_message_enter_struct *)p;
                if((obj_ptr != NULL) && (obj_human_ptr != NULL))
                {
                    free(obj_human_ptr->mesg_enter);
                    obj_human_ptr->mesg_enter = StringCopyAlloc(
                        p_human_message_enter->message
                    );
                }
                break;

              case SAR_PARM_HUMAN_REFERANCE:
                p_human_referance = (sar_parm_human_referance_struct *)p;
                if((obj_ptr != NULL) && (obj_human_ptr != NULL) &&
                   (p_human_referance->referance_name != NULL)
                )
                {
                    int human_ref_obj_num = -1;
                    const char *ref_name = (const char *)p_human_referance->referance_name;

                    /* Run towards? */
                    if(p_human_referance->flags & SAR_HUMAN_FLAG_RUN_TOWARDS)
                    {
                        obj_human_ptr->flags |= SAR_HUMAN_FLAG_RUN_TOWARDS;
                    }
                    /* Run away? */
                    else if(p_human_referance->flags & SAR_HUMAN_FLAG_RUN_AWAY)
                    {
                        obj_human_ptr->flags |= SAR_HUMAN_FLAG_RUN_AWAY;
                    }

                    /* Handle referance object name. */
                    if(!strcasecmp(ref_name, "player"))
                    {
                        /* Set special intercept code to intercept the player. */
                        obj_human_ptr->intercepting_object = -2;
                    }
                    else
                    {
                        /* All else match by object name. */
                        SARObjMatchPointerByName(
                            scene, *ptr, *total,
                            ref_name, &human_ref_obj_num
                        );
                        obj_human_ptr->intercepting_object = human_ref_obj_num;
                    }
                }
                break;

	      default:
		break;

	    }	/* Handle by parm type. */


	    /* If we lost the mission structure, then we need to give up.
	     * There was probably some fatal error encountered above.
	     */
	    if(mission == NULL)
		break;

	}	/* Itterate through loaded parms. */

        /* Deallocate all loaded parms. */
        SARParmDeleteAll(&parm, &total_parms);


	/* ********************************************************* */
	/* Post loading checks (warning, mission might be NULL). */

	/* Arrive at object exists? */
	if(arrive_objective != NULL)
	{
	    const char *cstrptr = arrive_objective->arrive_at_name;
	    if(cstrptr != NULL)
	    {
                obj_ptr = SARObjMatchPointerByName(
                    scene, *ptr, *total,
		    cstrptr, NULL
		);
		if(obj_ptr == NULL)
		{
		    fprintf(
			stderr,
 "%s: Warning: No object in scene matching arrive_at_name `%s'\n",
			filename, cstrptr
		    );
		}
	    }
	}
	/* Rescue objective. */
        else if(rescue_objective != NULL)
        {
            const char *cstrptr = rescue_objective->arrive_at_name;
            if(cstrptr != NULL)
            {
                obj_ptr = SARObjMatchPointerByName(
                    scene, *ptr, *total,
                    cstrptr, NULL
                );
                if(obj_ptr == NULL)
                {
                    fprintf(
                        stderr,
 "%s: Warning: No object in scene matching arrive_at_name `%s'\n",
                        filename, cstrptr
                    );
                }
            }    
	}


	/* Deallocate names. */
	free(begin_at_name);
	free(arrive_at_name);

	free(weather_preset_name);

#undef DO_RESET_SUBSTRUCTURE_PTRS

	return(mission);
}
