/*
                    SAR Objects and Scene Structures

            All coordinates and vectors use left hand rule.

 */

#ifndef OBJ_H
#define OBJ_H

#include <sys/types.h>

#include "../include/os.h"

#include "sfm.h"
#include "v3dtex.h"

#include "sound.h"


/*
 *	Engine flags type:
 */
#define sar_obj_flags_t	unsigned long

/*
 *	Gradient animation counter type:
 */
#define sar_grad_anim_t	u_int16_t

/*
 *	DMS units type:
 */
#define sar_dms_t	double


/*
 *	Air worthy states:
 */
#define SAR_AIR_WORTHY_NOT_FLYABLE	0	/* Pile of junk. */
#define SAR_AIR_WORTHY_OUT_OF_CONTROL	1
#define SAR_AIR_WORTHY_FLYABLE		2

/*
 *	Engine state:
 */
#define SAR_ENGINE_STATE_OFF		0
#define SAR_ENGINE_STATE_INIT		1
#define SAR_ENGINE_STATE_ON		2

/*
 *	Human color palette index and max:
 */
#define SAR_HUMAN_COLOR_FACE            0
#define SAR_HUMAN_COLOR_HAIR            1
#define SAR_HUMAN_COLOR_TORSO           2
#define SAR_HUMAN_COLOR_HIPS            3
#define SAR_HUMAN_COLOR_LEGS            4
#define SAR_HUMAN_COLOR_FEET            5
#define SAR_HUMAN_COLOR_ARMS            6
#define SAR_HUMAN_COLOR_HANDS           7

#define SAR_HUMAN_COLORS_MAX            8


/*
 *	Color structure:
 */
typedef struct {

	double a, r, g, b;

} sar_color_struct;

/*
 *	Visual model structure:
 */
typedef struct {

#define SAR_VISUAL_MODEL_NOT_LOADED	0
#define SAR_VISUAL_MODEL_LOADING	1
#define SAR_VISUAL_MODEL_LOADED		2

	int load_state;	/* One of SAR_VISUAL_MODEL_*. */

	int ref_count;	/* Referance count. */

	char *filename;	/* Can be NULL. */
	char *name;	/* Can be NULL. */

	void *data;	/* Really a GLuint which referances a GL list. */

} sar_visual_model_struct;

/*
 *	Position structure:
 */
typedef struct {

	double x, y, z;
	long sx, sy, sz;

} sar_position_struct;

/*
 *	Direction structure:
 */
typedef struct {

	/* In radians. */
	double heading, pitch, bank;

} sar_direction_struct;


/*
 *	Intercept coordinate structure:
 */
typedef struct {

	sar_obj_flags_t flags;
	double x, y, z;		/* Position of intercept in meters. Note
				 * that z value is intercept altitude in
				 * meters.
				 */
	double radius;		/* Size of intercept cylendrical in meters. */
	double urgency;		/* How fast must we get there, 0.0 to 1.0. */
	char *name;		/* Intercept label. */

} sar_obj_intercept_struct;

/*
 *	Light structure:
 */
typedef struct {

#define SAR_LIGHT_FLAG_ON		(1 << 0)
#define SAR_LIGHT_FLAG_STROBE		(1 << 1)
#define SAR_LIGHT_FLAG_ATTENUATE	(1 << 2)

	sar_obj_flags_t flags;

	sar_position_struct pos;
	sar_direction_struct dir;

	int radius;	/* In pixels. */

	sar_color_struct color;			/* Apperance color. */
	sar_color_struct attenuate_color;	/* Attenuate color. */

	/* On/off timmings, only for SAR_LIGHT_TYPE_STROBE. */
	time_t	next_on, int_on,
		next_off, int_off;

} sar_light_struct;


/*
 *	Physical contact and crash boundaries structure:
 *
 *	Note that any object can have this contact bounds, but will be
 *	have to follow some special rules if they do.
 *
 *	First if the object is a human or aircraft, it should have only
 *	the SAR_CRASH_FLAG_CRASH_OTHER flag set and it's contact
 *	shape should be SAR_CONTACT_SHAPE_CYLENDRICAL.
 *
 *	Buildings and other objects that you want to set
 *	SAR_CRASH_FLAG_CRASH_CAUSE can have any shape (recommend
 *	SAR_CONTACT_SHAPE_CYLENDRICAL or SAR_CONTACT_SHAPE_RECTANGULAR).
 *
 *	Now objects that you want to set landable or walkable (with
 *	crash_flags set to SAR_CRASH_FLAG_SUPPORT_SURFACE) may not be
 *	human or aircrafts (objects that themselves need to be checked
 *	if on top of something).
 *
 *	If an object is set SAR_CRASH_FLAG_SUPPORT_SURFACE then the object
 *	can only have its heading rotated.
 *
 *	Note that object's of type SAR_OBJ_TYPE_GROUND are always
 *	landable/walkable and their shape is always cylendrical.
 */
typedef struct {

        /* Crash detection flags, specifies if this object can collide
         * with other objects, if other objects can collide with
         * this object, and if this object produces a flat support
	 * surface that is landable/walkable (respectivly with defined
	 * flags).
	 *
	 * If no flags are set it can be considered a quick way
	 * (ie if(crash_flags == 0)) to determine this object has no
         * crash interaction with other objects.
         */
#define SAR_CRASH_FLAG_CRASH_OTHER      (1 << 0)
#define SAR_CRASH_FLAG_CRASH_CAUSE      (1 << 1)
#define SAR_CRASH_FLAG_SUPPORT_SURFACE	(1 << 2)
        sar_obj_flags_t crash_flags;

        /* This value used if and only if the SAR_CRASH_FLAG_CRASH_CAUSE
	 * flag is set in crash_flags. This value specifies what type of
         * crash it would be if some other object crashed into this
	 * object.
         */
#define SAR_CRASH_TYPE_OBSTRUCTION      0       /* General structure. */
#define SAR_CRASH_TYPE_GROUND           1
#define SAR_CRASH_TYPE_MOUNTAIN         2
#define SAR_GRASH_TYPE_BUILDING         3
#define SAR_GRASH_TYPE_AIRCRAFT         4
        int crash_type;

        /* Body contact shape, one of SAR_CONTACT_SHAPE_*. */
#define SAR_CONTACT_SHAPE_SPHERICAL     0   
#define SAR_CONTACT_SHAPE_CYLENDRICAL   1
#define SAR_CONTACT_SHAPE_RECTANGULAR	2
        int contact_shape;

        /* Radius value in meters for contact shapes
	 * SAR_CONTACT_SHAPE_SPHERICAL or SAR_CONTACT_SHAPE_CYLENDRICAL.
	 */
        double contact_radius;

        /* Cylendrical height value in meters for contact shape
	 * SAR_CONTACT_SHAPE_CYLENDRICAL.
         */
        double contact_h_min, contact_h_max;

	/* Rectangular contact shape in meters for contact shape
	 * SAR_CONTACT_SHAPE_RECTANGULAR (left hand rule).
	 */
	double	contact_x_min, contact_x_max,
		contact_y_min, contact_y_max,
		contact_z_min, contact_z_max;

	/* Value records for the object's inversed heading for the trig
	 * functions, this is used to speed up rotations about heading
	 * for SAR_CONTACT_SHAPE_RECTANGULAR shape contact.
	 *
	 * Ie cos_heading = cos(-heading) and sin_heading = sin(-heading)
	 */
	double cos_heading, sin_heading;

} sar_contact_bounds_struct;


/*
 *	Rotor/propellar structure:
 */
typedef struct {

/* Can pitch forward for duel propulsion direction aircraft like the
 * Ospray.
 */
#define SAR_ROTOR_FLAG_CAN_PITCH	(1 << 1)

/* Is in `forward' prop motion position (pitched forwards if flag set).
 * Note that this has no affect on actual flight physics, the member
 * flight_model_type on the aircraft structure will determine the
 * correct flight model type.
 */
#define SAR_ROTOR_FLAG_PITCH_STATE	(1 << 2)

/* Does not rotate (spin) about its Z axis, used for stationary
 * rotor housing.
 */
#define SAR_ROTOR_FLAG_NO_ROTATE	(1 << 3)

/* Pitches and banks along with control positions if any. */
#define SAR_ROTOR_FLAG_FOLLOW_PB	(1 << 4)

/* Blades will blur using the blades_blur_color if spinning too fast. */
#define SAR_ROTOR_FLAG_BLADES_BLUR_FAST		(1 << 5)

/* Blades always blur (overrides SAR_ROTOR_FLAG_BLADES_BLUR_FAST). */
#define SAR_ROTOR_FLAG_BLADES_BLUR_ALWAYS	(1 << 6)



	sar_obj_flags_t flags;

	/* Offset relative to center of object. */
        sar_position_struct pos;

	/* Rotation (always spins on Y axis). */
        sar_direction_struct dir;

	/* Radius in meters, for knowing size to make rotor wash. */
	double radius;

	/* Spin position, 0 to (sar_grad_anim_t)-1. */
	sar_grad_anim_t	anim_pos;

        sar_visual_model_struct *visual_model;

	/* Rotor pitch rotate position, 0 to (sar_grad_anim_t)-1,
	 * where (sar_grad_anim_t)-1 is pitched forwards
	 * (when SAR_ROTOR_FLAG_PITCH_STATE is set).
	 */
	sar_grad_anim_t pitch_anim_pos;

	/* Rotor wash animation position. */
	sar_grad_anim_t rotor_wash_anim_pos;

	/* Rotor wash texture number on scene, can be -1. */
	int rotor_wash_tex_num;

	/* Control position coefficients, used if
	 * SAR_ROTOR_FLAG_FOLLOW_PB is set.
	 */
	double control_coeff_pitch, control_coeff_bank;

	/* Blades blur color, used if spinning too fast and
	 *
	 */
	sar_color_struct blades_blur_color;

} sar_obj_rotor_struct;


/*
 *	Landing gear structure:
 */
typedef struct {

#define SAR_LGEAR_FLAG_STATE	(1 << 0)	/* True if gear is up. */
#define SAR_LGEAR_FLAG_FIXED	(1 << 1)	/* True if fixed (always down). */
#define SAR_LGEAR_FLAG_DAMAGED	(1 << 2)	/* True if damaged. */
#define SAR_LGEAR_FLAG_MISSING	(1 << 3)	/* Non-existant. */
#define SAR_LGEAR_FLAG_SKI	(1 << 4)
#define SAR_LGEAR_FLAG_FLOATS	(1 << 5)	/* Can land on water. */

	sar_obj_flags_t flags;

	/* Offset relative to center of object. */
	sar_position_struct pos;

	/* Gear extended direction (retracted direction is 0 0 0,
	 * values are in radians and can be negative to indicate
	 * rotating to in the opposite direction.
	 */
	sar_direction_struct extended_dir;

	sar_grad_anim_t	anim_pos,	/* 0 to (sar_grad_anim_t)-1. */
			anim_rate;	/* Units per cycle. */

	sar_visual_model_struct *visual_model;

} sar_obj_landing_gear_struct;


/*
 *	Door structure:
 */
typedef struct {

#define SAR_DOOR_FLAG_STATE	(1 << 0)        /* Door opened if set. */
#define SAR_DOOR_FLAG_FIXED	(1 << 1)	/* True if fixed (always closed). */
#define SAR_DOOR_FLAG_LOCKED	(1 << 2)	/* True if locked. */
#define SAR_DOOR_FLAG_STAY_OPEN	(1 << 3)	/* Do not close door on next loop
						 * check, ie if door was opened
						 * by player explicitly.
						 */
        sar_obj_flags_t flags;

	/* Positions, offsetted from object center. */
        sar_position_struct pos_closed, pos_opened;

	/* Rotation of door when opened. */
	sar_direction_struct dir_opened;

	/* Position of threshold, the base position for
	 * entering/leaving.
	 */
	sar_position_struct pos_threshold;

	sar_grad_anim_t	anim_pos,
			anim_rate;

	sar_visual_model_struct *visual_model;

} sar_obj_door_struct;

/*
 *	External fuel tank structure:
 *
 *	Note that this is a structure ON sar_aircraft_struct structures
 *	and not to be confused with sar_object_fueltank_struct which is
 *	a fuel tank dropped from an aircraft as an independant object.
 */
typedef struct {

#define SAR_EXTERNAL_FUELTANK_FLAG_FIXED	(1 << 0)	/* Not droppable. */
#define SAR_EXTERNAL_FUELTANK_FLAG_ONBOARD	(1 << 1)	/* Not jettesoned. */

	sar_obj_flags_t flags;

	/* Offset from center of object in meters. */
	sar_position_struct offset_pos;

	/* Spherical contact bounds radius in meters. */
	double radius;

	/* Mass, all in kg. */
	double dry_mass, fuel, fuel_max;

	/* Visual model. */
	sar_visual_model_struct *visual_model;

} sar_external_fueltank_struct;

/*
 *	Rescue hoist (rope and basket) structure:
 */
typedef struct {

	/* Offset from center of object. */
	sar_position_struct offset;

	/* Position (not offset) of basket. */
	sar_position_struct pos;

	/* Direction of basket. */
	sar_direction_struct dir;

	/* Rope extention (if <= 0.0 then implies it is fully retracted)
         * all units in meters.
	 */
	double rope_cur, rope_max;
	double rope_rate;	/* In meters per cycle. */

	/* Rope visual extension (ie when the rope_cur is longer than the
	 * distance to the landable ground (in meters).
	 */
	double rope_cur_vis;

	/* Indicates rope end is in contact with ground if true. */
	int on_ground;

	/* Hoist deployment type, determines what's on the end of the
	 * hoist. Can be either SAR_HOIST_DEPLOYMENT_BASKET or
	 * SAR_HOIST_DEPLOYMENT_DIVER. Note that contact bounds are always
	 * checked the same way regardless of what is on the end of the 
	 * rope.
	 */
#define SAR_HOIST_DEPLOYMENT_BASKET	0
#define SAR_HOIST_DEPLOYMENT_DIVER	1
	int deployment;

	/* Cylendrical contact area of rescue basket, in meters. */
	double contact_radius;
	double contact_z_min, contact_z_max;

	/* Load capacity, in kg. */
	double capacity;

	/* Referance to occupant (people) object index numbers.
	 * This does not include the diver.
	 */
	int *occupant;
	int total_occupants;

	/* Total mass of occupant object(s) in kg, this is to speed up
	 * calculations. The value is updated when a new occupant object
	 * is added to the hoist and reset to 0 when the hoist is deployed
	 * the next time.
	 */
	double	occupants_mass;

	/* Texture referance numbers on scene structure. */
	int	side_tex_num, end_tex_num, bottom_tex_num,
		water_ripple_tex_num;

	/* Diver color palette. */
	sar_color_struct diver_color[SAR_HUMAN_COLORS_MAX];

	/* Animation position and rate. */
	sar_grad_anim_t anim_pos, anim_rate;

} sar_obj_hoist_struct;



/*
 *	Object type codes:
 */
#define SAR_OBJ_TYPE_GARBAGE		0
#define SAR_OBJ_TYPE_STATIC		1
#define SAR_OBJ_TYPE_AUTOMOBILE		2
#define SAR_OBJ_TYPE_WATERCRAFT		3
#define SAR_OBJ_TYPE_AIRCRAFT		4
/* Note type 5 used to be airplane, it's now changed to be part of
 * type 4 (which was helicopter) so both 4 and 5 are all flying things
 * now.
 */

/* Cylendrical ground base with (optional) touchable heightfield. */
#define SAR_OBJ_TYPE_GROUND		6
/* Runway. */
#define SAR_OBJ_TYPE_RUNWAY		7
/* Helipad. */
#define SAR_OBJ_TYPE_HELIPAD		8

/* Human. */
#define SAR_OBJ_TYPE_HUMAN		9

/* Smoke trail. */
#define SAR_OBJ_TYPE_SMOKE		10
/* Fire. */
#define SAR_OBJ_TYPE_FIRE		11
/* Explosion. */
#define SAR_OBJ_TYPE_EXPLOSION		12
/* Chemical spray (water, fire-retardant, etc). */
#define SAR_OBJ_TYPE_CHEMICAL_SPRAY	13
/* Aircraft fuel tank (dropping from the sky, not a fuel tank that is on
 * an aircraft).
 */
#define SAR_OBJ_TYPE_FUELTANK		14

/* Premodeled object. */
#define SAR_OBJ_TYPE_PREMODELED		20


/*
 *	Flight model codes, used in member flight_model_type of
 *	aircraft structure sar_object_aircraft_struct.
 */
#define SAR_FLIGHT_MODEL_HELICOPTER	0
#define SAR_FLIGHT_MODEL_AIRPLANE	1
#define SAR_FLIGHT_MODEL_SLEW		2


/*
 *	SAR aircraft (helicopter/airplane) object sub structure:
 *
 *	This structure (and all object substructures) can be pointed to
 *	from member `data' in the sar_object_struct, determined by member
 *	`type'.
 */
typedef struct {

	/* Flight model type, one of SAR_FLIGHT_MODEL_*. */
	int flight_model_type;

	/* Previously set flight model type (for switching back and forth
	 * between flight model types.
	 */
	int last_flight_model_type;

	/* Sar flight model flight dynamics model structure. */
	SFMModelStruct *fdm;

        /* Current speed, in meters per cycle. */
        double speed;

	/* Stall speed, the speed at which a controllable stall begins.
	 * Loss in lift can still occure twice beyond this value, in
	 * meters per cycle.
	 */
	double speed_stall, stall_coeff;

	/* Maximum speed (this is *several* times the rated max), in
	 * meters per cycle.
	 */
	double speed_max;

	/* Expected maximum speed according to `specifications', this
	 * is the actual maximum speed.
	 */
	double speed_max_expected;

	/* Current velocity vector representing velocity in object
	 * relative vector direction, in meters per cycle.
	 */
        sar_position_struct vel;

	/* Minimum drag in meters per cycle, must be non-negative. */
	double min_drag;

	/* Acceleration response for when flight_model_type is set to
	 * SAR_FLIGHT_MODEL_HELICOPTER. Higher values produce less
	 * responsiveness).
	 */
	sar_position_struct accel_responsiveness;

	/* Acceleration response for when flight_model_type is set to
	 * SAR_FLIGHT_MODEL_AIRPLANE. Higher values produce less
	 * responsiveness).
	 */
	sar_position_struct airplane_accel_responsiveness;


	/* Attitude change rates, in radians per cycle. */
	sar_direction_struct attitude_change_rate;

	/* Pitch and bank leveling, in radians per cycle. */
	double pitch_leveling, bank_leveling;


        /* Cockpit offset position in object relative vector
	 * direction, in meters.
	 */
        sar_position_struct cockpit_offset_pos;

	/* Cockpit visual model. */
	sar_visual_model_struct *visual_model_cockpit;

	/* Belly to object relative xy plane in meters. */
        double belly_height;

	/* Height of landing gear in meters. */
        double gear_height;

	/* Mass. */
	double dry_mass;		/* In kg. */
	double fuel_rate;		/* In kg consumed per cycle. */
	double fuel, fuel_max;		/* In kg. */
	int crew, passengers, passengers_max;
	double passengers_mass;		/* Total mass of current massengers in kg. */

	/* Passengers pending to leave or drop from this aircraft,
	 * 0 means none and any positive value means this many passengers
	 * need to go when the aircraft's door is fully opened and its
	 * conditions are right (ie landed at the right helipad).
	 */
	int passengers_leave_pending, passengers_drop_pending;

	/* External fuel tanks. */
	sar_external_fueltank_struct **external_fueltank;
	int total_external_fueltanks;

	/* Engine data. */
	int engine_state;		/* One of SAR_ENGINE_STATE_*. */
	time_t next_engine_on;		/* Time till engine starts up (in ms). */
        double throttle;                /* 0.0 to 1.0. */
	double collective;		/* 0.0 to 1.0. */
	double collective_range;	/* 0.0 to 1.0. */
        double engine_power;            /* In kg * m / cycle^2. */
	int engine_can_pitch;		/* 1 if engines can pitch forwards/up. */
	char *engine_inside_sndpath;
	void *engine_inside_sndplay;
	char *engine_outside_sndpath;
	void *engine_outside_sndplay;

	/* Elevator trim, from -1.0 to 1.0. */
	double elevator_trim;

	/* Service ceiling, in meters. */
	double service_ceiling;

	/* Shadow visual model. */
	sar_visual_model_struct *visual_model_shadow;

	/* Distance from object's center to touchable ground in
	 * meters (usually negative).
	 */
	double center_to_ground_height;


	/* Rotors. */
	sar_obj_rotor_struct **rotor;
	int total_rotors;
	int rotor_wash_tex_num;	/* Texture number on scene struct. */

	/* Landing gears. */
	sar_obj_landing_gear_struct **landing_gear;
	int total_landing_gears;
	int landed;	/* Indicates object is on ground if true. */
	int on_water;	/* If landed is true, and this is true, then
			 * aircraft is on water.
			 */
	double gturn_radius;	/* Distance from farthest non-turning gear to
				 * turning gear in meters, can be negative.
				 * If 0 then it implies it cannot turn.
				 */
	/* Ground turning velocity optimul and maximum in meters per
	 * cycle.
	 */
	double gturn_vel_opt, gturn_vel_max;

	/* Rescue door. */
	sar_obj_door_struct rescue_door;

	/* Air brakes and wheel brakes:
	 * -1   non-existant
	 * 0    off
	 * 1    on
	 * 2    locked (parking brakes on)
	 */
	int	air_brakes_state,
		wheel_brakes_state;

	/* Air worthy state, one of SAR_AIR_WORTHY_*. */
	int air_worthy_state;

	/* Rescue hoist. */
	sar_obj_hoist_struct hoist;

	/* Intercept way points. */
	sar_obj_intercept_struct **intercept;
	int total_intercepts, cur_intercept;

} sar_object_aircraft_struct;

/*
 *	SAR ground elevation object sub structure:
 *
 *      This structure can be pointed to from member `data' in the
 *      sar_object_struct, determined by member `type'.
 */
typedef struct {

	/* Elevation from (above) local MSL in meters. If MSL was
	 * set to 10 meters and this elevation was set to 3 meters then
	 * the resulting elevation would be 13 meters to any object within
	 * contact of the ground object.
	 *
	 * Note that the ground object itself completely ignores the
	 * scene structure defined ground_elevation_msl value.
	 */
	double elevation;

	/* Translation of heightfield from center of object. */
	double x_trans, y_trans, z_trans;

	/* Heightfield data (only used if z_point_value is not NULL). */
	double	x_len,		/* grid_points_x * grid_x_spacing */
		y_len;		/* grid_points_y * grid_y_spacing */
	int	grid_points_x,	/* Number of grid points. */
		grid_points_y,
		grid_points_total;	/* grid_points_x * grid_points_y */
	double	grid_x_spacing,	/* Size of each grid in meters. */
		grid_y_spacing,
		grid_z_spacing;
	double	*z_point_value;	/* Heightfield z height map, each point
				 * value came from (image_data_pixel) /
				 * 0xff * grid_z_spacing.
				 */

        /* Value records for this ground object's inversed heading for the
	 * trig functions, this is used to speed up rotations of heading
         * for checking an object over this ground object's heightfield
	 * surface. It also limits ground objects to only have its
	 * heading rotated.
         *
         * Ie cos_heading = cos(-heading) and sin_heading = sin(-heading)
         */
        double cos_heading, sin_heading;

} sar_object_ground_struct;


/*
 *	Unit structures used in sar_object_smoke_struct.
 */
typedef struct {

	/* Position in world coordinates. */
	sar_position_struct pos;

	/* Current size in meters. */
	double radius;

	/* Current visibility of smoke unit, if it is 0.0 then it will
	 * not be drawn at all and is marked 'available for use'.
	 */
	double visibility;

} sar_object_smoke_unit_struct;
/*
 *	SAR smoke trails sub structure:
 */
typedef struct {

	/* Spawn offset position, this offset is applied to the actual
	 * location of the smoke trail object.
	 */
	sar_position_struct respawn_offset;

	/* All radius values in meters. */
	double	radius_start,
		radius_max,
		radius_rate;		/* Growth rate in meters per second. */

	/* If this is true then when a smoke unit has reached its 
	 * maximum size its visiblity will be reset to 0.0, marking it
	 * `available for use'.
	 */
	int hide_at_max;

	/* If this is true then this object will be marked for deletion
	 * (its life span set to 1) when all of its smoke units are
	 * no longer visible.
	 */
	int delete_when_no_units;

	/* Note that if respawn_int is 0 then no respawning will take
	 * place. This is often used to stop respawning but keep the
	 * already visible smoke units around.
	 */
	time_t respawn_int;	/* Respawn smoke unit interval in ms. */
	time_t respawn_next;	/* Next time to respawn smoke unit. */

        /* Texture number on scene structure. */
        int tex_num;

        /* Referance object that this smoke trail is to follow (can be -1
	 * for none/do not follow).
         */
        int ref_object;

	/* Each smoke unit forming this smoke trail. */
	sar_object_smoke_unit_struct *unit;
	int total_units;

} sar_object_smoke_struct;

/*
 *	SAR explosion sub structure:
 *
 *	Note this is also used for splashes.
 */
typedef struct {

	/* Spherical size of explosion in meters, this is also used to
	 * calculate the displayed explosion billboard.
	 */
	double radius;

	/* Explosion color emission, one of SAR_EXPLOSION_COLOR_EMISSION_*. */
#define SAR_EXPLOSION_COLOR_EMISSION_NONE	0	/* Ie splashes, interacts with light. */
#define SAR_EXPLOSION_COLOR_EMISSION_IS_LIGHT	1	/* Does not interact with light. */	
#define SAR_EXPLOSION_COLOR_EMISSION_EMIT_LIGHT	2	/* Same as IS_LIGHT and gives off light too. */
	int color_emission;

	/* Center gravity offset, one of SAR_EXPLOSION_GRAVITY_*.
	 * All gravity values are `upright' oriented.
	 */
#define SAR_EXPLOSION_GRAVITY_CENTER	0
#define SAR_EXPLOSION_GRAVITY_BASE	1
	int gravity_offset;

	time_t frame_inc_int;   /* Frame increment interval (in ms). */
	time_t next_frame_inc;	/* Time to increment next frame (in ms). */

	int cur_frame;          /* Current frame. */
	int frame_repeats;	/* Number of times animation has cycled. */

	/* Number of frame repeats, 0 or less to repeat forever (or
	 * when life span has exceeded.
	 */
	int total_frame_repeats;

	/* Texture number on scene structure. */
	int tex_num;

	/* Referance object that this explosion is to follow (can be -1
	 * for none/do not follow).
	 */
	int ref_object;

} sar_object_explosion_struct;

/* 
 *      SAR fire sub structure:
 */
typedef struct {

        /* Cylendrical size of fire in meters. This is also used to 
	 * calculate the fire billboard. Center is at base of fire.
	 */
        double radius, height;

        time_t frame_inc_int;   /* Frame increment interval (in ms). */
        time_t next_frame_inc;  /* Time to increment next frame (in ms). */

        int cur_frame;          /* Current frame. */
        int frame_repeats;      /* Number of times animation has cycled. */

        /* Number of frame repeats, 0 or less to repeat forever (or
         * when life span has exceeded.
         */
        int total_frame_repeats;

        /* Texture number on scene structure. */
        int tex_num;

	/* Referance object that this fire is to follow (can be -1 for
	 * none/do not follow).
	 */
        int ref_object; 

} sar_object_fire_struct;


/*
 *	Chemical spray object:
 *
 *	A single puff of chemical spray (ie water, fire-retardant, etc).
 */
typedef struct {

#define SAR_CHEMICAL_WATER		0
#define SAR_CHEMICAL_FIRE_RETARDANT	1

	int chemical_type;	/* One of SAR_CHEMICAL_*. */

	int owner;		/* Object that created this spray or -1 
				 * for none.
				 */

        /* Texture number on scene structure. */
	int tex_num;

} sar_object_chemical_spray_struct;

/*
 *	SAR aircraft fuel tank droped from an aircraft:
 *
 *	Note that this is not a fuel tank that is currently on an 
 *	aircraft, do not confuse this with the sar_external_fueltank_struct
 *	structure.
 */
typedef struct {

#define SAR_FUELTANK_FLAG_ON_GROUND	(1 << 1)

	sar_obj_flags_t flags;

        /* Current speed in meters per cycle. */
        double speed;

	/* Maximum negative vertical velocity in meters per cycle. */
	double vel_z_max;

        /* Current velocity vector representing velocity in object
         * relative vector direction, in meters per cycle.
         */
        sar_position_struct vel;

	/* Index of object of which this fuel tank fell off of
	 * (can be -1 for unknown).
	 */
	int ref_object;

	/* Mass, all in kg. */
	double	dry_mass,
		fuel, fuel_max;

} sar_object_fueltank_struct;


/*
 *      SAR runway sub structure:
 */
typedef struct {

#define SAR_RUNWAY_FLAG_NORTH_LABEL	(1 << 1)
#define SAR_RUNWAY_FLAG_SOUTH_LABEL	(1 << 2)
#define SAR_RUNWAY_FLAG_NORTH_DISPTHRES	(1 << 3)
#define SAR_RUNWAY_FLAG_SOUTH_DISPTHRES	(1 << 4)
#define SAR_RUNWAY_FLAG_EDGE_LIGHTING	(1 << 5)
#define SAR_RUNWAY_FLAG_NORTH_LIGHTING	(1 << 6)
#define SAR_RUNWAY_FLAG_SOUTH_LIGHTING	(1 << 7)

	sar_obj_flags_t flags;

	/* Size in meters. */
	double length, width;

	/* Runway end labels. */
	char *north_label, *south_label;

	/* Dash spacing in meters. */
	double	dash_on_spacing,
		dash_off_spacing;

	/* Lighting edge spacing in meters. */
	double light_spacing;

	/* Threshold displacements, from end to threshold in meters. */
	double	north_threshold_displacement,
		south_threshold_displacement;

        /* Texture number on scene structure. */
        int tex_num;

} sar_object_runway_struct;


/*
 *	SAR helipad sub structure:
 */
typedef struct {

#define SAR_HELIPAD_FLAG_LABEL			(1 << 1)
#define SAR_HELIPAD_FLAG_EDGE_LIGHTING		(1 << 2)
#define SAR_HELIPAD_FLAG_FUEL			(1 << 3)
#define SAR_HELIPAD_FLAG_REPAIR			(1 << 4)
#define SAR_HELIPAD_FLAG_DROPOFF		(1 << 5)	/* Drop off passengers. */
#define SAR_HELIPAD_FLAG_REF_OBJECT		(1 << 6)	/* Has ref object defined. */
#define SAR_HELIPAD_FLAG_FOLLOW_REF_OBJECT	(1 << 7)
#define SAR_HELIPAD_FLAG_RESTART_POINT		(1 << 8)	/* Player restarting point. */

        sar_obj_flags_t flags;

	/* Helipad style code, one of SAR_HELIPAD_STYLE_*. */
#define SAR_HELIPAD_STYLE_GROUND_PAVED	0	/* Default. */
#define SAR_HELIPAD_STYLE_GROUND_BARE	1	/* Unpaved. */
#define SAR_HELIPAD_STYLE_BUILDING	2	/* Ie roof top. */
#define SAR_HELIPAD_STYLE_VEHICLE	3	/* Ie helipad on a ship. */

	int style;

        /* Size in meters, specifying the landable area. */
        double length, width;

	/* Downward recession through helipad's landable ground, can be
	 * 0.0 or any positive value in meters. A value of 2.0 means 2
	 * meters down.
	 */
	double recession;

        /* Labels. */
        char *label;

        /* Lighting edge spacing in meters. */
        double light_spacing;

        /* Texture number on scene structure, specifying the texture
	 * for the helipad's main landable area.
	 */
        int tex_num;

	/* If SAR_HELIPAD_FLAG_REF_OBJECT is set then these members
	 * have affect. Note that ref_offset is applied first then
	 * ref_dir.
	 */
	int ref_object;		/* Object that this helipad `follows'. */
	/* These members are used if SAR_HELIPAD_FLAG_FOLLOW_REF_OBJECT
	 * is set. Note that ref_dir is applied first then ref_offset.
	 */
	sar_position_struct	ref_offset;	/* Rel to ref object. */
	sar_direction_struct	ref_dir;	/* Rel to ref object. */

} sar_object_helipad_struct;

/*
 *	SAR human structure:
 */
typedef struct {

#define SAR_HUMAN_FLAG_NEED_RESCUE	(1 << 1)
#define SAR_HUMAN_FLAG_SIT		(1 << 2)	/* Base at tush. */
#define SAR_HUMAN_FLAG_SIT_DOWN		(1 << 3)	/* Base at tush, feet out. */
#define SAR_HUMAN_FLAG_SIT_UP		(1 << 4)	/* Base at feet. */
#define SAR_HUMAN_FLAG_LYING		(1 << 5)
#define SAR_HUMAN_FLAG_ALERT		(1 << 6)	/* Is awake. */
#define SAR_HUMAN_FLAG_AWARE		(1 << 7)	/* Knows of surroundings. */
#define SAR_HUMAN_FLAG_IN_WATER		(1 << 8)
#define SAR_HUMAN_FLAG_ON_STREATCHER	(1 << 9)
#define SAR_HUMAN_FLAG_RUN		(1 << 10)	/* Animate as running. */
#define SAR_HUMAN_FLAG_RUN_TOWARDS	(1 << 11)	/* Does not imply actually running. */
#define SAR_HUMAN_FLAG_RUN_AWAY		(1 << 12)	/* Does not imply actually running. */
#define SAR_HUMAN_FLAG_PUSHING		(1 << 13)
#define SAR_HUMAN_FLAG_GRIPPED		(1 << 14)	/* In rescue basket for example. */
#define SAR_HUMAN_FLAG_DIVER_CATCHER	(1 << 15)

	sar_obj_flags_t flags;

	double mass;	/* In kg. */

	double height;	/* In meters. */

	sar_grad_anim_t anim_pos, anim_rate;

	/* Colors, see definations for SAR_HUMAN_COLOR_* and
	 * SAR_HUMAN_COLORS_MAX for index positions and maximum colors
	 * (respectivly).
	 */
	sar_color_struct color[SAR_HUMAN_COLORS_MAX];

	/* Water ripples texture number on scene structure. */
	int water_ripple_tex_num;

	/* Referance to object running towards or away from, the following
	 * values have special meaning:
	 *
	 *	-1	No intercepting
	 *	-2	Intercept player
	 *
	 * Works when flags SAR_HUMAN_FLAG_RUN_TOWARDS xor 
	 * SAR_HUMAN_FLAG_RUN_AWAY is set and intercepting_object is 
	 * valid and not the human object itself.
	 */
	int intercepting_object;

	/* Number of assisting humans following this human (0 for none).
	 * these are not other objects but rather groups of humans drawn
	 * with this human object to make it look like multiple
	 * humans.
	 */
	int assisting_humans;
        sar_color_struct assisting_human_color[SAR_HUMAN_COLORS_MAX];

	/* Messages. */
	char *mesg_enter;	/* Entering into aircraft or vehicle. */

} sar_object_human_struct;

/*
 *	SAR premodeled object:
 */
typedef struct {

#define SAR_OBJ_PREMODELED_BUILDING		1
#define SAR_OBJ_PREMODELED_CONTROL_TOWER	2
#define SAR_OBJ_PREMODELED_HANGAR		3

	int type;	/* One of SAR_OBJ_PREMODELED_*. */

	/* Values, depending on type, not all may be applicateable. */
	double width, length, height;

	/* Animation. */
	sar_grad_anim_t anim_pos, anim_rate;

        /* Texture number on scene structure. */
#define SAR_OBJ_PREMODEL_MAX_TEXTURES	10
        int tex_num[SAR_OBJ_PREMODEL_MAX_TEXTURES];

} sar_object_premodeled_struct;

/*
 *	SAR core object structure:
 */
typedef struct {

	int type;	/* Must be first member, one of SAR_OBJ_TYPE_*. */

#define SAR_OBJ_FLAG_NO_DEPTH_TEST	(1 << 1)
#define SAR_OBJ_FLAG_HIDE_DAY_MODEL	(1 << 2)	/* Hide day model when not day. */
#define SAR_OBJ_FLAG_HIDE_DAWN_MODEL	(1 << 3)	/* Hide dawn model when not dawn. */
#define SAR_OBJ_FLAG_HIDE_DUSK_MODEL	(1 << 4)	/* Hide dusk model when not dusk. */
#define SAR_OBJ_FLAG_HIDE_NIGHT_MODEL	(1 << 5)	/* Hide night model when not night. */

/* These two only work if SAR_OBJ_FLAG_HIDE_NIGHT_MODEL is set. */
#define SAR_OBJ_FLAG_NIGHT_MODEL_AT_DAWN	(1 << 6)	/* Show night modem at dawn. */
#define SAR_OBJ_FLAG_NIGHT_MODEL_AT_DUSK	(1 << 7)	/* Show night modem at dusk. */

#define SAR_OBJ_FLAG_SHADE_MODEL_SMOOTH	(1 << 8)	/* Use smooth shading. */

#define SAR_OBJ_FLAG_FAR_MODEL_DAY_ONLY	(1 << 9)	/* Display far model during
							 * time only.
							 */
#define SAR_OBJ_FLAG_POLYGON_OFFSET	(1 << 10)	/* Enable polygon offset. */
#define SAR_OBJ_FLAG_POLYGON_OFFSET_REVERSE	(1 << 11)	/* Same as SAR_OBJ_FLAG_POLYGON_OFFSET
								 * cept offsets in
								 * the other direction.
								 */
#define SAR_OBJ_FLAG_POLYGON_OFFSET_WRITE_DEPTH	(1 << 12)	/* Write depth if
								 * polygon offsetting.
								 */

	sar_obj_flags_t flags;

	/* Name of object (can be NULL). */
	char *name;

        /* Current position and direction. */
        sar_position_struct pos;
        sar_direction_struct dir;

	/* Visible range of this object in meters. */
	double range;

	/* Visible range for displaying the far model in meters,
	 * if and only if the visual_model_far is not NULL and
	 * the range is beyond this value is the far model
	 * displayed instead of the standard model.
	 */
	double range_far;


	/* Distance from sea level to touchable ground under
	 * object, in meters. (Not applicateable to object type
	 * SAR_OBJ_TYPE_GROUND).
	 */
	double ground_elevation_msl;


	/* Contact and crash bounds. */
	sar_contact_bounds_struct *contact_bounds;


	/* Time stamp of when this object was created in milliseconds
	 * and in systime seconds (respectivly). The value in milliseconds 
	 * may not be accurate when timmers are reset.
	 */
	time_t	birth_time_ms,
		birth_time_sec;

	/* When this object `dies' (0 if lives forever). */
	time_t life_span;

	/* Hit points. */
	double hit_points, hit_points_max;

	/* Visual models for day, dusk, night. */
	sar_visual_model_struct *visual_model;		/* Day. */
	sar_visual_model_struct *visual_model_far;	/* Far. */
	sar_visual_model_struct *visual_model_dawn;	/* Dawn. */
	sar_visual_model_struct *visual_model_dusk;	/* Dusk. */
	sar_visual_model_struct *visual_model_night;	/* Night. */

	/* Lights. */
	sar_light_struct **light;
	int total_lights;

	/* Pointer to additional data specific to the object type,
	 * the structure's type is specified by this structure's member
	 * type.
	 */
	void *data;

} sar_object_struct;



/*
 *	Scene ground base structure, contains visual models and tiling
 *	values for displaying of the ground base.
 */
typedef struct {

	/* Note, ground base moves with camera in modulous increments
	 * of the tiled width and height.
	 */

	/* Tiling size in meters. */
        int tile_width, tile_height;

	/* Tiling limited to this range in meters from origin of
	 * tiling.
	 */
	double close_range;


	/* Simple (farther away) solid color base plane and visual
	 * model.
	 */
	sar_color_struct color;
        sar_visual_model_struct *visual_model_simple;

	/* Close up texture tiled base plane visual model. */
	sar_visual_model_struct *visual_model_close;

} sar_scene_base_struct;


/*
 *	Scene horizon structure:
 */
typedef struct {

	/* Records the last time of day since midnight from
	 * the scene structure in 30 minute units as a whole number for
	 * horizon texture regeneration.
	 */
	int last_tod;

	/* Gradient textures, 0 is most highlighted and
	 * total_textures - 1 is darkest.
	 */
	v3d_texture_ref_struct **texture;
	int total_textures;

} sar_scene_horizon_struct;


/*
 *	Scene cloud layer structure:
 *
 *	Flat tiled layer of clouds as tiled textured quads.
 */
typedef struct {

        /* Note, cloud layer moves with camera in modulous           
         * of the tiled width and height on the xy plane.
         */

        /* Tiling size in meters. */
	int tile_width, tile_height;

        /* Tiling limited to this range in meters from origin of
         * tiling.
         */
        double range;

	/* Name of texture on scene structure. */
	char *tex_name;

	sar_visual_model_struct *visual_model;

	/* Height from MSL in meters. */
	double z;

} sar_cloud_layer_struct;


/*
 *      Scene cloud `billboard' structure:
 *
 *      `Billboard' cloud objects placed across the scene.
 */
typedef struct {

        /* Note, cloud billboard moves with camera in modulous
         * of the tiled width and height on the xy plane.
         */

        /* Tiling size in meters. */
        int tile_width, tile_height;

        /* Name of texture on scene structure. */
        char *tex_name;
	/* Matched texture on scene structure. */
	int tex_num;

	/* Position of billboard object relative to tiled center. The
	 * z value is relative to the ground ofcourse (in meters).
	 */
	double x, y, z;

	/* Size of cloud object in meters (note height not in feet). */
	double width, height;


	/* Lightning min interval (in ms), 0 for no lightning. */
	time_t lightning_min_int;

	/* Lightning on interval (in ms), how long does the lightning
	 * stay on.
	 */
	time_t lightning_on_int;

	/* When this value is greater than 0 it implies lightning
	 * is on and will stay on for the remaining duration (in ms).
	 */
	time_t lightning_on_left;

	/* Last time lightning was on in ms. */
	time_t lightning_last;

	/* Lightning points, in meters relative to cloud bb center. */
#define SAR_LIGHTNING_POINTS_MAX	6
	sar_position_struct lightning_point[SAR_LIGHTNING_POINTS_MAX];

} sar_cloud_bb_struct;


/*
 *	SAR scene structure:
 */
typedef struct {

	/* Time of day (in seconds) since midnight. */
	double tod;

	/* Time of day code, one of SARSceneTODCode*. */
#define SAR_SCENE_TOD_DAY		1
#define SAR_SCENE_TOD_DAWN		2
#define SAR_SCENE_TOD_NIGHT		3
#define SAR_SCENE_TOD_DUSK		4
	int tod_code;

        /* Base flags, determines the attributes of the landable ground
	 * base.
         */
#define SAR_SCENE_BASE_FLAG_IS_WATER	(1 << 1)	/* Opposite of land. */
	sar_obj_flags_t base_flags;

	/* CANT angle in radians, this value is only applied on readouts
	 * added to the internal angle value in question.
	 */
	double cant_angle;

	/* MSL elevation in meters. */
	double msl_elevation;

	/* Scene on planet values. */
	sar_dms_t	dms_x_offset,	/* -140.0 to 140.0. */
			dms_y_offset;	/* -90.0 to 90.0. */
	double planet_radius;		/* In meters. */

	/* List of visual models (structures containing GL list
	 * referances) each object or resource that has a visual
	 * model (has a GL list basically) will point to one of
	 * these structures. Only these structures may be deallocated.
	 */
	sar_visual_model_struct **visual_model;
	int total_visual_models;

	/* Sky and horizon gradient colors. */
	sar_color_struct	sky_nominal_color,
				sky_brighten_color,
				sky_darken_color;

	sar_color_struct	star_color,
				moon_color;

	/* Atmosphere (when enabled). */
	double	atmosphere_dist_coeff,	/* Coeff muliplied by max visiblity, */
		atmosphere_density_coeff;

	/* Referance to player object. */
	int player_obj_num;
	sar_object_struct *player_obj_ptr;
	int player_has_crashed;	/* True if player object crashed. */

	/* Pointers to SAR_OBJ_TYPE_GROUND objects, you may free
	 * the pointer array but not the pointer to structures!!
	 */
	sar_object_struct **ground_object;
	int total_ground_objects;

	/* Pointers to SAR_OBJ_TYPE_HUMAN objects that *need rescue*.
	 * You may free the pointer array but not the pointers to structures!!
	 */
	sar_object_struct **human_need_rescue_object;
        int total_human_need_rescue_objects;


	/* Camera referance. */
#define SAR_CAMERA_REF_COCKPIT	0
#define SAR_CAMERA_REF_SPOT	1
#define SAR_CAMERA_REF_TOWER	2
#define SAR_CAMERA_REF_MAP	3
#define SAR_CAMERA_REF_HOIST	4	/* Rescue hoist view. */

	int camera_ref;

	/* Camera field of view, in radians along Z axis (or
	 * you can say `about the X axis').
	 */
	double camera_fovz;

	/* Direction camera is looking while set to
	 * SAR_CAMERA_REF_COCKPIT. The bank member is ignored.
	 */
	sar_direction_struct camera_cockpit_dir;

	/* Direction from object to camera while set to
	 * SAR_CAMERA_REF_SPOT. The bank member is ignored.
	 */
	sar_direction_struct camera_spot_dir;
	double camera_spot_dist;

	/* Distance for SAR_CAMERA_REF_HOIST. */
	sar_direction_struct camera_hoist_dir;
	double camera_hoist_dist;

	/* Position of camera when set to SAR_CAMERA_REF_MAP. */
	sar_position_struct camera_map_pos;

	/* Position of camera when set to SAR_CAMERA_REF_TOWER. */
	sar_position_struct camera_tower_pos;

	/* Target object that camera is tracking (can be -1). */
	int camera_target;

	/* Stack of camera rotation matrixes (set in sardraw.c) to
	 * rotate any vertex relative to the camera's rotation, each
	 * a 3 * 3 matrix.
	 * Member camera_rotmatrix_count indicates the actual number
	 * of matrixes set, which is equal or less than 
	 * SAR_CAMERA_ROTMATRIX_MAX.
	 */
#define SAR_CAMERA_ROTMATRIX_MAX	5
	double camera_rotmatrix[SAR_CAMERA_ROTMATRIX_MAX][3 * 3];
	int camera_rotmatrix_count;

	/* Ear position, this is updated when the scene is drawn and
	 * the camera set.
	 */
	sar_position_struct ear_pos;


	/* Global primary light position, this specifies the offset that
	 * is to be applied to the camera.
	 */
	sar_position_struct light_pos;

	/* Global primary light intensity/color. */
	sar_color_struct light_color;


	/* Base ground. */
	sar_scene_base_struct base;

	/* Horizon. */
	sar_scene_horizon_struct horizon;

	/* Cloud layers. */
	sar_cloud_layer_struct **cloud_layer;
	int total_cloud_layers;

	/* Cloud `billboard' objects. */
	sar_cloud_bb_struct **cloud_bb;
	int total_cloud_bbs;

	/* Loaded textures list. */
	v3d_texture_ref_struct **texture_ref;
	int total_texture_refs;

	/* Default sounds object paths. */
	char	*land_wheel_skid_sndpath,
		*land_ski_skid_sndpath,		/* Skis skidding. */
		*land_ski_sndpath,		/* Skis just touching down. */
		*land_belly_sndpath,
		*crash_obstruction_sndpath,
		*crash_ground_sndpath,
		*splash_aircraft_sndpath,
		*splash_human_sndpath;

	/* Messages. */
	char **message;
	int total_messages;

	/* Display message untill this time runs out (in milliseconds), a
	 * value of 0 implies message not shown.
	 */
	time_t message_display_untill;

	/* Sticky banner message, for displaying crash reason or mission
	 * failed reason whenever it is not NULL.
	 */
	char **sticky_banner_message;
	int total_sticky_banner_messages;

	/* Name of object and/or description of what the camera is
	 * showing.
	 */
	char *camera_ref_title;
	time_t camera_ref_title_display_untill;

	/* Flight dynamics model realm structure, used by the SFM
	 * library as global referance while simulating all flight 
	 * dynamics models.
	 */
	SFMRealmStruct *realm;

} sar_scene_struct;



/*
 *	Code segment to get typed pointer to object's extended
 *	data.
 */
/*
	sar_object_struct *obj_ptr;
        sar_object_aircraft_struct *obj_aircraft_ptr = NULL;
	sar_object_ground_struct *obj_ground_ptr = NULL;
	sar_object_runway_struct *obj_runway_ptr = NULL;
        sar_object_helipad_struct *obj_helipad_ptr = NULL;
	sar_object_smoke_struct *obj_smoke_ptr = NULL;
	sar_object_explosion_struct *obj_explosion_ptr = NULL;
	sar_object_human_struct *obj_human_ptr = NULL;
 */
#define SAR_GET_EXT_DATA_PTR	\
/* obj_ptr must not be NULL. */ \
switch(obj_ptr->type) \
{ \
 case SAR_OBJ_TYPE_AIRCRAFT: \
  obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data; \
  break; \
 case SAR_OBJ_TYPE_GROUND: \
  obj_ground_ptr = (sar_object_ground_struct *)obj_ptr->data; \
  break; \
 case SAR_OBJ_TYPE_RUNWAY: \
  obj_runway_ptr = (sar_object_runway_struct *)obj_ptr->data; \
  break; \
 case SAR_OBJ_TYPE_HELIPAD: \
  obj_helipad_ptr = (sar_object_helipad_struct *)obj_ptr->data; \
  break; \
 case SAR_OBJ_TYPE_SMOKE: \
  obj_smoke_ptr = (sar_object_smoke_struct *)obj_ptr->data; \
  break; \
 case SAR_OBJ_TYPE_EXPLOSION: \
  obj_explosion_ptr = (sar_object_explosion_struct *)obj_ptr->data; \
  break; \
 case SAR_OBJ_TYPE_HUMAN: \
  obj_human_ptr = (sar_object_human_struct *)obj_ptr->data; \
  break; \
} \



/* In objutils.c */
extern void SARSetDirectionRadians(
	sar_direction_struct *dir,
	double heading, double pitch, double bank
);

extern int SARObjIsAllocated(
        sar_object_struct **ptr, int total,
        int n
);
extern sar_object_struct *SARObjGetPtr(
        sar_object_struct **ptr, int total,
        int n
);
extern int SARGetObjectNumberFromPointer(
        sar_scene_struct *scene,
        sar_object_struct **ptr, int total,
        sar_object_struct *obj_ptr
);
extern sar_object_struct *SARObjMatchPointerByName(
        sar_scene_struct *scene,
        sar_object_struct **ptr, int total,
        const char *name, int *obj_num
);

extern int SARIsTextureAllocated(
        sar_scene_struct *scene,
        int n
);
extern v3d_texture_ref_struct *SARGetTextureRefByName(
	sar_scene_struct *scene, const char *name
);
extern int SARGetTextureRefNumberByName(
        sar_scene_struct *scene, const char *name
);

extern sar_obj_rotor_struct *SARObjGetRotorPtr(
        sar_object_struct *obj_ptr, int n, int *total
);
extern sar_obj_landing_gear_struct *SARObjGetLandingGearPtr(
        sar_object_struct *obj_ptr, int n, int *total   
);
extern sar_obj_door_struct *SARObjGetDoorPtr(
        sar_object_struct *obj_ptr, int n, int *total 
);
extern sar_obj_hoist_struct *SARObjGetHoistPtr(
        sar_object_struct *obj_ptr, int n, int *total
);
extern int SARObjGetOnBoardPtr(
        sar_object_struct *obj_ptr,
        int **crew, int **passengers, int **passengers_max,
        double **passengers_mass,
        int **passengers_leave_pending, int **passengers_drop_pending
);
extern sar_external_fueltank_struct *SARObjGetFuelTankPtr(
        sar_object_struct *obj_ptr, int n, int *total
);


extern sar_visual_model_struct *SARVisualModelNew(
        sar_scene_struct *scene,
        const char *filename, const char *name
);
extern void *SARVisualModelNewList(sar_visual_model_struct *vmodel);
extern int SARVisualModelGetRefCount(sar_visual_model_struct *vmodel);
extern void SARVisualModelRef(sar_visual_model_struct *vmodel);
extern void SARVisualModelUnref(
	sar_scene_struct *scene, sar_visual_model_struct *vmodel
);
extern void SARVisualModelCallList(sar_visual_model_struct *vmodel);
extern void SARVisualModelDeleteAll(sar_scene_struct *scene);

extern sar_cloud_layer_struct *SARCloudLayerCreate(
	sar_scene_struct *scene,
	int tile_width, int tile_height,
	double range,		/* Tiling range in meters. */
	double altitude,	/* Altitude in meters. */
	const char *tex_name
);          
extern void SARCloudLayerDestroy(
	sar_scene_struct *scene,
        sar_cloud_layer_struct *cloud_layer_ptr
);

extern sar_cloud_bb_struct *SARCloudBBCreate(
        sar_scene_struct *scene,
        int tile_width, int tile_height,
        double x, double y, double z,   /* In meters. */
        double width, double height,    /* In meters. */
        const char *tex_name
);
extern void SARCloudBBDestroy(sar_cloud_bb_struct *cloud_bb_ptr);

extern int SARObjAddToGroundList(
        sar_scene_struct *scene,
        sar_object_struct *obj_ptr
);
extern void SARObjRemoveFromGroundList(
        sar_scene_struct *scene,
        sar_object_struct *obj_ptr
);
extern int SARObjAddToHumanRescueList(   
        sar_scene_struct *scene,
        sar_object_struct *obj_ptr
);
extern void SARObjRemoveFromHumanRescueList(
        sar_scene_struct *scene,
        sar_object_struct *obj_ptr
);

extern int SARObjAddContactBoundsSpherical(
        sar_object_struct *obj_ptr,
        sar_obj_flags_t crash_flags, int crash_type,
        double contact_radius
);
extern int SARObjAddContactBoundsCylendrical(
        sar_object_struct *obj_ptr,
        sar_obj_flags_t crash_flags, int crash_type,
        double contact_radius,
        double contact_h_min, double contact_h_max
);
extern int SARObjAddContactBoundsRectangular(
        sar_object_struct *obj_ptr,
        sar_obj_flags_t crash_flags, int crash_type,
        double contact_x_min, double contact_x_max,
        double contact_y_min, double contact_y_max,
        double contact_z_min, double contact_z_max
);

extern int SARObjCreateIntercept(
        sar_scene_struct *scene,
        sar_obj_intercept_struct ***ptr, int *total,
        sar_obj_flags_t flags,
        double x, double y, double z,
        double radius,
        double urgency,
        const char *name
);
extern int SARObjCreateLandingGear(
        sar_scene_struct *scene,
        sar_obj_landing_gear_struct ***ptr, int *total
);
extern int SARObjCreateExternalFuelTanks(
        sar_scene_struct *scene,
        sar_external_fueltank_struct ***ptr, int *total
);
extern int SARObjCreateRotor(  
        sar_scene_struct *scene,
        sar_obj_rotor_struct ***ptr, int *total
);
extern int SARObjCreate(
	sar_scene_struct *scene,
        sar_object_struct ***ptr, int *total,
        int type
);

extern void SARObjDeleteIntercepts(
	sar_scene_struct *scene,
        sar_obj_intercept_struct ***ptr, int *total
);
extern void SARObjDeleteLights(
	sar_scene_struct *scene,
	sar_light_struct ***ptr, int *total
);
extern void SARObjDeleteLandingGears(
        sar_scene_struct *scene,
        sar_obj_landing_gear_struct ***ptr, int *total
);
extern void SARObjDeleteExternalFuelTanks(
        sar_scene_struct *scene,
        sar_external_fueltank_struct ***ptr, int *total
);
extern void SARObjDeleteRotors(
        sar_scene_struct *scene,
        sar_obj_rotor_struct ***ptr, int *total
);
extern void SARObjDeleteDoors(
        sar_scene_struct *scene,
        sar_obj_door_struct ***ptr, int *total
);
extern void SARObjDelete(
	void *core_ptr,
        sar_object_struct ***ptr, int *total,
        int n
);

extern void SARObjGenerateTilePlane(
        double min, double max, /* In meters. */
        double tile_width, double tile_height
);


#endif	/* OBJ_H */
