#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>

#include "../include/string.h"

#include "sfm.h"
#include "v3dtex.h"

#include "sarreality.h"
#include "obj.h"

#include "sar.h"


void SARObjCreatePremodeledControlTower(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        int obj_num, sar_object_struct *obj_ptr,
        sar_object_premodeled_struct *obj_premodel_ptr,
        int argc, char **argv
);
void SARObjCreatePremodeledBuilding(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        int obj_num, sar_object_struct *obj_ptr,
        sar_object_premodeled_struct *obj_premodel_ptr,
        int argc, char **argv
);
int SARObjCreatePremodeled(
	sar_core_struct *core_ptr, sar_scene_struct *scene,
	const char *type_name, int argc, char **argv
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))



/*
 *	Creates a control tower.
 *
 *	Inputs assumed valid.
 */
void SARObjCreatePremodeledControlTower(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        int obj_num, sar_object_struct *obj_ptr,
        sar_object_premodeled_struct *obj_premodel_ptr,
        int argc, char **argv
)
{
        int i, n;

        /* Format:
         *
         * <range> <length> <width> <height> <walls_texture> <roof_texture>
         */

        obj_premodel_ptr->type = SAR_OBJ_PREMODELED_CONTROL_TOWER;

        /* Range. */
        i = 0;
        obj_ptr->range = ((i < argc) ? atof(argv[i]) : 1000.0);
            
        /* Length. */
        i = 1;
        obj_premodel_ptr->length = ((i < argc) ? atof(argv[i]) : 15.0);

        /* Width. */
        i = 2;
        obj_premodel_ptr->width = ((i < argc) ? atof(argv[i]) : 15.0);

        /* Height (in feet). */
        i = 3;
        obj_premodel_ptr->height = SFMFeetToMeters(
            (i < argc) ? atof(argv[i]) : 80.0
        );
  
        /* Walls texture. */
        i = 4;
        if(i < argc)
        {
            n = 0;
            if(n < SAR_OBJ_PREMODEL_MAX_TEXTURES)
                obj_premodel_ptr->tex_num[n] = SARGetTextureRefNumberByName(
                    scene, argv[i]
                );
        }

        /* Roof texture. */
        i = 5;
        if(i < argc)
        {
            n = 1;
            if(n < SAR_OBJ_PREMODEL_MAX_TEXTURES)
                obj_premodel_ptr->tex_num[n] = SARGetTextureRefNumberByName(
                    scene, argv[i]
                );
        }

        /* Set contact bounds. */
	SARObjAddContactBoundsRectangular(
	    obj_ptr,
	    SAR_CRASH_FLAG_CRASH_CAUSE | SAR_CRASH_FLAG_SUPPORT_SURFACE,
	    SAR_GRASH_TYPE_BUILDING,
	    -(obj_premodel_ptr->length / 2), (obj_premodel_ptr->length / 2),
	    -(obj_premodel_ptr->width / 2), (obj_premodel_ptr->width / 2),
	    0.0, obj_premodel_ptr->height
	);

	return;
}


/*
 *	Creates a building.
 *
 *	Inputs assumed valid.
 */
void SARObjCreatePremodeledBuilding(
	sar_core_struct *core_ptr, sar_scene_struct *scene,
	int obj_num, sar_object_struct *obj_ptr,
	sar_object_premodeled_struct *obj_premodel_ptr,
	int argc, char **argv
)
{
	int i, n;

	/* Format:
	 *
	 * <range> <length> <width> <height> <walls_texture>
	 * <walls_texture_night> <roof_texture>
	 */

	obj_premodel_ptr->type = SAR_OBJ_PREMODELED_BUILDING;

        /* Range. */
        i = 0;
	obj_ptr->range = ((i < argc) ? atof(argv[i]) : 1000.0);

	/* Length. */
	i = 1;
	obj_premodel_ptr->length = ((i < argc) ? atof(argv[i]) : 50.0);

	/* Width. */
	i = 2;
	obj_premodel_ptr->width = ((i < argc) ? atof(argv[i]) : 20.0);

	/* Height (in feet). */
	i = 3;
	obj_premodel_ptr->height = SFMFeetToMeters(
	    (i < argc) ? atof(argv[i]) : 50.0
	);

	/* Walls texture. */
	i = 4; n = 0;
	if(i < argc)
	{
	    if(n < SAR_OBJ_PREMODEL_MAX_TEXTURES)
	        obj_premodel_ptr->tex_num[n] = SARGetTextureRefNumberByName(
		    scene, argv[i]
	        );
	}

        /* Walls night. */
        i = 5; n = 1;
        if(i < argc)
        {
            if(n < SAR_OBJ_PREMODEL_MAX_TEXTURES)
                obj_premodel_ptr->tex_num[n] = SARGetTextureRefNumberByName(
                    scene, argv[i]
                );
        }

        /* Roof texture. */
        i = 6; n = 2;
        if(i < argc)
        {
            if(n < SAR_OBJ_PREMODEL_MAX_TEXTURES)
                obj_premodel_ptr->tex_num[n] = SARGetTextureRefNumberByName(
                    scene, argv[i]
                );
        }

        /* Set contact bounds. */
        SARObjAddContactBoundsRectangular(
            obj_ptr,
            SAR_CRASH_FLAG_CRASH_CAUSE | SAR_CRASH_FLAG_SUPPORT_SURFACE,
            SAR_GRASH_TYPE_BUILDING,
            -(obj_premodel_ptr->length / 2), (obj_premodel_ptr->length / 2),
            -(obj_premodel_ptr->width / 2), (obj_premodel_ptr->width / 2),
            0.0, obj_premodel_ptr->height
        );

	return;
}

/*
 *	Front end for creating a permodeled object.
 */
int SARObjCreatePremodeled(
	sar_core_struct *core_ptr, sar_scene_struct *scene,
	const char *type_name, int argc, char **argv
)
{
	int obj_num = -1;
	sar_object_struct *obj_ptr;
	sar_object_premodeled_struct *obj_premodel_ptr;


	if((core_ptr == NULL) || (scene == NULL) || (type_name == NULL))
	    return(obj_num);

	/* Create new premodeled object. */
	obj_num = SARObjCreate(
	    scene,
	    &core_ptr->object, &core_ptr->total_objects,
	    SAR_OBJ_TYPE_PREMODELED
        );
	if(obj_num < 0)
	    return(obj_num);
	else
 	    obj_ptr = core_ptr->object[obj_num];

	/* Get pointer to premodeled object data substructure. */
	obj_premodel_ptr = (sar_object_premodeled_struct *)obj_ptr->data;


	/* Handle by type name. */
	/* Standard building. */
	if(!strcasecmp(type_name, "building"))
	{
	    SARObjCreatePremodeledBuilding(
		core_ptr, scene,
		obj_num, obj_ptr, obj_premodel_ptr,
		argc, argv
	    );
	}
	/* Control tower. */
	else if(!strcasecmp(type_name, "control_tower"))
        {
	    SARObjCreatePremodeledControlTower(
                core_ptr, scene,
                obj_num, obj_ptr, obj_premodel_ptr,
                argc, argv
            );
	}
	else
	{
	    fprintf(
		stderr,
		"Unsupported permodeled type name `%s'.\n",
		type_name
	    );
	}

	return(obj_num);
}
